// --------------------------------------------------------------------
// Dialogs
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeq.h"
#include "ipestyle.h"
#include "ipeprefs.h"
#include "dialogs.h"

#include <QtGui>

// --------------------------------------------------------------------

/*! \class DialogDocumentProperties
  \brief Dialog to edit document properties
*/

//! The argument \c data is only updated if the dialog is accepted.
DialogDocumentProperties::DialogDocumentProperties
(QWidget* parent, IpeDocument::SProperties &data, const IpeStyleSheet *sheet)
  : QDialog(parent), iData(data), iSheet(sheet)
{
  setupUi(this);
  connect(iButtonHelp, SIGNAL(clicked()), SLOT(ShowHelp()));
  connect(iButtonOk, SIGNAL(clicked()), SLOT(UpdateData()));
  iAuthorEditor->setText(QIpe(iData.iAuthor));
  iTitleEditor->setText(QIpe(iData.iTitle));
  iSubjectEditor->setText(QIpe(iData.iSubject));
  iKeywordsEditor->setText(QIpe(iData.iKeywords));
  iLatexPreambleEditor->setPlainText(QIpe(iData.iPreamble));
  iFullScreenBox->setChecked(iData.iFullScreen);
  iNumberPages->setChecked(iData.iNumberPages);
  iCropBox->setChecked(iData.iCropBox);
  iCreated->setText(QIpe(iData.iCreated));
  iModified->setText(QIpe(iData.iModified));
}

void DialogDocumentProperties::UpdateData()
{
  iData.iAuthor = IpeQ(iAuthorEditor->text());
  iData.iTitle = IpeQ(iTitleEditor->text());
  iData.iSubject = IpeQ(iSubjectEditor->text());
  iData.iKeywords = IpeQ(iKeywordsEditor->text());
  iData.iPreamble = IpeQ(iLatexPreambleEditor->toPlainText());
  iData.iFullScreen = iFullScreenBox->isChecked();
  iData.iNumberPages = iNumberPages->isChecked();
  iData.iCropBox = iCropBox->isChecked();
}

void DialogDocumentProperties::ShowHelp()
{
  QWhatsThis::enterWhatsThisMode();
}

// --------------------------------------------------------------------

/*! \class DialogDirPreferences
  \brief Dialog to display Ipe directory settings (read only)
*/

//! Create dialog. \c data is only updated if dialog is accepted.
DialogDirPreferences::DialogDirPreferences(QWidget* parent,
					   IpePreferences &data)
  : QDialog(parent), iData(data)
{
  setupUi(this);
  connect(iButtonHelp, SIGNAL(clicked()), SLOT(ShowHelp()));
  // copy data
  iPreferencesEditor->setText(iData.iPrefsFileName);
  iPreferencesEditor->setReadOnly(true);
  iLatexDirEditor->setText(iData.iLatexDir);
  iLatexDirEditor->setReadOnly(true);
  iPdfLatexEditor->setText(iData.iPdfLatex);
  iPdfLatexEditor->setReadOnly(true);
  iDocDirEditor->setText(iData.iDocDir);
  iDocDirEditor->setReadOnly(true);
  iFontmapEditor->setText(iData.iFontmap);
  iFontmapEditor->setReadOnly(true);
  iStdKeysEditor->setText(iData.iStdKeysFileName);
  iStdKeysEditor->setReadOnly(true);
  iKeysEditor->setText(iData.iKeysFileName);
  iKeysEditor->setReadOnly(true);
  iIpeletListBox->clear();
  iIpeletListBox->addItems(iData.iIpeletPath);
}

void DialogDirPreferences::ShowHelp()
{
  QWhatsThis::enterWhatsThisMode();
}

// --------------------------------------------------------------------

/*! \class DialogPreferences
  \brief Dialog to edit Ipe preferences
*/

//! Create dialog. \c data is only updated if dialog is accepted.
DialogPreferences::DialogPreferences(QWidget* parent, IpePreferences &data,
				     int gridSize)
  : QDialog(parent), iData(data), iGridSize(gridSize)
{
  setupUi(this);
  connect(iButtonHelp, SIGNAL(clicked()), SLOT(ShowHelp()));
  connect(iButtonOk, SIGNAL(clicked()), SLOT(UpdateData()));
  connect(iButtonDirectory, SIGNAL(clicked()), SLOT(ShowDirectories()));
  connect(iTextFontButton, SIGNAL(clicked()), SLOT(SetFont()));
  connect(iGridButton, SIGNAL(clicked()), SLOT(SetDefaultGrid()));
  connect(iStyleSheetBrowseButton, SIGNAL(clicked()), SLOT(SetStyleSheet()));
  connect(iClearStyleSheetButton, SIGNAL(clicked()), SLOT(ClearStyleSheet()));
  // copy data
  iCompress->setChecked(iData.iCompressLevel > 0);
  iSelectDistance->setValue(iData.iSelectDistance);
  iSnapDistance->setValue(iData.iSnapDistance);
  iWhitePaper->setChecked(iData.iWhitePaper);
  iAntiAlias->setChecked(iData.iAntiAlias);
  iDoubleBuffer->setChecked(iData.iDoubleBuffer);
  iGridVisible->setChecked(iData.iGridVisible);
  iMaximize->setChecked(iData.iMaximize);
  iBitmapSize->setValue(iData.iBitmapSize);
  iPropertiesShape->addItem(tr("Two columns"));
  iPropertiesShape->addItem(tr("One column"));
  iPropertiesShape->addItem(tr("One row"));
  iPropertiesShape->setCurrentIndex(iData.iPropertiesShape);

  iStyleSheet->setReadOnly(true);
  iStyleSheetPath = iData.iStyleSheet;
  if (!iStyleSheetPath.isNull())
    iStyleSheet->setText(iStyleSheetPath);
  // find all languages
  QDir dir(iData.iLangDir);
  dir.setFilter(QDir::Files);
  dir.setNameFilters(QStringList(QLatin1String("ipe-*.qm")));
  QStringList langs = dir.entryList();
  iLanguage->addItem(QLatin1String("English (US)"));
  iLanguage->setCurrentIndex(0);
  for (int j = 0; j < langs.count(); ++j) {
    QString fn = dir.filePath(langs[j]);
    QFileInfo finfo(fn);
    iLanguage->addItem(finfo.baseName().mid(4));
    if (finfo.baseName().mid(4) == iData.iLanguage)
      iLanguage->setCurrentIndex(iLanguage->count() - 1);
  }
  iExport->addItem(tr("No auto-export"));
  iExport->addItem(tr("XML -> EPS"));
  iExport->addItem(tr("XML -> PDF"));
  iExport->addItem(tr("XML -> EPS & PDF"));
  iExport->addItem(tr("EPS -> PDF"));
  iExport->addItem(tr("PDF -> EPS"));
  iExport->setCurrentIndex(iData.iAutoExport);
  iExportExists->setChecked(iData.iExportExist);
}

void DialogPreferences::ShowHelp()
{
  QWhatsThis::enterWhatsThisMode();
}

void DialogPreferences::ShowDirectories()
{
  DialogDirPreferences *dialog = new DialogDirPreferences(this, iData);
  dialog->exec();
}

void DialogPreferences::SetStyleSheet()
{
  QString startIn = iStyleSheetPath;
  if (startIn.isNull())
    startIn = IpePreferences::Static()->iDialogDir;
  QString fn = QFileDialog::getOpenFileName
    (this, tr("Select default style sheet"), startIn,
     tr("Ipe style sheets (*.isy *.xml)"));
  if (!fn.isEmpty()) {
    iStyleSheetPath = fn;
    iStyleSheet->setText(fn);
  }
}

void DialogPreferences::SetDefaultGrid()
{
  IpePreferences *prefs = IpePreferences::Static();
  prefs->iGridSize = iGridSize;
  prefs->Save();
}

void DialogPreferences::ClearStyleSheet()
{
  iStyleSheetPath = QString::null;
  iStyleSheet->clear();
}

void DialogPreferences::SetFont()
{
  QFont font;
  font.fromString(iData.iFont);
  font = QFontDialog::getFont(0, font);
  iData.iFont = font.toString();
}

void DialogPreferences::UpdateData()
{
  // copy data
  iData.iCompressLevel = iCompress->isChecked() ? 9 : 0;
  iData.iSelectDistance = iSelectDistance->value();
  iData.iSnapDistance = iSnapDistance->value();
  iData.iWhitePaper = iWhitePaper->isChecked();
  iData.iPropertiesShape = iPropertiesShape->currentIndex();
  iData.iAntiAlias = iAntiAlias->isChecked();
  iData.iDoubleBuffer = iDoubleBuffer->isChecked();
  iData.iGridVisible = iGridVisible->isChecked();
  iData.iAutoExport = IpePreferences::TExportMode(iExport->currentIndex());
  iData.iExportExist = iExportExists->isChecked();
  iData.iMaximize = iMaximize->isChecked();
  iData.iBitmapSize = iBitmapSize->value();
  iData.iStyleSheet = iStyleSheetPath;
  if (iLanguage->currentIndex() == 0)
    iData.iLanguage = QString::null;
  else
    iData.iLanguage = iLanguage->currentText();
}

// --------------------------------------------------------------------

/*! \class DialogEffects
  \brief Dialog to edit transition effect of a page view
*/

/*! Create dialog, given an IpeView object.  */
DialogEffects::DialogEffects(QWidget* parent, IpeView &data)
  : QDialog(parent), iData(data)
{
  setupUi(this);
  connect(iButtonHelp, SIGNAL(clicked()), SLOT(ShowHelp()));
  connect(iButtonOk, SIGNAL(clicked()), SLOT(UpdateData()));

  // init effects combo box
  iEffectComboBox->addItem(tr("No effect"));
  iEffectComboBox->addItem(tr("Split horizontally inwards"));
  iEffectComboBox->addItem(tr("Split horizontally outwards"));
  iEffectComboBox->addItem(tr("Split vertically inwards"));
  iEffectComboBox->addItem(tr("Split vertically outwards"));
  iEffectComboBox->addItem(tr("Blinds horizontally"));
  iEffectComboBox->addItem(tr("Blinds vertically"));
  iEffectComboBox->addItem(tr("Box inwards"));
  iEffectComboBox->addItem(tr("Box outwards"));
  iEffectComboBox->addItem(tr("Wipe left-to-right"));
  iEffectComboBox->addItem(tr("Wipe bottom-to-top"));
  iEffectComboBox->addItem(tr("Wipe right-to-left"));
  iEffectComboBox->addItem(tr("Wipe top-to-bottom"));
  iEffectComboBox->addItem(tr("Dissolve"));
  iEffectComboBox->addItem(tr("Glitter left-to-right"));
  iEffectComboBox->addItem(tr("Glitter top-to-bottom"));
  iEffectComboBox->addItem(tr("Glitter diagonally"));

  iDurationSpinBox->setValue(iData.Duration());
  iTransitionSpinBox->setValue(iData.TransitionTime());
  iEffectComboBox->setCurrentIndex(int(iData.Effect()));

  iTransitionSpinBox->setEnabled(iData.Effect() != IpeView::ENormal);
  connect(iEffectComboBox, SIGNAL(highlighted(int)), SLOT(UpdateFields(int)));
}

// --------------------------------------------------------------------

//! Update the transition field when the effect has changed.
void DialogEffects::UpdateFields(int effect)
{
  iTransitionSpinBox->setEnabled(effect != 0);
}

void DialogEffects::UpdateData()
{
  iData.SetEffect(IpeView::TEffect(iEffectComboBox->currentIndex()));
  iData.SetTransitionTime(iTransitionSpinBox->value());
  iData.SetDuration(iDurationSpinBox->value());
}

void DialogEffects::ShowHelp()
{
  QWhatsThis::enterWhatsThisMode();
}

// --------------------------------------------------------------------

DialogLatexError::DialogLatexError(QWidget* parent, const QString &log)
  : QDialog(parent)
{
  setupUi(this);
  iLogEditor->setReadOnly(true);
  iLogEditor->setFocus();
  QTextDocument *doc = new QTextDocument(iLogEditor);
  doc->setPlainText(log);
  QTextCursor cursor(doc);
  int curPos = 0;
  int errNo = 0;
  for (;;) {
    int nextErr = log.indexOf(QLatin1String("\n!"), curPos);
    if (nextErr < 0)
      break;

    int lines = 0;
    while (curPos < nextErr + 1) {
      if (log[curPos++] == QLatin1Char('\n'))
	++lines;
    }
    cursor.movePosition(QTextCursor::Down, QTextCursor::MoveAnchor, lines);
    int pos = cursor.position();
    cursor.movePosition(QTextCursor::Down);
    cursor.setPosition(pos, QTextCursor::KeepAnchor);
    ++errNo;
    QString s;
    s.sprintf("err%d", errNo);
    QTextCharFormat format;
    format.setBackground(Qt::yellow);
    format.setAnchorName(s);
    format.setAnchor(true);
    cursor.setCharFormat(format);
  }
  iLogEditor->setDocument(doc);
  iLogEditor->scrollToAnchor(QLatin1String("err1"));
}

// --------------------------------------------------------------------

DialogPage::DialogPage(QWidget* parent, QStringList &sections, bool *useTitle)
  : QDialog(parent), iSections(sections), iUseTitle(useTitle)
{
  setupUi(this);
  connect(iButtonHelp, SIGNAL(clicked()), SLOT(ShowHelp()));
  connect(iButtonOk, SIGNAL(clicked()), SLOT(UpdateData()));
  iTitleEditor->insertPlainText(iSections[0]);
  iSectionEditor->setText(iSections[1]);
  iSubsectionEditor->setText(iSections[2]);
  iUseForSection->setChecked(iUseTitle[0]);
  iUseForSubsection->setChecked(iUseTitle[1]);
  connect(iUseForSection, SIGNAL(stateChanged(int)),
	  SLOT(useTitleChanged(int)));
  connect(iUseForSubsection, SIGNAL(stateChanged(int)),
	  SLOT(useTitleChanged(int)));
  // init
  useTitleChanged(0);
}

void DialogPage::UpdateData()
{
  iSections[0] = iTitleEditor->toPlainText();
  iSections[1] = iSectionEditor->text();
  iSections[2] = iSubsectionEditor->text();
  iUseTitle[0] = iUseForSection->isChecked();
  iUseTitle[1] = iUseForSubsection->isChecked();
}

void DialogPage::useTitleChanged(int)
{
  if (!iSectionEditor->isEnabled() && !iUseForSection->isChecked())
    iSectionEditor->setText(QString());
  iSectionEditor->setEnabled(!iUseForSection->isChecked());
  if (!iSectionEditor->isEnabled())
    iSectionEditor->setText(iTitleEditor->toPlainText());
  if (!iSubsectionEditor->isEnabled() && !iUseForSubsection->isChecked())
    iSubsectionEditor->setText(QString());
  iSubsectionEditor->setEnabled(!iUseForSubsection->isChecked());
  if (!iSubsectionEditor->isEnabled())
    iSubsectionEditor->setText(iTitleEditor->toPlainText());
}

void DialogPage::ShowHelp()
{
  QWhatsThis::enterWhatsThisMode();
}

// --------------------------------------------------------------------
