 
/***

fltmgr.c - filter list management routines

Copyright (c) Gerard Paul Java 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <curses.h>
#include <panel.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "attrs.h"
#include "error.h"
#include "deskman.h"
#include "input.h"
#include "menurt.h"
#include "dirs.h"
#include "fltdefs.h"
#include "fltmgr.h"
#include "instances.h"

extern int daemonized;

void makestdfiltermenu(struct MENU *menu)
{
    initmenu(menu, 8, 31, (LINES - 8) / 2, (COLS - 31) / 2 + 15);
    additem(menu, " ^D^efine new filter...",
	    "Defines a new set of filter parameters");
    additem(menu, " ^A^pply filter...",
	    "Applies a defined filter");
    additem(menu, " Detac^h^ filter",
	    "Removes the currently applied filter");
    additem(menu, " ^E^dit filter...",
	    "Modifies existing filter data");
    additem(menu, " Dele^t^e filter...",
	    "Removes a TCP filter from the filter list");
    additem(menu, " E^x^it menu", "Returns to the main menu");
}

void makeudpfiltermenu(struct MENU *menu)
{
    initmenu(menu, 9, 38, (LINES - 11) / 2 + 1, (COLS - 38) / 2 + 18);
    additem(menu, " Show ^a^ll UDP packets",
	    "Permits display of all UDP packets");
    additem(menu, " Show ^n^o UDP packets",
	    "Omits all UDP packets from display");
    additem(menu, " ^D^efine custom UDP filter...",
	    "Defines a custom UDP filter");
    additem(menu, " Apply ^c^ustom UDP filter...",
	    "Applies a custom UDP filter");
    additem(menu, " ^E^dit custom UDP filter...",
	    "Modifies a custom UDP filter");
    additem(menu, " Dele^t^e custom UDP filter...",
	    "Removes a custom UDP filter");
    additem(menu, " E^x^it menu", "Returns to the main menu");
}

void makemainfiltermenu(struct MENU *menu)
{
    initmenu(menu, 9, 15, (LINES - 8) / 2, (COLS - 15) / 2 - 5);
    additem(menu, " ^T^CP...", "Transmission Control Protocol");
    additem(menu, " ^U^DP...", "User Datagram Protocol");
    additem(menu, " ^O^ther IP...", "Other IP protocols");
    additem(menu, " ^A^RP", "Address Resolution Protocol");
    additem(menu, " RAR^P^ ", "Reverse Address Resolution Protocol");
    additem(menu, " ^N^on-IP", "Non-IP packets");
    additem(menu, " E^x^it menu", "Return to main menu");
}

/*
 * Generate a string representation of a number to be used as a name.
 */

void genname(unsigned long n, char *m)
{
    sprintf(m, "%lu", n);
}

int mark_filter_change(int protocol)
{
    int resp;

    if (protocol == F_TCP) {
	if (!facility_active(TCPFLTIDFILE, ""))
	    mark_facility(TCPFLTIDFILE, "TCP filter change", "");
	else {
	    errbox("TCP filter data file in use; try again later",
		   ANYKEY_MSG, &resp);
	    return 0;
	}
    } else if (protocol == F_UDP) {
	if (!facility_active(UDPFLTIDFILE, ""))
	    mark_facility(UDPFLTIDFILE, "UDP filter change", "");
	else {
	    errbox("UDP filter data file in use; try again later",
		   ANYKEY_MSG, &resp);
	    return 0;
	}
    } else {
        if (!facility_active(OTHIPFLTIDFILE, ""))
            mark_facility(OTHIPFLTIDFILE, "Other IP filter change", "");
        else {
            errbox("Other IP protocol filter data file in use; try again later",
                   ANYKEY_MSG, &resp);
            return 0;
        }
    }
    return 1;
}

void clear_flt_tag(int protocol)
{
    if (protocol == F_TCP)
	unmark_facility(TCPFLTIDFILE, "");
    else if (protocol == F_UDP)
	unmark_facility(UDPFLTIDFILE, "");
    else
        unmark_facility(OTHIPFLTIDFILE, "");
}

void listfileerr(int code)
{
    if (code == 1)
	write_error("Error loading filter list file", daemonized);
    else
	write_error("Error writing filter list file", daemonized);
}

unsigned long int nametoaddr(char *ascname, int *err)
{
    unsigned long int result;
    struct hostent *he;
    char imsg[45];
    struct in_addr inp;
    int resolv_err = 0;

    resolv_err = inet_aton(ascname, &inp);
    if (resolv_err == 0) {
	snprintf(imsg, 45, "Resolving %s", ascname);
	indicate(imsg);
	he = gethostbyname(ascname);
	if (he != NULL)
	    bcopy((he->h_addr_list)[0], &result, he->h_length);
	else {
	    snprintf(imsg, 45, "Unable to resolve %s", ascname);
	    write_error(imsg, daemonized);
	    *err = 1;
	    return (-1);
	}
    } else
	result = inp.s_addr;

    return (result);
    *err = 0;
}

int loadfilterlist(unsigned int protocol, struct ffnode **fltfile)
{
    int pfd = 0;
    int result = 0;

    struct ffnode *ffiles = NULL;
    struct ffnode *ptemp;
    struct ffnode *tail = NULL;

    int br;

    if (protocol == F_TCP)
	pfd = open(TCPFLNAME, O_RDONLY);
    else if (protocol == F_UDP)
	pfd = open(UDPFLNAME, O_RDONLY);
    else if (protocol == F_OTHERIP)
        pfd = open(OTHIPFLNAME, O_RDONLY);

    if (pfd < 0) {
	*fltfile = NULL;
	return 1;
    }
    do {
	ptemp = malloc(sizeof(struct ffnode));
	br = read(pfd, &(ptemp->ffe), sizeof(struct filterfileent));

	if (br > 0) {
	    if (ffiles == NULL) {
		ffiles = ptemp;
		ffiles->prev_entry = NULL;
	    } else {
		tail->next_entry = ptemp;
		ptemp->prev_entry = tail;
	    }

	    ptemp->next_entry = NULL;
	    tail = ptemp;
	} else {
	    free(ptemp);

	    if (br < 0)
		result = 1;
	}
    } while (br > 0);

    close(pfd);
    *fltfile = ffiles;

    if (ffiles == NULL)
	result = 1;

    return result;
}

void destroyfilterlist(struct ffnode *fltlist)
{
    struct ffnode *fftemp;

    if (fltlist != NULL) {
	fftemp = fltlist->next_entry;

	do {
	    free(fltlist);
	    fltlist = fftemp;
	    if (fftemp != NULL)
		fftemp = fftemp->next_entry;
	} while (fltlist != NULL);
    }
}

void save_filterlist(unsigned int protocol, struct ffnode *fltlist)
{
    struct ffnode *fltfile;
    struct ffnode *ffntemp;
    int fd;
    int bw;

    if (protocol == F_TCP)
	fd =
	    open(TCPFLNAME, O_WRONLY | O_CREAT | O_TRUNC,
		 S_IRUSR | S_IWUSR);
    else if (protocol == F_UDP)
	fd =
	    open(UDPFLNAME, O_WRONLY | O_CREAT | O_TRUNC,
		 S_IRUSR | S_IWUSR);
    else
	fd =
	    open(OTHIPFLNAME, O_WRONLY | O_CREAT | O_TRUNC,
		 S_IRUSR | S_IWUSR);
    
    if (fd < 0) {
	listfileerr(2);
	clear_flt_tag(protocol);
	return;
    }
    fltfile = fltlist;
    while (fltfile != NULL) {
	bw = write(fd, &(fltfile->ffe), sizeof(struct filterfileent));

	if (bw < 0) {
	    listfileerr(2);
	    clear_flt_tag(protocol);
	    return;
	}
	ffntemp = fltfile;
	fltfile = fltfile->next_entry;
	free(ffntemp);
    }

    close(fd);
}

void displayfilters(struct ffnode *ffiles, WINDOW * win)
{
    unsigned int row = 0;
    struct ffnode *pptr;

    pptr = ffiles;

    wattrset(win, STDATTR);
    do {
	wmove(win, row, 2);
	wprintw(win, pptr->ffe.desc);
	row++;
	pptr = pptr->next_entry;
    } while ((row < 8) && (pptr != NULL));

    update_panels();
    doupdate();
}

void operate_select(struct ffnode *ffiles, WINDOW * win,
		    struct ffnode **item, int *aborted)
{
    unsigned int row = 0;
    struct ffnode *pptr;
    int ch;
    int exitloop = 0;

    listkeyhelp();
    update_panels();
    doupdate();
    pptr = ffiles;
    do {
	wattrset(win, PTRATTR);
	wmove(win, row, 1);
	waddch(win, ACS_RARROW);
	ch = wgetch(win);
	wmove(win, row, 1);
	wprintw(win, " ");
	wattrset(win, STDATTR);

	switch (ch) {
	case KEY_UP:
	    if (pptr->prev_entry != NULL) {
		if (row > 0)
		    row--;
		else {
		    wscrl(win, -1);
		    wmove(win, 0, 0);
		    wprintw(win, "%58c", ' ');
		    wmove(win, 0, 2);
		    wprintw(win, pptr->prev_entry->ffe.desc);
		}
		pptr = pptr->prev_entry;
	    }
	    break;
	case KEY_DOWN:
	    if (pptr->next_entry != NULL) {
		if (row < 7)
		    row++;
		else {
		    wscrl(win, 1);
		    scrollok(win, 0);
		    wmove(win, 7, 0);
		    wprintw(win, "%58c", ' ');
		    scrollok(win, 1);
		    wmove(win, 7, 2);
		    wprintw(win, pptr->next_entry->ffe.desc);
		}
		pptr = pptr->next_entry;
	    }
	    break;
	case 12:
	case 'l':
	case 'L':
	    refresh_screen();
	    break;
	case 13:
	    exitloop = 1;
	    *aborted = 0;
	    break;
	case 27:
	case 24:
	case 'X':
	case 'x':
	case 'Q':
	case 'q':
	    exitloop = 1;
	    *aborted = 1;
	    break;
	}
	update_panels();
	doupdate();
    } while (!exitloop);

    *item = pptr;
}

void pickafilter(struct ffnode *ffiles,
		 struct ffnode **fltfile, int *aborted)
{
    WINDOW *borderwin;
    PANEL *borderpanel;
    WINDOW *selectwin;
    PANEL *selectpanel;

    borderwin = newwin(10, 60, (LINES - 10) / 2 - 2, (COLS - 60) / 2 - 2);
    borderpanel = new_panel(borderwin);
    wattrset(borderwin, BOXATTR);
    box(borderwin, ACS_VLINE, ACS_HLINE);

    selectwin = newwin(8, 58, (LINES - 8) / 2 - 2, (COLS - 58) / 2 - 2);
    selectpanel = new_panel(selectwin);
    wattrset(selectwin, STDATTR);
    colorwin(selectwin);
    stdwinset(selectwin);
    wtimeout(selectwin, -1);

    displayfilters(ffiles, selectwin);
    operate_select(ffiles, selectwin, fltfile, aborted);

    del_panel(borderpanel);
    delwin(borderwin);
    del_panel(selectpanel);
    delwin(selectwin);
    update_panels();
    doupdate();
}

void selectfilter(unsigned int protocol,
		  struct filterfileent *ffe, int *aborted)
{
    struct ffnode *fltfile;
    struct ffnode *ffiles;

    if (loadfilterlist(protocol, &ffiles) == 1) {
	listfileerr(1);
	*aborted = 1;
	destroyfilterlist(ffiles);
	return;
    }
    pickafilter(ffiles, &fltfile, aborted);

    if (!(*aborted))
	*ffe = fltfile->ffe;

    destroyfilterlist(ffiles);
}


void get_filter_description(char *description, int *aborted,
			    char *pre_edit)
{
    struct FIELDLIST descfield;
    int dlgwintop;
    WINDOW *dlgwin;
    PANEL *dlgpanel;
    int resp = 0;
    
    dlgwintop = (LINES - 9) / 2;
    dlgwin = newwin(7, 42, dlgwintop, (COLS - 42) / 2 - 10);
    dlgpanel = new_panel(dlgwin);
    wattrset(dlgwin, DLGBOXATTR);
    colorwin(dlgwin);
    box(dlgwin, ACS_VLINE, ACS_HLINE);
    wattrset(dlgwin, DLGTEXTATTR);
    wmove(dlgwin, 2, 2);
    wprintw(dlgwin, "Enter a description for this filter");
    wmove(dlgwin, 5, 2);
    stdkeyhelp(dlgwin);
    update_panels();
    doupdate();

    initfields(&descfield, 1, 35, dlgwintop + 3, (COLS - 42) / 2 - 8);
    addfield(&descfield, 33, 0, 0, pre_edit);
    
    do {
        fillfields(&descfield, aborted);
    
        if ((descfield.list->buf[0] == '\0') && (!(*aborted)))
            errbox("Enter an appropriate description for this filter", ANYKEY_MSG, &resp);
    } while ((descfield.list->buf[0] == '\0') && (!(*aborted)));
    
    if (!(*aborted))
        strcpy(description, descfield.list->buf);

    destroyfields(&descfield);
    del_panel(dlgpanel);
    delwin(dlgwin);
    update_panels();
    doupdate();
}

