/*************************************************************************
 * $Id: config.c,v 1.3 2000/02/07 18:50:39 dpotter Exp $
 *
 * config.c -- Runtime configuration management
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "tools.h"
#include "log.h"
#include "config.h"


/*************************************************************************
 * GLOBALS
 */
char config_file[512] = "";
config_entry_t *config_list = NULL;


/*************************************************************************
 * GET A CONFIG KEY'S VALUE
 */
char *config_getstr (char *key, char *defaultvalue)
{
	config_entry_t *ent;
	for (ent=config_list; ent; ent=ent->next)
		if (!strcasecmp(ent->key, key)) {
			log_printf(LOG_NOISYDEBUG, "config_getstr(): '%s': '%s'\n", ent->key, ent->value);
			return ent->value;
		}
	log_printf(LOG_NOISYDEBUG, "config_getstr(): queried unknown key '%s'\n", key);
	return defaultvalue;
}

int config_getnum (char *key, int defaultvalue)
{
	char *s = config_getstr(key, NULL);
	return s ? atoi(s) : defaultvalue;
}


/*************************************************************************
 * ADD A CONFIG KEY/VALUE PAIR
 * Note: we cannot use log_printf here, because the logfile may
 *       not have been opened yet.
 */
void config_add (char *key, char *value)
{
	config_entry_t *ent;

	ent = malloc(sizeof(config_entry_t));
	if (!ent)
		return;
	ent->key = malloc(strlen(key)+1);
	if (!ent->key) {
		free(ent);
		return;
	}
	strcpy(ent->key, key);
	ent->value = malloc(strlen(value)+1);
	if (!ent->value) {
		free(ent->key);
		free(ent);
		return;
	}
	strcpy(ent->value, value);
	ent->next = config_list;
	config_list = ent;
}


/*************************************************************************
 * READ THE CONFIG FILE
 * Note: we cannot use log_printf here, because the logfile may
 *       not have been opened yet.
 */
int config_read (char *configfile)
{
	int fd;
	char buf[255], *s;

	// open config file
	strncpy(config_file, configfile, sizeof(config_file)-1);
	fd = open(config_file, O_RDONLY);
	if (fd < 0)
		return -1;

	// read the config file
	while (readline(fd, buf, sizeof(buf)) >= 0) {
		trim(buf);
		if (!*buf || *buf == '#')
			continue;
		s = strchr(buf, ':');
		if (!s)
			continue;
		*s++ = 0;
		config_add(trim(buf), trim(s));
	}

	// close config file
	close(fd);
	return 0;
}


/*************************************************************************
 * FREE ALL CONFIG ENTRIES
 */
void config_free (void)
{
	config_entry_t *n, *ent = config_list;
	while (ent) {
		if (ent->value)
			free(ent->value);
		if (ent->key)
			free(ent->key);
		n = ent->next;
		free(ent);
		ent = n;
	}
	config_list = NULL;
	log_printf(LOG_DEBUG, "config_free(): freed all config entries\n");
}


/*************************************************************************
 * RELOAD THE CONFIG FILE
 */
void config_reload (void)
{
	config_free();
	config_read(config_file);
	log_printf(LOG_DEBUG, "config_reload(): reloaded config file\n");
}


/*************************************************************************
 * EOF
 */
