/*
 gui-channels.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "gui-nicklist.h"

gchar *gui_channel_get_name(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, NULL);

    if (channel->type == CHANNEL_TYPE_CHANNEL || channel->type == CHANNEL_TYPE_QUERY)
        return channel->name; /* channel/query */

    if (channel->level & (MSGLEVEL_ALL ^ MSGLEVEL_MSGS))
        return "status"; /* status window */

    else if (channel->level & MSGLEVEL_MSGS)
        return "msgs"; /* msgs window */

    return NULL;
}

gboolean gui_channel_active(CHANNEL_REC *channel)
{
    return CHANNEL_PARENT(channel)->active == channel &&
	WINDOW_VIEW(CHANNEL_PARENT(channel)) == WINDOW_GUI(CHANNEL_PARENT(channel))->active;
}

static void window_redraw_tab(WINDOW_REC *window)
{
    GtkWidget *label;
    GString *str;
    GList *tmp, *views;

    g_return_if_fail(window != NULL);

    label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(WINDOW_GUI(window)->parent->notebook),
                                       WINDOW_GUI(window)->window);
    g_return_if_fail(label != NULL);

    str = g_string_new(NULL);

    for (views = WINDOW_GUI(window)->views; views != NULL; views = views->next)
    {
	WINDOW_VIEW_REC *view = views->data;

	g_string_sprintfa(str, str->len == 0 ? "%d" : " %d",
			  g_list_index(windows, view->window)+1);
	for (tmp = g_list_first(view->window->channels); tmp != NULL; tmp = tmp->next)
	{
	    CHANNEL_REC *rec = tmp->data;

	    g_string_sprintfa(str, " %s", rec->name);
	}
    }
    gtk_label_set_text(GTK_LABEL(label), str->str);
    g_string_free(str, TRUE);
}

static gboolean gui_channel_init(CHANNEL_REC *channel)
{
    GUI_CHANNEL_REC *rec;

    g_return_val_if_fail(channel != NULL, FALSE);

    rec = g_new0(GUI_CHANNEL_REC, 1);
    channel->gui_data = rec;

    rec->font_normal = font_normal;
    rec->font_bold = font_bold;

    window_redraw_tab(CHANNEL_PARENT(channel));

    if (channel->type == CHANNEL_TYPE_EMPTY)
        rec->servermenu = TRUE;
    if (channel->type == CHANNEL_TYPE_CHANNEL && setup_get_bool("toggle_show_nicklist"))
    {
	gtk_widget_show(WINDOW_GUI(CHANNEL_PARENT(channel))->nlist_widget);
	CHANNEL_GUI(channel)->nicklist = TRUE;
    }
    if (channel->type == CHANNEL_TYPE_QUERY)
	rec->autoraise = setup_get_bool("toggle_autoraise_msgs_window");

    signal_emit("gui channel created", 1, channel);
    return TRUE;
}

static gboolean signal_channel_child_destroyed(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    window_redraw_tab(CHANNEL_PARENT(channel));
    return TRUE;
}

static gboolean signal_channel_destroyed(CHANNEL_REC *channel)
{
    GUI_CHANNEL_REC *gui;
    GList *tmp;

    g_return_val_if_fail(channel != NULL, FALSE);

    gui = CHANNEL_GUI(channel);

    /* remove all who reqests left for the channel */
    if (channel->server != NULL)
    {
	for (tmp = g_list_first(gui->who_tags); tmp != NULL; tmp = tmp->next)
	    server_idle_remove(channel->server, GPOINTER_TO_INT(tmp->data));
    }
    g_list_free(gui->who_tags);

    g_free(channel->gui_data);
    channel->gui_data = NULL;
    return TRUE;
}

static gboolean signal_channel_topic_changed(CHANNEL_REC *channel)
{
    WINDOW_REC *window;
    GString *tmp;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    if (!gui_channel_active(channel)) return TRUE;

    tmp = g_string_new(channel->name);

    if (channel->topic == NULL || *channel->topic == '\0')
    {
	/* No topic in channel */
	gtk_entry_set_text(GTK_ENTRY(WINDOW_GUI(window)->topicentry), "");
    }
    else
    {
	gtk_entry_set_text(GTK_ENTRY(WINDOW_GUI(window)->topicentry), channel->topic);
	gtk_entry_set_position(GTK_ENTRY(WINDOW_GUI(window)->topicentry), 0);
	g_string_sprintfa(tmp, " - %s", channel->topic);
    }

    if (window == WINDOW_GUI(window)->parent->active)
        gtk_window_set_title(GTK_WINDOW(WINDOW_GUI(window)->parent->window), tmp->str);
    g_string_free(tmp, TRUE);
    return TRUE;
}

static gboolean signal_channel_mode_changed(CHANNEL_REC *channel)
{
    GUI_WINDOW_REC *gui;
    WINDOW_REC *window;
    gchar *str;
    gint num;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    if (channel->type != CHANNEL_TYPE_CHANNEL || !gui_channel_active(channel))
	return TRUE;
    gui = WINDOW_GUI(window);

    for (num = 0; num < 8; num++)
        gtk_object_set_data(GTK_OBJECT(gui->modebuttons[num]), "toggling", GINT_TO_POINTER(TRUE));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[0]), channel->mode_secret);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[1]), channel->mode_private);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[2]), channel->mode_moderate);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[3]), channel->mode_invite);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[4]), channel->mode_nomsgs);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[5]), channel->mode_optopic);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[6]), channel->limit != 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[7]), channel->key != NULL);

    /* display limit */
    str = g_strdup_printf(channel->limit ? "%d" : "", channel->limit);
    gtk_entry_set_text(GTK_ENTRY(gui->limit_entry), str);
    g_free(str);

    /* display key */
    gtk_entry_set_text(GTK_ENTRY(gui->key_entry), channel->key == NULL ? "" : channel->key);

    for (num = 0; num < 8; num++)
        gtk_object_set_data(GTK_OBJECT(gui->modebuttons[num]), "toggling", GINT_TO_POINTER(FALSE));

    return TRUE;
}

static gboolean signal_channel_change_name(CHANNEL_REC *channel, gchar *name)
{
    g_return_val_if_fail(channel != NULL, TRUE);

    window_redraw_tab(CHANNEL_PARENT(channel));
    signal_emit("channel topic changed", 1, channel);
    return TRUE;
}

static gboolean signal_channel_open(CHANNEL_REC *channel)
{
    WINDOW_REC *window;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    if (WINDOW_GUI(WINDOW_GUI(window)->parent->active) != WINDOW_GUI(window))
    {
        /* change to right window tab */
        gint page;

        page = gtk_notebook_page_num(GTK_NOTEBOOK(WINDOW_GUI(window)->parent->notebook), WINDOW_GUI(window)->window);
        gtk_notebook_set_page(GTK_NOTEBOOK(WINDOW_GUI(window)->parent->notebook), page);
    }
    gdk_window_show(WINDOW_GUI(window)->parent->window->window);

    if (WINDOW_GUI(window)->active != WINDOW_VIEW(window))
    {
	/* change the view */
        WINDOW_GUI(window)->active = WINDOW_VIEW(window);
	WINDOW_GUI(window)->parent->active = window;
        signal_emit("window focused", 1, window);
        window->active = NULL;
    }

    if (window->active != channel)
    {
        window->active = channel;
        signal_emit("channel focused", 1, channel);
    }
    return TRUE;
}

/* changed to different channel in same window (or just creating the channel) */
static gboolean signal_channel_focused(CHANNEL_REC *channel)
{
    if (WINDOW_GUI(CHANNEL_PARENT(cur_channel)) == WINDOW_GUI(CHANNEL_PARENT(channel)))
    {
	/* active channel changed in the current window */
	cur_channel = channel;
	signal_emit("channel changed", 1, cur_channel);
    }

    if (channel->type == CHANNEL_TYPE_CHANNEL)
    {
        /* redraw channel widgets */
        signal_emit("channel topic changed", 1, channel);
        signal_emit("channel mode changed", 1, channel);
        gui_nicklist_redraw(channel);
    }

    return TRUE;
}

/* window destroyed - numbers in channel tabs have changed, redraw them.. */
static gboolean signal_window_destroyed(WINDOW_REC *window)
{
    GList *tmp;

    for (tmp = g_list_first(windows); tmp != NULL; tmp = tmp->next)
	window_redraw_tab(tmp->data);

    return TRUE;
}

void gui_channels_init(void)
{
    signal_add("gui channel open", (SIGNAL_FUNC) signal_channel_open);
    signal_add("gui channel init", (SIGNAL_FUNC) gui_channel_init);
    signal_add("channel child destroyed", (SIGNAL_FUNC) signal_channel_child_destroyed);
    signal_add("channel destroyed", (SIGNAL_FUNC) signal_channel_destroyed);
    signal_add("channel name changed", (SIGNAL_FUNC) signal_channel_change_name);
    signal_add("channel mode changed", (SIGNAL_FUNC) signal_channel_mode_changed);
    signal_add("channel topic changed", (SIGNAL_FUNC) signal_channel_topic_changed);
    signal_add("channel focused", (SIGNAL_FUNC) signal_channel_focused);
    signal_add("window destroyed", (SIGNAL_FUNC) signal_window_destroyed);
}

void gui_channels_deinit(void)
{
    signal_remove("gui channel open", (SIGNAL_FUNC) signal_channel_open);
    signal_remove("gui channel init", (SIGNAL_FUNC) gui_channel_init);
    signal_remove("channel child destroyed", (SIGNAL_FUNC) signal_channel_child_destroyed);
    signal_remove("channel destroyed", (SIGNAL_FUNC) signal_channel_destroyed);
    signal_remove("channel name changed", (SIGNAL_FUNC) signal_channel_change_name);
    signal_remove("channel mode changed", (SIGNAL_FUNC) signal_channel_mode_changed);
    signal_remove("channel topic changed", (SIGNAL_FUNC) signal_channel_topic_changed);
    signal_remove("channel focused", (SIGNAL_FUNC) signal_channel_focused);
    signal_remove("window destroyed", (SIGNAL_FUNC) signal_window_destroyed);
}
