/*
 * screenshot_list.cpp - implementation of screenshot-list for side-bar
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include <qdir.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qpushbutton.h>
#include <qwhatsthis.h>

#include "screenshot_list.h"
#include "client.h"
#include "system_environment.h"
#include "client_manager.h"
#include "embed.h"

#include "screenshot_list.moc"


screenshotList::screenshotList( QWidget * _parent ) :
	sideBarWidget( tr( "Screenshots you made" ), embed::getIconPixmap( "screenshots" ), _parent )
{
	QFont f;
	f.setPixelSize( 12 );

	m_list = new QListBox( contentParent() );
	m_list->setFont( f );
	m_list->setSelectionMode( QListBox::Single );
	QWhatsThis::add( m_list, tr( "Here are all screenshots listed you made. You can make screenshots by clicking with the "
					"right mousebutton on a client-window and select \"Make screenshot\" in the menu. "
					"These screenshots can be managed using the buttons below." ) );
	connect( m_list, SIGNAL( highlighted( const QString & ) ), this, SLOT( screenshotSelected( const QString & ) ) );
	connect( m_list, SIGNAL( selected( const QString & ) ), this, SLOT( screenshotDoubleClicked( const QString & ) ) );

	m_preview = new QLabel( contentParent() );
	m_preview->setScaledContents( TRUE );

	QHBox * hbox = new QHBox( contentParent() );
	QVBox * vbox1 = new QVBox( hbox );
	QVBox * vbox2 = new QVBox( hbox );

	f.setBold( TRUE );
	f.setItalic( TRUE );

	QLabel * user_lbl = new QLabel( vbox1 );
	QLabel * date_lbl = new QLabel( vbox1 );
	QLabel * time_lbl = new QLabel( vbox1 );
	QLabel * client_lbl = new QLabel( vbox1 );
	user_lbl->setText( tr( "User" )+":" );
	date_lbl->setText( tr( "Date" )+":" );
	time_lbl->setText( tr( "Time" )+":" );
	client_lbl->setText( tr( "Client" )+":" );
	user_lbl->setFont( f );
	date_lbl->setFont( f );
	time_lbl->setFont( f );
	client_lbl->setFont( f );

	f.setBold( FALSE );
	f.setItalic( FALSE );

	m_user = new QLabel( vbox2 );
	m_date = new QLabel( vbox2 );
	m_time = new QLabel( vbox2 );
	m_client = new QLabel( vbox2 );
	m_user->setFont( f );
	m_date->setFont( f );
	m_time->setFont( f );
	m_client->setFont( f );

	vbox1->setFixedWidth( 72 );

	m_showBtn = new QPushButton( embed::getIconPixmap( "client_show" ), tr( "Show screenshot" ), contentParent() );
	connect( m_showBtn, SIGNAL( clicked() ), this, SLOT( showScreenshot() ) );
	QWhatsThis::add( m_showBtn, tr( "When clicking on this button, the selected screenshot is being displayed in full size." ) );

	m_deleteBtn = new QPushButton( embed::getIconPixmap( "cancel" ), tr( "Delete screenshot" ), contentParent() );
	connect( m_deleteBtn, SIGNAL( clicked() ), this, SLOT( deleteScreenshot() ) );
	QWhatsThis::add( m_deleteBtn, tr( "When clicking on this button, the selected screenshot is being deleted." ) );

	m_reloadBtn = new QPushButton( embed::getIconPixmap( "reload" ), tr( "Reload list" ), contentParent() );
	connect( m_reloadBtn, SIGNAL( clicked() ), this, SLOT( reloadList() ) );
	QWhatsThis::add( m_reloadBtn, tr( "When clicking on this button, the screenshot-list is being reloaded immediately. "
						"You should never need this function since iTALC automatically reloads this list, "
						"after making a new screenshot." ) );

	reloadList();
}




screenshotList::~screenshotList()
{
}





void screenshotList::screenshotSelected( const QString & _s )
{
	m_preview->setPixmap( QDir::home().path()+SCREENSHOT_PATH+_s );
	m_preview->setFixedHeight( m_preview->width()*3/4 );
	m_user->setText( systemEnvironment::realUserName( _s.section( '_', 0, 0 ) ) + " (" + _s.section( '_', 0, 0 ) + ")" );
 	m_client->setText( _s.section( '_', 1, 1 ) );
	m_date->setText( QDate::fromString( _s.section( '_', 2, 2 ), Qt::ISODate ).toString( Qt::LocalDate ) );
	m_time->setText( _s.section( '_', 3, 3 ).section( '.', 0, 0 ) );
}




void screenshotList::screenshotDoubleClicked( const QString & _s )
{
	// maybe the user clicked on "show screenshot" and selected no screenshot...
	if( _s == "" )
		return;

	QWidget * w = new QWidget( NULL, NULL, WDestructiveClose );
	w->move( 0, 0 );
	w->setCaption( _s );

	QLabel * img_label = new QLabel( w );
	img_label->setPixmap( QDir::home().path() + SCREENSHOT_PATH + _s );
	if( img_label->pixmap() != NULL )
	{
		w->setFixedSize( img_label->pixmap()->width(), img_label->pixmap()->height() );
		img_label->setFixedSize( img_label->pixmap()->width(), img_label->pixmap()->height() );
	}

	w->show();
}




void screenshotList::showScreenshot( void )
{
#if QT_VERSION >= 0x030100
	QString screenshot = m_list->text( m_list->index( m_list->selectedItem() ) );
#else
	QString screenshot = m_list->currentText();
#endif
	screenshotDoubleClicked( screenshot );
}




void screenshotList::deleteScreenshot( void )
{
#if QT_VERSION >= 0x030100
	QString s = m_list->text( m_list->index( m_list->selectedItem() ) );
#else
	QString s = m_list->currentText();
#endif

	// maybe the user clicked on "delete screenshot" and selected no screenshot...
	if( s == "" )
		return;

	QFile screenshot( QDir::home().path()+SCREENSHOT_PATH+s );
	screenshot.remove();

	reloadList();
}




void screenshotList::reloadList( void )
{
	QDir sdir( QDir::home().path() + SCREENSHOT_PATH, "*.png", QDir::Name|QDir::IgnoreCase, QDir::Files|QDir::Readable );

	m_list->clear();
	m_list->insertStringList( sdir.entryList() );
}
