#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <getopt.h>

/* gcc -o class2c class2c.c */

static int verbose = 1;

static void
usage(int argc, char *argv[]) {
	fprintf(stderr, "usage: %s [-q] -o <outfile> -c class_var_name MyJava.class\n", argv[0]);
	return;
}

static int
dumpfile(FILE *fp_out, FILE *fp, const char *classname, const char *outfile) {
	unsigned char buffer[4096];
	unsigned int bytecount;
	int first;

	first = 1;
	bytecount = 0;
	fprintf(fp_out, "const unsigned char %s_data[] = {\n", classname);

	for(;;) {
		int n, i;

		n = fread(buffer, 1, sizeof(buffer), fp);
		if(n < 0)
			return -1;
		if(n == 0)
			break;

		i = 0;
		while(i < n) {
			if(first)
				first = 0;
			else
				fprintf(fp_out, ", ");
			fprintf(fp_out, " 0x%02x", buffer[i]);

			bytecount++;
			if(bytecount > 0 && (bytecount % 16) == 0) {
				fprintf(fp_out, ",\n");
				first = 1;
			}
			i++;
		}
	}
	fprintf(fp_out, "\n");

	fprintf(fp_out, "};\n");
	fprintf(fp_out, "\n");
	fprintf(fp_out, "const unsigned int %s_len = %u;\n", classname, bytecount);

	if(verbose > 0)
		fprintf(stderr, "bytecount=%d outfile=%s\n", bytecount, outfile);

	if(fflush(fp_out))
		return -1;
	if(ferror(fp_out))
		return -1;

	return 0;
}

int
main(int argc, char *argv[])
{
	const char *infile;
	const char *outfile;
	const char *classname;
	FILE *fp_out;
	FILE *fp;
	int error;
	int i;

	error = 0;
	while((i = getopt(argc, argv, "c:o:q")) != EOF) {
		switch(i) {
		case 'c':
			classname = optarg;
			break;

		case 'o':
			outfile = optarg;
			break;

		case 'q':
			verbose = 0;
			break;

		default:
			error = 1;
			break;
		}
	}

	if(error || outfile == NULL || classname == NULL || argc <= optind) {
		usage(argc, argv);
		return 1;
	}

	infile = argv[optind];
	if((fp = fopen(infile, "r")) == NULL) {
		fprintf(stderr, "fopen(\"%s\"): %s\n", infile, strerror(errno));
		return 1;
	}

	if((fp_out = fopen(outfile, "w")) == NULL) {
		fprintf(stderr, "fopen(\"%s\"): %s\n", outfile, strerror(errno));
		return 1;
	}

	fprintf(fp_out, "/* GENERATED FILE - DO NOT EDIT */\n");
	fprintf(fp_out, "/* -o %s -c %s %s */\n", outfile, classname, infile);
	fprintf(fp_out, "/* */\n");
	fprintf(fp_out, "\n");

	dumpfile(fp_out, fp, classname, outfile);
	fprintf(fp_out, "\n");
	fprintf(fp_out, "/* THEEND */\n");

	if(fclose(fp_out) != 0) {
		fprintf(stderr, "fclose(\"%s\"): %s\n", outfile, strerror(errno));
		fclose(fp);
	}

	fclose(fp);

	return 0;
}
