/***************************************************************************
 $RCSfile: rsacard.cpp,v $
 -------------------
 cvs         : $Id: rsacard.cpp,v 1.15 2003/05/07 22:27:22 aquamaniac Exp $
 begin       : Sat Dec 14 2002
 copyright   : (C) 2002 by Martin Preuss
 email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#include "rsacard.h"
#include "ctmisc.h"
#include <engine/chameleon/error.h>
#include <engine/chameleon/debug.h>


RSACard::KeyDescriptor::KeyDescriptor()
  :_chg(false)
  ,_status(KEY_STATUS_INTERNAL_UNUSED)
  ,_isSignKey(false)
  ,_keynum(-1)
  ,_keyver(-1)
{
}


RSACard::KeyDescriptor::KeyDescriptor(unsigned int st,
				      bool isSignKey,
				      int keynum,
				      int keyver)
:_status(st)
,_isSignKey(isSignKey)
,_keynum(keynum)
,_keyver(keyver)
{
}


RSACard::KeyDescriptor::KeyDescriptor(const string &data)
:_chg(false)
,_status(KEY_STATUS_INTERNAL_UNUSED)
,_isSignKey(false)
,_keynum(-1)
,_keyver(-1)
{
  if (data.length()>7) {
    _status=(unsigned char)data[0];
    _isSignKey=(data[1]==0x53);
    if (data[2])
      _keynum=CTMisc::string2num(data.substr(2,3));
    else
      _keynum=-1;
    if (data[5])
      _keyver=CTMisc::string2num(data.substr(5,3));
    else
      _keyver=-1;
  }
}


RSACard::KeyDescriptor::~KeyDescriptor(){
}


string RSACard::KeyDescriptor::toString() const{
  string r;

  r+=_status;
  if (_isSignKey)
    r+=0x53;
  else
    r+=0x56;

  if (_keynum==-1)
    r+=string(3,(char)0x00);
  else
    r+=CTMisc::num2string(_keynum,"%03d");
  if (_keyver==-1)
    r+=string(3,(char)0x00);
  else
    r+=CTMisc::num2string(_keyver,"%03d");

  return r;
}


string RSACard::KeyDescriptor::dump(){
  string r;

  r+="Keydescriptor:";
  r+="\nStatus     : ";
  r+=CTMisc::num2string(_status,"%04x");
  r+="\nSignKey    : ";
  if (_isSignKey)
    r+="yes";
  else
    r+="no";
  r+="\nKey Number : ";
  r+=CTMisc::num2string(_keynum);
  r+="\nKey Version: ";
  r+=CTMisc::num2string(_keyver);
  r+="\n";

  return r;
}








RSACard::BankDescription::BankDescription()
:_chg(false)
,_country(280)
,_service(2)
{
}


RSACard::BankDescription::BankDescription(const string &data)
:_chg(false)
,_country(0)
,_service(2)
{
  if (data.length()>153) {
    if ((unsigned char)(data[0])!=0xff) {
      _country=CTMisc::string2num(data.substr(0,3));
      _bankName=data.substr(3,30);
      CTMisc::removeBlanks(_bankName);
      _userId=data.substr(33, 30);
      CTMisc::removeBlanks(_userId);
      _service=data[63];
      _address=data.substr(64, 28);
      CTMisc::removeBlanks(_address);
      _addressSuffix=data.substr(92,2);
      CTMisc::removeBlanks(_addressSuffix);
      _bankId=data.substr(94, 30);
      CTMisc::removeBlanks(_bankId);
      _systemId=data.substr(124,30);
      CTMisc::removeBlanks(_systemId);
    }
  }
}


RSACard::BankDescription::~BankDescription(){
}


string RSACard::BankDescription::toString() const{
  string r;
  string t;

  r+=CTMisc::num2string(_country,"%03d");
  t=_bankName.substr(0,30);
  t+=string(30-t.length(),(char)32);
  r+=t;
  t=_userId.substr(0,30);
  t+=string(30-t.length(),(char)32);
  r+=t;
  r+=_service;
  t=_address.substr(0,28);
  t+=string(28-t.length(),(char)32);
  r+=t;
  t=_addressSuffix.substr(0,2);
  t+=string(2-t.length(),(char)32);
  r+=t;
  t=_bankId.substr(0,30);
  t+=string(30-t.length(),(char)32);
  r+=t;
  t=_systemId.substr(0,30);
  t+=string(30-t.length(),(char)32);
  r+=t;

  return r;
}


string RSACard::BankDescription::dump(){
  string r;

  r+="BankDescription:";
  r+="\nCountry        : ";
  r+=CTMisc::num2string(_country);
  r+="\nBank Name      : ";
  r+=_bankName;
  r+="\nUser Id        : ";
  r+=_userId;
  r+="\nService        : ";
  r+=_service;
  r+="\nAddress        : ";
  r+=_address;
  r+="\nAddress Suffix : ";
  r+=_addressSuffix;
  r+="\nBank Id        : ";
  r+=_bankId;
  r+="\nSystem Id      : ";
  r+=_systemId;
  r+="\n";

  return r;
}


RSACard::RSACard(const CTCard &c)
:CTProcessorCard(c)
{
}


RSACard::~RSACard(){
}


CTError RSACard::reopenCard(){
  CTError err;

  err=_openCard();
  if (!err.isOk())
    return CTError("RSACard::_openCard",err);
  return CTError();
}


CTError RSACard::openCard(){
  CTError err;

  err=CTProcessorCard::openCard();
  if (!err.isOk())
    return CTError("CTProcessorCard::openCard",err);
  err=_openCard();
  if (!err.isOk()) {
    CTProcessorCard::closeCard();
    return CTError("RSACard::openCard",err);
  }
  return CTError();
}


CTError RSACard::closeCard(bool force){
  return CTProcessorCard::closeCard(force);
}


string RSACard::cardType(){
  return "RSACard";
}


string RSACard::cardTypes(){
  return CTProcessorCard::cardTypes()+",RSACard";
}


CTError RSACard::_readFile(unsigned short fid, string &data){
  CTError err;
  string fcp;

  // select file
  err=selectFile(fcp, fid);
  if (!err.isOk())
    return CTError("RSACard::_readFile",err);

  // read file
  err=execCommand("read_file",
		_cmdReadFile,
		data);
  if (!err.isOk(0x62, 0x82))
    return CTError("RSACard::_readFile",err);

  return CTError();
}


CTError RSACard::_getInitialPin(string &pin){
  CTError err;
  string fcp;
  string resp;

  // read EF_GD0
  err=_readFile(0x2f02, resp);
  if (!err.isOk())
    return CTError("RSACard::_getInitialPin",err);

  if (resp.length()<12)
    return CTError("RSACard::_getInitialPin()",
		   k_CTERROR_INVALID,0,0,
		   "Response too short");

  pin=resp.substr(6,5);
  return CTError();
}


CTError RSACard::_openCard(){
  CTError err;
  string fcp;
  char fid[9]={0xd2,0x76,0x00,0x00,0x74,0x48,0x42,0x01,0x10};
  string fids;

  err=selectFile(fcp, 0x3f00);
  if (!err.isOk())
    return CTError("RSACard::_openCard",err);
  err=_getInitialPin(_firstPin);
  if (!err.isOk())
    return CTError("RSACard::_openCard",err);

  fids.assign(fid,sizeof(fid));
  err=selectById(fcp, fids);
  if (!err.isOk()) {
    DBG_ERROR("ERROR is: %s", err.errorString().c_str());
    return CTError("RSACard::_openCard",err);
  }
  return CTError();
}


unsigned int RSACard::readSeq(int bank){
  CTError err;
  string response;
  unsigned int s;

  err=selectFile(response, 0xa601);
  if (!err.isOk())
    throw CTError("RSACard::_readSeq",err);
  err=readRecord(response, 1+bank);
  if (!err.isOk())
    throw CTError("RSACard::_readSeq",err);
  if (response.length()!=4)
    throw CTError("RSACard::_readSeq",
		  k_CTERROR_INVALID,0,0,
		  "Bad response size");
  s=(unsigned char)(response[0])<<24;
  s+=(unsigned char)(response[1])<<16;
  s+=(unsigned char)(response[2])<<8;
  s+=(unsigned char)(response[3]);
  return s;
}


CTError RSACard::writeSeq(int bank, unsigned int seq){
  CTError err;
  string response;
  string data;

  err=selectFile(response, 0xa601);
  if (!err.isOk())
    return CTError("RSACard::_readSeq",err);

  data+=(seq>>24) & 0xff;
  data+=seq>>16 & 0xff;
  data+=seq>>8 & 0xff;
  data+=seq & 0xff;

  err=execCommand("update_record",
		_cmdUpdateRecord,
		response,
		CTMisc::num2string(1+bank),
		CTMisc::bin2hex(data));
  if (!err.isOk())
    return CTError("RSACard::_writeSeq",err);
  return CTError();
}


int RSACard::_findPublicKey(unsigned int kid){
  int kc;
  KeyLogStatus st;
  CTError err;
  int i;
  string response;

  st=readKeyLogStatus();
  // select IPF
  err=selectFile(response, 0xb300);
  if (!err.isOk())
    throw CTError("RSACard::_findPublicKey",err);

  // maximum number of available keys
  kc=(st.maxEntries*4)+2;
  for (i=0; i<kc; i++) {
    // read key id
    err=readBinaryRaw(response, (i*121)+1,1);
    if (!err.isOk())
      throw CTError("RSACard::_findPublicKey",err);
    if (response.length()!=1)
      throw CTError("RSACard::_findPublicKey",
		    k_CTERROR_INVALID,0,0,
		    "Bad size of response");
    if ((unsigned char)(response[0])==kid)
      return i;
  } // for

  // not found
  return -1;
}


int RSACard::_getKeyPos_EF_LOG(int kid) {
  int i;

  i=0;
  if ((kid & 15)>5) // cryptkey
    i=(kid & 15)-6;
  else
    i=(kid & 15)-1;
  i*=32;
  if (isBankKey(kid))
    i+=16;
  if (isSignKey(kid))
    i+=8;
  i++;
  return i;
}


CTError RSACard::deleteKeyDescriptor(int kid) {
  int i;
  CTError err;
  string response;

  i=_getKeyPos_EF_LOG(kid);

  // select EF_KEY_LOG
  err=selectFile(response, 0xa602);
  if (!err.isOk())
    return CTError("RSACard::deleteKey",err);

  // write key descriptor - status byte
  err=execCommand("update_binary",
		_cmdUpdateBinary,
		response,
		CTMisc::num2string(i),
		"08");
  if (!err.isOk())
    return CTError("RSACard::deleteKey",err);

  // write key descriptor - status byte
  err=execCommand("update_binary",
		_cmdUpdateBinary,
		response,
		CTMisc::num2string(i+2),
		"000000000000");
  if (!err.isOk())
    return CTError("RSACard::deleteKey",err);
  return CTError();
}


unsigned int RSACard::readKeyStatus(int kid) {
  int i;
  string response;
  CTError err;

  i=_getKeyPos_EF_LOG(kid);

  // select EF_KEY_LOG
  err=selectFile(response, 0xa602);
  if (!err.isOk())
    throw CTError("RSACard::getKeyStatus",err);

  // write key descriptor - status byte
  err=readBinaryRaw(response, i, 1);
  if (!err.isOk())
    throw CTError("RSACard::getKeyStatus",err);
  if (response.length()<1)
    throw CTError("RSACard::getKeyStatus",
		  k_CTERROR_INVALID,0,0,
		  "Bad size of response");
  return (unsigned char)(response[0]);
}


CTError RSACard::writeKeyStatus(int kid, unsigned int st) {
  int i;
  string response;
  CTError err;

  i=_getKeyPos_EF_LOG(kid);

  // select EF_KEY_LOG
  err=selectFile(response, 0xa602);
  if (!err.isOk())
    return CTError("RSACard::setKeyStatus",err);

  // write key descriptor - status byte
  err=execCommand("update_binary",
		_cmdUpdateBinary,
		response,
		CTMisc::num2string(i),
		CTMisc::num2string(st));
  if (!err.isOk())
    return CTError("RSACard::deleteKey",err);

  if (!err.isOk())
    return CTError("RSACard::deleteKey",err);
  return CTError();
}


CTError RSACard::writeKeyDescriptor(int kid, const string &kd) {
  int i;
  string response;
  CTError err;

  i=_getKeyPos_EF_LOG(kid);
  if (kd.length()!=8)
    return CTError("RSACard::writeKeyDescriptor",
		   k_CTERROR_INVALID,0,0,
		   "Bad size of data");

  // select EF_KEY_LOG
  err=selectFile(response, 0xa602);
  if (!err.isOk())
    return CTError("RSACard::deleteKey",err);

  // write key descriptor
  err=execCommand("update_binary",
		_cmdUpdateBinary,
		response,
		CTMisc::num2string(i),
		CTMisc::bin2hex(kd));
  if (!err.isOk())
    return CTError("RSACard::writeKeyDescriptor",err);
  return CTError();
}


string RSACard::readKeyDescriptor(int kid) {
  int i;
  string response;
  CTError err;

  i=_getKeyPos_EF_LOG(kid);

  // select EF_KEY_LOG
  err=selectFile(response, 0xa602);
  if (!err.isOk())
    throw CTError("RSACard::getKeyStatus",err);

  // write key descriptor - status byte
  err=readBinaryRaw(response, i, 8);
  if (!err.isOk())
    throw CTError("RSACard::getKeyStatus",err);
  if (response.length()<8)
    throw CTError("RSACard::getKeyStatus",
		  k_CTERROR_INVALID,0,0,
		  "Bad size of response");
  return response;
}


string RSACard::createKey(int kid, bool overwrite) {
  KeyLogStatus st;
  int bkid;
  string response;
  string modulus;
  CTError err;
  int to;

  // check for EF_KEY_LOG status
  st=readKeyLogStatus();

  if ((isSignKey(kid) && (st.oldDSfree!=0)) ||
      (!isSignKey(kid) && (st.oldENfree!=0))) {
    /*
    if (!overwrite)
      throw CTError("2:RSACard::createUserKey",
		    k_CTERROR_INVALID,0,0,
		    "Key in use");
                    */
    st.oldENfree=0;
    st.oldDSfree=0;
    err=writeKeyLogStatus(st);
    if (!err.isOk())
      throw CTError("3:RSACard::createUserKey",err);
  }

  if (readKeyStatus(kid)!=0x08) {
    if (!overwrite)
      throw CTError("4:RSACard::createUserKey",
		    k_CTERROR_INVALID,0,0,
		    "Key in use");
    // set status to "free"
    err=writeKeyStatus(kid, 0x08);
    if (!err.isOk())
      throw CTError("5:RSACard::createUserKey",err);
  }

  // create key in buffer
  if (isSignKey(kid))
    bkid=0x8f;
  else
    bkid=0x8e;

  // generate key pair
  DBG_VERBOUS("Key id: %02x",bkid);
  to=timeout();
  setTimeout(60);
  err=execCommand("generate_keypair",
		_cmdGenerateKeyPair,
		modulus,
		CTMisc::num2string(bkid),
		"768",
		"96");
  setTimeout(to);
  if (!err.isOk())
    throw CTError("6:RSACard::createUserKey",err);

  return modulus;
}


CTError RSACard::activateKey(int kid,
			     int num, int ver){
  int bkid;
  string response;
  string data;
  CTError err;

  KeyDescriptor kd(0x10, isSignKey(kid), num, ver);
  DBG_VERBOUS("Is Sign key: %d",isSignKey(kid));
  if (isSignKey(kid))
    bkid=0x8f;
  else
    bkid=0x8e;

  DBG_VERBOUS("Key id: %02x\n",bkid);

  // set status to "active"
  err=execCommand("activate_key",
		_cmdActivateKey,
		response,
		CTMisc::num2string(bkid),
		CTMisc::num2string(kid),
		CTMisc::bin2hex(kd.toString()));
  if (!err.isOk())
    return CTError("2:RSACard::activateKey",err);

  return CTError();
}


int RSACard::findFreeBankDescription() {
  CTError err;
  string response;
  KeyLogStatus st;
  unsigned int i;

  st=readKeyLogStatus();
  // select EF_BNK
  err=selectFile(response, 0xa603);
  if (!err.isOk())
    throw CTError("RSACard::_findFreeBank",err);

  for (i=0; i<st.maxEntries; i++) {
    // read bank description
    err=readRecord(response, i+1, 1);
    if (!err.isOk())
      throw CTError("RSACard::_findFreeBank",err);
    if (response.length()!=1)
      throw CTError("RSACard::_findFreeBank",
		    k_CTERROR_INVALID,0,0,
		    "Bad size of response");
    if ((unsigned char)(response[0])==0xff)
      return i;
  } // for

  return -1;
}


RSACard::KeyLogStatus RSACard::readKeyLogStatus() {
  string data;
  string fcp;
  CTError err;
  KeyLogStatus s;

  // select EF_KEY_LOG
  err=selectFile(fcp, 0xa602);
  if (!err.isOk())
    throw CTError("RSACard::_readKeylogStatus",err);

  // read status byte
  err=readBinaryRaw(data, 0, 1);
  if (!err.isOk())
    throw CTError("RSACard::_readKeylogStatus",err);
  if (data.length()!=1)
    throw CTError("RSACard::_readKeylogStatus",
		  k_CTERROR_INVALID,0,0,
		  "Bad size of response");
  s.maxEntries=data[0]&7;
  s.oldENfree=(data[0]>>3)&1;
  s.entries=(data[0]>>4)&7;
  s.oldDSfree=(data[0]>>7)&1;
  return s;
}


CTError RSACard::writeKeyLogStatus(RSACard::KeyLogStatus s) {
  string data;
  string fcp;
  CTError err;
  char c;
  string response;

  if (s.maxEntries==0)
    return CTError("RSACard::_readKeylogStatus",
		   k_CTERROR_INVALID,0,0,
		   "Bad key log status");
  c=0;
  c|=(s.maxEntries)&7;
  c|=(s.oldENfree&1)<<3;
  c|=(s.entries&7)<<4;
  c|=(s.oldENfree&1)<<7;

  // select EF_KEY_LOG
  err=selectFile(response, 0xa602);
  if (!err.isOk())
    throw CTError("RSACard::_writeKeylogStatus",err);

  // write status byte
  data+=c;
  err=execCommand("update_binary",
		_cmdUpdateBinary,
                response,
		"0",
		CTMisc::bin2hex(data));
  if (!err.isOk())
    return CTError("RSACard::_writeKeylogStatus",err);
  if (data.length()!=1)
    return CTError("RSACard::_writeKeylogStatus",
		   k_CTERROR_INVALID,0,0,
		   "Bad size of response");
  return CTError();
}


bool RSACard::isBankKey(int kid) {
  return (kid>0x90);
}


bool RSACard::isSignKey(int kid) {
  return ((kid&15)<6);
}


int RSACard::getKeyId(int bank, bool pub, bool sign){
  int k;

  k=0x81;
  if (pub)
    k+=0x10;
  if (!sign)
    k+=5;
  k+=bank;
  return k;
}


string RSACard::readPublicKey(int kid){
  CTError err;
  int i;
  string response;
  string modulus;
  int kpos;

  // check status
  if (readKeyStatus(kid)==0x08)
    throw CTError("2:RSACard::_getPublicKey",
		  k_CTERROR_INVALID,0,0,
		  "Key not in use");
  // find key
  kpos=_findPublicKey(kid);
  if (kpos==-1)
    throw CTError("1:RSACard::_getPublicKey",
		  k_CTERROR_INVALID,0,0,
		  "Key not found");

  // read modulus length
  err=readBinaryRaw(response, (kpos*121)+1+14, 1);
  if (!err.isOk())
    throw CTError("3:RSACard::_getPublicKey",err);
  if (response.length()!=1)
    throw CTError("RSACard::_getPublicKey",
		  k_CTERROR_INVALID,0,0,
		  "Bad size of response");

  i=(unsigned int)(response[0]);
  if (i) {
    // read modulus
    err=readBinaryRaw(modulus, (kpos*121)+1+20, i);
    if (!err.isOk())
      throw CTError("4:RSACard::_getPublicKey",err);
    if ((int)modulus.length()!=i)
      throw CTError("5:RSACard::_getPublicKey",
		    k_CTERROR_INVALID,0,0,
		    "Bad size of response");

    // read algo byte
    err=readBinaryRaw(response, (kpos*121)+1+6, 1);
    if (!err.isOk())
      throw CTError("6:RSACard::_getPublicKey",err);
    if ((int)response.length()!=1)
      throw CTError("7:RSACard::_getPublicKey",
		    k_CTERROR_INVALID,0,0,
		    "Bad size of response");
    if (response[0]==0x09) {
      // modulus is in LSB first, so revert it
      response=modulus;
      modulus.erase();
      for (i=response.length()-1; i>=0; i--)
	modulus+=response[i];
    }
  }
  else
    modulus.erase();

  return modulus;
}


CTError RSACard::writePublicKey(int kid, const string &modulus){
  CTError err;
  int i;
  string response;
  int kpos;
  string newmod;
  string data;

  // check modulus size, exponent
  if (modulus.length()!=96)
    return CTError("1:RSACard::_putPublicKey",
		   k_CTERROR_INVALID,0,0,
		   "modulus must have 96 bytes");

  // find key
  kpos=_findPublicKey(kid);
  if (kpos==-1)
    return CTError("2:RSACard::_putPublicKey",
		   k_CTERROR_INVALID,0,0,
		   "Key not found",
		   CTMisc::num2string(kid));

  // read algo byte
  err=readBinaryRaw(response, (kpos*121)+1+6, 1);
  if (!err.isOk())
    throw CTError("3:RSACard::_getPublicKey",err);
  if ((int)response.length()!=1)
    throw CTError("4:RSACard::_putPublicKey",
		  k_CTERROR_INVALID,0,0,
		  "Bad size of response");
  if (response[0]==0x09) {
    newmod.erase();
    // modulus is in LSB first, so revert it
    response=modulus;
    for (i=response.length()-1; i>=0; i--)
      newmod+=response[i];
  }
  else
    newmod=modulus;

  // store modulus
  err=execCommand("update_binary",
		_cmdUpdateBinary,
		response,
		CTMisc::num2string((kpos*121)+1+20),
		CTMisc::bin2hex(newmod));
  if (!err.isOk())
    return CTError("5:RSACard::_putPublicKey",err);

  // store size of modulus
  data.erase();
  data+=(char)(modulus.length());
  err=execCommand("update_binary",
		_cmdUpdateBinary,
		response,
		CTMisc::num2string((kpos*121)+1+14),
		CTMisc::bin2hex(data));
  if (!err.isOk())
    return CTError("6:RSACard::_putPublicKey",err);

  // store size of free space
  data.erase();
  data+=(char)(0x60-modulus.length());
  err=execCommand("update_binary",
		_cmdUpdateBinary,
		response,
		CTMisc::num2string((kpos*121)+1+18),
		CTMisc::bin2hex(data));
  if (!err.isOk())
    return CTError("7:RSACard::_putPublicKey",err);

  return CTError();
}


CTError RSACard::_manageSE(int tmpl, int kids, int kidp, int ar){
  string response;
  CTError err;

  DBG_VERBOUS("tmpl=%02x, kids=%02x, kidp=%02x, ar=%02x",
	      tmpl, kids, kidp, ar);
  if (kids==-1)
    err=execCommand("select_pubkey",
		  _cmdSelectPubKey,
		  response,
		  CTMisc::num2string(tmpl),
		  CTMisc::num2string(kidp),
		  CTMisc::num2string(ar));
  else if (kidp==-1)
    err=execCommand("select_privkey",
		  _cmdSelectPrivKey,
		  response,
		  CTMisc::num2string(tmpl),
		  CTMisc::num2string(kids),
		  CTMisc::num2string(ar));
  else
    err=execCommand("manage_se",
		  _cmdManageSE,
		  response,
		  CTMisc::num2string(tmpl),
		  CTMisc::num2string(kids),
		  CTMisc::num2string(kidp),
		  CTMisc::num2string(ar));
  if (!err.isOk())
    return CTError("RSACard::manage_SE",err);

  return CTError();
}


string RSACard::sign(int kid, const string &data){
  CTError err;
  string response;

  DBG_VERBOUS("Signing with key %02x", kid);
  err=_manageSE(0xb6, kid, kid, 0x25);
  if (!err.isOk())
    throw CTError("RSACard::sign", err);

  err=execCommand("put_hash",
		_cmdPutHash,
		response,
		CTMisc::bin2hex(data));
  if (!err.isOk())
    throw CTError("RSACard::signData",err);

  err=execCommand("sign",
		_cmdSign,
		response);
  if (!err.isOk())
    throw CTError("RSACard::signData",err);

  return response;
}


CTError RSACard::verify(int kid, const string &data,
			const string &signature){
  CTError err;
  string response;

  DBG_VERBOUS("Verifying with key %02x", kid);
  err=_manageSE(0xb6, -1, kid, 0x25);
  if (!err.isOk())
    return CTError("RSACard::verify", err);

  err=execCommand("put_hash",
		_cmdPutHash,
		response,
		CTMisc::bin2hex(data));
  if (!err.isOk())
    return CTError("RSACard::verify",err);

  err=execCommand("verify",
		_cmdVerify,
		response,
		CTMisc::bin2hex(signature));
  if (!err.isOk())
    return CTError("RSACard::verify",err);

  return CTError();
}


string RSACard::getRandom(int s){
  CTError err;
  string response;

  // check for open bank
  err=execCommand("challenge",
		_cmdChallenge,
		response,
		CTMisc::num2string(s));
  if (!err.isOk())
    throw CTError("RSACard::getRandom",err);

  return response;
}


string RSACard::encrypt(int kid, const string &data){
  CTError err;
  string response;

  DBG_VERBOUS("Encrypting with key %02x", kid);
  err=_manageSE(0xb8, -1, kid, 0x03);
  if (!err.isOk())
    throw CTError("RSACard::encrypt", err);

  err=execCommand("encipher",
		_cmdEncipher,
		response,
		CTMisc::bin2hex(data));
  if (!err.isOk())
    throw CTError("RSACard::encrypt",err);
  return response;
}


string RSACard::decrypt(int kid, const string &data){
  CTError err;
  string response;

  DBG_VERBOUS("Decrypting with key %02x (length=%d, data=%s)", kid,
	      data.length(), CTMisc::bin2hex(data).c_str());

  err=_manageSE(0xb8, kid, kid, 0x003);
  if (!err.isOk())
    throw CTError("RSACard::decrypt", err);

  err=execCommand("decipher",
		_cmdDecipher,
		response,
		CTMisc::bin2hex(data));
  if (!err.isOk())
    throw CTError("RSACard::decrypt",err);

  return response;
}


RSACard::BankDescription RSACard::readBankDescription(int idx){
  CTError err;
  string response;

  err=selectFile(response, 0xa603);
  if (!err.isOk())
    throw CTError("RSACard::readBankDescription",err);

  err=readRecord(response,1+idx);
  if (!err.isOk())
    throw CTError("RSACard::readBankDescription",err);
  return BankDescription(response);
}


CTError RSACard::writeBankDescription(int idx, const BankDescription &bd){
  CTError err;
  string response;

  err=selectFile(response, 0xa603);
  if (!err.isOk())
    return CTError("RSACard::readBankDescription",err);

  err=execCommand("update_record",
		_cmdUpdateRecord,
		response,
		CTMisc::num2string(1+idx),
		CTMisc::bin2hex(bd.toString()));
  if (!err.isOk())
    return CTError("RSACard::writeBankDescription",err);
  return CTError();
}


CTError RSACard::deleteBankDescription(int idx){
  CTError err;
  string response;

  err=selectFile(response, 0xa603);
  if (!err.isOk())
    return CTError("RSACard::deleteBankDescription",err);

  err=execCommand("update_record",
		_cmdUpdateRecord,
		response,
		CTMisc::num2string(1+idx),
		string(154*2,'F'));
  if (!err.isOk())
    return CTError("RSACard::deleteBankDescription",err);
  return CTError();
}


CTError RSACard::_changePin(int pinid, const string &oldpin,
			    const string &newpin) {
  CTError err;
  string resp;

  err=execCommand("change_pin",
		_cmdChangePin,
		resp,
		CTMisc::num2string(pinid),
		CTMisc::bin2hex(oldpin),
		CTMisc::bin2hex(newpin));
  if (!err.isOk())
    return CTError("RSACard::_changePin",err);

  return CTError();
}


CTError RSACard::_changePin(int pinid) {
  CTError err;
  string resp;
  int to;

  to=timeout();
  setTimeout(60);
  err=execCommand("secure_change_pin",
		_cmdSecureChangePin,
		resp,
		CTMisc::num2string(pinid));
  setTimeout(to);
  if (!err.isOk())
    return CTError("RSACard::_changePin",err);
  return CTError();
}


CTError RSACard::_verifyPin(int pinid, const string &pin){
  CTError err;
  string resp;

  err=execCommand("verify_pin",
		_cmdVerifyPin,
		resp,
		CTMisc::num2string(pinid),
		CTMisc::bin2hex(pin));
  if (!err.isOk())
    return CTError("RSACard::_verifyPin",err);
  return CTError();
}


CTError RSACard::_verifyPin(int pinid) {
  CTError err;
  string resp;
  int to;

  to=timeout();
  setTimeout(60);
  err=execCommand("secure_verify_pin",
		_cmdSecureVerifyPin,
		resp,
		CTMisc::num2string(pinid));
  setTimeout(to);
  if (!err.isOk())
    return CTError("RSACard::_verifyPin",err);
  return CTError();
}


CTError RSACard::verifyPin(int kid, const string &pin){
  CTError err;

  err=_verifyPin(kid,
		 pin);
  if (!err.isOk())
    return CTError("RSACard::verifyPin",err);
  return CTError();
}


CTError RSACard::changePin(int kid, const string &oldpin,
			   const string &newpin) {
  CTError err;

  err=_changePin(kid,
		 oldpin,
		 newpin);
  if (!err.isOk())
    return CTError("RSACard::changePin",err);
  return CTError();
}


CTError RSACard::changePin(int kid) {
  CTError err;

  err=_changePin(kid);
  if (!err.isOk())
    return CTError("RSACard::changePin",err);
  return CTError();
}


CTError RSACard::verifyPin(int kid){
  CTError err;

  err=_verifyPin(kid);
  if (!err.isOk())
    return CTError("RSACard::verifyPin",err);
  return CTError();
}


CTError RSACard::pinStatus(int &maxerr, int &errleft){
  CTError err;
  string resp;

  err=execCommand("pin_status",
		  _cmdPinStatus,
		  resp);
  if (!err.isOk())
    return CTError("RSACard::pinStatus",err);
  if (resp.length()<1) {
    return CTError("RSACard::pinStatus",
		   k_CTERROR_INVALID,0,0,
		   "Response too short");
  }
  maxerr=((unsigned char)(resp[0])>>4)&0xf;
  errleft=((unsigned char)(resp[0]))&0xf;
  return CTError();
}




