/* 
 * $Id: ctkeditable.c,v 1.27 2000/08/14 19:03:29 cbond Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <string.h>

#include "ctk.h"
#include "ctkcolor.h"
#include "ctkutil.h"

gboolean ctk_editable_focus_in_event (CtkWidget* object, gpointer event, gpointer data);
gboolean ctk_editable_focus_out_event(CtkWidget* object, gpointer event, gpointer data);

gboolean ctk_editable_key_press   (CtkWidget* widget, CdkEventKey*    event, gpointer data);
gboolean ctk_editable_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data);
gboolean ctk_editable_noop        (CtkWidget* widget, CdkEventButton* event, gpointer data);

/* Initialize the Editable Widget */
void ctk_editable_init(CtkEditable* editable)
{
	ctk_widget_init(&editable->widget);		
	CTK_OBJECT(editable)->type = CtkTypeEditable;

	CTK_WIDGET(editable)->sensitive = TRUE;
	CTK_WIDGET(editable)->set_real_size = &ctk_editable_real_size;
	
	editable->editable = TRUE;
	
	editable->text       = NULL;
	editable->curpos     = 0;
	editable->text_start = NULL;
	editable->visible    = TRUE;
	editable->max_length = 0;
	
	ctk_signal_new("activate", CtkTypeEditable);
	ctk_signal_new("changed",  CtkTypeEditable);

	ctk_signal_connect(CTK_OBJECT(editable), "key_press_event", 
			   CTK_SIGNAL_FUNC(&ctk_editable_key_press), NULL);
	ctk_signal_connect(CTK_OBJECT(editable), "button_press_event", 
			   CTK_SIGNAL_FUNC(&ctk_editable_button_press), NULL);
	ctk_signal_connect(CTK_OBJECT(editable), "button_release_event", 
			   CTK_SIGNAL_FUNC(&ctk_editable_noop), NULL);
	ctk_signal_connect(CTK_OBJECT(editable), "ctk_drag", 
			   CTK_SIGNAL_FUNC(&ctk_editable_noop), NULL);
	ctk_signal_connect(CTK_OBJECT(editable), "focus_in_event",
	                   CTK_SIGNAL_FUNC(&ctk_editable_focus_in_event), NULL);
	ctk_signal_connect(CTK_OBJECT(editable), "focus_out_event",
	                   CTK_SIGNAL_FUNC(&ctk_editable_focus_out_event), NULL);
			   
}

void ctk_editable_set_editable(CtkEditable *editable, gboolean is_editable)
{
	editable->editable = is_editable;
}

/* Move cursor to the right */
void ctk_editable_key_right(CtkWidget *widget, gint width)
{
	CtkEditable *editable;
	
	if (!widget)
	    return;
	
	editable = CTK_EDITABLE(widget);
	
	if (editable->curpos < width)
	    editable->curpos += 1;
	else {
		editable->text_start += 1;
		ctk_draw_mark_changed(widget);
	}
}

/* Move cursor to the left */
void ctk_editable_key_left(CtkWidget *widget)
{
	CtkEditable *editable;

	if (!widget)
	    return;

	editable = CTK_EDITABLE(widget);
	
	if (editable->curpos > 0)
	    editable->curpos -= 1;
	else if (editable->text) {
		if (editable->text_start != editable->text) {
			editable->text_start -= 1;
			ctk_draw_mark_changed(widget);
		}
	}
}

/* Handle keyboard events for entry widget */
gboolean ctk_editable_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
	CtkEditable* editable;
	guint        t_len     = 0;
	gint         start_cur = 0;
	gchar*       ptr;
	gint         i;
	gint         res;
	gint         widget_width;
	gboolean     changed;
	
	changed = FALSE;
	
	if (!widget || !((CtkEditable *)widget)->editable)
	    return FALSE;
	
	editable = CTK_EDITABLE(widget);

	if (((CtkObject *)widget)->type == CtkTypeSpinButton)
	    widget_width = widget->width-3;
	else
	    widget_width = widget->width-1;
	
	/* Check for UP and Down Keys if widget == SpinButton so that
	 * we can handle them properly */

	/* RIGHT key Move cursor to the right */
	if ((event->keyval == AK_ARROW_RIGHT) &&
	    (editable->text) && (*(editable->text_start+editable->curpos)
				 != 0)) {
		ctk_editable_key_right(widget,widget_width);
	}
	else if (event->keyval == AK_ENTER) {
		ctk_signal_emit_by_name(CTK_OBJECT(widget), "activate");
	}
	/* LEFT KEY Move cursor to the left */
	else if (event->keyval == AK_ARROW_LEFT) {
		ctk_editable_key_left(widget);
	}
	/* CTRL-E or END Move cursor to the end of the string */
	else if (((event->keyval == AK_END) ||
		  (event->keyval == AK_CTRL('E'))) && editable->text) {
		t_len = strlen(editable->text);
		if (t_len <= widget_width)
		    editable->curpos = t_len;
		else {
			editable->curpos = widget_width;
			editable->text_start = editable->text+(t_len-(widget_width+1))+1;
			ctk_draw_mark_changed(widget);
		}
	}
	/* CTRL-A or HOME Move cursor to the beginning of the string */
	else if (((event->keyval == AK_HOME) ||
		 (event->keyval == AK_CTRL('A'))) && (editable->text)) {
		editable->curpos = 0;
		editable->text_start = editable->text;
		ctk_draw_mark_changed(widget);
	}
	/* Enter characters into the buffer */
	else if ((event->keyval >= ' ') && (event->keyval <= '~')) {
		if (!editable->text) {
			editable->text = g_malloc(2);
			*editable->text = event->keyval;
			*(editable->text+1) = 0;
			editable->text_start = editable->text;
			ctk_editable_key_right(widget,widget_width);
			ctk_size_mark_changed(widget);
		} else if (editable->text && (editable->max_length > strlen(editable->text) ||
					      editable->max_length == 0)) {
			start_cur = editable->text_start-editable->text;
			ctk_util_strinschr(&editable->text,
				      editable->text_start+editable->curpos,
				      event->keyval);
			editable->text_start = editable->text + start_cur;

			if (editable->text && *(editable->text_start+editable->curpos) != 0)
			    ctk_editable_key_right(widget,widget_width);
			ctk_size_mark_changed(widget);
		}		
		changed = TRUE;
	}
	/* CTRL-K Cut some text and put it into a clickboard */
	else if (event->keyval == AK_CTRL('K')) {
		if (CtkClipboard.text)
		    free(CtkClipboard.text);
		CtkClipboard.text = NULL;
		CtkClipboard.text_len = 0;
		
		if (editable->text_start) {
			CtkClipboard.text = g_strdup(editable->text_start+editable->curpos);
			CtkClipboard.text_len = strlen(CtkClipboard.text);
			ptr = editable->text_start+editable->curpos;
			while(*ptr != 0) {
				*ptr = 0;
				ptr += 1;
			}
			ctk_size_mark_changed(widget);
		}
		    
		changed = TRUE;
	}
	/* CTRL-Y Paste text back into the editable box */
	else if (event->keyval == AK_CTRL('Y') && CtkClipboard.text) {
		if (!editable->text_start)
		    editable->text_start = editable->text;
		
		start_cur = editable->text_start-editable->text;
		
		res = ctk_util_strinsstr(&editable->text,
				    editable->text_start+editable->curpos,
				    CtkClipboard.text);
		
		editable->text_start = editable->text + start_cur;
		
		for (i = 0; res > 0 && i < CtkClipboard.text_len; i += 1) {
			if (editable->curpos < widget_width)
			    editable->curpos += 1;
			else if (*editable->text_start != 0)
			    editable->text_start += 1;
		}
		
		ctk_size_mark_changed(widget);
		changed = TRUE;
	}
	/* CTRL-H or BACKSPACE */
	else if (((event->keyval == AK_BACKSPACE) || event->keyval == 272) &&
		 (editable->text) && *editable->text && editable->curpos > 0) {
		start_cur = editable->text_start-editable->text;	       
		ctk_util_strdelch(&editable->text,
				   editable->text_start+editable->curpos);
		editable->text_start = editable->text + start_cur;
		
		ctk_editable_key_left(widget);
		
		ctk_size_mark_changed(widget);
		changed = TRUE;
	}
	/* CTRL-D or DEL */
	else if ((event->keyval == AK_DELETE) ||
		 ((event->keyval == 275) && editable->text && *editable->text)){
		if (*(editable->text_start+editable->curpos) != 0) {
			start_cur = editable->text_start-editable->text;
			ctk_util_strdelch(&editable->text,
				     editable->text_start+editable->curpos+1);
			editable->text_start = editable->text + start_cur;
			
			ctk_size_mark_changed(widget);
		}
		changed = TRUE;
	}
	
	if (changed)
	{
		ctk_signal_emit_by_name(CTK_OBJECT(widget), "changed");
	}

	ctk_draw_mark_changed(widget);
	return TRUE;
}

gboolean ctk_editable_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
	CtkEditable *editable = CTK_EDITABLE(widget);
	gint widget_width;

	if (((CtkObject *)widget)->type == CtkTypeSpinButton)
	    widget_width = widget->width-3;
	else
	    widget_width = widget->width-1;

	editable->curpos = event->x - widget->col;
	if (editable->curpos > widget_width) editable->curpos = widget_width;
       	if (editable->curpos <= 0)           editable->curpos = 0;

        if (editable->text_start)
	{
	      if (editable->curpos > strlen(editable->text_start))
		    editable->curpos = strlen(editable->text_start);
	}
	else
	{
	      editable->curpos = 0;
	}

	ctk_draw_mark_changed(widget);
	return TRUE;
}

gboolean ctk_editable_noop(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      return TRUE;
}

/* Get Chars */
gchar* ctk_editable_get_chars(CtkEditable *editable, gint start_pos, gint end_pos)
{

      /* 
	 according to gtk, the string returned by this should be 
	 dynamically allocated, and must be freed by the caller. 
      */
        gchar *text = NULL;
	
	int i;
	gchar *sptr = NULL;
	gchar *eptr = NULL;

	if (!editable || !editable->text)
	    return NULL;

	for (i = 0; *(editable->text+i); i += 1) {
		if (i == start_pos)
		    sptr = editable->text+i;
		if (i == end_pos)
		    eptr = editable->text+i;
	}
	
	if (!sptr)
	    return NULL;
	if (!eptr)
	    eptr = editable->text+i;
	
	if (sptr > eptr)
	    return NULL;
	
	if (text) g_free(text);
	
	/* text[0] through text[eptr-sptr] are allocated */
	text = g_malloc((eptr-sptr)+1);
	/* text[0] through text[eptr-sptr-1] are assigned */
	strncpy(text,sptr,eptr-sptr);
	/* don't forget that \0 ! */
	text[eptr-sptr] = '\0';
	
	return(text);
}

/* Destroy Editable Data */
gpointer ctk_editable_destroy(CtkEditable* editable)
{
	if (editable->text)
	    free(editable->text);

	editable->text = NULL;
	
	return NULL;
}

gboolean ctk_editable_focus_in_event(CtkWidget* object, gpointer event, gpointer data)
{
	arr_show_cursor();
	return TRUE;
}

gboolean ctk_editable_focus_out_event(CtkWidget* object, gpointer event, gpointer data)
{
	arr_hide_cursor();
	return TRUE;
}

void ctk_editable_real_size(CtkWidget* widget)
{
	CtkWindow* window = ctk_window_get_focus_window();
	
	ctk_widget_real_size(widget);
	
	if (window && window->focus_widget == widget)
	{
		arr_set_cursor_pos(widget->col+CTK_EDITABLE(widget)->curpos,
			widget->row);
	}
}
