/* $Id: gic.h,v 1.7 2005/06/20 06:49:34 cegger Exp $
******************************************************************************

   Generic Input Configuration library for GII.

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#ifndef _GGI_GIC_H
#define _GGI_GIC_H


#include <ggi/gii.h>
#include <ggi/gic-defs.h>


__BEGIN_DECLS

#ifndef _BUILDING_LIBGIC
typedef void *gic_handle_t;
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ggi/errors.h>
#include <ggi/gii.h>
#include <ggi/gic_structs.h>
#include <ggi/gic-defs.h>

/* General Library control
 */
GGIGICAPIFUNC int gicInit(void);
GGIGICAPIFUNC int gicExit(void);

/* Open a new GIC instance.
 */
GGIGICAPIFUNC gic_handle_t gicOpen(const char *recognizers,...);
GGIGICAPIFUNC int gicClose(gic_handle_t hand);

/*
 * Miscellaneous stuff that can be associated to a GIC handle:
 */

/* Register a GII gii_input_t with the GIC instance. This is needed by some
 * recognizers to be able to store their configuration in a portable way.
 */
GGIGICAPIFUNC int gicInputRegister(gic_handle_t hand,gii_input_t inp);

/*
 * Recognizerdrivers. The work-monkeys. The recognizers given in gicOpen are automatically
 * open. But you might want to adjust that list.
 */

GGIGICAPIFUNC int gicRecognizerDriverRegister(
	gic_handle_t hand,gic_recognizerdriver *driver);
GGIGICAPIFUNC int gicRecognizerDriverUnregister(
	gic_handle_t hand,gic_recognizerdriver *driver);
GGIGICAPIFUNC gic_recognizerdriver *gicRecognizerDriverLookup(
	gic_handle_t hand,const char *name);

/*
 * Stage 5: Recognizers
 * the helper libraries that decide whether some event is matched.
 */

GGIGICAPIFUNC int gicRecognizerWrite(
	gic_handle_t hand,gic_recognizer *recognizer, FILE *file);
GGIGICAPIFUNC int gicRecognizerTrain(
	gic_handle_t hand,gic_recognizer **recognizerlist,gii_event *event);

#define	gicRecognizerTrainStart(hand,x) gicRecognizerTrain(hand,x,NULL)
#define	gicRecognizerTrainStop(hand,x)  gicRecognizerTrain(hand,x,NULL)

GGIGICAPIFUNC int gicRecognizerTrainAdd(gic_handle_t hand,
	gic_recognizer **recognizerlist,gic_recognizer *newone);
GGIGICAPIFUNC int gicRecognizerTrainMove(gic_handle_t hand,
	gic_recognizer **recognizerlist,gic_recognizer *newone);
GGIGICAPIFUNC int gicRecognizerFindConflict(gic_handle_t hand,
	gic_recognizer *recognizer, gic_recognizer *compare_to);
GGIGICAPIFUNC int gicRecognizerGetName(gic_handle_t hand,
	gic_recognizer *recognizer, char *string, size_t maxlen);
GGIGICAPIFUNC int gicRecognizerGetOpposite(gic_handle_t hand,
	gic_recognizer *recognizer,gic_recognizer **opposite);

/*
 * Stage 5B: Actions
 * the functions called, when features get active
 */
GGIGICAPIFUNC int gicActionWrite(gic_handle_t hand,
	gic_actionlist *aclist,FILE *where);
GGIGICAPIFUNC int gicActionMapActions(gic_handle_t hand,
	gic_actionlist *aclist, gic_actionlist *actions);

/*
 * Stage 4: Features.
 * Features are individual "moves" that control something. They can have multiple
 * actions and recognizers attached.
 */

GGIGICAPIFUNC gic_feature *gicFeatureAllocate(gic_handle_t hand,
	const char *name,const char *shortname);
GGIGICAPIFUNC void gicFeatureFree            (gic_handle_t hand,
	gic_feature *feature);

GGIGICAPIFUNC int gicFeatureAttachRecognizerDriver(gic_handle_t hand,
	gic_feature *feature, gic_recognizerdriver *driver,void *privdata);
GGIGICAPIFUNC int gicFeatureAttachRecognizer(gic_handle_t hand,
	gic_feature *feature, gic_recognizer *recognizer);
GGIGICAPIFUNC int gicFeatureDetachRecognizer(gic_handle_t hand,
	gic_feature *feature, gic_recognizer *recognizer);
GGIGICAPIFUNC gic_recognizer *gicFeatureGetRecognizer(gic_handle_t hand,
	gic_feature *feature, int number);
GGIGICAPIFUNC int gicFeatureNumRecognizers(gic_handle_t hand,
	gic_feature *feature);
GGIGICAPIFUNC int gicFeatureGetName(gic_handle_t hand,
	gic_feature *feature, char *string, size_t maxlen);

GGIGICAPIFUNC int gicFeatureAttachAction(gic_handle_t hand,
	gic_feature *feature,
	void (*action)(gic_handle_t hand, gic_actionlist *action,
		gic_feature *feature,gic_state newstate,
		gic_flag flag,int recnum),
	void *privdata, const char *name);
GGIGICAPIFUNC int gicFeatureDetachAction(gic_handle_t hand,
	gic_feature *feature,
	void (*action)(gic_handle_t hand, gic_actionlist *action,
		gic_feature *feature,gic_state newstate,
		gic_flag flag,int recnum),
	void *privdata, char *name);

GGIGICAPIFUNC int gicFeatureWrite(gic_handle_t hand,
	gic_feature *feature, FILE *file);
GGIGICAPIFUNC gic_feature *gicFeatureRead(gic_handle_t hand,
	FILE *file);

GGIGICAPIFUNC int gicFeatureMapActions(gic_handle_t hand,
	gic_feature *feature, gic_actionlist *actions);
GGIGICAPIFUNC int gicFeatureFindConflict(gic_handle_t hand,
	gic_feature *feature, gic_recognizer *rec,
	gic_recognizer **start_and_return);

/* "Fake" a feature activation. Pretty much like a SendEvent.
 */
GGIGICAPIFUNC int gicFeatureActivate(gic_handle_t hand,
	gic_feature *feature,gic_state newstate,gic_flag flag,int recnum);

/* Check for a feature activation. Tries all recognizers. RC is number of 
 * activations, i.e. zero for an unmatched event.
 */
GGIGICAPIFUNC int gicFeatureHandleEvent(gic_handle_t hand,
	gic_feature *feature, gii_event *event);

/*
 * Stage 3: Controls.
 * Contexts contain all "Features" that make up a given set like all things
 * you might need to do for navigating a menu.
 */

GGIGICAPIFUNC gic_control *gicControlAllocate(gic_handle_t hand,
	char *name,char *shortname);
GGIGICAPIFUNC void gicControlFree(gic_handle_t hand, gic_control *control);
GGIGICAPIFUNC int gicControlAttachFeature(gic_handle_t hand,
	gic_control *control,gic_feature *feature);
GGIGICAPIFUNC int gicControlDetachFeature(gic_handle_t hand,
	gic_control *control,gic_feature *feature);
GGIGICAPIFUNC gic_feature *gicControlLookupFeature(gic_handle_t hand,
	gic_control *context,const char *name);
GGIGICAPIFUNC gic_feature *gicControlGetFeature(gic_handle_t hand,
	gic_control *context,int number);
GGIGICAPIFUNC int gicControlNumFeatures(gic_handle_t hand,
	gic_control *context);
GGIGICAPIFUNC int gicControlGetName(gic_handle_t hand,
	gic_control *control, char *string, size_t maxlen);
GGIGICAPIFUNC int gicControlWrite(gic_handle_t hand,
	gic_control *control,FILE *file);
GGIGICAPIFUNC gic_control *gicControlRead(gic_handle_t hand, FILE *file);
GGIGICAPIFUNC int gicControlMapActions(gic_handle_t hand,
	gic_control *control,gic_actionlist *actions);
GGIGICAPIFUNC int gicControlFindConflict(gic_handle_t hand,
	gic_control *control,gic_recognizer *feature,
	gic_recognizer **start_and_return,gic_feature **optional);

/* Check for a control activation. Loops through all features.
 * RC is number of total activations, i.e. zero for an unmatched event.
 */
GGIGICAPIFUNC int gicControlHandleEvent(gic_handle_t hand,
	gic_control *control, gii_event *event);

/*
 * Stage 2: Contexts.
 * Contexts describe _all_ "Controls" that are applicable in a given state of
 * an application.
 */

GGIGICAPIFUNC gic_context *gicContextAllocate(gic_handle_t hand,char *name);
GGIGICAPIFUNC void gicContextFree(gic_handle_t hand,gic_context *context);
GGIGICAPIFUNC int gicContextAttachControl(gic_handle_t hand,
	gic_context *context,gic_control *control);
GGIGICAPIFUNC int gicContextDetachControl(gic_handle_t hand,
	gic_context *context,gic_control *control);
GGIGICAPIFUNC gic_control *gicContextLookupControl(gic_handle_t hand,
	gic_context *context,const char *name);
GGIGICAPIFUNC gic_control *gicContextGetControl(gic_handle_t hand,
	gic_context *context,int number);
GGIGICAPIFUNC int gicContextNumControls(gic_handle_t hand,
	gic_context *context);
GGIGICAPIFUNC int gicContextGetName(gic_handle_t hand,
	gic_context *context, char *string, size_t maxlen);
GGIGICAPIFUNC int gicContextWrite(gic_handle_t hand,
	gic_context *context,FILE *file);
GGIGICAPIFUNC gic_context *gicContextRead(gic_handle_t hand,FILE *file);
GGIGICAPIFUNC int gicContextMapActions(gic_handle_t hand,
	gic_context *context,gic_actionlist *actions);
GGIGICAPIFUNC int gicContextFindConflict(gic_handle_t hand,
	gic_context *context,gic_recognizer *rec,
	gic_recognizer **start_and_return,gic_feature **optional);

/* Check for a context activation. Loops through all controls and features.
 * RC is number of total activations, i.e. zero for an unmatched event.
 */
GGIGICAPIFUNC int gicContextHandleEvent(gic_handle_t hand,
	gic_context *context, gii_event *event);

/*
 * Stage 1: Heads.
 * This is a convenience structure that can be used to group contexts.
 * There are some convenience functions that will load/restore complete heads
 * allowing to save configuration with a single call.
 */

GGIGICAPIFUNC gic_head    *gicHeadAllocate     (gic_handle_t hand,
	char *name);
GGIGICAPIFUNC void         gicHeadFree         (gic_handle_t hand,
	gic_head *head);
GGIGICAPIFUNC int          gicHeadAttachContext(gic_handle_t hand,
	gic_head *head,gic_context *context);
GGIGICAPIFUNC int          gicHeadDetachContext(gic_handle_t hand,
	gic_head *head,gic_context *context);
GGIGICAPIFUNC gic_context *gicHeadLookupContext(gic_handle_t hand,
	gic_head *head,const char *name);
GGIGICAPIFUNC gic_context *gicHeadGetContext   (gic_handle_t hand,
	gic_head *head,int number);
GGIGICAPIFUNC int          gicHeadNumContexts  (gic_handle_t hand,
	gic_head *head);
GGIGICAPIFUNC int          gicHeadWrite        (gic_handle_t hand,
	gic_head *head,FILE *file);
GGIGICAPIFUNC gic_head    *gicHeadRead         (gic_handle_t hand,
	FILE *file);
GGIGICAPIFUNC int          gicHeadMapActions   (gic_handle_t hand,
	gic_head *head,gic_actionlist *actions);

/* 
 * Miscellaneous:
 */

/* This function is NOT part of LibGIC. It is autogenerated by gic2c and used
 * for embedding a default configuration in the program executable itself.
 * You _have_ to generate the source for it with gic2c and link your program 
 * with it.
 */
size_t gicWriteDefaultConfig(FILE *outfile);

__END_DECLS

#endif /* _GGI_GIC_H */
