/**
 * @file libgnurdf/description.h Description functions
 *
 * $Id: description.h,v 1.3 2001/08/23 08:51:22 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBGNURDF_DESCRIPTION_H_
#define _LIBGNURDF_DESCRIPTION_H_

#include <libgnurdf/types.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Returns the first description in the schema.
 *
 * @param schema The schema containing the descriptions.
 *
 * @return The first description in the schema.
 *
 * @see rdfNextDescription()
 */
RdfDescription *rdfFirstDescription(RdfSchema *schema);

/**
 * Returns the next description in the schema.
 *
 * @param desc The previous description.
 *
 * @return The next description.
 *
 * @see rdfFirstDescription()
 */
RdfDescription *rdfNextDescription(RdfDescription *desc);

/**
 * Adds a description to the schema.
 *
 * @param schema The schema to add the description to.
 * @param id     The id of the new description.
 * @param about  The about URI of the new description.
 *
 * @return The new description.
 */
RdfDescription *rdfAddDescription(RdfSchema *schema,
								  const char *id,
								  const char *about);

/**
 * Returns the description with the specified about URI.
 *
 * @param schema The schema containing the descriptions.
 * @param about  The about URI to search for.
 *
 * @return The description, or NULL if not found.
 */
RdfDescription *rdfGetDescriptionFromAbout(RdfSchema *schema,
										   const char *about);

/**
 * Returns the description with the specified ID.
 *
 * @param schema The schema containing the descriptions.
 * @param id     The ID to search for.
 *
 * @return The description, or NULL if not found.
 */
RdfDescription *rdfGetDescriptionFromId(RdfSchema *schema, const char *id);

/**
 * Sets a description's ID.
 *
 * @param desc The description.
 * @param id   The new ID.
 */
void rdfSetDescriptionId(RdfDescription *desc, const char *id);

/**
 * Returns a description's ID.
 *
 * Note: The returned ID must be freed.
 *
 * @param desc The description containing the ID.
 *
 * @return The ID of the description.
 */
char *rdfGetDescriptionId(RdfDescription *desc);

/**
 * Sets a description's About URI.
 *
 * If this finds an \c About or \c about attributes
 * (in that order), it will set that attribute. Otherwise,
 * it will set the \c About attribute.
 *
 * @param desc  The description.
 * @param about The new About URI.
 */
void rdfSetDescriptionAbout(RdfDescription *desc, const char *about);

/**
 * Returns a description's About URI.
 *
 * Note: The returned About URI must be freed.
 *
 * @param schema The document schema.
 * @param desc   The description containing the About URI.
 *
 * @return The About URI of the description.
 */
char *rdfGetDescriptionAbout(RdfDescription *desc);

#ifdef __cplusplus
}
#endif

#endif /* _LIBGNURDF_DESCRIPTION_H_ */

