/*
 * @(#)IMethodCodeUTestI.java
 *
 * Copyright (C) 2002,2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2;

import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.module.DefaultAnalysisMetaData;
import net.sourceforge.groboutils.junit.v1.iftc.ImplFactory;
import net.sourceforge.groboutils.junit.v1.iftc.InterfaceTestCase;
import net.sourceforge.groboutils.junit.v1.iftc.InterfaceTestSuite;

import org.apache.bcel.classfile.Method;


/**
 * Tests the IMethodCode interface.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:27 $
 * @since     December 28, 2002
 */
public class IMethodCodeUTestI extends InterfaceTestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = IMethodCodeUTestI.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public IMethodCodeUTestI( String name, ImplFactory f )
    {
        super( name, IMethodCode.class, f );
    }

    
    public IMethodCode createIMethodCode()
    {
        return (IMethodCode)createImplObject();
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testGetOriginalMethod1()
    {
        IMethodCode mc = createIMethodCode();
        Method m = mc.getOriginalMethod();
        assertNotNull(
            "Must not return null original method.",
            m );
    }
    
    
    public void testGetMethod1()
    {
        IMethodCode mc = createIMethodCode();
        assertNotNull(
            "Method name is null.",
            mc.getMethodName() );
    }
    
    
    public void testGetClassName1()
    {
        IMethodCode mc = createIMethodCode();
        assertNotNull(
            "Class name is null.",
            mc.getClassName() );
    }
    
    
    public void testGetInstructionCount1()
    {
        IMethodCode mc = createIMethodCode();
        int count = mc.getInstructionCount();
        assertTrue(
            "Count is not valid",
            count >= 0 );
    }
    
    
    public void testMethodName1()
    {
        IMethodCode mc = createIMethodCode();
        Method m = mc.getOriginalMethod();
        String name = mc.getMethodName();
        assertEquals(
            "Method name doesn't match returned method's name",
            m.getName()+m.getSignature(),
            name
            );
    }
    
    
    public void testInstructionConsistency1()
    {
        IMethodCode mc = createIMethodCode();
        int count = mc.getInstructionCount();
        for (int i = 0; i < count; ++i)
        {
            assertNotNull(
                "Null instruction at position "+i+".",
                mc.getInstructionAt( i ) );
        }
    }
    
    
    public void testInstructionConsistency2()
    {
        IMethodCode mc = createIMethodCode();
        int count = mc.getInstructionCount();
        try
        {
            mc.getInstructionAt( -1 );
            fail( "Did not throw excption w/ -1 index." );
        }
        catch (IndexOutOfBoundsException e)
        {
            // test exception
        }
        
        try
        {
            mc.getInstructionAt( count );
            fail( "Did not throw exception w/ "+count+" index." );
        }
        catch (IndexOutOfBoundsException e)
        {
            // test exception
        }
    }
    
    
    public void testMarkInstruction1()
    {
        IMethodCode mc = createIMethodCode();
        int count = mc.getInstructionCount();
        if (count <= 0)
        {
            DOC.getLog().warn( "No instructions to mark for '"+mc+"'." );
            return;
        }
        
        // something to test
        //   note: we can mark the last index + one over it!
        for (int i = 0; i <= count; ++i)
        {
            IAnalysisMetaData meta = new DefaultAnalysisMetaData(
                "", "", (byte)0 );
            mc.markInstruction( i, meta );
        }
    }
    
    
    public void testMarkInstruction2()
    {
        IMethodCode mc = createIMethodCode();
        
        try
        {
            IAnalysisMetaData meta = new DefaultAnalysisMetaData(
                "", "", (byte)0 );
            mc.markInstruction( -1, meta );
            fail( "Did not throw IndexOutOfBoundsException." );
        }
        catch (IndexOutOfBoundsException e)
        {
            // test exception
        }
    }
    
    
    public void testMarkInstruction4()
    {
        IMethodCode mc = createIMethodCode();
        int count = mc.getInstructionCount();
        
        try
        {
            IAnalysisMetaData meta = new DefaultAnalysisMetaData(
                "", "", (byte)0 );
            mc.markInstruction( count+1, meta );
            fail( "Did not throw IndexOutOfBoundsException." );
        }
        catch (IndexOutOfBoundsException e)
        {
            // test exception
        }
    }
    
    
    public void testMarkInstruction5()
    {
        IMethodCode mc = createIMethodCode();
        int count = mc.getInstructionCount();
        
        // even if the instruction count is == 0, we can still mark it,
        // hence this will be a valid test for 'null' metadata.
        
        // something to test
        try
        {
            mc.markInstruction( 0, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static InterfaceTestSuite suite()
    {
        InterfaceTestSuite suite = new InterfaceTestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

