/******************************************************************************************************************************************
 ctreeview.c
******************************************************************************************************************************************/

#include "ctreeview.h"

#define N_FIELDS 		3	// N_FIELDS per column / 1 displayed that corresponds to the item field value (the first attribute)

#define TEXT_FIELD 		0	// CItemFieldValueString
#define ACTIVE_FIELD 		0	// CItemFieldValueBoolean
#define PROGRESS_FIELD		0	// CItemFieldValueProgress
#define PIXBUF_FIELD		0	// CItemFieldValuePixbuf

#define EDITABLE_FIELD 		2	// CItemFieldValueString
#define ACTIVATABLE_FIELD 	2	// CItemFieldValueBoolean
#define CAPTION_FIELD		2	// CItemFieldValueProgress

#define VISIBLE_FIELD 		1	// all

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CTreeViewItemFieldValueAPIListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener handling on api item field state modification
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeViewItemFieldValueAPIListener::OnStateChange (CObject *inItemField)
{
	// retreive the item field value instance
	CItemFieldValue *inItemFieldValue = static_cast <CItemFieldValue *> (inItemField);

	// retreive the generic tree view item instance
	CTreeViewItem *inTreeViewItem = static_cast <CTreeViewItem *> (inItemFieldValue -> GetOwner());

	// set the tree view item its new item fields values and states
	if (inTreeViewItem != NULL) inTreeViewItem -> SetItemFieldValues (CItemFieldValues (inTreeViewItem -> GetItemFieldValues()));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CTreeViewItem);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeViewItem::CTreeViewItem 	 (CTreeView *inOwner, const CItemFieldValues &inFieldValues, const CObjectListener *inListener)
	      :CControl	     	 (NULL, inListener),
	       m_ItemFieldValues (inFieldValues),
	       m_GtkTreeIter	 ()
{
	// set the tree view item values and create the associated gtk tree iter of the tree view
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeViewItem::CTreeViewItem	 (CTreeViewItem *inOwner, const CItemFieldValues &inFieldValues, const CObjectListener *inListener)
	      :CControl	    	 (NULL, inListener),
	       m_ItemFieldValues (inFieldValues),
	       m_GtkTreeIter	 ()
{
	// set the tree view item values and create the associated gtk tree iter of the tree view
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeViewItem::~CTreeViewItem ()
{
	// lock the spine
	CComponent::Lock();

	// get the whole sub items as we cannot wait for the ccomponent destructor to do this job until this tree iter will be null
	// when the sub ones will try to remove their values from...
	CComponents inSubItems (GetSubComponents (__metaclass(CTreeViewItem)) - this);

	// delete each one : CTreeViewItem::~CTreeViewItem on each
	for (size_t i=inSubItems.GetLength(); i>0; i--) delete *inSubItems[i-1];

	// get the gtkol tree view owner of this instance
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// remove the item value from the gtk widget tree view
	if (inTreeView != NULL && inTreeView -> GetGtkWidget() != NULL)
		::gtk_tree_store_remove (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(inTreeView->GetGtkWidget()))), 
					 &m_GtkTreeIter);

	// delete the handled tree view item instances values and their associated api listener
	for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) delete *m_ItemFieldValues[i-1];

	// unlock the spine
	CComponent::Unlock();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CTreeViewItem::OwnerMustBe () const
{
	return __metaclasses(CTreeView) + __metaclass(CTreeViewItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected children type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CTreeViewItem::ChildMustBe () const
{
	return __metaclasses(CTreeViewItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// owner affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CTreeViewItem::SetOwner (CComponent *inOwner, const SInt16 inIndex)
{
	// owner type check !
	if (!CheckSetOwner (inOwner)) return false;

	// new tree view owner
	CTreeView *newTreeView = inOwner != NULL ? inOwner -> ClassIs (__metaclass(CTreeView)) ? 
				 	static_cast <CTreeView *> (inOwner) : 
				 	static_cast <CTreeView *> (inOwner -> GetOwner (__metaclass(CTreeView))) : NULL;

	// old owner (tree view or tree view item)
	CComponent *oldOwner = GetOwner ();

	// old tree view owner (that may differ from the new one)
	CTreeView *oldTreeView = oldOwner != NULL ? oldOwner -> ClassIs (__metaclass(CTreeView)) ? 
					static_cast <CTreeView *> (oldOwner) : 
				 	static_cast <CTreeView *> (oldOwner -> GetOwner (__metaclass(CTreeView))) : NULL;

	// if the tree views differ...
	if (oldTreeView != NULL && newTreeView != NULL && oldTreeView != newTreeView)
	{
		// get the repsective models
		CMetaClasses inOldModel (oldTreeView -> GetModel());
		CMetaClasses inNewModel (newTreeView -> GetModel());

		// remove any reference to a pack field specification
		for (size_t i=0; i<inOldModel.GetLength(); i++)
			if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *inOldModel[i]))
				inOldModel.Delete (i--, 1);
		for (size_t i=0; i<inNewModel.GetLength(); i++)
			if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *inNewModel[i]))
				inNewModel.Delete (i--, 1);

		// model conformity check
		if (inOldModel.GetLength() != inNewModel.GetLength()) return false; 
		for (size_t i=inNewModel.GetLength(); i>0; i--)
			if (*inNewModel[i-1] != *inOldModel[i-1]) return false;
	}

	// set the new hierarchical component owner
	if (!CControl::SetOwner (inOwner, inIndex)) return false;

	// if there were an old tree view widget for this tree view item instance
	if (oldTreeView != NULL)
	
		// remove the item from its old owner
		::gtk_tree_store_remove (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(oldTreeView->GetGtkWidget()))), 
					 &m_GtkTreeIter);

	// if the new owner is a tree view
	if (inOwner != NULL && inOwner -> ClassIs (__metaclass(CTreeView)))
	{
		// append requested
		if (inIndex < 0)

			// append this tree view item as a new root of the tree view widget
			::gtk_tree_store_append (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(newTreeView->GetGtkWidget()))), 
						 &m_GtkTreeIter, NULL);

		// insert requested (index range checked by gtk itself)
		else
			
			// insert this tree view item as a new root of the tree view widget
			::gtk_tree_store_insert (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(newTreeView->GetGtkWidget()))), 
						 &m_GtkTreeIter, NULL, inIndex);
	}
	// if the new owner is a tree view item
	else if (inOwner != NULL && inOwner -> ClassIs (__metaclass(CTreeViewItem)))
	{
		// append requested
		if (inIndex < 0)
		
			// append this tree view item as a new child of the specified one
			::gtk_tree_store_append (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(newTreeView->GetGtkWidget()))),
				 		 &m_GtkTreeIter, &(static_cast <CTreeViewItem *> (inOwner) -> m_GtkTreeIter));

		// insert requested
		else

			// insert this tree view item as a new child of the specified one
			::gtk_tree_store_insert (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(newTreeView->GetGtkWidget()))),
				 		 &m_GtkTreeIter, &(static_cast <CTreeViewItem *> (inOwner) -> m_GtkTreeIter), inIndex);
	}

	// item field values affectation
	if (!SetItemFieldValues (CItemFieldValues (m_ItemFieldValues))) return false;

	// if invoked, the gtk_tree_store_remove has removed the whole descendant hierarchy that has to be created again
	if (oldOwner != NULL && newTreeView != NULL)
	{
		// potential sub components items of the current tree view item, do not consider this instance as the field values have 
		// already been affected
		CComponents inSubComponents (GetSubComponents (__metaclass(CTreeViewItem)) - this);

		// go through the sub components items
		for (size_t i=inSubComponents.GetLength(), j=0; i>0; i--, j++)
		{
			// the tree view item
			CTreeViewItem *inTreeViewItem = static_cast <CTreeViewItem *> (*inSubComponents[j]);

			// its owner
			inOwner = inTreeViewItem -> GetOwner ();

			// owner type analyse
			if (inOwner -> ClassIs (__metaclass(CTreeView)))
				::gtk_tree_store_append 
					(GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(newTreeView->GetGtkWidget()))), 
					&(inTreeViewItem->m_GtkTreeIter), NULL);
			else if (inOwner -> ClassIs (__metaclass(CTreeViewItem)))
				::gtk_tree_store_append 
					(GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(newTreeView->GetGtkWidget()))),
					&(inTreeViewItem->m_GtkTreeIter), &(static_cast <CTreeViewItem *> (inOwner) -> m_GtkTreeIter));

			// set the field values
			if (!inTreeViewItem -> SetItemFieldValues (CItemFieldValues (inTreeViewItem -> m_ItemFieldValues))) return false;
		}
	}

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get the tree view item bounds
//-----------------------------------------------------------------------------------------------------------------------------------------
TBounds CTreeViewItem::GetBounds () const
{
	// get the gtkol tree view owner
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// the gtk tree view widget
	GtkTreeView *inGtkTreeView = NULL;

	// pointer check
	if (inTreeView == NULL || (inGtkTreeView = GTK_TREE_VIEW(inTreeView->GetGtkWidget())) == NULL) return TBounds();

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_tree_view_get_model(inGtkTreeView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));
	// pointer check
	if (inGtkTreePath == NULL) return TBounds();

	// get the tree view item height
	GdkRectangle inGdkRectangle; ::gtk_tree_view_get_background_area (inGtkTreeView, inGtkTreePath, NULL, &inGdkRectangle);

	// release the path pointer
	::gtk_tree_path_free (inGtkTreePath);

	// ok, return the tree view item bounds
	return TBounds (0, inGdkRectangle.y, inTreeView -> GetBounds().w, inGdkRectangle.height);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// tree view item pixbuf representation
//-----------------------------------------------------------------------------------------------------------------------------------------
CPixbuf * CTreeViewItem::GetControlPixbuf () const
{
	// get the tree view owner
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// the gtk tree view
	GtkTreeView *inGtkTreeView = NULL;

	// pointer check
	if (inTreeView == NULL || (inGtkTreeView = GTK_TREE_VIEW(inTreeView->GetGtkWidget())) == NULL) return NULL;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_tree_view_get_model(inGtkTreeView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));

	// create the associated pixmap
	GdkPixmap *inGdkPixmap = ::gtk_tree_view_create_row_drag_icon (inGtkTreeView, inGtkTreePath);

	// get the pixmap size
	gint inWidth, inHeight; ::gdk_drawable_get_size (inGdkPixmap, &inWidth, &inHeight);

	// create the pixbuf instance
	CPixbuf *outPixbuf = new CPixbuf (inGdkPixmap, PIXELFORMAT_32, TBounds(0,0,inWidth,inHeight));

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);

	// relase the pixmap pointer
	::gdk_pixmap_unref (inGdkPixmap);

	// be paranoïd...
	if (outPixbuf -> GetPixelFormat() != PIXELFORMAT_32)
	{
		// delete it !
		delete outPixbuf;

		// stop
		return NULL;
	}

	// keep dark pixels (that is, darker than gray)
	UInt8 *Line = outPixbuf -> GetBaseAddress();
	for (size_t i=inHeight; i>0; i--)
	{
		UInt32 *Pixel = reinterpret_cast <UInt32 *> (Line);
		for (size_t j=inWidth; j>0; j--, Pixel++)
			if ((*Pixel&0x00FFFFFF) <= 0x007F7F7F) *Pixel = 0xFF7F7F7F; else *Pixel = 0L;
		Line += outPixbuf -> GetRowBytes();
	}

	// set the default border representation of the dragged control (opaque gray borders), direct draw on
	// the pixbuf's buffer (see static declaration of CPixbuf::DrawLine and CPixbuf::DrawRectangle function for details in cgraphics.h)
	CPixbuf::DrawRectangle (TBounds (0L, 0L, inWidth-1L, inHeight-1L), outPixbuf->GetBaseAddress(), PIXELFORMAT_32, 
				outPixbuf->GetRowBytes(), 0xFF7F7F7F, 1.0f);

	// ok
	return outPixbuf;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// GtkTreeIter access
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkTreeIter * CTreeViewItem::GetGtkTreeIter () const
{
	return const_cast <GtkTreeIter *> (&m_GtkTreeIter);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtkol tree view item search
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeViewItem *	CTreeViewItem::GetTreeViewItem (const GtkTreeIter *inGtkTreeIter)
{
	// pointer check
	if (inGtkTreeIter == NULL) return NULL;

	// get the whole tree view item instances
	CComponents inComponents (CComponent::GetComponents (__metaclass(CTreeViewItem)));

	// go through
	for (size_t i=inComponents.GetLength(), j=0; i>0; i--, j++)
		if (static_cast <CTreeViewItem *> (*inComponents[j]) -> m_GtkTreeIter.user_data == inGtkTreeIter->user_data)
			return static_cast <CTreeViewItem *> (*inComponents[j]);

	// not found
	return NULL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// field values affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CTreeViewItem::SetItemFieldValues (const CItemFieldValues &inItemFieldValues)
{
	// get the tree view widget this item is under
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// delete the old values if any
	if (&inItemFieldValues != &m_ItemFieldValues)
		for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) 
			if (*inItemFieldValues[i-1] != *m_ItemFieldValues[i-1]) 
				delete *m_ItemFieldValues[i-1];

	// pointers local copy
	m_ItemFieldValues = inItemFieldValues;

	// the tree view corresponding representation model
	CMetaClasses inModel;

	// preliminary checks
	if (inTreeView != NULL)
	{
		// get the tree view corresponding representation model
		inModel = inTreeView -> GetModel();

		// remove any reference to a pack specification
		for (size_t i=0; i<inModel.GetLength(); i++)
			if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *inModel[i]))
				inModel.Delete (i--, 1);

		// empty or part empty buffer list check
		if (m_ItemFieldValues.GetLength() < inModel.GetLength())
		{
			// go through the model and set default field value instances
			for (size_t i=m_ItemFieldValues.GetLength(); i<inModel.GetLength(); i++)
			{
				if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueString), *inModel[i]))
					m_ItemFieldValues += new CItemFieldValueString ();
				if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueBoolean), *inModel[i]))
					m_ItemFieldValues += new CItemFieldValueBoolean ();
				if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePixbuf), *inModel[i]))
					m_ItemFieldValues += new CItemFieldValuePixbuf (new CPixbuf("null"));
				if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueCombo), *inModel[i]))
					// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
					// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
					// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
					;
				if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueProgress), *inModel[i]))
					m_ItemFieldValues += new CItemFieldValueProgress ();	
			}
		}
	}

	// go through the field values instances and set the friend protected attributes we are supposed to handle
	for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--)
	{ 
		// our tree view item handles the field value instance
		if ((*m_ItemFieldValues[i-1]) -> m_Owner != this) 
			(*m_ItemFieldValues[i-1]) -> m_Owner = this;

		// set the item field value api listener so that when the programmer performs modifications on that field, the gtkol engine 
		// can handle gui modifications accordinaly
		if ((*m_ItemFieldValues[i-1]) -> m_Listener == NULL) 
			(*m_ItemFieldValues[i-1]) -> AssignListener (new CTreeViewItemFieldValueAPIListener());
	}
	
	// pointer check, memorization done anyway
	if (inTreeView == NULL) return true;

	// go through the model and set the associated values
	for (size_t i=inModel.GetLength(), j=0; i>0 && j<m_ItemFieldValues.GetLength(); i--, j++)
	{
		// get the field value of the incoming values
		CItemFieldValue *inItemFieldValue (*m_ItemFieldValues[j]);

		// model correspondance check
		if (!inItemFieldValue -> ClassIs (*inModel[j])) return false;

		// field type analyse, string field value
		if (inItemFieldValue -> ClassIs (__metaclass(CItemFieldValueString)))
		{
			// set the field values
			::gtk_tree_store_set (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(inTreeView->GetGtkWidget()))),
				&m_GtkTreeIter, 
				j * N_FIELDS + TEXT_FIELD, 
					static_cast <CItemFieldValueString *> (inItemFieldValue) -> m_FieldValue.Get(),
				j * N_FIELDS + VISIBLE_FIELD, 
					static_cast <CItemFieldValueString *> (inItemFieldValue) -> m_Visible,
				j * N_FIELDS + EDITABLE_FIELD, 
					static_cast <CItemFieldValueString *> (inItemFieldValue) -> m_Enabled, -1);
		}
		// boolean field value
		else if (inItemFieldValue -> ClassIs (__metaclass(CItemFieldValueBoolean)))
		{
			// set the field values
			::gtk_tree_store_set (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(inTreeView->GetGtkWidget()))),
				&m_GtkTreeIter, 
				j * N_FIELDS + ACTIVE_FIELD, 
					static_cast <CItemFieldValueBoolean *> (inItemFieldValue) -> m_FieldValue,
				j * N_FIELDS + VISIBLE_FIELD, 
					static_cast <CItemFieldValueBoolean *> (inItemFieldValue) -> m_Visible,
				j * N_FIELDS + ACTIVATABLE_FIELD, 
					static_cast <CItemFieldValueBoolean *> (inItemFieldValue) -> m_Enabled, -1);
		}
		// pixbuf field value
		else if (inItemFieldValue -> ClassIs (__metaclass(CItemFieldValuePixbuf)))
		{
			::gtk_tree_store_set (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(inTreeView->GetGtkWidget()))),
				&m_GtkTreeIter, 
				j * N_FIELDS + PIXBUF_FIELD, 
					static_cast <CItemFieldValuePixbuf *> (inItemFieldValue) -> m_FieldValue -> GetPixbuf(),
				j * N_FIELDS + VISIBLE_FIELD, 
					static_cast <CItemFieldValuePixbuf *> (inItemFieldValue) -> m_Visible, -1);
		}
		// combo field value
		else if (inItemFieldValue -> ClassIs (__metaclass(CItemFieldValueCombo)))
		{
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
		}
		// progress field value
		else if (inItemFieldValue -> ClassIs (__metaclass(CItemFieldValueProgress)))
		{
			// set the field values
			::gtk_tree_store_set (GTK_TREE_STORE(::gtk_tree_view_get_model(GTK_TREE_VIEW(inTreeView->GetGtkWidget()))),
				&m_GtkTreeIter, 
				j * N_FIELDS + PROGRESS_FIELD, 
					static_cast <CItemFieldValueProgress *> (inItemFieldValue) -> m_FieldValue * 100,
				j * N_FIELDS + VISIBLE_FIELD, 
					static_cast <CItemFieldValueProgress *> (inItemFieldValue) -> m_Visible,
				j * N_FIELDS + CAPTION_FIELD, 
					static_cast <CItemFieldValueProgress *> (inItemFieldValue) -> m_Caption.Get(), -1);
		}
	}

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// item field values access
//-----------------------------------------------------------------------------------------------------------------------------------------
CItemFieldValues CTreeViewItem::GetItemFieldValues () const
{
	return m_ItemFieldValues;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// 0 indexed field value direct access
//-----------------------------------------------------------------------------------------------------------------------------------------
CItemFieldValue & CTreeViewItem::operator [] (const size_t inIndex) const
{
	return **m_ItemFieldValues[inIndex];
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// select / unselect
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeViewItem::Select (const bool inSelect)
{
	// get the tree view owner
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// the gtk tree view
	GtkTreeView *inGtkTreeView = NULL;

	// pointer check
	if (inTreeView == NULL || (inGtkTreeView = GTK_TREE_VIEW(inTreeView->GetGtkWidget())) == NULL) return;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_tree_view_get_model (inGtkTreeView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));

	// select or unselect the tree item	
	if (inSelect)
		::gtk_tree_selection_select_path (::gtk_tree_view_get_selection (inGtkTreeView), inGtkTreePath);
	else
		::gtk_tree_selection_unselect_path (::gtk_tree_view_get_selection (inGtkTreeView), inGtkTreePath);

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// selected ?
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CTreeViewItem::IsSelected () const
{
	// get the tree view owner
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// the gtk tree view
	GtkTreeView *inGtkTreeView = NULL;

	// pointer check
	if (inTreeView == NULL || (inGtkTreeView = GTK_TREE_VIEW(inTreeView->GetGtkWidget())) == NULL) return false;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_tree_view_get_model (inGtkTreeView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));

	// get the selected value
	bool outSelected = ::gtk_tree_selection_path_is_selected (::gtk_tree_view_get_selection (inGtkTreeView), inGtkTreePath);

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);

	// ok
	return outSelected;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// set the cursor on the specified column and start editing if enable to
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeViewItem::SetCursor (const size_t inCol, const bool inEdit)
{
	// get the tree view owner
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// the gtk tree view
	GtkTreeView *inGtkTreeView = NULL;

	// pointer check
	if (inTreeView == NULL || (inGtkTreeView = GTK_TREE_VIEW(inTreeView->GetGtkWidget())) == NULL) return;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_tree_view_get_model (inGtkTreeView), 
									const_cast <GtkTreeIter *> (&m_GtkTreeIter));

	// get the gtk tree column
	GtkTreeViewColumn *inGtkTreeViewColumn = ::gtk_tree_view_get_column (GTK_TREE_VIEW(inGtkTreeView), inCol);

	// perform the query
	::gtk_tree_view_set_cursor (GTK_TREE_VIEW(inGtkTreeView), inGtkTreePath, inGtkTreeViewColumn, inEdit);

	// release the path
	::gtk_tree_path_free (inGtkTreePath);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expand a tree view item
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeViewItem::Expand (const bool inAll)
{
	// get the tree view owner
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// the gtk tree view
	GtkTreeView *inGtkTreeView = NULL;

	// pointer check
	if (inTreeView == NULL || (inGtkTreeView = GTK_TREE_VIEW(inTreeView->GetGtkWidget())) == NULL) return;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_tree_view_get_model (inGtkTreeView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));
	// expansion requested
	::gtk_tree_view_expand_row (inGtkTreeView, inGtkTreePath, inAll);

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// collapse a tree view item
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeViewItem::Collapse ()
{
	// get the tree view owner
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// the gtk tree view
	GtkTreeView *inGtkTreeView = NULL;

	// pointer check
	if (inTreeView == NULL || (inGtkTreeView = GTK_TREE_VIEW(inTreeView->GetGtkWidget())) == NULL) return;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_tree_view_get_model (inGtkTreeView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));
	// expansion requested
	::gtk_tree_view_collapse_row (inGtkTreeView, inGtkTreePath);

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// is the tree view item expanded ?
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CTreeViewItem::IsExpanded () const
{
	// get the tree view owner
	CTreeView *inTreeView = static_cast <CTreeView *> (GetOwner (__metaclass(CTreeView)));

	// the gtk tree view
	GtkTreeView *inGtkTreeView = NULL;

	// pointer check
	if (inTreeView == NULL || (inGtkTreeView = GTK_TREE_VIEW(inTreeView->GetGtkWidget())) == NULL) return false;

	// get the gtk tree iter path
	GtkTreePath *inGtkTreePath = ::gtk_tree_model_get_path (::gtk_tree_view_get_model (inGtkTreeView), 
								const_cast <GtkTreeIter *> (&m_GtkTreeIter));
	// expansion requested
	Bool outExpanded = ::gtk_tree_view_row_expanded (inGtkTreeView, inGtkTreePath);

	// free the path pointer
	::gtk_tree_path_free (inGtkTreePath);

	// ok
	return outExpanded;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// tree view item xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeViewItem::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic serialization first
	CControl::Serialize (ioXMLElementNode, inMode);

	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_TREEVIEWITEM_ELEMENT);

			// modify the io xml node so that the overwritten definitions would continue under this new xml node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// instanciate the fields xml element under the current xml node
			newXMLElement = new CXMLElement (ioXMLElementNode, XML_TREEVIEWITEM_FIELDS_ELEMENT);

			// foreach of the fields, delegate the serialization process
			for (size_t i=m_ItemFieldValues.GetLength(), j=0; i>0; i--, j++)
			{
				// each of the field serializes under the current node
				CXMLElementNode *inXMLNode = newXMLElement -> GetXMLElementNode ();

				// serialize the field
				(*m_ItemFieldValues[j]) -> Serialize (inXMLNode, XML_WRITE);
			}
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the node this instance should read from
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_TREEVIEWITEM_ELEMENT);

			// check we got it
                        if (inXMLNode == NULL)
                                throw new CException (CString("CTreeViewItem::Serialize, specified xml node is not a \"") +
                                                              XML_TREEVIEWITEM_ELEMENT + CString("\" element one."), 
						      	      __exception(XMLPARSE));

			// modify the io xml node so that the overwritten definitions would continue under this new xml node
			ioXMLElementNode = inXMLNode;

			// get the fields xml child element node
			inXMLNode = ::xml_node_get_child (inXMLNode, XML_TREEVIEWITEM_FIELDS_ELEMENT);

			// check we got it
			if (inXMLNode != NULL)
			{
				// delete the handled tree view item instances values if any
				if (m_ItemFieldValues.GetLength() > 0) 
				{
					for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) delete *m_ItemFieldValues[i-1];
					m_ItemFieldValues.Delete (0, m_ItemFieldValues.GetLength());
				}

				// get the "fields" xml element children nodes
				CXMLElementNodes inXMLNodes (::xml_node_get_children (inXMLNode));

				// foreach child of the "children" element, launch the global serialization process
				for (size_t i=inXMLNodes.GetLength(), j=0; i>0; i--, j++)
				{
					// launch the instanciation process
					CSerialized *inSerialized = CSerialized::Instanciate (*inXMLNodes[j]) THROWABLE;

					// pointer check
					if (inSerialized == NULL) continue;

					// check the pointer type
					if (!inSerialized -> ClassIs (__metaclass(CItemFieldValue))) { delete inSerialized; continue; }

					// add the field value to the local buffer
					m_ItemFieldValues += static_cast <CItemFieldValue *> (inSerialized);
				}
			}

			// set the item field values
			SetItemFieldValues (CItemFieldValues (m_ItemFieldValues));
		}
		break;
	}	
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CTreeViewListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeViewListener::CTreeViewListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeViewListener::~CTreeViewListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemEdited
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::OnItemEdited (GtkCellRendererText *inGtkCellRendererText, const gchar *inPath, const gchar *inNewValue, gpointer inData)
{
	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// pointer check, be paranoïd...
	if (inTreeView == NULL || inTreeView -> GetGtkWidget() == NULL) return;

	// get the gtk tree view used model
	GtkTreeModel *inGtkTreeModel = ::gtk_tree_view_get_model (GTK_TREE_VIEW(inTreeView -> GetGtkWidget()));

	// get the gtk tree path
	GtkTreePath *inGtkTreePath = ::gtk_tree_path_new_from_string (inPath);

	// get the gtk tree iter target
	GtkTreeIter inGtkTreeIter; ::gtk_tree_model_get_iter (inGtkTreeModel, &inGtkTreeIter, inGtkTreePath);

	// get the associated gtkol tree view item
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (&inGtkTreeIter);

	// get the cell renderer associated column index (attribute set in the model affectation function)
	gint *inColumn = reinterpret_cast <gint *> (::g_object_get_data (G_OBJECT(inGtkCellRendererText), "column"));

	// the associated item field value !
	CItemFieldValueString *inItemFieldValueString = 
		static_cast <CItemFieldValueString *> (*inTreeViewItem -> m_ItemFieldValues[reinterpret_cast <size_t> (inColumn)]);

	// the new field value...
	CString ioNewValue (inNewValue);

	// default to modify
	Bool ioDoChange = true;

	// send the request to the listener if any
	if (inTreeView -> GetListener() != NULL)
		static_cast <CTreeViewListener *> (inTreeView -> GetListener()) -> 
			OnItemEdited (inTreeView, inTreeViewItem, reinterpret_cast <size_t> (inColumn), ioNewValue, ioDoChange);

	// if the modification should be done
	if (ioDoChange)
	{
		// modify the field value attribute 
		inItemFieldValueString -> m_FieldValue = ioNewValue;

		// modify the tree item field value
		::gtk_tree_store_set (GTK_TREE_STORE(inGtkTreeModel), &inGtkTreeIter, 
				      reinterpret_cast <size_t> (inColumn) * N_FIELDS + TEXT_FIELD, 
				      inItemFieldValueString -> m_FieldValue.Get(), -1);
	}

	// free the gtk tree path
	::gtk_tree_path_free (inGtkTreePath);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemToggled
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::OnItemToggled (GtkCellRendererToggle *inGtkCellRendererToggle, gchar *inPath, gpointer inData)
{
	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// check pointer
	if (inTreeView == NULL || inTreeView -> GetGtkWidget() == NULL) return;

	// get the gtk tree view used model
	GtkTreeModel *inGtkTreeModel = ::gtk_tree_view_get_model (GTK_TREE_VIEW(inTreeView -> GetGtkWidget()));

	// get the gtk tree path
	GtkTreePath *inGtkTreePath = ::gtk_tree_path_new_from_string (inPath);

	// get the gtk tree iter target
	GtkTreeIter inGtkTreeIter; ::gtk_tree_model_get_iter (inGtkTreeModel, &inGtkTreeIter, inGtkTreePath);

	// get the associated gtkol tree view item
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (&inGtkTreeIter);

	// get the cell renderer associated column index (attribute set in the model affectation function)
	gint *inColumn = reinterpret_cast <gint *> (::g_object_get_data (G_OBJECT(inGtkCellRendererToggle), "column"));

	// the associated item field value !
	CItemFieldValueBoolean *inItemFieldValueBoolean = 
		static_cast <CItemFieldValueBoolean *> (*inTreeViewItem -> m_ItemFieldValues[reinterpret_cast <size_t> (inColumn)]);

	// default to modify
	Bool ioDoChange = true;

	// send the request to the listener if any
	if (inTreeView -> GetListener() != NULL)
		static_cast <CTreeViewListener *> (inTreeView -> GetListener()) -> 
			OnItemToggled (inTreeView, inTreeViewItem, reinterpret_cast <size_t> (inColumn), ioDoChange);

	// if the modification should be done
	if (ioDoChange)
	{
		// modify the field value attribute 
		inItemFieldValueBoolean -> m_FieldValue = !inItemFieldValueBoolean -> m_FieldValue;

		// modify the tree item field value
		::gtk_tree_store_set (GTK_TREE_STORE(inGtkTreeModel), &inGtkTreeIter, 
				      reinterpret_cast <size_t> (inColumn) * N_FIELDS + ACTIVE_FIELD, 
				      inItemFieldValueBoolean -> m_FieldValue, -1);
	}

	// free the gtk tree path
	::gtk_tree_path_free (inGtkTreePath);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemActivated
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::OnItemActivated (GtkTreeView *inGtkTreeView, GtkTreePath *inGtkTreePath, GtkTreeViewColumn *, gpointer inData)
{
	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// check pointer
	if (inTreeView == NULL || inTreeView -> GetListener() == NULL) return;

	// get the gtk tree iter target
	GtkTreeIter inGtkTreeIter; ::gtk_tree_model_get_iter (::gtk_tree_view_get_model (inGtkTreeView), &inGtkTreeIter, inGtkTreePath);

	// get the associated tree view item
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (&inGtkTreeIter);

	// send the notification
	static_cast <CTreeViewListener *> (inTreeView -> GetListener()) -> OnItemActivated (inTreeView, inTreeViewItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemQueryExpand
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CTreeView::OnItemQueryExpand (GtkTreeView *inGtkTreeView, GtkTreeIter *inGtkTreeIter, GtkTreePath *, gpointer inData)
{
	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// get the gtkol tree view item
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (inGtkTreeIter);

	// defaults to do expand
	Bool ioDoExpand = true;

	// send the notification to the listener if any
	if (inTreeView != NULL && inTreeView -> GetListener() != NULL && inTreeViewItem != NULL)
		static_cast <CTreeViewListener *> (inTreeView -> GetListener()) ->
			OnItemQueryExpand (inTreeView, inTreeViewItem, ioDoExpand);

	// return false to allow expansion, true to disable it...
	return !ioDoExpand;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemQueryCollapse
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CTreeView::OnItemQueryCollapse (GtkTreeView *inGtkTreeView, GtkTreeIter *inGtkTreeIter, GtkTreePath *, gpointer inData)
{
	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// get the gtkol tree view item
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (inGtkTreeIter);

	// defaults to do expand
	Bool ioDoCollapse = true;

	// send the notification to the listener if any
	if (inTreeView != NULL && inTreeView -> GetListener() != NULL && inTreeViewItem != NULL)
		static_cast <CTreeViewListener *> (inTreeView -> GetListener()) ->
			OnItemQueryCollapse (inTreeView, inTreeViewItem, ioDoCollapse);

	// return false to allow expansion, true to disable it...
	return !ioDoCollapse;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemExpanded
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::OnItemExpanded (GtkTreeView *inGtkTreeView, GtkTreeIter *inGtkTreeIter, GtkTreePath *, gpointer inData)
{
	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// get the gtkol tree view item
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (inGtkTreeIter);

	// send the notification to the listener if any
	if (inTreeView != NULL && inTreeView -> GetListener() != NULL && inTreeViewItem != NULL)
		static_cast <CTreeViewListener *> (inTreeView -> GetListener()) -> OnItemExpanded (inTreeView, inTreeViewItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemCollapsed
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::OnItemCollapsed (GtkTreeView *inGtkTreeView, GtkTreeIter *inGtkTreeIter, GtkTreePath *, gpointer inData)
{
	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// get the gtkol tree view item
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (inGtkTreeIter);

	// send the notification to the listener if any
	if (inTreeView != NULL && inTreeView -> GetListener() != NULL && inTreeViewItem != NULL)
		static_cast <CTreeViewListener *> (inTreeView -> GetListener()) -> OnItemCollapsed (inTreeView, inTreeViewItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemQuerySelect
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CTreeView::OnItemQuerySelect (GtkTreeSelection *inGtkTreeSelection, GtkTreeModel *inGtkTreeModel, GtkTreePath *inGtkTreePath,
				       gboolean inCurrent, gpointer inData)
{
	// defaults to do select / unselect
	Bool ioDoToggle = true;

	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// pointer check
	if (inTreeView == NULL || inTreeView -> GetListener() == NULL) return ioDoToggle;

	// get the gtk tree iter target
	GtkTreeIter inGtkTreeIter; ::gtk_tree_model_get_iter (inGtkTreeModel, &inGtkTreeIter, inGtkTreePath);

	// get the associated tree view item
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (&inGtkTreeIter);

	// pointer check
	if (inTreeViewItem == NULL) return ioDoToggle;

	// send the notification
	static_cast <CTreeViewListener *> (inTreeView -> GetListener()) -> 
		OnItemQuerySelect (inTreeView, inTreeViewItem, inCurrent, ioDoToggle);

	// if it is to be selected, ensure it is visible
	if (!inCurrent && ioDoToggle)
		::gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW(inTreeView->GetGtkWidget()), inGtkTreePath, NULL, false, 0, 0);

	// ok
	return ioDoToggle;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnItemSelectionChanged
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::OnItemSelectionChanged (GtkTreeSelection *inGtkTreeSelection, gpointer inData)
{
	// retreive the gtkol instance
	CTreeView *inTreeView = reinterpret_cast <CTreeView *> (inData);

	// send the notification to the listener if any
	if (inTreeView != NULL && inTreeView -> GetListener() != NULL)
		static_cast <CTreeViewListener *> (inTreeView -> GetListener()) -> OnItemSelectionChanged (inTreeView);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CTreeView);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeView::CTreeView	    (CContainer *inOwner, const CMetaClasses &inModel, const CTreeViewListener *inListener) THROWABLE
	  :CContainer	    (inOwner, inListener),
	   m_Model    	    (inModel),
	   m_ColumnsTitle   ()
{
	// specified metaclasses model check in...
	for (size_t i=m_Model.GetLength(), j=0; i>0; i--, j++)
		if (!CMetaClass::MetaClassIs (__metaclass(CItemFieldValue), *m_Model[j]) || *m_Model[j] == __metaclass(CItemFieldValue))
			throw new CException ("CTreeView : the input model is not handled; the specified class type \"" + 
					      (*m_Model[j])->ClassName + "\" is not a CItemFieldValue derived one. Aborting...");

	// instanciation process requested
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeView::~CTreeView ()
{
	// gtk widget deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CTreeView::PerformWidgetInstanciate ()
{
	// ok, model affectation into the initialize section
	return ::gtk_tree_view_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::PerformWidgetInitialize ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check in, be paranoïd...
	if (inGtkWidget == NULL) return;

	// default to non visible titles
	::gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(inGtkWidget), false);

	// model affectation
	SetModel (CMetaClasses (m_Model));

	// default selection mode
	SetSelectionMode (GTK_SELECTION_SINGLE);

	// signal connections / static to dynamic listening implementation
	::gtk_tree_selection_set_select_function (::gtk_tree_view_get_selection(GTK_TREE_VIEW(inGtkWidget)), 
						  				   CTreeView::OnItemQuerySelect, 	this, NULL);
	::g_signal_connect (G_OBJECT(inGtkWidget), "row-activated", 	G_CALLBACK(CTreeView::OnItemActivated), 	this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "test-expand-row", 	G_CALLBACK(CTreeView::OnItemQueryExpand), 	this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "test-collapse-row", G_CALLBACK(CTreeView::OnItemQueryCollapse), 	this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "row-expanded", 	G_CALLBACK(CTreeView::OnItemExpanded),	 	this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "row-collapsed", 	G_CALLBACK(CTreeView::OnItemCollapsed),	 	this);
	::g_signal_connect (G_OBJECT(::gtk_tree_view_get_selection(GTK_TREE_VIEW(inGtkWidget))), 
						   "changed",		G_CALLBACK(CTreeView::OnItemSelectionChanged), 	this);

	// show the job
	Show ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CTreeView::ListenerMustBe () const
{
	return __metaclass(CTreeViewListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CTreeView::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}
 
//-----------------------------------------------------------------------------------------------------------------------------------------
// expected children type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CTreeView::ChildMustBe () const
{
	return __metaclasses(CTreeViewItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// tree view item at specified relative point
//-----------------------------------------------------------------------------------------------------------------------------------------
CControl * CTreeView::GetControlAtPoint (const TPoint &inRelativePoint) const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return const_cast <CTreeView *> (this);

	// convert the given coords
	gint inX, inY; 
	//::gtk_tree_view_tree_to_widget_coords (GTK_TREE_VIEW(inGtkWidget), inRelativePoint.x, inRelativePoint.y, &inX, &inY);

	inX = inRelativePoint.x;
	inY = inRelativePoint.y;

	// get the path under the specified point
	GtkTreePath *inGtkTreePath = NULL;
	::gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW(inGtkWidget), inX, inY, &inGtkTreePath, NULL, NULL, NULL);

	// check the path
	if (inGtkTreePath == NULL) return const_cast <CTreeView *> (this);
	
	// get the tree view model
	GtkTreeModel *inModel = (GtkTreeModel *) ::gtk_tree_view_get_model (GTK_TREE_VIEW(inGtkWidget));

	// get the associated tree iter
	GtkTreeIter inGtkTreeIter; ::gtk_tree_model_get_iter (GTK_TREE_MODEL(inModel), &inGtkTreeIter, inGtkTreePath);

	// release the path
	::gtk_tree_path_free (inGtkTreePath);

	// get the associated CTreeViewItem
	CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (&inGtkTreeIter);

	// ok ?
	return inTreeViewItem != NULL ? static_cast <CControl *> (inTreeViewItem) : const_cast <CTreeView *> (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// model affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CTreeView::SetModel (const CMetaClasses &inModel)
{
	// model definition number check in...
	if (inModel.GetLength() == 0) return false;

	// model check in, all the specified metaclasses must derive the CItemFieldValue definition
	for (size_t i=inModel.GetLength(), j=0; i>0; i--, j++)
		if (!CMetaClass::MetaClassIs (__metaclass(CItemFieldValue), *inModel[j]) || __metaclass(CItemFieldValue) == *inModel[j])
			return false;

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check 
	if (inGtkWidget == NULL) return false;

	// tmp copy the metaclasses
	CMetaClasses oldModel (m_Model);

	// remove any reference to the field pack specification
	for (size_t i=0; i<oldModel.GetLength(); i++)
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *oldModel[i]))
			oldModel.Delete (i--, 1);

	// remove the old columns if any
	for (size_t i=oldModel.GetLength(); i>0; i--)
	{
		// get the column
		GtkTreeViewColumn *inGtkTreeViewColumn = ::gtk_tree_view_get_column (GTK_TREE_VIEW(inGtkWidget), 0);

		// retrait de cette première colonne
		if (inGtkTreeViewColumn != NULL) ::gtk_tree_view_remove_column (GTK_TREE_VIEW(inGtkWidget), inGtkTreeViewColumn);
	}

	// model input local copy
	m_Model = inModel;

	// GType buffer for the associated gtk tree store instance
	TBuffer <GType> outGTypeBuffer;

	// model input analyse
	for (size_t i=m_Model.GetLength(), j=0; i>0; i--, j++)
	{
		// string field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueString), *m_Model[j]))
		{
			outGTypeBuffer += G_TYPE_STRING;	// field value
			outGTypeBuffer += G_TYPE_BOOLEAN;	// visible field
			outGTypeBuffer += G_TYPE_BOOLEAN;	// enabled / editable field
		}
		// boolean field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueBoolean), *m_Model[j]))
		{
			outGTypeBuffer += G_TYPE_BOOLEAN;	// field value
			outGTypeBuffer += G_TYPE_BOOLEAN;	// visible field
			outGTypeBuffer += G_TYPE_BOOLEAN;	// enabled / activatable field
		}
		// pixbuf field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePixbuf), *m_Model[j]))
		{
			outGTypeBuffer += GDK_TYPE_PIXBUF;	// field value
			outGTypeBuffer += G_TYPE_BOOLEAN;	// visible field
			outGTypeBuffer += G_TYPE_BOOLEAN;	// [unused]
		}
		// combo field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueCombo), *m_Model[j]))
		{
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
		}
		// progress field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueProgress), *m_Model[j]))
		{
			outGTypeBuffer += G_TYPE_FLOAT;		// field value
			outGTypeBuffer += G_TYPE_BOOLEAN;	// visible field
			outGTypeBuffer += G_TYPE_STRING;	// caption fiel
		}
	}

	// gtk tree store instanciation
	GtkTreeStore *inGtkTreeStore = ::gtk_tree_store_newv (outGTypeBuffer.GetLength(), outGTypeBuffer.Get());

	// gtk tree store affectation
	::gtk_tree_view_set_model (GTK_TREE_VIEW(inGtkWidget), GTK_TREE_MODEL(inGtkTreeStore));

	// delete local reference to the gtk tree store
	::g_object_unref (inGtkTreeStore);

	// foreach field value specification, keep a temporary reference on the allocated column as field values may be packed together
	// in the same cell layout...
	GtkTreeViewColumn *inGtkTreeViewColumn = NULL; bool doFieldValuePack = false;

	// create the tree view model associated columns
	for (size_t i=m_Model.GetLength(), j=0, k=0; i>0; i--, j++)
	{
		// pack field specification
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *m_Model[j]))
		{
			// do a field value pack between the already allocated column renderer and the next to come
			if (inGtkTreeViewColumn != NULL) doFieldValuePack = true;
		}
		// string field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueString), *m_Model[j]))
		{
			// allocate the cell renderer
			GtkCellRenderer *inGtkCellRenderer = ::gtk_cell_renderer_text_new ();

			// default properties affectation to the cell renderer
			::g_object_set (G_OBJECT(inGtkCellRenderer), "xalign", 0.0, NULL);

			// for commodity, does not point to anything after the loop but must be reinterpret as an int and not an int * !
			::g_object_set_data (G_OBJECT(inGtkCellRenderer), "column", GINT_TO_POINTER(k));

			// signal connection
			::g_signal_connect (G_OBJECT(inGtkCellRenderer), "edited", G_CALLBACK(CTreeView::OnItemEdited), this);

			// if not to be packed with a previous renderer
			if (!doFieldValuePack)
			{
				// allocate a gtk tree view column
				inGtkTreeViewColumn = ::gtk_tree_view_column_new_with_attributes 
					((m_ColumnsTitle.GetLength()>k) ? m_ColumnsTitle[k]->Get() : "", inGtkCellRenderer, 
					"text",     k * N_FIELDS + TEXT_FIELD, 
					"visible",  k * N_FIELDS + VISIBLE_FIELD, 
					"editable", k * N_FIELDS + EDITABLE_FIELD, NULL);

				// append the newly allocated column to the tree view
				::gtk_tree_view_append_column (GTK_TREE_VIEW(inGtkWidget), inGtkTreeViewColumn);

				// clickable tree view column
				::gtk_tree_view_column_set_clickable (inGtkTreeViewColumn, true);
			}
			// to be packed...
			else 
			{
				// pack it into the current cell layout
				::gtk_tree_view_column_pack_start (inGtkTreeViewColumn, inGtkCellRenderer, true);

				// set the attributes
				::gtk_tree_view_column_set_attributes (inGtkTreeViewColumn, inGtkCellRenderer,
					"text",     k * N_FIELDS + TEXT_FIELD, 
					"visible",  k * N_FIELDS + VISIBLE_FIELD, 
					"editable", k * N_FIELDS + EDITABLE_FIELD, NULL);
			}

			// next field value
			k++; doFieldValuePack = false;
		}
		// boolean field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueBoolean), *m_Model[j]))
		{
			// allocate the cell renderer
			GtkCellRenderer *inGtkCellRenderer = ::gtk_cell_renderer_toggle_new ();

			// default properties affectation to the cell renderer
			::g_object_set (G_OBJECT(inGtkCellRenderer), "xalign", 0.0, NULL);

			// for commodity, does not point to anything after the loop but must be reinterpret as an int and not an int * !
			::g_object_set_data (G_OBJECT(inGtkCellRenderer), "column", GINT_TO_POINTER(k));

			// signal connection
			::g_signal_connect (G_OBJECT(inGtkCellRenderer), "toggled", G_CALLBACK(CTreeView::OnItemToggled), this);

			// if not to be packed with a previous renderer
			if (!doFieldValuePack)
			{
				// allocate a gtk tree view column
				inGtkTreeViewColumn = ::gtk_tree_view_column_new_with_attributes 
					((m_ColumnsTitle.GetLength()>k) ? m_ColumnsTitle[k]->Get() : "", inGtkCellRenderer, 
					"active",      k * N_FIELDS + ACTIVE_FIELD, 
					"visible",     k * N_FIELDS + VISIBLE_FIELD, 
					"activatable", k * N_FIELDS + ACTIVATABLE_FIELD, NULL);

				// append the newly allocated column to the tree view
				::gtk_tree_view_append_column (GTK_TREE_VIEW(inGtkWidget), inGtkTreeViewColumn);

				// clickable tree view column
				::gtk_tree_view_column_set_clickable (inGtkTreeViewColumn, true);
			}
			// to be packed field value
			else
			{
				// pack it into the current cell layout
				::gtk_tree_view_column_pack_start (inGtkTreeViewColumn, inGtkCellRenderer, true);

				// set the attributes
				::gtk_tree_view_column_set_attributes (inGtkTreeViewColumn, inGtkCellRenderer,
					"active",      k * N_FIELDS + ACTIVE_FIELD, 
					"visible",     k * N_FIELDS + VISIBLE_FIELD, 
					"activatable", k * N_FIELDS + ACTIVATABLE_FIELD, NULL);
			}

			// next field value
			k++; doFieldValuePack = false;
		}
		// pixbuf field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePixbuf), *m_Model[j]))
		{
			// instanciate the cell renderer
			GtkCellRenderer *inGtkCellRenderer = ::gtk_cell_renderer_pixbuf_new ();

			// default properties affectation to the cell renderer
			::g_object_set (G_OBJECT(inGtkCellRenderer), "xalign", 0.0, NULL);

			// for commodity, does not point to anything after the loop but must be reinterpret as an int and not an int * !
			::g_object_set_data (G_OBJECT(inGtkCellRenderer), "column", GINT_TO_POINTER(k));

			// if not to be packed with a previous renderer
			if (!doFieldValuePack)
			{
				// allocate a gtk tree view column
				inGtkTreeViewColumn = ::gtk_tree_view_column_new_with_attributes 
					((m_ColumnsTitle.GetLength()>k) ? m_ColumnsTitle[k]->Get() : "", inGtkCellRenderer, 
					"pixbuf",       k * N_FIELDS + PIXBUF_FIELD, 
					"visible",      k * N_FIELDS + VISIBLE_FIELD, 
					// UNUSED FIELD
					"follow-state", k * N_FIELDS + VISIBLE_FIELD, NULL);

				// append the newly allocated column to the tree view
				::gtk_tree_view_append_column (GTK_TREE_VIEW(inGtkWidget), inGtkTreeViewColumn);

				// clickable tree view column
				::gtk_tree_view_column_set_clickable (inGtkTreeViewColumn, true);
			}
			// to be packed field value
			else
			{
				// pack it into the current cell layout
				::gtk_tree_view_column_pack_start (inGtkTreeViewColumn, inGtkCellRenderer, true);

				// set the attributes
				::gtk_tree_view_column_set_attributes (inGtkTreeViewColumn, inGtkCellRenderer,
					"pixbuf",       k * N_FIELDS + PIXBUF_FIELD, 
					"visible",      k * N_FIELDS + VISIBLE_FIELD, 
					// UNUSED FIELD
					"follow-state", k * N_FIELDS + VISIBLE_FIELD, NULL);
			}

			// next field value
			k++; doFieldValuePack = false;
		}
		// combo field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueCombo), *m_Model[j]))
		{
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
			// TODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODOTODO
		}
		// progress field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueProgress), *m_Model[j]))
		{
			// allocate the cell renderer
			GtkCellRenderer *inGtkCellRenderer = ::gtk_cell_renderer_progress_new ();

			// default properties affectation to the cell renderer
			::g_object_set (G_OBJECT(inGtkCellRenderer), "xalign", 0.0, NULL);

			// for commodity, does not point to anything after the loop but must be reinterpret as an int and not an int * !
			::g_object_set_data (G_OBJECT(inGtkCellRenderer), "column", GINT_TO_POINTER(k));

			// if not to be packed with a previous renderer
			if (!doFieldValuePack)
			{
				// allocate a gtk tree view column
				inGtkTreeViewColumn = ::gtk_tree_view_column_new_with_attributes 
					((m_ColumnsTitle.GetLength()>k) ? m_ColumnsTitle[k]->Get() : "", inGtkCellRenderer, 
					"value",   k * N_FIELDS + PROGRESS_FIELD, 
					"visible", k * N_FIELDS + VISIBLE_FIELD, 
					"text",    k * N_FIELDS + CAPTION_FIELD, NULL);

				// append the newly allocated column to the tree view
				::gtk_tree_view_append_column (GTK_TREE_VIEW(inGtkWidget), inGtkTreeViewColumn);

				// clickable tree view column
				::gtk_tree_view_column_set_clickable (inGtkTreeViewColumn, true);
			}
			// to be packed field value
			else
			{
				// pack it into the current cell layout
				::gtk_tree_view_column_pack_start (inGtkTreeViewColumn, inGtkCellRenderer, true);

				// set the attributes
				::gtk_tree_view_column_set_attributes (inGtkTreeViewColumn, inGtkCellRenderer,
					"value",   k * N_FIELDS + PROGRESS_FIELD, 
					"visible", k * N_FIELDS + VISIBLE_FIELD, 
					"text",    k * N_FIELDS + CAPTION_FIELD, NULL);
			}

			// next field value
			k++; doFieldValuePack = false;
		}
	}

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// model access
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CTreeView::GetModel () const
{
	return m_Model;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// columns titles
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::SetColumnsTitle	(const CStrings &inTitles)
{
	// keep a local copy of the titles
	m_ColumnsTitle = inTitles;

	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return;

	// go through the specified titles
	for (size_t i=m_ColumnsTitle.GetLength(), j=0; i>0; i--, j++)
	{
		// retreive the gtk column if any
		GtkTreeViewColumn *inGtkTreeViewColumn = ::gtk_tree_view_get_column (GTK_TREE_VIEW(inGtkWidget), j);

		// set its assoicated title
		if (inGtkTreeViewColumn != NULL) ::gtk_tree_view_column_set_title (inGtkTreeViewColumn, m_ColumnsTitle[j]->Get());
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// columns titles
//-----------------------------------------------------------------------------------------------------------------------------------------
CStrings CTreeView::GetColumnsTitle () const
{
	return m_ColumnsTitle;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// rules hint
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::SetRulesHint (const bool inRulesHint)
{
	// get gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// set rules hint
	if (inGtkWidget) ::gtk_tree_view_set_rules_hint (GTK_TREE_VIEW(inGtkWidget), inRulesHint);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// rules hint
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CTreeView::GetRulesHint () const
{
	// get rules hint
	return GetGtkWidget() ? ::gtk_tree_view_get_rules_hint (GTK_TREE_VIEW(GetGtkWidget())) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// headers visible
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::SetHeadersVisible (const bool inVisible)
{
	// get gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// set headers visible
	if (inGtkWidget) ::gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(inGtkWidget), inVisible);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// headers visible
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CTreeView::GetHeadersVisible () const
{
	// get headers visible
	return GetGtkWidget() ? ::gtk_tree_view_get_headers_visible (GTK_TREE_VIEW(GetGtkWidget())) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// tree view selection mode affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::SetSelectionMode (const GtkSelectionMode inSelectionMode)
{
	// get the gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// ok, attribute affectation
	::gtk_tree_selection_set_mode (::gtk_tree_view_get_selection (GTK_TREE_VIEW(inGtkWidget)), inSelectionMode);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// tree view item selection mode access
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkSelectionMode CTreeView::GetSelectionMode () const
{
	// get the gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return GTK_SELECTION_NONE;

	// ok
	return ::gtk_tree_selection_get_mode (::gtk_tree_view_get_selection (GTK_TREE_VIEW(inGtkWidget)));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get tree view current selection items
//-----------------------------------------------------------------------------------------------------------------------------------------
CTreeViewItems CTreeView::GetSelection () const
{
	// out items list
	CTreeViewItems outTreeViewItems;

	// get the gtk widget
	GtkTreeView *inGtkTreeView = GTK_TREE_VIEW(GetGtkWidget());

	// pointer check
	if (inGtkTreeView == NULL) return outTreeViewItems;

	// get the gtk tree model
	GtkTreeModel *inGtkTreeModel = reinterpret_cast <GtkTreeModel *> (::gtk_tree_view_get_model (inGtkTreeView));

	// get the glist of the gtk selected rows
	GList *inGList 	   = ::gtk_tree_selection_get_selected_rows (::gtk_tree_view_get_selection (inGtkTreeView), NULL);
	GList *inGListItem = inGList;

	// go through the given glist
	while (inGListItem != NULL)
	{
		// the requested gtk tree iter element
		GtkTreeIter inGtkTreeIter; 
		::gtk_tree_model_get_iter (inGtkTreeModel, &inGtkTreeIter, reinterpret_cast <GtkTreePath *> (inGListItem -> data));

		// get the associated gtkol tree view item
		CTreeViewItem *inTreeViewItem = CTreeViewItem::GetTreeViewItem (&inGtkTreeIter);

		// output buffer addon
		if (inTreeViewItem != NULL) outTreeViewItems += inTreeViewItem;

		// next item
		inGListItem = inGListItem -> next;
	}

	// delete the pointers
	::g_list_foreach (inGList, reinterpret_cast <void (*) (void *, void *)> (::gtk_tree_path_free), NULL);
	::g_list_free 	 (inGList);

	// ok
	return outTreeViewItems;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CTreeView::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// generic call first
			CContainer::Serialize (ioXMLElementNode, XML_WRITE);

			// create a new xml element node for the ctreeview serialization
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_TREEVIEW_ELEMENT);

			// get the selection mode
			CString inSelectionMode; switch (GetSelectionMode())
			{
				case GTK_SELECTION_NONE		: inSelectionMode = CString("none");	 break;
				case GTK_SELECTION_SINGLE	: inSelectionMode = CString("single");	 break;
				case GTK_SELECTION_BROWSE	: inSelectionMode = CString("browse"); 	 break;
				case GTK_SELECTION_MULTIPLE 	: inSelectionMode = CString("multiple"); break;
			}

			// get the selected items if any
			CString inSelection; CTreeViewItems inCurrentSelection (GetSelection());
			for (size_t i=inCurrentSelection.GetLength(), j=0; i>0; i--, j++)
			{
				SInt16 index = CComponent::GetInOwnerOccurence (this, *inCurrentSelection[j], __metaclass(CTreeViewItem));
				if (index >= 0)
				{
					inSelection += CString ((UInt32)index);
					if (i > 1) inSelection += CString(",");
				}
			}

			// get the whole tree view items
			CComponents inTreeViewItems (GetSubComponents (__metaclass(CTreeViewItem)));

			// go through the list and keep a local copy of those who are expanded
			CString inExpanded; for (size_t i=inTreeViewItems.GetLength(), j=0; i>0; i--, j++)
			{
				// check expanded state
				if (static_cast <CTreeViewItem *> (*inTreeViewItems[j]) -> IsExpanded())
				{
					if (j > 0 && inExpanded != CString()) inExpanded += CString(",");
					inExpanded += CString((UInt32)j);
				}
			}

			// add the attributes the node is interested in
			newXMLElement -> AddAttribute (XML_TREEVIEW_ATTR_SELECTION_MODE,  inSelectionMode);
			newXMLElement -> AddAttribute (XML_TREEVIEW_ATTR_HEADERS_VISIBLE, CString(GetHeadersVisible()?"true":"false"));
			newXMLElement -> AddAttribute (XML_TREEVIEW_ATTR_RULES_HINT, 	  CString(GetRulesHint()?"true":"false"));
			newXMLElement -> AddAttribute (XML_TREEVIEW_ATTR_SELECTION,	  inSelection);
			newXMLElement -> AddAttribute (XML_TREEVIEW_ATTR_EXPAND,	  inExpanded);

			// modify the io xml element node so that the potential overloaded definitions will continue under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// create a new xml element node for the ctreeview model serialization
			newXMLElement = new CXMLElement (ioXMLElementNode, XML_TREEVIEW_MODEL_ELEMENT);

			// serialize each model field
			for (size_t i=m_Model.GetLength(), j=0, k=0; i>0; i--, j++)
			{
				// instanciate a new xml element for the field model dump
				CXMLElement *newXMLModelElement = new CXMLElement (newXMLElement -> GetXMLElementNode(), 
										   XML_TREEVIEW_FIELDCLASS_ELEMENT);

				// append the attributes this instance is interested in
				newXMLModelElement -> AddAttribute (XML_TREEVIEW_FIELDCLASS_ATTR_NAME, 
								   (*m_Model[j])->ClassName);
				newXMLModelElement -> AddAttribute (XML_TREEVIEW_FIELDCLASS_ATTR_TAG, 
								    ::ClassTagToString((*m_Model[j])->ClassTag));

				// if the model item is not an abstract pack one
				if (!CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *m_Model[j]))

					// check for a potential column title
					if (k < m_ColumnsTitle.GetLength())

						// add the title attribute
						newXMLModelElement -> AddAttribute (XML_TREEVIEW_FIELDCLASS_ATTR_TITLE,
										    *m_ColumnsTitle[k++]);
					// add an empty one
					else	newXMLModelElement -> AddAttribute (XML_TREEVIEW_FIELDCLASS_ATTR_TITLE, CString());
			}
		}
		break;

		// xml load
		case XML_READ :
		{
			// we first have to find the tree view xml node because the tree view model must be set before the owned items
			// are instanciated, so request the expected xml node
			CXMLElementNode *inXMLNode = ::xml_node_search (ioXMLElementNode, XML_TREEVIEW_ELEMENT);

			// check we got an expected ctreeview node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CTreeView::Serialize, specified xml node is not a \"") +
                                                              XML_TREEVIEW_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// get the selection mode and the selected items attributes
			CString inSelectionMode (::xml_node_get_attribute (inXMLNode, XML_TREEVIEW_ATTR_SELECTION_MODE).GetValue());
			CString inSelection	(::xml_node_get_attribute (inXMLNode, XML_TREEVIEW_ATTR_SELECTION).	GetValue());
			CString inExpanded	(::xml_node_get_attribute (inXMLNode, XML_TREEVIEW_ATTR_EXPAND).	GetValue());

			// keep a reference on the node
			CXMLElementNode *theXMLNode = inXMLNode;

			// reset the columns title if any
			m_ColumnsTitle.Reset();

			// get the model xml node
			inXMLNode = ::xml_node_get_child (inXMLNode, XML_TREEVIEW_MODEL_ELEMENT);

			// check we got the expected xml node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CTreeView::Serialize, specified xml node is not a \"") +
                                                              XML_TREEVIEW_MODEL_ELEMENT + CString("\" element one."), 
							      __exception(XMLPARSE));

			// get the tree view model
			m_Model = CMetaClasses(); for (size_t i=xml_node_get_children_number(inXMLNode), j=0; i>0; i--, j++)
			{
				// get the xml node
				CXMLElementNode *inChildXMLNode = ::xml_node_get_child (inXMLNode, j);

				// check it is the expected xml element node
				if (::xml_node_get_name(inChildXMLNode) != XML_TREEVIEW_FIELDCLASS_ELEMENT) continue;

				// get the associated metaclass
				const CMetaClass *inMetaClass = CMetaClass::GetMetaClass (::StringToClassTag (
					(::xml_node_get_attribute (inChildXMLNode, XML_TREEVIEW_FIELDCLASS_ATTR_TAG).GetValue())));

				// check we got the metaclass
				if (inMetaClass != NULL)
				{
					// add it to the input model list
					m_Model += inMetaClass;

					// if it is not pack abstract specification
					if (!CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), inMetaClass))
					{
						// retreive the potential title attribute
						CString inTitle = ::xml_node_get_attribute (inChildXMLNode,
										    XML_TREEVIEW_FIELDCLASS_ATTR_TITLE).GetValue();

						// add it to the list
						m_ColumnsTitle += inTitle;
					}
				}
			}

			// set the tree view model from the serialization read in (and the columns title)
			SetModel (m_Model);

			// the tree view model is set so we can call the generic definition that will indirectly instanciate the items
			CContainer::Serialize (ioXMLElementNode, XML_READ);

			// set the selection mode of the tree view
			if (inSelectionMode == CString("single"  )) SetSelectionMode (GTK_SELECTION_SINGLE  ); else
			if (inSelectionMode == CString("browse"  )) SetSelectionMode (GTK_SELECTION_BROWSE  ); else
			if (inSelectionMode == CString("multiple")) SetSelectionMode (GTK_SELECTION_MULTIPLE); else
			SetSelectionMode (GTK_SELECTION_NONE);

			// set the tree view headers visible and rules hint state
			SetRulesHint      (::xml_node_get_attribute (theXMLNode, XML_TREEVIEW_ATTR_RULES_HINT).	    GetValue().ToBool());
			SetHeadersVisible (::xml_node_get_attribute (theXMLNode, XML_TREEVIEW_ATTR_HEADERS_VISIBLE).GetValue().ToBool());

			// analyse the expanded items
			CStrings inExpansion (inExpanded.Cut (CString(","), true));

			// get the tree view items
			CComponents inTreeViewItems (GetSubComponents (__metaclass(CTreeViewItem)));

			// foreach one, expand it !
			if (inExpanded.GetLength() > 0) for (size_t i=inExpansion.GetLength(), j=0; i>0; i--, j++)
			{
				// get the component of specified index
				CComponent *inComponent = inExpansion[j]->ToULong() < inTreeViewItems.GetLength() ? 
							  *inTreeViewItems[inExpansion[j]->ToULong()] : NULL;

				// check it we got it
				if (inComponent != NULL)

					// expand it, partial requested...
					static_cast <CTreeViewItem *> (inComponent) -> Expand (false);

				// not found or not a tree view item
				else

					// throw an exception, we can't afford that !
					throw new CException ("CTreeView::Serialize, specified ctreeview item index \"" +
							      (*inExpansion[j]) + "\" on \"" +
                                                              XML_TREEVIEW_ELEMENT + 
							      CString("\" element cannot be handled, wrong \"" + 
							      XML_TREEVIEW_ATTR_EXPAND + "\" attribute specification."), 
							      __exception(XMLPARSE));
			}

			// analyse the specified items selection i.e. separate the tree view items id
			CStrings inSelections (inSelection.Cut (CString(","), true));

			// foreach one, select it !
			if (inSelection.GetLength() > 0) for (size_t i=inSelections.GetLength(), j=0; i>0; i--, j++)
			{
				// get the component of specified index
				CComponent *inComponent = inSelections[j]->ToULong() < inTreeViewItems.GetLength() ? 
							  *inTreeViewItems[inSelections[j]->ToULong()] : NULL;

				// check we got it
				if (inComponent != NULL)

					// select it...
					static_cast <CTreeViewItem *> (inComponent) -> Select (true);

				// not found or not a tree view item
				else

					// throw an exception, we can't afford that !
					throw new CException ("CTreeView::Serialize, specified ctreeview item index \"" +
							      (*inSelections[j]) + "\" on \"" +
                                                              XML_TREEVIEW_ELEMENT + 
							      CString("\" element cannot be handled, wrong \"" + 
							      XML_TREEVIEW_ATTR_SELECTION + "\" attribute specification."), 
							      __exception(XMLPARSE));
			}
		}
		break;
	}
}







