/*
 * IRiver ifp supporting functions, a basic (primative) API.
 * $Id: prim.c,v 1.26.2.1 2005/02/19 17:18:03 oakhamg Exp $
 *
 * Copyright (C) Geoff Oakham, 2004; <oakhamg@users.sourceforge.net>
 */

#include "ifp.h"
#include "ifp_os.h"
#include "prim.h"

//Constants needed for autodetection.  (Platform-indendent, I might add.)

//This list is a guestimate.  We've confirmed all models except
//iFP-10xx.
int ifp_product_ids[IFP_PRODUCT_IDs] = {
	0x1001,
	0x1003,
	0x1005,
	0x1007,
	0x1008,
	0x1009,
	0x1010,
	0x1011,
};
char * ifp_product_strings[IFP_PRODUCT_IDs] = {
	"iFP-1xx",
	"iFP-3xx",
	"iFP-5xx",
	"iFP-7xx",
	"iFP-8xx",
	"iFP-9xx",
	"iFP-10xx",
	"N10",
};

#ifdef IFP_DEBUG_USB_SNOOPING
static char const * ifp_human_commands[] = {
	"echo (ping)", //0x00
	"golf (goodbye)", //0x01
	"02_command", //0x02
	"0x03", //0x03
	"0x04", //0x04
	"f.open", //0x05
	"f.new", //0x06
	"download", //0x07
	"upload", //0x08
	"0x09", //0x09
	"0x0a", //0x0a
	"f.size", //0x0b
	"0x0c", //0x0c
	"f.close", //0x0d
	"delete", //0x0e
	"d.open", //0x0f
	"d.next", //0x10
	"d.close", //0x11
	"mkdir", //0x12
	"rmdir", //0x13
	"capacity", //0x14
	"availa", //0x15
	"format", //0x16
	"firmware update", //0x17
	"buffsz", //0x18
	"0x19", //0x19
	"getFAT page", //0x1a
	"d.next_debug", //0x1b
	"setFAT page", //0x1c
	"getRadio stations", //0x1d
	"setRadio stations", //0x1e
	"0x1f", //0x1f
};

static char const * ifp_human_02_commands[] = {
	"string", //0x00
	"02_01", //0x01
	"02_02", //0x02
	"firmwr", //0x03
	"delta", //0x04
	"02_05", //0x05
	"flush", //0x06
	"02_07", //0x07
	"battery", //0x08
};

static int _ifp_control_debug(struct ifp_device * dev,
	int command, int arg1, int arg2, int r1, int * pr2,
	int ivalue
	)
{
	int i = 0;
	int r2 = -1;
	char const * s_command = NULL;


	if (pr2) {
		r2 = *pr2;
	}

	if (command == 0x02 && arg2 >= 0 && arg2 <= 0x08) {
		s_command = ifp_human_02_commands[arg2];
	} else if (command >= 0 && command <= 0x1f) {
		s_command = ifp_human_commands[command];
	} else {
		s_command = "";
	}

	ifp_print("ifp: %-6.6s c0%02x %04x %04x | %04x",
		s_command, command, arg1, arg2, r1);

	if (pr2) {
		ifp_print(" %04x", r2);
	}

	if (ivalue) {
		ifp_print(" i=%d", ivalue);
	}

	ifp_print("\n");
	return i;
}

static inline int ifp_control_debug_info(struct ifp_device * dev,
	int command, int arg1, int arg2, int r1, int * pr2,
	int ivalue
	)
{
	return _ifp_control_debug(dev, command, arg1, arg2, r1, pr2, ivalue);
}
#else //IFP_DEBUG_USB_SNOOPING
static inline int ifp_control_debug_info(struct ifp_device * dev,
	int command, int arg1, int arg2, int r1, int * pr2, int ivalue)
{
	return 0;
}
#endif //IFP_DEBUG_USB_SNOOPING

int ifp_control_send(struct ifp_device * dev, int command,
	int arg1, int arg2, int * pr2)
{
	int i, r1;
	i = ifp_os_control_send(dev, command, arg1, arg2, &r1, pr2);
	ifp_control_debug_info(dev, command, arg1, arg2, r1, pr2, i);
	if (i < 0) {
		ifp_err_i(i, "error sending control value");
	} else if (i) {
		//caller allready warned about unexpected value.
		i = 0;
	}

	if (i == 0) {
		i = r1;
	}

	return i;
}

//Convience method to convert ifp's "success" return value to 0
int ifp_control_send_bool(struct ifp_device * dev, int command, int arg1, int arg2,
	int * ret)
{
	int i = ifp_control_send(dev, command, arg1, arg2, ret);
	if (i == 0) {
		//FIXME: change to i = 1;
		//i = -1;
		i = 1;
	} else if (i == 1) {
		i = 0;
	}
	return i;
}


#ifdef IFP_DEBUG_USB_RAWDATA
static int hexdump_line(void * pp, int n, int lw)
{
	uint8_t * p = pp;
	int j, c;

	for (j=0; j!=lw; j++) {
		if (j >= n) {
			ifp_print("  ");
		} else {
			ifp_print("%02x", (int)(p[j]));
		}
		if (j % 4 == 3) {
			ifp_print(" ");
		}
	}

	for (j=0; j!=lw; j++) {
		if (j >= n) {
			break;
			//printk(" ");
		} else {
			c = (int)p[j];
			if (isprint(c)) {
				ifp_print("%c", c);
			} else {
				ifp_print(".");
			}
		}
	}

	return 0;
}

static int hexdump(char * prefix, void * pp, int n)
{
	uint8_t * p = pp;
	int const lw = 32;
	int zeros = 0;
	int j = n;
	int total = n;

	while (p[j-1] == 0) {
		j--;
	}
	if (n-j > 3) {
		zeros = n - j;
		n = j;
	}
	
	while (n > 0) {
		ifp_print("%s", prefix);
		hexdump_line(p, n, lw);
		n -= lw;
		if (n > 0) {
			p += lw;
		} else {
			if (zeros > 0) {
				ifp_print(" [%d]", total);
			}
		}
		ifp_print("\n");
	}
	return 0;
}
#endif


//
// primatives
//

//Returns a negative number on error, 0 on success, and the number of
//bytes written (if the whole block wasn't written).
static inline int _ifp_push(struct ifp_device * dev, void * p, int n)
{
	int i = 0;

	i = ifp_os_push(dev, p, n);

#ifdef IFP_DEBUG_USB_RAWDATA
	hexdump("<--dn  ", p, n);
#endif

	return i;
}

/* Note that 'n' must be a power of 2. */
static inline int _ifp_pop(struct ifp_device * dev, void * p, int n)
{
	int i = 0;

	i = ifp_os_pop(dev, p, n);

#ifdef IFP_DEBUG_USB_RAWDATA
	hexdump("-->up  ", p, n);
#endif

	return i;
}

static inline int _ifp_push_unicode(struct ifp_device * dev, const char * s, int blocksize)
{
	int i;
	uint8_t * buf = dev->b1;

	memset(buf, 0, IFP_BUFFER_SIZE);

	//Warning about possible buffer overrun: here 'n' is the
	//size of the buffer 's', but we have no way of preventing
	//'buf' from being overrun.  (We can tell afterwards, but by
	//then it's too late.)
	
	i = ifp_utf8_to_utf16(buf, IFP_BUFFER_SIZE, s, strlen(s)+1);
	ifp_err_jump(i, err, "character conversion failed");

	i = ifp_os_push(dev, buf, blocksize);

err:
#ifdef IFP_DEBUG_USB_SNOOPING
	ifp_print("<--dn '%s'[unicode]\n", s);
#endif // IFP_DEBUG_USB_SNOOPING
	return i;
}

static inline int _ifp_pop_unicode(struct ifp_device * dev, char * s, int n,
		int blocksize)
{
	int i;
	uint8_t * buf = dev->b1;

	memset(buf, 0, IFP_BUFFER_SIZE);
	i = ifp_os_pop(dev, buf, blocksize);
	if (i) {
		//Either an error occured, or fewer bytes than 'n' was read.
		if (i < 0) {
			ifp_err_i(i, "pop error.");
		} else {
			ifp_err("pop read only %d bytes.", i);
		}
		return i;
	}
	i = ifp_utf16_to_utf8(s, n, buf, IFP_BUFFER_SIZE);
	ifp_err_jump(i, err, "character conversion failed");
err:
#ifdef IFP_DEBUG_USB_SNOOPING
	ifp_print("-->up '%s'[unicode]\n", s);
#endif // IFP_DEBUG_USB_SNOOPING

	return 0;
}

#if 0
//Returns a negative number on error, 0 on success, and the number of
//bytes written (if the whole block wasn't written).
static inline int _ifp_set_buffer_size(struct ifp_device * dev, int n,
	int force)
{
	int i = 0, allowed;
	if (dev->last_buffer_size != n || force) {
		i = ifp_control_send(dev, IFP_SET_BUFFER, n, 0, &allowed);
		if (allowed >= 0) {
			dev->last_buffer_size = allowed;
		}
		if (i == 1 && allowed == n) {
			return 0;
		}
	}
	//return 0;
	return i ? i : 0;
}
#endif

static inline int _ifp_file_open(struct ifp_device * dev, int type)
{ return ifp_control_send_bool(dev, IFP_FILE_OPEN, type, 0, NULL); }

static inline int _ifp_file_open_new(struct ifp_device * dev, int size)
{ return ifp_control_send_bool(dev, IFP_FILE_OPEN_NEW, size, 0, NULL); }

static inline int _ifp_file_close(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_FILE_CLOSE, 0, 0, NULL); }

static inline int _ifp_file_size(struct ifp_device * dev)
{
	int i, n;
	i = ifp_control_send_bool(dev, IFP_FILE_SIZE, 0, 0, &n);
	if (i) {
		return i;
	}
	return n;
}

#if 0
/* returns 0 on success, 1 on "device not ready!" and negative on error. */
static inline int _ifp_file_download(struct ifp_device * dev, int bytes, int * actual)
{
	int i;
	i = ifp_control_send(dev, IFP_FILE_DOWNLOAD, bytes, 0, NULL);
	if (actual) {
		*actual = i;
	}
	if (i < 0) {
		ifp_err_i(i, "error sending code to download block.");
		return i;
	} else if (i == bytes) {
		return 0;
	} else if (i > bytes) {
		ifp_err_i(i, "Something's wierd.  The return value is larger than %d", bytes);
		return -1;
	} else if (i == 0) {
		ifp_wrn("warning: return value is 0 instead of %d, which is often a sign of corruption.", bytes);
		return 1;
	} else {
		//ifp_wrn("[_ifp_file_download] warning: got %d instead of %d.\n", i, bytes);
		return 0;
	}
}
#endif

static inline int _ifp_dir_open(struct ifp_device * dev)
{ return ifp_control_send(dev, IFP_LS_OPEN, 0, 0, NULL); }

static inline int _ifp_dir_next(struct ifp_device * dev, int mode)
{ return ifp_control_send(dev, IFP_LS_NEXT, mode, 0, NULL); }

static inline int _ifp_dir_close(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_LS_CLOSE, 0, 0, NULL); }


//Returns a negative number on error, 0 on success, and the number of
//bytes written (if the whole block wasn't written).
static inline int _ifp_file_upload(struct ifp_device * dev, int bytes)
{
	int i;
	i = ifp_control_send(dev, IFP_FILE_UPLOAD, bytes, 0, NULL);
	if (i >= 0 && i == bytes) {
		return 0;
	}
	return i < 0 ? i : -1;
}

static inline int _ifp_file_flush(struct ifp_device * dev)
{
	int n;
	return ifp_control_send_bool(dev, IFP_02_COMMAND, 0, IFP_02_UPLOAD_FLUSH, &n);
	//printk("[_ifp_file_flush] info, n=%d\n", n);
}

static inline int _ifp_delete(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_FILE_DELETE, 0, 0, NULL); }

static inline int _ifp_mkdir(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_DIR_CREATE, 0, 0, NULL); }

static inline int _ifp_rmdir(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_DIR_DELETE, 0, 0, NULL); }

static inline int _ifp_format(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_FORMAT, 0, 0, NULL); }

static inline int _ifp_update_firmware(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_FIRMWARE_UPDATE, 0, 0, NULL); }

//return 1 on success, 0 on 'done', and negative on error
static inline int _ifp_dir_next_debug(struct ifp_device * dev, int mode)
{ return ifp_control_send(dev, IFP_LS_NEXT_DEBUG, mode, 0, NULL); }

static inline int _ifp_get_fat_page(struct ifp_device * dev, int dir, int page)
{ return ifp_control_send_bool(dev, IFP_GET_FAT_PAGE, dir, page, NULL); }

static inline int _ifp_set_fat_page(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_SET_FAT_PAGE, 0, 0, NULL); }

static inline int _ifp_get_tuner_preset(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_GET_PRESET, 0, 0, NULL); }

static inline int _ifp_set_tuner_preset(struct ifp_device * dev)
{ return ifp_control_send_bool(dev, IFP_SET_PRESET, 0, 0, NULL); }


static inline int get_control_value(struct ifp_device * dev,
	int cmd, int arg1, int arg2)
{
	int i,n;
	i = ifp_control_send_bool(dev, cmd, arg1, arg2, &n);
	if (i < 0) {
		return i;
	}
	return n;
}

static inline int _ifp_echo(struct ifp_device * dev)
{ return ifp_control_send(dev, IFP_ECHO, 0, 0, NULL); }

static inline int _ifp_golf(struct ifp_device * dev)
{ return ifp_control_send(dev, IFP_GOLF, 0, 0, NULL); }

static inline int _ifp_battery(struct ifp_device * dev)
{ return get_control_value(dev, IFP_02_COMMAND, 0, IFP_02_BATTERY); }

static inline int _ifp_capacity(struct ifp_device * dev)
{ return get_control_value(dev, IFP_GET_CAPACITY, 0, 0); }

static inline int _ifp_freespace(struct ifp_device * dev)
{ return get_control_value(dev, IFP_GET_FREE, 0, 0); }

static inline int _ifp_model(struct ifp_device * dev)
{ return get_control_value(dev, IFP_02_COMMAND, 0, IFP_02_STRING); }

static inline int _ifp_firmware_version(struct ifp_device * dev)
{ return get_control_value(dev, IFP_02_COMMAND, 0, IFP_02_FIRMWARE); }

static inline int _ifp_delta(struct ifp_device * dev)
{ return get_control_value(dev, IFP_02_COMMAND, 0, IFP_02_DELTA); }

//
//public functions (to this module)
//

/*
 * Called on startup by the windows program.  Returns 0 the first time it is
 * called, and 1 every time after that until you powercycle the device.
 */
int ifp_echo(struct ifp_device * dev)
{
	return _ifp_echo(dev);
}

/*
 * Called on startup by the windows program, but only the second time.  (ie, when ifp_echo would
 * be expected to return 1). Called before ifp_echo; returns 1.
 *
 * (Note: returns 0 if called the first time, just like ifp_echo.)
 * (further note: does not appear to help reset the device.)
 */
int ifp_golf(struct ifp_device * dev)
{
	return _ifp_golf(dev);
}

/** \brief Initializes device.
 
  Initialzies and tests the device for use with device_handle.
  (::ifp_finalize should be called when you're finished with 'dev'.)
 
  \param dev Unitialized memory ready for use as an ifp_device.
  \param device_handle The iFP USB hardware device handle.
 
  \return ::IFP_ERR_DEV_FUBAR if the self-test failed.  (Ask the user to jiggle the handle.)
  \return 0 on success.
 */
int ifp_init(struct ifp_device * dev, void * device_handle) {
	int i = 0;

	if (dev == NULL) {
		ifp_err("Um, dev is NULL.");
		i = -EINVAL;
		goto out;
	}
	if (device_handle == NULL) {
		ifp_err("Um, device_handle is NULL.");
		i = -EINVAL;
		goto out;
	}

	i = ifp_os_init(dev, device_handle);
	if (i) {
		ifp_err_i(i, "ifp_os_init error.");
		//reserve postive values for special errors.
		i = i < 0 ? i : -1;
		goto out;
	}

	dev->last_buffer_size = 0;
	dev->mode = IFP_MODE_NONE;
	dev->download_pipe_errors = 0;

#if 0
	i = ifp_golf(dev);
	ifp_dbg("golf returned %d", i);

	i = ifp_echo(dev);
	ifp_dbg("echo returned %d", i);
#endif

	i = ifp_selftest(dev);
	if (i) {
		ifp_err_i(i, "self test failed.");
		i = IFP_ERR_DEV_FUBAR;
		goto out_err;
	}

	return 0;

out_err:
	ifp_os_finalize(dev);
out:
	return i;
}
IFP_EXPORT(ifp_init);

/** \brief Releases device.
 * Releases any resources aquired by ::ifp_init.  Basically, when
 * ::ifp_init returns 0 (success), ::ifp_finalize must be called 
 * after you're finished with dev.
 */
int ifp_finalize(struct ifp_device * dev) {
	int i = 0;
	
	if (dev->download_pipe_errors) {
		ifp_dbg("%d pipe errors were counted", dev->download_pipe_errors);
	}

	i = ifp_os_finalize(dev);
	ifp_err_jump(i, out, "ifp_os_finalize returned an error.");

out:
	return i;
}
IFP_EXPORT(ifp_finalize);

/** returns -ENOENT */
int ifp_file_open(struct ifp_device * dev, const char * s)
{
	int i;
	//i = _ifp_set_buffer_size(dev, n, 1);
	i = _ifp_set_buffer_size(dev, IFP_PATH_XFER_SIZE, 1);
	ifp_err_jump(i, out, "set buffer failed");
	i = _ifp_push_unicode(dev, s, IFP_PATH_XFER_SIZE);
	ifp_err_jump(i, out, "push failed");
	i = _ifp_file_open(dev, IFP_FILE);
	if (i == 1) {
		i = -ENOENT;
	} else if (i != 0) {
		ifp_err_i(i, "open directive failed");
	}

out:
	return i;
}

static int check_path_string(const char * f, int max) {
	int n;
	int pn;
	char * c;

	if (strpbrk(f, "/:*?\"<>|")) {
		return IFP_ERR_BAD_FILENAME;
	}

	n = strlen(f);
	c = strrchr(f, '\\');
	if (c == NULL) {
		if (n >= IFP_MAX_FILENAME) {
			//ifp_dbg("filename too long (%d) for ifp:\\%s",n,f);
			return -1;
		}
		return 0;
	}
	pn = (int)(c-f);
	
	if (n - pn >= IFP_MAX_FILENAME) {
		//ifp_dbg("filename too long (%d) for ifp:\\%s",n-pn,f);
		return IFP_ERR_BAD_FILENAME;
	}
	if (n >= IFP_MAX_PATH) {
		//ifp_dbg("path too long (%d) for ifp:\\%s",n,f);
		return IFP_ERR_BAD_FILENAME;
	}
	return 0;
}

/** returns -EEXIST and IFP_ERR_BAD_FILENAME */
int ifp_file_open_new(struct ifp_device * dev, const char * s, int filesize)
{
	int i;
	//i = _ifp_set_buffer_size(dev, n, 1);
	i = _ifp_set_buffer_size(dev, IFP_PATH_XFER_SIZE, 0);
	if (i) { ifp_err_i(i, "set buffer failed"); return i; }
	i = _ifp_push_unicode(dev, s, IFP_PATH_XFER_SIZE);
	if (i) { ifp_err_i(i, "push failed"); return i; }
	i = _ifp_file_open_new(dev, filesize);
	if (i == 1) {
		i = check_path_string(s, IFP_PATH_XFER_SIZE);
		if (i == 0) {
			i = -EEXIST;
		}
	} else if (i != 0) {
		ifp_err_i(i, "open_new directive failed");
	}

	return i;
}

int ifp_file_close(struct ifp_device * dev)
{
	return _ifp_file_close(dev);
}

int ifp_file_size(struct ifp_device * dev)
{
	return _ifp_file_size(dev);
}

//I'd like to move this out of this file, eventually
int ifp_read_seek_forward(struct ifp_device * dev, int count, int blocksize)
{
	int i, j, actual;
	i = _ifp_set_buffer_size(dev, blocksize, 1);
	if (i) {
		ifp_err_i(i, "set buffer failed");
		return i > 0 ? -EIO : i;
	}
	for (j=0; j != count; j++) {
		i = _ifp_file_download(dev, blocksize, &actual);
		if (i) {
			ifp_err_i(i, "download control message failed");
			return i;
		}
		if (actual != blocksize) {
			i = -EIO;
			ifp_err("seek failed,  I can't handle "
				"getting %d bytes instead of %d\n", i, blocksize
			);
			return -EIO;
		}
	}
	return 0;
}


//returns the number of bytes received, or a negative number on error.
int ifp_file_download(struct ifp_device * dev, void * p, int n)
{
	int i;
	int actual;

	if (n == 0) {
		ifp_err("refusing to download 0 bytes.");
		return -1;
	}

	i = _ifp_set_buffer_size(dev, n, 1);
	if (i) {
		ifp_err_i(i, "set buffer failed");
		return i > 0 ? -EIO : i;
	}
	i = _ifp_file_download(dev, n, &actual);
	if (i) {
		ifp_err_i(i, "download control code failed");
		return i;
	}

	if (actual == 0) {
		ifp_wrn("warning: zero bytes available for download.");
		return 0;
	}
	//printk("[ifp_file_download] download returned %d\n",i);
	n = actual;

	i = _ifp_pop(dev, p, n);
	if (i) {
		//Either an error occured, or fewer bytes than 'n' was read.
		if (i < 0) {
			ifp_err_i(i, "pop failed");
		} else {
			ifp_err_i(i, "pop returned an unexpected value (fewer bytes read than expected.)");
			i = -1;
		}
		return i;
	}

	return n;
}

int ifp_file_upload(struct ifp_device * dev, void * p, int n)
{
	int i;
	if (n == 0) {
		ifp_err("refusing to upload 0 bytes.");
		return -1;
	}
	i = _ifp_set_buffer_size(dev, n, 1);
	ifp_err_jump(i, out, "set buffer failed");
	i = _ifp_push(dev, p, n);
	ifp_err_jump(i, out, "push failed");
	i = _ifp_file_upload(dev, n);
	ifp_err_jump(i, out, "upload control code failed");

out:
	return i;
}

int ifp_file_flush(struct ifp_device * dev)
{
	unsigned char buf[4];
	int i;
	int status;
	int n = 0;
	int const obsene_limit = 1000;

	do {
		//Sleep even on the first loop.. it's good for soul.. call it "kerma"
		i = ifp_os_sleep(100);
		ifp_err_jump(i, out, "trouble falling asleep, loading Dickens..");

		i = _ifp_file_flush(dev);
		ifp_err_jump(i, out, "error issuing 'flush' command");

		i = _ifp_pop(dev, buf, sizeof(buf));
		ifp_err_jump(i, out, "pop failed");

		//flush returns 1 when it's ready to continue, but 2 when
		//it's still writing to flash.
		status = ifp_os_le32_to_cpup(buf);
		if (status == 1) {
			//done
			return 0;
		}
		if (n == 5) {
			ifp_wrn("[ifp_file_flush] waiting for device 'ready'.");
			ifp_wrn("[ifp_file_flush] (normally it doesn't take this long).");
		}

		//infinite loop plug
		n++;
		if (n == obsene_limit) {
			ifp_err("timeout: hit obscene limit");
			i = -1; goto out;
		}
	} while(status == 2);

	ifp_err("unrecognized return value %d.",i);
	i = -1;
out:
	return i;
}

/** \brief Reports the battery's status on the scale from 0 to 4.
 *
 * Typical values are 4, 0 and occasionally 2. */
int ifp_battery(struct ifp_device * dev)
{
	uint8_t buf[4];
	int n, i = 0;
	n = _ifp_battery(dev);
	if (n < 0) {
		ifp_err_i(n, "error reading battery");
		return n;
	}
	IFP_BUG_ON(n != sizeof(buf));
	i = _ifp_pop(dev, buf, sizeof(buf));
	ifp_err_jump(i, out, "pop failed");

	n = ifp_os_le32_to_cpup(buf);

	return n;
out:
	return i;
}

/** returns -ENOENT if dir doesn't exist */
int ifp_dir_open(struct ifp_device * dev, const char * s)
{
	int i;
	//i = _ifp_set_buffer_size(dev, IFP_BUFFER_SIZE, 1);
	i = _ifp_set_buffer_size(dev, IFP_PATH_XFER_SIZE, 1);
	ifp_err_jump(i, out, "set buffer failed");
	i = _ifp_push_unicode(dev, s, IFP_PATH_XFER_SIZE);
	ifp_err_jump(i, out, "push failed");
	i = _ifp_dir_open(dev);
	if (i < 0) {
		ifp_err_i(i, "open directive failed");
	} else if (i == 0) {
		//dir doesn't exist, so it wasn't opened.  This isn't
		//an error, so don't print any messages to the user.
		i = -ENOENT;
	} else if (i == 1) {
		//dir found.. open succeeded.
		i = 0;
	} else {
		ifp_err_i(i, "unexpected return value");
		i = -1;
	}

out:
	return i;
}

int ifp_dir_next(struct ifp_device * dev, void * s, int n, int mode)
{
	int i;
	//printk("[ifp_dir_next] buffer is %#03x, n=%#03x.\n", sizeof(buf), n);
	//i = _ifp_set_buffer_size(dev, IFP_BUFFER_SIZE, 1);
	
	//FIXME: IFP_MAXLSLEN   0x80
	//use it for 'n' in here.  Or check that n is 0x80
	i = _ifp_dir_next(dev, mode);
	if (i < 0) {
		ifp_err_i(i, "error requesting next filename");
		return i;
	} else if (i == 0) {
		//no more files to return.  Just to be nice,
		//we're putting a null terminator on s.
		if (n > 0) {
			*((char *)s) = 0;
		}
		return i;
	}
	mode = i;

	i = _ifp_pop_unicode(dev, s, n, IFP_PATH_XFER_SIZE);
	if (i) {
		ifp_err_i(i, "pop failed");
		return i > 0 ? -1 : i;
	}
	return mode;
}

int ifp_dir_close(struct ifp_device * dev)
{
	return _ifp_dir_close(dev);
}

/** \brief Delete the file f.
 *
 * Returns -ENOENT if f doesn't exist.
 */
int ifp_delete(struct ifp_device * dev, const char * f)
{
	int i = 0;
	i = _ifp_set_buffer_size(dev, IFP_PATH_XFER_SIZE, 1);
	ifp_err_jump(i, out, "set buffer failed");
	i = _ifp_push_unicode(dev, f, IFP_PATH_XFER_SIZE);
	ifp_err_jump(i, out, "push failed");
	i = _ifp_delete(dev);
	if (i < 0) {
		ifp_err_i(i, "open request failed");
		return i;
	} else if (i == 1) {
		//ifp_err("file not found");
		return -ENOENT;
	}
	i = 0;

out:
	return i;
}
IFP_EXPORT(ifp_delete);

/** \brief Creates a new directory, f.
 *
 * Returns -ENOENT if f's parent doesn't exist, -EEXISTS the
 * dirname 'f' is allready in use, and IFP_ERR_BAD_FILENAME
 * if 'f' contains unsupported characters.
 */
int ifp_mkdir(struct ifp_device * dev, const char * f)
{
	int i;

	i = check_path_string(f, IFP_PATH_XFER_SIZE);
	if (i == IFP_ERR_BAD_FILENAME) {
		return i;
	}

	i = _ifp_set_buffer_size(dev, IFP_PATH_XFER_SIZE, 1);
	ifp_err_jump(i, out, "set buffer failed");
	i = _ifp_push_unicode(dev, f, IFP_PATH_XFER_SIZE);
	ifp_err_jump(i, out, "push failed");
	i = _ifp_mkdir(dev);
	if (i < 0) {
		ifp_err_i(i, "open failed");
		return i;
	} else if (i == 1) {
		//ifp_err("no such directory");

		i = ifp_exists(dev, f);
		if (i == 1 || i == 2) {
			//ifp_dbg("directory exists, i=%d, f='%s'", i,f);
			return -EEXIST;
		} else if (i == 0) {
			return -ENOENT;
		} else {
			ifp_err("can't determine cause of error");
			return -1;
		}
	}
	

	i = 0;
out:
	return i;
}
IFP_EXPORT(ifp_mkdir);

/* \brief Deletes the directory f.
 *
 * (f must be empty.)*/
int ifp_rmdir_nocheck(struct ifp_device * dev, const char * f)
{
	int i;
	i = _ifp_set_buffer_size(dev, IFP_PATH_XFER_SIZE, 1);
	ifp_err_jump(i, out, "set buffer failed");
	i = _ifp_push_unicode(dev, f, IFP_PATH_XFER_SIZE);
	ifp_err_jump(i, out, "push failed");
	i = _ifp_rmdir(dev);
	if (i < 0) {
		ifp_err_i(i, "open failed");
		return i;
	} else if (i == 1) {
		ifp_err("no such directory");
		return -ENOENT;
	}

	i = 0;
out:
	return i;
}

/** \brief Reports the device's capacity in bytes. */
int ifp_capacity(struct ifp_device * dev)
{ return _ifp_capacity(dev); }
IFP_EXPORT(ifp_capacity);
/** \brief Reports the device's available free space in bytes. */
int ifp_freespace(struct ifp_device * dev)
{ return _ifp_freespace(dev); }
IFP_EXPORT(ifp_freespace);

/** \brief Reformats the device's storage media.
 *
 * (Deletes all your stuff.)  Returns 0 on success, 1 on error.
 * This function hasn't been tested--please report if you've successfully used
 * it.
 */
int ifp_format(struct ifp_device * dev)
{
	int i = 0;
	//implemenation note: I don't know if it actually returns 0 on success.
	i = _ifp_format(dev);

	//FIXME: figure out how to send the previous URB with a much longer timeout.
	if (i == -ETIMEDOUT) {
		i = 0;
	}
	return i;
}
IFP_EXPORT(ifp_format);

/** \brief Reformats the device's storage media.
 *
 * (Deletes all your stuff.)  Returns 0 on success, 1 on error.
 * This function hasn't been tested--please report if you've successfully used
 * it.
 */
int ifp_update_firmware_raw(struct ifp_device * dev)
{
	int i = 0;

	//implemenation note: I don't know if it actually returns 0 on success.
	i = _ifp_update_firmware(dev);
	//FIXME: figure out how to send the previous URB with a much longer timeout.
	if (i == -ETIMEDOUT) {
		i = 0;
	}
	return i;
}
IFP_EXPORT(ifp_update_firmware_raw);

/** \brief (experimental) retrieves a mystery value.
 *
 * I've coined this mystery value "Delta" until a better name is chosen.
 *
 * \param values an empty int[4] for the output values.
 *
 * Integers returned in 'values' are my interpretation of the data.
 * The actual raw data has been varried widely:
 *
 *  \code
 * 0108 0312 ffff ffff
 * 0108 0415 ffff ffff
 * 0108 0616 ffff ffff
 * 0108 0417 0000 0000
 * 0108 0418 ffff ffff
 *  \endcode
 *
 * Two devices returned 4 bytes instead of 8.
 */
int ifp_delta(struct ifp_device * dev, int * values)
{
	uint8_t buf[8];
	int i = 0;
	int bytes;
	unsigned int nn;

	bytes = _ifp_delta(dev);
	if (bytes < 0) {
		i = bytes;
		ifp_err_i(i, "error sending control code");
		goto out;
	} else if (bytes > 8) {
		i = -1;
		ifp_err("unexpected buffer size of %d, which is more than %d",
			bytes, (int)sizeof(buf));
		goto out;
	}
	if (bytes != 8 && bytes != 4) {
		ifp_wrn("interesting, %d bytes are being sent.", bytes);
	}

	i = _ifp_pop(dev, buf, bytes);
	ifp_err_jump(i, out, "pop failed");

	values[0] = buf[0];
	values[1] = buf[1];
	values[2] = buf[2];
	values[3] = buf[3];

	if (bytes > 4) {
		nn = ifp_os_le32_to_cpup(buf + 4);
		if ( nn != 0xffffffff ) {
			ifp_wrn("interesting, the last 4 bytes are %08x.", nn);
		}
	} else {
		ifp_wrn("interesting, there were only %d bytes.", bytes);
	}

out:
	return i;
}
IFP_EXPORT(ifp_delta);

/** \brief Reads in the device's model number into 's'.
 *
 * ('size' is the size of the buffer s points to.)
 * Typical results look like "IFP-590T".
 */
int ifp_model(struct ifp_device * dev, char * s, int size)
{
	int n, i = 0;
	n = _ifp_model(dev);
	if (n < 0) {
		ifp_err_i(n, "error reading device model string");
		return n;
	}
	if (n >= size) {
		ifp_wrn("warning: the buffer is too small for the model string.  Truncating.  (%d instead of %d.)", n, size);
		n = size - 1;
	}
	i = _ifp_pop(dev, s, n);
	ifp_err_jump(i, out, "pop failed");

	s[n] = '\0';

out:
	return i;
}
IFP_EXPORT(ifp_model);

/** \brief Reads the device's firmware version.
 *  The firmware version is returned in raw BCD.  For human consumption,
 *  I suggest:
 *
 *  \code
 *  	sprintf(s, "%x.%02x", r/0x0100, r%0x100)
 *  \endcode
 */
int ifp_firmware_version(struct ifp_device * dev)
{
	uint8_t * s = dev->b1;
	const int transfer_size = 64;
	int n, i;

	n = _ifp_firmware_version(dev);
	if (n < 0) {
		ifp_err_i(n, "error reading device model string");
		return n;
	}
	//printk("[ifp_firmware_version] info, asking for %d bytes.\n", n);
	if (n >= transfer_size) {
		ifp_wrn("warning: the buffer is too small for the firmware string.  Truncating.  (%d instead of %d.)", n, transfer_size);
		n = transfer_size - 1;
	}
	i = _ifp_pop(dev, s, n);
	if (i) {
		ifp_err_i(i, "pop failed");
		return i >= 0 ? -1 : i;
	}
	//s[n] = '\0';
	
	return ifp_os_le32_to_cpup(s);
}

// 0x0100 >= sizeof(raw) >= 0x0010
/*
 * dir : directory code
 * slot : the first slot assigned to this entry.
 * size : number of slots used in this directory entry.  (between 1 and 9?)
 *
 * eg:
 * 	If an entry returned 'slot=42' and 'size=5', you can expect to find
 * 	data for this entry in slots 42-46, inclusive:
 *
 * 	40. .............
 * 	41. .............
 * 	42. extra data #4
 * 	43. extra data #3
 * 	44. extra data #2
 * 	45. extra data #1
 * 	46. main entry
 * 	47. .............
 * 	48. .............
 * 	49. .............
 */
int ifp_dir_next_debug(struct ifp_device * dev, char * s, int n, int mode,
	int * dir, int * slot, int * size)
{
	uint8_t * spec_buff = dev->b1;

	int i;
	if (dev == NULL) {
		ifp_err("dev is NULL..");
	}

	//mode = 0;
	i = _ifp_dir_next_debug(dev, mode);
	if (i == 0) {
		//no more files to return.  Just to be nice,
		//we're putting a null terminator on s.
		if (n > 0) {
			*((char *)s) = 0;
		}
		return 0;
	} else if (i < 0 || i != 1) {
		ifp_err_i(i, "error getting next file");
		return i < 0 ? i : -1;
	}

	i = _ifp_pop_unicode(dev, s, n, IFP_PATH_XFER_SIZE);
	if (i) {
		//Either an error occured, or fewer bytes than 'n' was read.
		if (i < 0) {
			ifp_err_i(i, "pop error");
		} else {
			ifp_err_i(i, "unexpected pop return value");
		}
		return i;
	}

	i = _ifp_dir_next_debug(dev, mode);
	if (i != 1) {
		ifp_err_i(i, "error requesting file debug info for %s", s);
		return i < 0 ? i : -1;
	}
	//buffer is actually IFP_BUFFER_SIZE
	i = _ifp_pop(dev, spec_buff, 0x0100);
	if (i) {
		ifp_err_i(i, "error getting file debug info for %s", s);
		return i < 0 ? i : -1;
	}
	if (dir) {
		*dir = ifp_os_le16_to_cpup(spec_buff);

		if (*dir != ifp_os_le16_to_cpup(spec_buff + 2)) {
			ifp_wrn("warning %d != %d [2]",
				*dir, ifp_os_le16_to_cpup(spec_buff + 2));
		}
		if (*dir != ifp_os_le16_to_cpup(spec_buff + 4)) {
			ifp_wrn("warning %d != %d [4]",
				*dir, ifp_os_le16_to_cpup(spec_buff + 4));
		}
		//BUG_ON(*dir != ifp_os_le16_to_cpup(spec_buff + 2));
		//BUG_ON(*dir != ifp_os_le16_to_cpup(spec_buff + 4));
	}

	if (size) {
		*size = ifp_os_le16_to_cpup(spec_buff + 6);
	}

	//Combined with the directory number, this is effectively an 'inode number'.
	i = ifp_os_le16_to_cpup(spec_buff + 8);

	//Point to the begining of the last entry, not the end of the last (and
	//all) entries
	if (slot) {
		*slot = i - *size;
		if (*slot < 0) {
			ifp_err("slot calculated as %d, i=%d, size=%d", *slot, i, *size);
			return -1;
		}
	}
	//IFP_FAT_SLOTS_PER_PAGE;

	i = ifp_os_le16_to_cpup(spec_buff + 10);
	if (i != 0x0100) {
		ifp_wrn("warning: the field at +10 is %04x instead of 0x0100."
			"  (For %s)", i, s);
	}

	mode = ifp_os_le16_to_cpup(spec_buff + 12);

	if (mode == 0) {
		ifp_err_i(mode, "more error (on %s)", s);
		return -1;
	}

	return mode;
}

/**
 * Loads a page from a directory's FAT.  Pages are numbered from 0 onwards.
 * Each page holds ::IFP_FAT_SLOTS_PER_PAGE (16) slots, so you can calculate
 * which page you will find which slots.  Eg, page 0 has slots 0-15, page 1
 * slots 16-31.. etc.
 *
 * Note that fat pages are IFP_FAT_PAGE_SIZE bytes each.
 */
int ifp_get_fat_page(struct ifp_device * dev, int dir, int page,
	void * p, int n)
{
	const int DOWNLOAD_PAGE_SIZE = IFP_FAT_PAGE_SIZE/2;
	int i = 0;

	IFP_BUG_ON(n < DOWNLOAD_PAGE_SIZE*2);
	i = _ifp_get_fat_page(dev, dir, page);
	ifp_err_jump(i, out,
		"error requesting chuck #1 of (%#x, %#x)\n", dir, page);

	i = _ifp_pop(dev, p, DOWNLOAD_PAGE_SIZE);
	ifp_err_jump(i, out,
		"error downloading chuck #1 of (%#x, %#x)\n", dir, page);

	i = _ifp_get_fat_page(dev, dir, page);
	ifp_err_jump(i, out,
		"error requesting chuck #2 of (%#x, %#x)\n", dir, page);

	i = _ifp_pop(dev, (uint8_t *)p + DOWNLOAD_PAGE_SIZE, DOWNLOAD_PAGE_SIZE);
	ifp_err_jump(i, out,
		"error downloading chuck #2 of (%#x, %#x)\n", dir, page);

out:
	return i;
}

/* Saves a fat page, overwriting previous contents.
 * Note that fat pages are IFP_FAT_PAGE_SIZE bytes each.
 */
int ifp_set_fat_page(struct ifp_device * dev, int dir, int page,
	void * p, int n)
{
	const int HEADER = 0x0010;
	uint8_t * buf = dev->b1;
	int i = 0;

	IFP_BUG_ON(n < IFP_FAT_PAGE_SIZE);

	//Create the upload payload.
	memset(buf, 0, HEADER);
	memcpy(buf + HEADER, p, IFP_FAT_PAGE_SIZE);
	((uint16_t *)(buf))[0] = ifp_os_cpu_to_le16(dir);
	((uint16_t *)(buf))[1] = ifp_os_cpu_to_le16(page);

	i = _ifp_set_buffer_size(dev, IFP_FAT_PAGE_SIZE + HEADER, 1);
	ifp_err_jump(i, out, "error setting buffer size");

	i = _ifp_push(dev, buf, IFP_FAT_PAGE_SIZE + HEADER);
	ifp_err_jump(i, out, "error pushing data for (%d, %d)",
		dir, page);

	i = _ifp_set_fat_page(dev);
	ifp_err_jump(i, out, "error setting FAT page (%d, %d)", dir, page);

out:
	return i;
}

#define IFP_PRESET_GET_BUF 0x0100
#define IFP_PRESET_SET_BUF 0x0200
#define IFP_PRESET_ENTRY 12 

/** \brief Retrieves the tuner preset file into 'data'.
 *
 * 'data' is a buffer of 'n' bytes.. n must be at least
 * ::IFP_TUNER_PRESET_DATA bytes.
 */
int ifp_get_tuner_presets(struct ifp_device * dev, void * data, int n)
{
	int i = 0;
	uint8_t * buf = dev->b1;

	if (n < IFP_TUNER_PRESET_DATA) {
		ifp_err("buffer too small");
		return -1;
	}

	i = _ifp_get_tuner_preset(dev);
	ifp_err_jump(i, out, "problem requesting first chunk");

	i = _ifp_pop(dev, buf, IFP_PRESET_GET_BUF);
	ifp_err_jump(i, out, "problem retrieving first chunk");
	memcpy(data, buf+2*IFP_PRESET_ENTRY, 10*IFP_PRESET_ENTRY);


	i = _ifp_get_tuner_preset(dev);
	ifp_err_jump(i, out, "problem requesting second chunk");

	i = _ifp_pop(dev, buf, IFP_PRESET_GET_BUF);
	ifp_err_jump(i, out, "problem retrieving second chunk");
	memcpy(data + 10*IFP_PRESET_ENTRY, buf, 10*IFP_PRESET_ENTRY);

out:
	if (i > 0) {
		ifp_err("returning silent error");
	}
	return i>0 ? -1 : i;
}

/** \brief Stores the tuner preset file 'data' on the device.
 *
 * 'data' is tuner preset file to be stored.. it is extactly
 * ::IFP_TUNER_PRESET_DATA bytes.  'n' must be exactly ::IFP_TUNER_PRESET_DATA
 *
 */
int ifp_set_tuner_presets(struct ifp_device * dev, void * data, int n)
{
	//20*12bytes = 240bytes.
	int i = 0;
	uint8_t * buf = dev->b1;

	#if IFP_BUFFER_SIZE < IFP_PRESET_SET_BUF
		#error "assumptions no longer true."
	#endif

	if (n != IFP_TUNER_PRESET_DATA) {
		ifp_err("this buffer is %d instead of %d bytes.",n,
			IFP_TUNER_PRESET_DATA);
		return -1;
	}
	memset(buf, 0, IFP_BUFFER_SIZE);
	memcpy(buf, data, IFP_TUNER_PRESET_DATA);

	i = _ifp_set_buffer_size(dev, IFP_PRESET_SET_BUF, 1);
	ifp_err_jump(i, out, "error setting buffer size");

	i = _ifp_push(dev, buf, IFP_PRESET_SET_BUF);
	ifp_err_jump(i, out, "error sending data");

	i = _ifp_set_tuner_preset(dev);
	ifp_err_jump(i, out, "problem sending 'save preset' command");

out:
	if (i > 0) {
		ifp_err("returning silent error");
	}
	return i>0 ? -1 : i;
}


