/*  Inti-GConf: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*  BasicGConfApp demonstrates how to use Inti-GConf to access the GConf database. The key
 *  thing is that the main window and the preference dialog have NO KNOWLEDGE of one another
 *  as far as configuration values are concerned; they don't even have to be in the same 
 *  process. That is, the GConf::Client acts as the data "model" for configuration information;
 *  the main application is a "view" of the model; and the preference dialog is a "controller."
 *
 *  You can tell if your application has done this correctly by using "gconftool" instead of
 *  your preferences dialog to set preferences. For example, try:
 *
 *      gconftool --type=string --set /apps/basic-gconf-app/foo "My string"
 *
 *  If that doesn't work every bit as well as setting the value via the prefs dialog, then you
 *  aren't doing things right. ;-)
 *
 *  If you really want to be mean to your application, make it survive this:
 *
 *      gconftool --break-key /apps/basic-gconf-app/foo
 *
 *  Remember, the GConf database is just like an external file or the network - it may have
 *  bogus values in it. GConf admin tools will let people put in whatever they can think of.
 *
 *  GConf does guarantee that string values will be valid UTF-8, for convenience.
 *
 *  Throughout, this program is letting GConf::Client use the default error handlers rather
 *  than checking for errors or attaching custom handlers to the "unreturned_error" signal.
 *
 *  Note: special mention of an idiom often used in GTK+ apps that does not work right with
 *  GConf but may appear to at first:
 *
 *      bool i_am_changing_value;
 *
 *      i_am_changing_value = true;
 *      change_value(value);
 *      i_am_changing_value = false;
 *
 *  This breaks for several reasons: notification of changes may be asynchronous, you may get
 *  notifications that are not caused by change_value() while change_value() is running, since
 *  GConf will enter the main loop, and also if you need this code to work you are probably 
 *  going to have issues when someone other than yourself sets the value.
 *
 *  A robust solution in this case is often to compare the old
 *  and new values to see if they've really changed, thus avoiding
 *  whatever loop you were trying to avoid.
 */

#include <inti/main.h>
#include <inti/core.h>
#include <inti/gconf/client.h>
#include <inti/gconf/value.h>
#include <inti/gtk/entry.h>
#include <inti/gtk/frame.h>
#include <inti/gtk/label.h>

using namespace Inti;

/* ConfigEntry is used to edit the given config key
 */

class ConfigEntry : public Gtk::HBox
{
	String key;
	Gtk::Entry *entry;

public:
	ConfigEntry(const char *config_key);
	~ConfigEntry();
	
	const String& get_key();
	Gtk::Entry* get_entry();
};

/*  PreferenceDialog
 */

class PreferenceDialog : public Gtk::Dialog
{
	GConf::Client& client_;

protected:
	bool on_entry_focus_out(GdkEventFocus *event, ConfigEntry *entry);
	void on_entry_activate(ConfigEntry *entry);
	virtual void on_response(int response_id);

public:
	PreferenceDialog(GConf::Client& client, Gtk::Window& parent);
	~PreferenceDialog();

	ConfigEntry* create_config_entry(const char *config_key, bool focus = false);
};

/*  ConfigurableFrame creates a Gtk::Label inside a frame, that we can "configure" 
 *  to display the value of the config key.
 */

class ConfigurableFrame : public Gtk::Frame
{
	GConf::Client& client_;

	Pointer<GConf::Client::NotifySlot> notify_slot;
	unsigned int notify_id;

	Gtk::Label *label;

protected:
	void on_frame_notify(unsigned int cnxn_id, const GConf::Entry& entry);

public:
	ConfigurableFrame(GConf::Client& client, const char *config_key);
	~ConfigurableFrame();
};

/*  BasicGConfApp
 */

class BasicGConfApp : public Gtk::Window
{
	Pointer<GConf::Client> client;

protected:
	void on_button_clicked();

public:
	BasicGConfApp();
	virtual ~BasicGConfApp();
};

