/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/viewport.h
//! @brief A GtkViewport C++ wrapper interface.
//! 
//! Provides Viewport, a widget that allows you to see part of a child widget and scroll
//! the contents of the child.

#ifndef INTI_GTK_VIEWPORT_H
#define INTI_GTK_VIEWPORT_H

#ifndef INTI_GTK_BIN_H
#include <inti/gtk/bin.h>
#endif

#ifndef __GTK_VIEWPORT_H__
#include <gtk/gtkviewport.h>
#endif

namespace Inti {

namespace Gtk {

class ViewportClass;

//! @class Viewport viewport.h inti/gtk/viewport.h
//! @brief A GtkViewport C++ wrapper class.
//!
//! The Viewport widget provides you with the capability to see part of a child widget
//! and scroll the contents of the child. It is normally not accessed directly but used
//! by other widgets, particularly the ScrollWindow, for child widgets that do not have
//! their own scrolling capability.

class Viewport : public Bin
{
	friend class G::Object;
	friend class ViewportClass;

	Viewport(const Viewport&);
	Viewport& operator=(const Viewport&);

protected:
//! @name Constructors
//! @{

	explicit Viewport(GtkViewport *viewport, bool reference = false);
	//!< Construct a new Viewport from an existing GtkViewport.
	//!< @param viewport A pointer to a GtkViewport.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>viewport</EM> can be a newly created GtkViewport or an existing
	//!< GtkViewport (see G::Object::Object).
	
//! @}
//! @name Signal Handlers
//! @{

	virtual void on_set_scroll_adjustments(Adjustment *hadjustment, Adjustment *vadjustment);
	//!< Called when a viewport is added to a scrolling aware parent.
	//!< @param hadjustment The horizontal adjustment.
	//!< @param vadjustment The vertical adjustment.
	
//! @}
//  Properties

	typedef G::Property<Adjustment*, G::Object*> HAdjustmentPropertyType;
	typedef G::PropertyProxy<G::Object, HAdjustmentPropertyType> HAdjustmentPropertyProxy;
	static const HAdjustmentPropertyType hadjustment_property;

	typedef G::Property<Adjustment*, G::Object*> VAdjustmentPropertyType;
	typedef G::PropertyProxy<G::Object, VAdjustmentPropertyType> VAdjustmentPropertyProxy;
	static const VAdjustmentPropertyType vadjustment_property;

	typedef G::Property<ShadowType, int> ShadowTypePropertyType;
	typedef G::PropertyProxy<G::Object, ShadowTypePropertyType> ShadowTypePropertyProxy;
	static const ShadowTypePropertyType shadow_type_property;

//  Signals

	typedef G::Signal2<void, GtkAdjustment*, GtkAdjustment*> SetScrollAdjustmentsSignalType;
	typedef G::SignalProxy<TypeInstance, SetScrollAdjustmentsSignalType> SetScrollAdjustmentsSignalProxy;
	static const SetScrollAdjustmentsSignalType set_scroll_adjustments_signal;

public:
//! @name Constructors
//! @{

	Viewport();
	//!< Construct a new viewport.

	Viewport(Adjustment *hadjustment, Adjustment *vadjustment);
	//!< Construct a new viewport with the given adjustments.
	//!< @param hadjustment The horizontal adjustment. 
	//!< @param vadjustment The vertical adjustment. 
	
	virtual ~Viewport();
	//!< Destructor.
	
//! @}	
//! @name Accessors
//! @{

	GtkViewport* gtk_viewport() const { return (GtkViewport*)instance; }
	//!< Get a pointer to the GtkViewport structure.
	
	GtkViewportClass* gtk_viewport_class() const;
	//!< Get a pointer to the GtkViewportClass structure.
	
	operator GtkViewport* () const;
	//!< Conversion operator; safely converts a Viewport to a GtkViewport pointer.

	Adjustment* get_hadjustment() const;
	//!< Returns the horizontal adjustment of the viewport. If the adjustment has not
	//!< been set a default adjustment with all values set to zero will be returned.

	Adjustment* get_vadjustment() const;
	//!< Returns the vertical adjustment of the viewport. If the adjustment has not
	//!< been set a default adjustment with all values set to zero will be returned.

	ShadowType get_shadow_type() const;
	//!< Returns the shadow type of the viewport (see set_shadow_type()).

//! @}
//! @name Methods
//! @{

	void set_hadjustment(Adjustment *adjustment);
	//!< Sets the horizontal adjustment of the viewport.
	//!< @param adjustment An Adjustment.
	 	
	void set_vadjustment(Adjustment *adjustment);
	//!< Sets the vertical adjustment of the viewport.
	//!< @param adjustment An Adjustment.
	
	void set_shadow_type(ShadowType type);
	//!< Sets the shadow type of the viewport.
	//!< @param type The new shadow type.
	
//! @}	
//! @name Property Proxies
//! @{
	
	const HAdjustmentPropertyProxy prop_hadjustment()
	{
		return HAdjustmentPropertyProxy(this, &hadjustment_property);
	}
	//!< The Adjustment that determines the values of the horizontal position
	//!< for this viewport (Adjustment* : Read / Write).

	const VAdjustmentPropertyProxy prop_vadjustment()
	{
		return VAdjustmentPropertyProxy(this, &vadjustment_property);
	}
	//!< The Adjustment that determines the values of the vertical position
	//!< for this viewport (Adjustment* : Read / Write).	

	const ShadowTypePropertyProxy prop_shadow_type()
	{
		return ShadowTypePropertyProxy(this, &shadow_type_property);
	}
	//!< Determines how the shadowed box around the viewport is drawn (ShadowType : Read / Write).
	
//! @}	
//! @name Signal Proxies
//! @{
	
	const SetScrollAdjustmentsSignalProxy sig_set_scroll_adjustments()
	{
		return SetScrollAdjustmentsSignalProxy(this, &set_scroll_adjustments_signal);
	}
	//!< Connect to the set_scroll_adjustments_signal; emitted when a viewport is added to 
	//!< a scrolling aware parent.
	
//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_VIEWPORT_H

