/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management;

import java.io.Serializable;

import mx4j.util.Utils;

/**
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.12 $
 */
public class MBeanInfo implements Cloneable, Serializable
{
   private static final long serialVersionUID = -6451021435135161911L;

   private static final MBeanConstructorInfo[] EMPTY_CONSTRUCTORS = new MBeanConstructorInfo[0];
   private static final MBeanAttributeInfo[] EMPTY_ATTRIBUTES = new MBeanAttributeInfo[0];
   private static final MBeanOperationInfo[] EMPTY_OPERATIONS = new MBeanOperationInfo[0];
   private static final MBeanNotificationInfo[] EMPTY_NOTIFICATIONS = new MBeanNotificationInfo[0];

   /**
    * @serial The MBean class name
    */
   private String className;
   /**
    * @serial The MBean description
    */
   private String description;
   /**
    * @serial The MBean constructors
    */
   private MBeanConstructorInfo[] constructors;
   /**
    * The MBean attributes
    */
   private MBeanAttributeInfo[] attributes;
   /**
    * The MBean operations
    */
   private MBeanOperationInfo[] operations;
   /**
    * The MBean notifications
    */
   private MBeanNotificationInfo[] notifications;

   public MBeanInfo(String className, String description, MBeanAttributeInfo[] attributes, MBeanConstructorInfo[] constructors, MBeanOperationInfo[] operations, MBeanNotificationInfo[] notifications)
   {
      this.className = className;
      this.description = description;
      this.constructors = constructors == null || constructors.length == 0 ? EMPTY_CONSTRUCTORS : constructors;
      this.attributes = attributes == null || attributes.length == 0 ? EMPTY_ATTRIBUTES : attributes;
      this.operations = operations == null || operations.length == 0 ? EMPTY_OPERATIONS : operations;
      this.notifications = notifications == null || notifications.length == 0 ? EMPTY_NOTIFICATIONS : notifications;
   }

   public Object clone()
   {
      // This class is read only, so no need to clone also data members
      try
      {
         return super.clone();
      }
      catch (CloneNotSupportedException x)
      {
         return null;
      }
   }

   public String getClassName()
   {
      return className;
   }

   public String getDescription()
   {
      return description;
   }

   public MBeanConstructorInfo[] getConstructors()
   {
      return constructors;
   }

   public MBeanAttributeInfo[] getAttributes()
   {
      return attributes;
   }

   public MBeanOperationInfo[] getOperations()
   {
      return operations;
   }

   public MBeanNotificationInfo[] getNotifications()
   {
      return notifications;
   }

   public int hashCode()
   {
      int hash = className == null ? 0 : className.hashCode();
      if (description != null) hash ^= description.hashCode();
      if (constructors != null) hash ^= Utils.arrayHashCode(constructors);
      if (attributes != null) hash ^= Utils.arrayHashCode(attributes);
      if (operations != null) hash ^= Utils.arrayHashCode(operations);
      if (notifications != null) hash ^= Utils.arrayHashCode(notifications);
      return hash;
   }

   public boolean equals(Object obj)
   {
      if (obj == null) return false;
      if (obj == this) return true;

      try
      {
         MBeanInfo other = (MBeanInfo)obj;
         if ((className == null && other.className == null) || (className != null && className.equals(other.className)))
         {
            // Description ignored for equality

            if (!Utils.arrayEquals(constructors, other.constructors)) return false;
            if (!Utils.arrayEquals(attributes, other.attributes)) return false;
            if (!Utils.arrayEquals(operations, other.operations)) return false;
            if (!Utils.arrayEquals(notifications, other.notifications)) return false;
            return true;
         }
      }
      catch (ClassCastException ignored)
      {
      }
      return false;
   }
}
