/* TableColumnButton.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.tables;

import java.awt.Component;
import java.awt.EventQueue;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.event.EventListenerList;

/**
 * Table column decoration that behaves like a button. Clients may
 * register action listeners with a prototype button which will then
 * fire for every clone made. The action listener can distinguish between
 * clones by the value returned by {@link #getColumnIndex}.
 */
public class TableColumnButton implements TableColumnDecoration, Cloneable {
    
    //
    private final Icon iconRollover;
    
    //
    private final Icon iconSelected;
    
    //
    private final Icon icon;
    
    //
    private Icon currentIcon;
    
    //
    public TableColumnButton copy() {
        try {
            return (TableColumnButton)super.clone ();
        } catch (CloneNotSupportedException e) {
            assert false : "Unexpected exception";
            return null;
        }
    }
    
    //
    private String actionCommand;
    
    /**
     * Return the action command for this button.
     */
    public String getActionCommand() {
        return actionCommand;
    }
    
    /**
     * Set the action command for this button. An action command set for
     * the prototype will be propagated to all clones.
     */
    public void setActionCommand(String actionCommand) {
        this.actionCommand = actionCommand;
    }
    
    /**
     * Create a button with given icons. All parameters must have
     * the same dimensions. Parameters cannot be null, but it is allowed
     * for two or three parameters to be identical.
     */
    public TableColumnButton(Icon icon, Icon iconRollover, Icon iconSelected) {
        this.icon = icon;
        this.iconRollover = iconRollover;
        this.iconSelected = iconSelected;
        if (icon.getIconHeight() != iconRollover.getIconHeight()
        || icon.getIconHeight() != iconSelected.getIconHeight()
        || icon.getIconWidth() != iconRollover.getIconWidth()
        || icon.getIconWidth() != iconSelected.getIconWidth())
            throw new IllegalArgumentException("All three icons should have the same size");
        
        this.currentIcon = icon;
        this.pressed = false;
        this.inBounds = false;
        this.columnIndex = -1;
    }
    
    private static final Class<TableColumnButton> TABLE_COLUMN_BUTTON_CLASS 
            = TableColumnButton.class;
    
    
    private final static Icon ICON_CLOSE
            = new ImageIcon(TABLE_COLUMN_BUTTON_CLASS.getResource("close.png"));
    private final static Icon ICON_CLOSE_ROLLOVER
            = new ImageIcon(TABLE_COLUMN_BUTTON_CLASS.getResource("close-rollover.png"));
    private final static Icon ICON_CLOSE_SELECTED
            = new ImageIcon(TABLE_COLUMN_BUTTON_CLASS.getResource("close-selected.png"));
        // if lazy instantiation is needed, use initailize-on-demand holder class
    
    /**
     * Return a button with the look and feel of a 'close' button.
     * Convenience method which enables the use of standard icons.
     */
    public static TableColumnButton createCloseButton() {
        return new TableColumnButton(ICON_CLOSE, ICON_CLOSE_ROLLOVER, ICON_CLOSE_SELECTED);
    }
    
    //
    public int getDecorationHeight() {
        return icon.getIconHeight();
    }
    
    //
    public int getDecorationWidth() {
        return icon.getIconWidth();
    }
    
    //
    public void paintDecoration(Component c, Graphics g, int x, int y) {
        currentIcon.paintIcon(c,g,x,y);
    }
    
    //
    private boolean pressed;
    
    //
    private boolean inBounds;
    
    // trivial implementation
    public void mouseReleased() {
        if (inBounds) {
            EventQueue.invokeLater(new Runnable() {
                public void run() {
                    fireActionPerformed();
                }
            });
        }
        currentIcon = icon;
        pressed = false;
    }
    
    // trivial implementation
    public void mousePressed() {
        currentIcon = iconSelected;
        pressed = true;
        inBounds = true;
    }
    
    // trivial implementation
    public void mouseExited() {
        currentIcon = icon;
        inBounds = false;
    }
    
    // trivial implementation
    public void mouseEntered() {
        currentIcon = pressed ? iconSelected : iconRollover;
        inBounds = true;
    }
    
    //
    private int columnIndex;
    
    //
    public void setColumnIndex(int columnIndex) {
        this.columnIndex = columnIndex;
    }
    
    /**
     * Return the column index for this button. Only valid for clones,
     * the prototype returns -1.
     */
    public int getColumnIndex() {
        return columnIndex;
    }
    
    //
    public String toString() {
        return "TableColumnButton[column " + getColumnIndex() + "]";
    }
    
    /* ============================================================
     * LISTENER MANAGEMENT
     * ============================================================ */
    
    private final EventListenerList listenerList = new EventListenerList();
    
    /**
     * Adds an <code>ActionListener</code> to the button. Action listeners
     * are shared by all clones.
     */
    public void addActionListener(ActionListener l) {
        listenerList.add(ActionListener.class, l);
    }
    
    /**
     * Removes an <code>ActionListener</code> from the button and all its clones.
     */
    public void removeActionListener(ActionListener l) {
        listenerList.remove(ActionListener.class, l);
    }
    
    //
    protected void fireActionPerformed() {
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        ActionEvent ev = null;
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==ActionListener.class) {
                // Lazily create the event:
                if (ev == null)
                    ev = new ActionEvent(this,
                            ActionEvent.ACTION_PERFORMED,
                            actionCommand);
                ((ActionListener)listeners[i+1]).actionPerformed(ev);
            }
        }
    }
    
}
