/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "syncml.h"

#include "syncml_internals.h"
#include "sml_notification_internals.h"
#include "sml_transport_internals.h"
#include "sml_command_internals.h"

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

/** @brief Function description
 * 
 * @param parameter This parameter does great things
 * @returns What you always wanted to know
 * 
 */

SmlNotification *smlNotificationNew(SmlNotificationVersion version, SmlNotificationUIMode mode, SmlNotificationInitiator init, unsigned int sessionID, const char *identifier, const char *target, SmlMimeType type, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %i, %i, %i, %s, %s, %i, %p)", __func__, version, mode, init, sessionID, identifier, target, type, error);

	SmlNotification *san = smlTryMalloc0(sizeof(SmlNotification), error);
	if (!san)
		goto error;
	
	san->version = version;
	san->mode = mode;
	san->init = init;
	san->sessionID = sessionID;
	san->identifier = g_strdup(identifier);
	san->type = type;
	
	san->target = smlLocationNew(target, NULL, error);
	if (!san->target)
		goto error_free_san;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, san);
	return san;

error_free_san:

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

void smlNotificationFree(SmlNotification *san)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, san);
	smlAssert(san);
	
	while (san->alerts) {
		SmlSanAlert *alert = san->alerts->data;
		san->alerts = g_list_remove(san->alerts, alert);
		smlNotificationFreeAlert(alert);
	}
	
	if (san->target)
		smlLocationUnref(san->target);
	
	g_free(san->identifier);
	g_free(san);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlBool smlNotificationNewAlert(SmlNotification *san, SmlAlertType type, const char *contenttype, const char *serverURI, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, error);
	smlAssert(type >= 206 && type <= 210);

	SmlSanAlert *alert = smlTryMalloc0(sizeof(SmlSanAlert), error);
	if (!alert)
		goto error;

	alert->type = type;
	alert->contenttype = g_strdup(contenttype);
	alert->serverURI = g_strdup(serverURI);

	san->alerts = g_list_append(san->alerts, alert);

	smlTrace(TRACE_EXIT, "%s: %p", __func__, alert);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlNotificationFreeAlert(SmlSanAlert *alert)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, alert);
	smlAssert(alert);
	
	g_free(alert->contenttype);
	g_free(alert->serverURI);
	g_free(alert);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlNotification *smlNotificationParse(const char *data, unsigned int size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p)", __func__, data, size, error);
	smlAssert(data);
	smlAssert(size);
	SmlSanAlert *alert = NULL;
	
	if (size < 25) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Size too small");
		goto error;
	}
	
	unsigned int idLength = (uint8_t)data[23];
	if (size < (25 + idLength)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Size too small2");
		goto error;
	}
	
	SmlNotification *san = smlTryMalloc0(sizeof(SmlNotification), error);
	if (!san)
		goto error;

	/* Version */
	unsigned int version = ((uint8_t)data[16]) << 2;
	version = version | ((uint8_t)data[17]) >> 6;
	switch (version) {
		case 12:
			san->version = SML_SAN_VERSION_12;
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown version");
			goto error_free_san;
	}
	
	/* ui mode */
	san->mode = (((uint8_t)data[17]) & 0x30) >> 4;
	
	/* initiator */
	san->init = (((uint8_t)data[17]) & 0x08) >> 3;

	/* session id */
	san->sessionID = ((uint8_t)data[21]) << 8;
	san->sessionID = san->sessionID | (uint8_t)data[22];
	
	/* server identifier */
	if (idLength) {
		san->identifier = smlTryMalloc0(idLength, error);
		if (!san->identifier)
			goto error_free_san;
	
		memcpy(san->identifier, data + 24, idLength);
	}
	
	unsigned int numSync = ((uint8_t)data[24 + idLength]) >> 4;
	data += 25 + idLength;
	unsigned int alertLength = 25 + idLength;

	int i = 0;
	for (i = 0; i < numSync; i++) {
		if (size < alertLength + 5) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Size too small3");
			goto error_free_san;
		}
		
		idLength = (uint8_t)data[4];
		if (size < (alertLength + 5 + idLength)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Size too small4");
			goto error_free_san;
		}
		alertLength = alertLength + 5 + idLength;
		
		alert = smlTryMalloc0(sizeof(SmlSanAlert), error);
		if (!alert)
			goto error_free_san;
		san->alerts = g_list_append(san->alerts, alert);
		
		/* Alert type + future */
		alert->type = (((uint8_t)data[0]) >> 4) + 200;
		if (alert->type < 206 || alert->type > 210) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Wrong alert type");
			goto error_free_san;
		}
		
		unsigned int contenttype = ((uint8_t)data[1]) << 16;
		contenttype = contenttype | ((uint8_t)data[2]) << 8;
		contenttype = contenttype | ((uint8_t)data[3]);
		
		/* The contenttype */
		switch (contenttype) {
			case 0x07:
				alert->contenttype = g_strdup(SML_ELEMENT_TEXT_VCARD);
				break;
			case 0x06:
				alert->contenttype = g_strdup(SML_ELEMENT_TEXT_VCAL);
				break;
			case 0x03:
				alert->contenttype = g_strdup(SML_ELEMENT_TEXT_PLAIN);
				break;
			default:
				smlErrorSet(error, SML_ERROR_GENERIC, "Unknown content type");
				goto error_free_san;
		}
		
		/* The server uri */
		if (idLength) {
			alert->serverURI = smlTryMalloc0(idLength, error);
			if (!alert->serverURI)
				goto error_free_san;
				
			memcpy(alert->serverURI, data + 5, idLength);
		}
		data += 5 + idLength;
	}

	if (size > alertLength) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Extra chunk at the end of the buffer");
		goto error_free_san;
	}

	smlTrace(TRACE_EXIT, "%s: %p", __func__, san);
	return san;
	
error_free_san:
	while (san->alerts) {
		smlNotificationFreeAlert(san->alerts->data);
		san->alerts = g_list_delete_link(san->alerts, san->alerts);
	}
	g_free(san->identifier);
	g_free(san);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool _smlNotificationAssemble11(SmlNotification *san, char **data, unsigned int *size, SmlProtocolVersion version, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %i, %p)", __func__, san, data, size, version, error);
	smlAssert(san);
	smlAssert(data);
	smlAssert(size);
	SmlSanAlert *alert = NULL;
	SmlCommand *cmd = NULL;

	SmlAssembler *assm = smlAssemblerNew(san->type, 0, error);
	if (!assm)
		goto error;
	
	SmlLocation *source = smlLocationNew(san->identifier, NULL, error);
	if (!source)
		goto error_free_assm;
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, san->type, version, SML_PROTOCOL_SYNCML, source, san->target, "1", 0, error);
	if (!session) {
		smlLocationUnref(source);
		goto error_free_assm;
	}

	if (!smlAssemblerStart(assm, session, error))
		goto error_free_session;
	
	
	if (!smlAssemblerAddHeader(assm, session, error))
		goto error_free_session;
	
	GList *a = NULL;
	int i = 1;
	for (a = san->alerts; a; a = a->next) {
		alert = a->data;
		
		
		SmlLocation *loc = smlLocationNew(alert->serverURI, NULL, error);
		if (!loc)
			goto error_free_session;
		
		cmd = smlCommandNewAlert(SML_ALERT_TWO_WAY_BY_SERVER, NULL, loc, NULL, NULL, alert->contenttype, error);
		if (!cmd) {
			smlLocationUnref(loc);
			goto error_free_session;
		}
		
		cmd->cmdID = i;
		i++;
		
		if (!smlAssemblerStartCommand(assm, NULL, cmd, error) == SML_ASSEMBLER_RESULT_OK)
			goto error_free_cmd;
			
		if (!smlAssemblerEndCommand(assm, NULL, error))
			goto error_free_cmd;
			
		smlLocationUnref(loc);
		
		smlCommandUnref(cmd);
	}
	
	if (!smlAssemblerRun(assm, data, size, NULL, TRUE, error))
		goto error_free_session;
	
	smlAssemblerFlush(assm);
	
	smlSessionUnref(session);
	
	smlLocationUnref(source);
	
	smlAssemblerFree(assm);
	
	smlTrace(TRACE_INTERNAL, "San packet assembled: %s", smlPrintHex(*data, *size));
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error_free_cmd:
	smlCommandUnref(cmd);
error_free_session:
	smlSessionUnref(session);
error_free_assm:
	smlAssemblerFree(assm);
error:
	*data = NULL;
	*size = 0;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlNotificationAssemble12(SmlNotification *san, char **data, unsigned int *size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, san, data, size, error);
	smlAssert(san);
	smlAssert(data);
	smlAssert(size);
	unsigned int numsync = 0;
	SmlSanAlert *alert = NULL;
	
	/* Calculate the length of the san data */
	unsigned int length = 16 /* Digest */ + 8 /* Hdr */ + strlen(san->identifier) + 1 /* Identifier */ + 1 /* num-sync + future */;	
	GList *a = NULL;
	for (a = san->alerts; a; a = a->next) {
		alert = a->data;
		length += 5 + strlen(alert->serverURI) + 1;
		numsync++;
	}
	
	/* Malloc the necessary space */
	char *buffer = smlTryMalloc0(length, error);
	if (!buffer)
		goto error;
	*data = buffer;
	*size = length;
	
	/* FIXME: Now create the digest */
	
	/* Create Header */
	/* Version */
	buffer[16] = 0x03;

	/* ui mode */
	buffer[17] = buffer[17] | (san->mode << 4);
	
	/* initiator */
	buffer[17] = buffer[17] | (san->init << 3);
	
	/* future use is 0 for now */

	/* session id */
	buffer[21] = san->sessionID >> 8;
	buffer[22] = san->sessionID;
	
	/* server identifier */
	buffer[23] = strlen(san->identifier) + 1;
	strcpy(buffer + 24, san->identifier);
	
	buffer += 24 + strlen(san->identifier) + 1;
	
	/* Num syncs + future */
	buffer[0] = numsync << 4;
	buffer++;
	
	for (a = san->alerts; a; a = a->next) {
		alert = a->data;
		/* Alert type + future */
		buffer[0] = (alert->type - 200) << 4;
		
		/* The contenttype */
		if (!strcmp(alert->contenttype, SML_ELEMENT_TEXT_VCARD))
			buffer[3] = 0x07;
		else if (!strcmp(alert->contenttype, SML_ELEMENT_TEXT_VCAL))
			buffer[3] = 0x06;
		else if (!strcmp(alert->contenttype, SML_ELEMENT_TEXT_PLAIN))
			buffer[3] = 0x03;
		else {
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown content type");
			goto error_free_data;
		}
		
		/* The server uri */
		buffer[4] = strlen(alert->serverURI) + 1;
		strcpy(buffer + 5, alert->serverURI);
		buffer += 5 + strlen(alert->serverURI) + 1;
	}
	
	smlTrace(TRACE_INTERNAL, "San packet assembled: %s", smlPrintHex(*data, *size));
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_data:
	g_free(*data);
error:
	*data = NULL;
	*size = 0;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlNotificationAssemble(SmlNotification *san, char **data, unsigned int *size, SmlError **error)
{
	switch (san->version) {
		case SML_SAN_VERSION_12:
			return _smlNotificationAssemble12(san, data, size, error);
		case SML_SAN_VERSION_10:
			return _smlNotificationAssemble11(san, data, size, SML_VERSION_10, error);
		case SML_SAN_VERSION_11:
			return _smlNotificationAssemble11(san, data, size, SML_VERSION_11, error);
		case SML_SAN_VERSION_UNKNOWN:
			break;
	}
	
	return FALSE;
}

SmlBool smlNotificationSend(SmlNotification *san, SmlTransport *tsp, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, san, tsp, error);
	smlAssert(san);
	smlAssert(tsp);

	char *data = NULL;
	unsigned int size = 0;
	if (!smlNotificationAssemble(san, &data, &size, error))
		goto error;

	SmlTransportData *tspdata = NULL;
	if (san->version == SML_SAN_VERSION_12)
		tspdata = smlTransportDataNew(data, size, SML_MIMETYPE_SAN, TRUE, error);
	else
		tspdata = smlTransportDataNew(data, size, san->type, TRUE, error);
	
	if (!tspdata)
		goto error_free_data;
	data = NULL;
	
	if (!smlTransportSend(tsp, NULL, tspdata, error))
		goto error_free_tspdata;

	g_free(data);

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_tspdata:
	smlTransportDataDeref(tspdata);
error_free_data:
	g_free(data);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlNotificationVersion smlNotificationGetVersion(SmlNotification *san)
{
	smlAssert(san);
	return san->version;
}

SmlNotificationUIMode smlNotificationGetMode(SmlNotification *san)
{
	smlAssert(san);
	return san->mode;
}

SmlNotificationInitiator smlNotificationGetInitiator(SmlNotification *san)
{
	smlAssert(san);
	return san->init;
}

unsigned int smlNotificationGetSessionID(SmlNotification *san)
{
	smlAssert(san);
	return san->sessionID;
}

const char *smlNotificationGetIdentifier(SmlNotification *san)
{
	smlAssert(san);
	return san->identifier;
}

unsigned int smlNotificationNumAlerts(SmlNotification *san)
{
	smlAssert(san);
	return g_list_length(san->alerts);
}

SmlSanAlert *smlNotificationNthAlert(SmlNotification *san, unsigned int nth)
{
	smlAssert(san);
	return g_list_nth_data(san->alerts, nth);
}

SmlAlertType smlSanAlertGetType(SmlSanAlert *alert)
{
	smlAssert(alert);
	return alert->type;
}

const char *smlSanAlertGetContentType(SmlSanAlert *alert)
{
	smlAssert(alert);
	return alert->contenttype;
}
const char *smlSanAlertGetServerURI(SmlSanAlert *alert)
{
	smlAssert(alert);
	return alert->serverURI;
}

/*@}*/
