/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeProxMatrixConstructionStrategy.h
    This file contains structures and definitions about construction strategies of proximity matrix 
*/

#ifndef TeProxMatrixConstructionStrategy_H
#define TeProxMatrixConstructionStrategy_H

#include "TeProxMatrixImplementation.h"
#include "TeNetwork.h"

class TeSTElementSet;


//! An abstract class to representate construction strategies of proximity matrix   
class TeProxMatrixConstructionStrategy  
{

protected:

	//! Set of objetcs used to construct the matrix
	TeSTElementSet*   objects_;

	//! Geometry representation of the objects used to construct the matrix
	TeGeomRep	geomRep_;

	//! Type of the construction strategy
	string		type_;

	//! Construct
	TeProxMatrixConstructionStrategy(); 

	//! Construct
	TeProxMatrixConstructionStrategy(TeSTElementSet*  objects, TeGeomRep geomRep, const string& type);

	//! Copy construct
	TeProxMatrixConstructionStrategy(const TeProxMatrixConstructionStrategy& st);

public:

	//! Construct the proximity matrix 
	virtual bool Construct(TeProxMatrixImplementation*){return false;}

	//! Destructor
	virtual ~TeProxMatrixConstructionStrategy() {}

	//! Return the type of the strategy
	string Type() {return type_;}

	//! Set the set of objects and its geometry representation
	void setSTObjects (TeSTElementSet*  objects, TeGeomRep  geomRep); 
	
	//! Get the objects used to construct the matrix
	TeSTElementSet* objects(); 

	//! Get the geometry representation
	TeGeomRep geomRep() { return geomRep_; }
	
	//! Verify if the type of the strategy, the object set and its geometry representation are equal
	virtual bool IsEqual (const TeProxMatrixConstructionStrategy& other) const;  
};


//! A class to implement the local adjacency strategy of proximity matrix
class TeProxMatrixLocalAdjacencyStrategy : public  TeProxMatrixConstructionStrategy
{

private:

	//! If it must calculate or not the distance between neighbors
	bool  calcDistance_;
	
public:

	//! Constructor
	TeProxMatrixLocalAdjacencyStrategy (bool calcDistance=false); 

	//! Constructor
	TeProxMatrixLocalAdjacencyStrategy (TeSTElementSet*  objects, TeGeomRep	geomRep, bool calcDistance=false);
	
	//! Copy constructor
	TeProxMatrixLocalAdjacencyStrategy (TeProxMatrixLocalAdjacencyStrategy& st);  
	 
	 //! Destructor
	virtual ~TeProxMatrixLocalAdjacencyStrategy() {}
	
	//! Construct the proximity matrix through local adjacency strategy 
	virtual bool Construct (TeProxMatrixImplementation* imp);
		
	//! Equal operator 
	bool operator== (const TeProxMatrixLocalAdjacencyStrategy& rhs) const;  
	
	//! Assignment operator 
	TeProxMatrixLocalAdjacencyStrategy& operator= (const TeProxMatrixLocalAdjacencyStrategy& rhs); 

	bool calculateDistance()
	{ return calcDistance_; }
};


//! A class to implement the local distance strategy of proximity matrix
class TeProxMatrixLocalDistanceStrategy : public  TeProxMatrixConstructionStrategy
{
private:
	double			max_distance_;

public:

	//! Constructor
	TeProxMatrixLocalDistanceStrategy (double max_distance = 0);
	
	//! Constructor
	TeProxMatrixLocalDistanceStrategy (TeSTElementSet*  objects, TeGeomRep geomRep, double max_distance); 
	
	//! Copy constructor
	TeProxMatrixLocalDistanceStrategy (const TeProxMatrixLocalDistanceStrategy& st); 
	
	//! Destructor
	virtual ~TeProxMatrixLocalDistanceStrategy(){}
	
	//! Construct the proximity matrix through local distance strategy 
	virtual bool Construct (TeProxMatrixImplementation* imp); 

	//! Verify if is equal 
	virtual bool IsEqual (const TeProxMatrixConstructionStrategy& other) const;
		
	//! Equal operator 
	bool operator== (const TeProxMatrixLocalDistanceStrategy& s) const; 
		
	//! Assignment operator
	TeProxMatrixLocalDistanceStrategy& operator= (const TeProxMatrixLocalDistanceStrategy& rhs);

};


//! A class to implement the closed network strategy of proximity matrix
class TeProxMatrixClosedNetworkStrategy : public  TeProxMatrixConstructionStrategy
{
private:
	double	max_local_distance_;
	double	max_net_distance_;
	double  max_connection_distance_;


	TeGraphNetwork*		net_;
 
public:

	//! Constructor
	TeProxMatrixClosedNetworkStrategy (	TeSTElementSet*  objects, TeGeomRep rep, double max_local_distance,
										double max_net_distance, double  max_connection_distance, 
										TeGraphNetwork* input_net); 

	//! Copy constructor
	TeProxMatrixClosedNetworkStrategy (const TeProxMatrixClosedNetworkStrategy& rhs);
		  
	//! Destructor
	virtual ~TeProxMatrixClosedNetworkStrategy()
	{	
		if(net_)
			delete (net_); 
	}

	//! Construct the proximity matrix through closed network strategy 
	virtual bool Construct (TeProxMatrixImplementation* imp); 
	
	//! Verify if is equal
	virtual bool IsEqual (const TeProxMatrixConstructionStrategy& other) const; 
	
	//! Assignment operator
	TeProxMatrixClosedNetworkStrategy& operator= (const TeProxMatrixClosedNetworkStrategy& rhs);

	//! Equal operator
	bool operator== (const TeProxMatrixClosedNetworkStrategy& other) const;  

	//! Get the objects used to construct the matrix
	TeSTElementSet* objects() { return objects_; }
};

//! A class to implement the open network strategy of proximity matrix (among a set of objetcs)
class TeProxMatrixOpenNetworkStrategy : public  TeProxMatrixConstructionStrategy
{
private:
	double			max_local_distance_;
	double			max_net_distance_;
	double			max_connection_distance_;

	TeGraphNetwork*		net_;
public:

	//! Constructor
	TeProxMatrixOpenNetworkStrategy (	TeSTElementSet*  objects, TeGeomRep rep, double max_local_distance, 
										double	max_net_distance, double  max_connetion_distance, 
										TeGraphNetwork* input_net); 
	//! Copy constructor
	TeProxMatrixOpenNetworkStrategy (const TeProxMatrixOpenNetworkStrategy& rhs); 
	
	//! Destructor
	virtual ~TeProxMatrixOpenNetworkStrategy()
	{
		if(net_)
			delete (net_); 
	}

	//! Construct the proximity matrix through open network strategy
	virtual bool Construct (TeProxMatrixImplementation* imp); 

	//! Verify if is equal
	virtual bool IsEqual (const TeProxMatrixConstructionStrategy& other) const; 

	//! Equal operator
	bool operator== (const TeProxMatrixOpenNetworkStrategy& other) const; 
	
	//! Assignment operator
	TeProxMatrixOpenNetworkStrategy& operator= (const TeProxMatrixOpenNetworkStrategy& rhs);
};


//! A class to implement the open network strategy of proximity matrix (relationships among objetc of two differnt sets)
class TeProxMatrixOpenNetworkStrategy2 : public  TeProxMatrixConstructionStrategy
{
private:
	TeSTElementSet*		objects2_;
	TeGeomRep			geomRep2_;
	double			max_local_distance_;
	double			max_net_distance_;
	double			max_connection_distance_;

	TeGraphNetwork*		net_;
public:

	//! Constructor
	TeProxMatrixOpenNetworkStrategy2 (	TeSTElementSet*  objects1, TeGeomRep rep1,
										TeSTElementSet*  objects2, TeGeomRep rep2,
										double max_local_distance, 
										double	max_net_distance, double  max_connetion_distance, 
										TeGraphNetwork* input_net); 
	//! Copy constructor
	TeProxMatrixOpenNetworkStrategy2 (const TeProxMatrixOpenNetworkStrategy2& rhs); 
	
	//! Destructor
	virtual ~TeProxMatrixOpenNetworkStrategy2 ()
	{
		if(net_)
			delete (net_); 
	}

	//! Construct the proximity matrix through open network strategy
	virtual bool Construct (TeProxMatrixImplementation* imp); 

	//! Verify if is equal
	virtual bool IsEqual (const TeProxMatrixConstructionStrategy& other) const; 

	//! Equal operator
	bool operator== (const TeProxMatrixOpenNetworkStrategy2& other) const; 
	
	//! Assignment operator
	TeProxMatrixOpenNetworkStrategy2& operator= (const TeProxMatrixOpenNetworkStrategy2& rhs);

};

#endif
