/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.web;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Properties;
import java.util.StringTokenizer;
import xjavadoc.XClass;
import xjavadoc.XTag;

import xdoclet.DocletContext;
import xdoclet.DocletTask;
import xdoclet.XDocletException;

import xdoclet.XDocletTagSupport;
import xdoclet.tagshandler.ClassTagsHandler;
import xdoclet.tagshandler.PackageTagsHandler;

/**
 * @author               Marcus Brito (pazu@animegaiden.com.br)
 * @author               Christoph G. Jung (christoph.jung@infor.de)
 * @created              Jun 28, 2002
 * @version              $Revision: 1.4 $
 * @xdoclet.taghandler   namespace="Web"
 */
public class WebTagsHandler extends XDocletTagSupport
{

    /**
     * Returns short version of the servlet name of the clazz.
     *
     * @param clazz  the class we want its short servlet name
     * @return       The shortName value
     * @see          #shortName()
     */
    public static String shortServletName(XClass clazz)
    {
        // Find the last part of the name
        StringTokenizer ejbNameTokens = new StringTokenizer(servletName(clazz), ":./\\-");
        String name;

        do {
            name = ejbNameTokens.nextToken();
        } while (ejbNameTokens.hasMoreTokens());

        return name;
    }

    public static String serviceEndpoint(XClass clazz) throws XDocletException
    {
        String fileName = clazz.getContainingPackage().getName();
        String name_pattern = null;
        String package_pattern = null;
        String component_interface = null;

        component_interface = clazz.getDoc().getTagAttributeValue(ServiceEndpointSubTask.WEB_SERVLET, ServiceEndpointSubTask.SERVICE_ENDPOINT + "-class");
        if (component_interface != null) {
            return component_interface;
        }

        name_pattern = clazz.getDoc().getTagAttributeValue(ServiceEndpointSubTask.WEB_SERVLET, ServiceEndpointSubTask.SERVICE_ENDPOINT + "-pattern");
        if (name_pattern == null) {
            ServiceEndpointSubTask seintf_subtask = ((ServiceEndpointSubTask) DocletContext.getInstance().
                getSubTaskBy(DocletTask.getSubTaskName(ServiceEndpointSubTask.class)));

            if (seintf_subtask != null) {
                name_pattern = seintf_subtask.getServiceEndpointClassPattern();
            }
            else {
                name_pattern = ServiceEndpointSubTask.DEFAULT_SERVICE_ENDPOINT_CLASS_PATTERN;
            }
        }

        package_pattern = clazz.getDoc().getTagAttributeValue(ServiceEndpointSubTask.WEB_SERVLET, ServiceEndpointSubTask.SERVICE_ENDPOINT + "-package");

        String ejb_name = null;

        if (name_pattern.indexOf("{0}") != -1) {
            ejb_name = MessageFormat.format(name_pattern, new Object[]{shortServletName(clazz)});
        }
        else {
            ejb_name = name_pattern;
        }

        String subtask_name = null;

        // Fix package name
        fileName = choosePackage(fileName, package_pattern, subtask_name);
        fileName += "." + ejb_name;
        return fileName;
    }

    public static String serviceEndpoint(Properties properties) throws XDocletException
    {
        XClass clazz = getCurrentClass();

        return serviceEndpoint(clazz);
    }

    public static String servletName(Properties properties)
    {
        return servletName(getCurrentClass());
    }

    public static String servletName(XClass clazz)
    {
        XTag beanTag = clazz.getDoc().getTag(ServiceEndpointSubTask.WEB_SERVLET);
        String paramValue = null;

        if (beanTag != null) {
            paramValue = beanTag.getAttributeValue("name");
        }

        if (paramValue == null) {
            paramValue = clazz.getName();
        }
        return paramValue;
    }

    public static String choosePackage(String packageName, String packagePattern, String subtask)
    {

        ArrayList packageSubstitutions = PackageTagsHandler.getPackageSubstitutions(subtask);

        if (packagePattern != null) {
            // later we may do some parametric {0} fancy stuff here
            return packagePattern;
        }
        else {
            for (int i = 0; i < packageSubstitutions.size(); i++) {
                PackageTagsHandler.PackageSubstitution ps = (PackageTagsHandler.PackageSubstitution) packageSubstitutions.get(i);
                StringTokenizer st = new StringTokenizer(ps.getPackages(), ",", false);

                while (st.hasMoreTokens()) {
                    String packages = st.nextToken();
                    String suffix = "." + packages;

                    if (packageName.endsWith(suffix)) {
                        packageName = packageName.substring(0, packageName.length() - suffix.length()) + '.' + ps.getSubstituteWith();
                        break;
                    }
                }
            }
        }

        return packageName;
    }

    /**
     * Process the tag body for each web:ejb-ref tag in all source files. Please note that this tag already iterates
     * over all available sources; it should <em>not</em> be enclosed by a &lt;XDtClass:forAllClasses> tag or any other
     * that process classes. This tag does not process tags with duplicated name attributes. If such tags occurs, only
     * the first tag is processed, and further tags will only emit a warning message.
     *
     * @param template           The body of the block tag
     * @throws XDocletException  if something goes wrong
     * @doc.tag                  type="block"
     */
    public void forAllEjbRefs(String template) throws XDocletException
    {
        ClassTagsHandler.forAllDistinctClassTags(getEngine(), template, "web:ejb-ref", "name");
    }

    /**
     * Process the tag body for each web:ejb-local-ref tag in all source files. Look at forAllEjbRefs for some notes
     * about the behavior of this tag.
     *
     * @param template           The body of the blocktag
     * @throws XDocletException  if something goes wrong
     * @doc.tag                  type="block"
     */
    public void forAllEjbLocalRefs(String template) throws XDocletException
    {
        ClassTagsHandler.forAllDistinctClassTags(getEngine(), template, "web:ejb-local-ref", "name");
    }

    /**
     * Process the tag body for each web:resource-ref tag in all source files. Look at forAllEjbRefs for some notes
     * about the behavior of this tag.
     *
     * @param template           The body of the blocktag
     * @throws XDocletException  if something goes wrong
     * @doc.tag                  type="block"
     */
    public void forAllResourceRefs(String template) throws XDocletException
    {
        ClassTagsHandler.forAllDistinctClassTags(getEngine(), template, "web:resource-ref", "name");
    }

    /**
     * Process the tag body for each web:resource-env-ref tag in all source files. Look at forAllEjbRefs for some notes
     * about the behavior of this tag.
     *
     * @param template           The body of the blocktag
     * @throws XDocletException  if something goes wrong
     * @doc.tag                  type="block"
     */
    public void forAllResourceEnvRefs(String template) throws XDocletException
    {
        ClassTagsHandler.forAllDistinctClassTags(getEngine(), template, "web:resource-env-ref", "name");
    }

    /**
     * Process the tag body for each web:security-role tag in all source files. Look at forAllEjbRefs for some notes
     * about the behavior of this tag.
     *
     * @param template           The body of the blocktag
     * @throws XDocletException  if something goes wrong
     * @doc.tag                  type="block"
     */
    public void forAllSecurityRoles(String template) throws XDocletException
    {
        ClassTagsHandler.forAllDistinctClassTags(getEngine(), template, "web:security-role", "role-name");
    }

    /**
     * Process the tag body for each web:env-entry tag in all source files. Look at forAllEjbRefs for some notes about
     * the behavior of this tag.
     *
     * @param template           The body of the blocktag
     * @throws XDocletException  if something goes wrong
     * @doc.tag                  type="block"
     */
    public void forAllEnvEntries(String template) throws XDocletException
    {
        ClassTagsHandler.forAllDistinctClassTags(getEngine(), template, "web:env-entry", "name");
    }

    /**
     * Returns the name of the class home interface extends.
     *
     * @param attributes
     * @return                      The name of generated PK class.
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String extendsFrom(Properties attributes) throws XDocletException
    {
        String def_base_class_name = getTagValue(
            FOR_CLASS,
            getCurrentClass().getDoc(),
            ServiceEndpointSubTask.WEB_SERVLET,
            "service-endpoint-extends",
            null,
            "java.rmi.Remote",
            true,
            false);

        return def_base_class_name;
    }

}
