/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: random.c,v 1.11 2003/05/30 14:13:13 sasa Exp $
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/random.h>
#include <zorp/zorplib.h>
#include <zorp/log.h>

#include <glib.h>
// TODO

#ifdef G_OS_WIN32
#define _WIN32_WINNT 0x0500
  #include <windows.h>
  #include <wincrypt.h>
  #include <malloc.h>
  #define alloca _alloca
#endif

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif

typedef gboolean (*ZRandomGet)(guchar *buf, guint buflen, gpointer user_data);

typedef struct _ZRandom
{
  ZRandomGet strong_get;
  ZRandomGet basic_get;
  ZRandomGet weak_get;

  gpointer strong_data;
  gpointer basic_data;
  gpointer weak_data;

  gchar strong_buf;
  guchar strong_unused_bit;
  gchar basic_buf;
  guchar basic_unused_bit;
  gchar weak_buf;
  guchar weak_unused_bit;
} ZRandom;

// static GStaticPrivate randoms = G_STATIC_PRIVATE_INIT;

static ZRandom *global_random = NULL;

static GStaticMutex global_random_lock = G_STATIC_MUTEX_INIT;

static guchar kettohatvany[] = {0, 1, 3, 7, 15, 31, 63, 127, 255};

#ifdef G_OS_WIN32

static gboolean
z_random_entropy_cryptoapi_get(guchar *target,
                                guint  len,
                            gpointer  user_data G_GNUC_UNUSED)
{
  HCRYPTPROV prov;
  BOOL res;

  // try the user's default key container ...
  res = CryptAcquireContext(&prov, NULL, NULL, PROV_RSA_FULL, 0);

  // ... if that fails, try to create the user's default container ...
  if (!res)
    res = CryptAcquireContext(&prov, NULL, NULL, PROV_RSA_FULL, CRYPT_NEWKEYSET);

  // ... if that fails as well, exit; target will be left as is.
  if (!res)
    {
//      z_log(NULL, CORE_ERROR, 1, "%s: CryptAcquireContext failed with error code %x",
 //           __FUNCTION__, GetLastError());
      return FALSE;
    }

  res = CryptGenRandom(prov, len, (BYTE *) target);
  if (!res)
    return FALSE;

  CryptReleaseContext(prov, 0);
  return TRUE;
}

inline static gpointer
z_random_entropy_cryptoapi_init(guint type G_GNUC_UNUSED)
{
  return NULL;
}

#define z_random_strong_get z_random_entropy_cryptoapi_get
#define z_random_strong_init z_random_entropy_cryptoapi_init

#define z_random_basic_get z_random_entropy_cryptoapi_get
#define z_random_basic_init z_random_entropy_cryptoapi_init

#define z_random_weak_get z_random_entropy_cryptoapi_get
#define z_random_weak_init z_random_entropy_cryptoapi_init

#else



static gboolean
z_random_entropy_devurandom_get(guchar *target,
                                 guint  len,
                              gpointer  user_data G_GNUC_UNUSED)
{
  int fd;
  
  fd = open("/dev/urandom", O_RDONLY);
  if (fd < 0)
    return FALSE;
  read(fd, target, len);
  close(fd);
  target[len] = '\0';
  return TRUE;
}

inline static gpointer
z_random_entropy_devurandom_init(guint type G_GNUC_UNUSED)
{
  return NULL;
}

static gboolean
z_random_entropy_devrandom_get(guchar *target,
                                guint  len,
                             gpointer  user_data G_GNUC_UNUSED)
{
  int fd;
  guint readed_len = 0;
  
  fd = open("/dev/random", O_RDONLY);
  if (fd < 0)
    return FALSE;
  while (readed_len < len)
    readed_len += read(fd, target + readed_len, len - readed_len);
  close(fd);
  target[len] = '\0';
  
  return TRUE;
}

inline static gpointer
z_random_entropy_devrandom_init(guint type G_GNUC_UNUSED)
{
  return NULL;
}

#define z_random_strong_get z_random_entropy_devrandom_get
#define z_random_strong_init z_random_entropy_devrandom_init

#define z_random_basic_get z_random_entropy_devurandom_get
#define z_random_basic_init z_random_entropy_devurandom_init

#define z_random_weak_get z_random_entropy_devurandom_get
#define z_random_weak_init z_random_entropy_devurandom_init

#endif

static ZRandom *
z_random_new(void)
{
  ZRandom *self = g_new0(ZRandom, 1);
  
  self->strong_get = z_random_strong_get;
  self->strong_data = z_random_strong_init(Z_RANDOM_STRONG);
  self->basic_get = z_random_basic_get;
  self->basic_data = z_random_basic_init(Z_RANDOM_BASIC);
  self->weak_get = z_random_weak_get;
  self->weak_data = z_random_weak_init(Z_RANDOM_WEAK);
  
  return self;
}

gboolean
z_random_sequence_get_bounded(guchar *target,
                               guint  len,
                              guchar  min,
                              guchar  max,
                               guint  type)
{
  ZRandomGet random_get;
  guchar random_buf;
  guchar random_unused_len;
  gpointer random_user_data;
  guint bound;
  guchar num_bits = 0;
  guint needed_byte;
  guchar *buf;
  guchar offset;
  guint i, j;
  guchar tmptarget;

  z_enter();
  
  buf = alloca(len + 1);

  g_static_mutex_lock(&global_random_lock);

  if (!global_random)
    {
      global_random = z_random_new();
    }
  
  switch (type)
    {
      case Z_RANDOM_STRONG:
        random_get = global_random->strong_get;
        random_buf = global_random->strong_buf;
        random_unused_len = global_random->strong_unused_bit;
        random_user_data = global_random->strong_data;
        break;
        
      case Z_RANDOM_BASIC:
        random_get = global_random->basic_get;
        random_buf = global_random->basic_buf;
        random_unused_len = global_random->basic_unused_bit;
        random_user_data = global_random->basic_data;
        break;
        
      case Z_RANDOM_WEAK:
        random_get = global_random->weak_get;
        random_buf = global_random->weak_buf;
        random_unused_len = global_random->weak_unused_bit;
        random_user_data = global_random->weak_data;
        break;
      
      default:
        z_leave();
        return FALSE;
    }
  
  bound = max - min;
  
  while (bound)
    {
      num_bits++;
      bound >>= 1;
    }
  
  memset(buf, 0, sizeof(buf));
  needed_byte = ((num_bits * len) - random_unused_len + 7) / 8;
  if (!random_get(buf, needed_byte, random_user_data))
    {
      z_leave();
      return FALSE;
    }
  
  for (i = 0, j = 0; i < len; i++)
    {
      if (num_bits < random_unused_len)
        {
          target[i] = random_buf & kettohatvany[num_bits];
          random_buf >>= num_bits;
          /* FIXMEE MSVC make warning for -= */
          random_unused_len = random_unused_len - num_bits;
        }
      else
        {
          target[i] = random_buf << (num_bits - random_unused_len);
          offset = num_bits - random_unused_len;
          random_buf = buf[j++];
          random_unused_len = 8;
          target[i] |= random_buf & kettohatvany[offset];
          random_buf >>= offset;
          /* FIXMEE MSVC make warning for -= */
          random_unused_len = random_unused_len - offset;
        }
      
      tmptarget = target[i];
      tmptarget = (tmptarget * (max - min)) / (kettohatvany[num_bits]);

      /* FIXMEE MSVC make warning for += */
      tmptarget = tmptarget + min;
      target[i] = tmptarget;
      if (tmptarget > max)
        {
          /*LOG
	    This message indicates that an internal error occurred, during random number
	    generation. Please report this event to the Balabit QA Team (devel@balabit.com).
           */
          z_log(NULL, CORE_ERROR, 3, "Bad random sequence;");
          target[i] = max;
        }
    }

  g_static_mutex_unlock(&global_random_lock);
  z_leave();
  return TRUE;
}
