/*
Sparse matrix functionality for octave, based on the SuperLU package  
Copyright (C) 1998-2000 Andy Adler

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with Octave; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.

In addition to the terms of the GPL, you are permitted to link
this program with any Open Source program, as defined by the
Open Source Initiative (www.opensource.org)

$Id: make_sparse.cc,v 1.3 2001/10/14 03:06:31 aadler Exp $

*/

#define  SPARSE_COMPLEX_CODE
#include "make_sparse.h"
//
//   These functions override those in SuperLU/SRC/util.c
//

void *
oct_sparse_malloc(int size) {
   // avoid zero byte alloc requests, request a minimum of
   // 1 byte - this is ok becuause free should handle it
   size= MAX(size,1);
#ifdef USE_DMALLOC   
   return _malloc_leap(__FILE__, __LINE__, size);
#else   
   return malloc( MAX(size,1) );
#endif   
#if 0   
   void * vp= malloc( size );
   printf ("allocated %04X : %d\n", (int) vp, size);
   return vp;
#endif   
}  

void
oct_sparse_fatalerr(char *msg) {
   SP_FATAL_ERR( msg );
}  

void
oct_sparse_free(void * addr) {
#ifdef USE_DMALLOC   
   if(addr) _free_leap(__FILE__, __LINE__, addr);
#else   
   if (addr) free( addr );
#endif   
#if 0
   DEBUGMSG("sparse - oct_sparse_free");
   printf ("freeing %04X\n", (int) addr );
   free( addr );
#endif   
}  


//
// Utility methods for sparse ops
//

void
oct_sparse_expand_bounds( int lim, int& bound,
                          int*& idx,
                          void*& coef, int varsize)
{   
   const int mem_expand = 2;

   DEBUGMSG("growing bounds"); 
   bound*= mem_expand;
   int *   t_idx = (int  *) oct_sparse_malloc((bound) * sizeof(int)); 
   void * t_coef = (void *) oct_sparse_malloc((bound) * varsize    ); 
   if ((t_idx==NULL) || (t_coef == NULL) ) 
      SP_FATAL_ERR("memory error in check_bounds");
 
   memcpy( t_idx , idx , lim*sizeof(int) );
   memcpy( t_coef, coef, lim*varsize     );

   free( idx);
   idx= t_idx;
   free( coef);
   coef= t_coef;
}      


void
oct_sparse_maybe_shrink( int lim, int bound,
                         int*& idx,
                         void*& coef, int varsize) {
   if ( (lim < bound) && (lim > 0) ) {
      idx = (int    *) realloc( idx,  lim*sizeof(int) );
      coef= (void   *) realloc( coef, lim*varsize );
      assert (idx != NULL);
      assert (coef != NULL);
   }
   else if (lim==0) {      
      free( idx ); idx= NULL;
      free( coef); coef=NULL;
   }
}   

void
oct_sparse_Destroy_SuperMatrix( SuperMatrix X) {
   switch( X.Stype ) { 
      case NC:  Destroy_CompCol_Matrix(&X);   break;
      case DN:  Destroy_Dense_Matrix(&X);     break;
      case SC:  Destroy_SuperNode_Matrix(&X); break;
      case NCP: Destroy_CompCol_Permuted(&X); break;
      default:  SP_FATAL_ERR("Bad SuperMatrix Free"); 
   }
}

#ifdef ANDYS_SEGFAULT_OVERRIDE
#include <signal.h>
#endif

DEFUN_DLD (sparse, args, ,
  "sparse_val = sparse (...)\n\
SPARSE: create a sparse matrix\n\
\n\
sparse can be called in the following ways:\n\
\n\
1: S = sparse(A), where 'A' is a full matrix\n\
\n\
2: S = sparse(i,j,s,m,n,nzmax), where\n\
        i,j   are integer index vectors (1 x nnz)\n\
        s     is the vector of real or complex entries (1 x nnz)\n\
        m,n   are the scalar dimentions of S\n\
        nzmax is ignored (for compatability with Matlab)\n\
\n\
3: S = sparse(i,j,s,m,n),     same as (2) above\n\
\n\
4: S=  sparse(i,j,s),         uses m=max(i), n=max(j)\n\
\n\
5: S=  sparse(m,n),           does sparse([],[],[],m,n,0)\n\
\n\
s, and i or j may be scalars, in which case they are expanded\n\
so they all have the same length ")
{
#ifdef ANDYS_SEGFAULT_OVERRIDE
signal( SIGSEGV, SIG_DFL );
#endif
   
   static bool sparse_type_loaded         = false;
   static bool complex_sparse_type_loaded = false;

   octave_value retval;

   int nargin= args.length();
   if (nargin < 1) {
      print_usage ("sparse");
      return retval;
   }

// note: sparse_type needs to be loaded in all cases,
// because complex_sparse * sparse operations need to be defined
   if (! sparse_type_loaded) {
      octave_sparse::register_type ();

#ifdef VERBOSE
      cout << "installing sparse type at type-id = "
           << octave_sparse::static_type_id () << "\n";
#endif          
      install_sparse_ops() ;
      sparse_type_loaded= true;
   }

   bool use_complex = false;
   if (nargin > 2)
      use_complex= args(2).is_complex_type();
   else
      use_complex= args(0).is_complex_type();


   if (use_complex) {
      if (! complex_sparse_type_loaded) {
         octave_complex_sparse::register_type ();

#ifdef VERBOSE
         cout << "installing complex sparse type at type-id = "
              << octave_complex_sparse::static_type_id () << "\n";
#endif          
         install_complex_sparse_ops() ;
         complex_sparse_type_loaded= true;

         assert( 0==complex_sparse_verify_doublecomplex_type() );
      }
   }

   if (nargin == 1) {
      if (use_complex) {
         ComplexMatrix A = args(0).complex_matrix_value ();
         SuperMatrix sm= oct_matrix_to_sparse( A ) ;
         retval = new octave_complex_sparse ( sm );
      } else {
         Matrix A = args(0).matrix_value ();
         SuperMatrix sm= oct_matrix_to_sparse( A ) ;
         retval = new octave_sparse ( sm );
      }
   }
   else {
      int m=0,n=0;
      ColumnVector coefA, ridxA, cidxA;
      ComplexColumnVector coefAC;

      if (nargin == 2) {
         m= (int) args(0).double_value();
         n= (int) args(1).double_value();
         cidxA = ColumnVector ();
         ridxA = ColumnVector ();
         coefA = ColumnVector ();
      }
      else {
// 
//  I use this clumsy construction so that we can use
//  any orientation of args
//
         { ColumnVector x( args(0).vector_value() ); ridxA= x; }
         { ColumnVector x( args(1).vector_value() ); cidxA= x; }
         if (use_complex) 
            { ComplexColumnVector x( args(2).complex_vector_value() ); coefAC= x; }
         else
            { ColumnVector x( args(2).vector_value() ); coefA= x; }

         if (nargin == 3) {
            m= (int) ridxA.max();
            n= (int) cidxA.max();
         } else {
            m= (int) args(3).double_value();
            n= (int) args(4).double_value();
         }
      }

      if (use_complex) 
         retval = new octave_complex_sparse (
               assemble_sparse( n, m, coefAC, ridxA, cidxA ) );
      else
         retval = new octave_sparse (
               assemble_sparse( n, m, coefA, ridxA, cidxA ) );
   }

   return retval;
}


DEFINE_OCTAVE_ALLOCATOR (octave_sparse);

DEFINE_OV_TYPEID_FUNCTIONS_AND_DATA (octave_sparse, "sparse");

DEFINE_OCTAVE_ALLOCATOR (octave_complex_sparse);

DEFINE_OV_TYPEID_FUNCTIONS_AND_DATA (octave_complex_sparse, "complex_sparse");
/*
 * $Log: make_sparse.cc,v $
 * Revision 1.3  2001/10/14 03:06:31  aadler
 * fixed memory leak in complex sparse solve
 * fixed malloc bugs for zero size allocs
 *
 * Revision 1.2  2001/10/12 02:24:28  aadler
 * Mods to fix bugs
 * add support for all zero sparse matrices
 * add support fom complex sparse inverse
 *
 * Revision 1.7  2001/09/23 17:46:12  aadler
 * updated README
 * modified licence to GPL plus link to opensource programmes
 *
 * Revision 1.6  2001/04/04 02:13:46  aadler
 * complete complex_sparse, templates, fix memory leaks
 *
 * Revision 1.5  2001/03/30 04:36:30  aadler
 * added multiply, solve, and sparse creation
 *
 * Revision 1.4  2001/03/15 15:47:58  aadler
 * cleaned up duplicated code by using "defined" templates.
 * used default numerical conversions
 *
 * Revision 1.3  2001/02/27 03:01:52  aadler
 * added rudimentary complex matrix support
 *
 * Revision 1.2  2000/12/18 03:31:16  aadler
 * Split code to multiple files
 * added sparse inverse
 *
 */
