/***
 ***  curl-helper.c
 ***
 ***  Copyright (c) 2003, Lars Nilsson, <lars@quantumchamaeleon.com>
 ***/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <curl/curl.h>

#include <caml/alloc.h>
#include <caml/memory.h>
#include <caml/mlvalues.h>
#include <caml/callback.h>
#include <caml/fail.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

void leave_blocking_section(void);
void enter_blocking_section(void);

typedef struct Connection Connection;
typedef struct ConnectionList ConnectionList;

struct Connection
{
     CURL *connection;
     Connection *next;
     Connection *prev;

     value ocamlWriteCallback;
     value ocamlReadCallback;
     value ocamlErrorBuffer;
     value ocamlPostFields;
     value ocamlHTTPHeader;
     value ocamlHTTPPost;
     value ocamlQuote;
     value ocamlPostQuote;
     value ocamlHeaderCallback;
     value ocamlProgressCallback;
     value ocamlPasswdCallback;
     value ocamlDebugCallback;
     value ocamlHTTP200Aliases;
     value ocamlHTTPAuth;
     value ocamlProxyAuth;

     char *url;
     char *proxy;
     char *userPwd;
     char *proxyUserPwd;
     char *range;
     char *errorBuffer;
     char *postFields;
     int postFieldSize;
     char *referer;
     char *userAgent;
     char *ftpPort;
     char *cookie;
     struct curl_slist *httpHeader;
     struct curl_httppost *httpPostFirst;
     struct curl_httppost *httpPostLast;
     struct curl_slist *httpPostStrings;
     char *sslCert;
     char *sslCertType;
     char *sslCertPasswd;
     char *sslKey;
     char *sslKeyType;
     char *sslKeyPasswd;
     char *sslEngine;
     struct curl_slist *quote;
     struct curl_slist *postQuote;
     char *cookieFile;
     char *customRequest;
     char *interface;
     char *caInfo;
     char *caPath;
     char *randomFile;
     char *egdSocket;
     char *cookieJar;
     char *sslCipherList;
     char *private;
     struct curl_slist *http200Aliases;
     char *netrcFile;
};

struct ConnectionList
{
     Connection *head;
     Connection *tail;
};

static ConnectionList connectionList = {NULL, NULL};

typedef struct CURLErrorMapping CURLErrorMapping;

struct CURLErrorMapping
{
     char *name;
     CURLcode error;
};

CURLErrorMapping errorMap[] =
{
#if HAVE_DECL_CURLE_UNSUPPORTED_PROTOCOL
     {"CURLE_UNSUPPORTED_PROTOCOL", CURLE_UNSUPPORTED_PROTOCOL},
#else
     {"CURLE_UNSUPPORTED_PROTOCOL", -1},
#endif
#if HAVE_DECL_CURLE_FAILED_INIT
     {"CURLE_FAILED_INIT", CURLE_FAILED_INIT},
#else
     {"CURLE_FAILED_INIT", -1},
#endif
#if HAVE_DECL_CURLE_URL_MALFORMAT
     {"CURLE_URL_MALFORMAT", CURLE_URL_MALFORMAT},
#else
     {"CURLE_URL_MALFORMAT", -1},
#endif
#if HAVE_DECL_CURLE_URL_MALFORMAT_USER
     {"CURLE_URL_MALFORMAT_USER", CURLE_URL_MALFORMAT_USER},
#else
     {"CURLE_URL_MALFORMAT_USER", -1},
#endif
#if HAVE_DECL_CURLE_COULDNT_RESOLVE_PROXY
     {"CURLE_COULDNT_RESOLVE_PROXY", CURLE_COULDNT_RESOLVE_PROXY},
#else
     {"CURLE_COULDNT_RESOLVE_PROXY", -1},
#endif
#if HAVE_DECL_CURLE_COULDNT_RESOLVE_HOST
     {"CURLE_COULDNT_RESOLVE_HOST", CURLE_COULDNT_RESOLVE_HOST},
#else
     {"CURLE_COULDNT_RESOLVE_HOST", -1},
#endif
#if HAVE_DECL_CURLE_COULDNT_CONNECT
     {"CURLE_COULDNT_CONNECT", CURLE_COULDNT_CONNECT},
#else
     {"CURLE_COULDNT_CONNECT", -1},
#endif
#if HAVE_DECL_CURLE_FTP_WEIRD_SERVER_REPLY
     {"CURLE_FTP_WEIRD_SERVER_REPLY", CURLE_FTP_WEIRD_SERVER_REPLY},
#else
     {"CURLE_FTP_WEIRD_SERVER_REPLY", -1},
#endif
#if HAVE_DECL_CURLE_FTP_ACCESS_DENIED
     {"CURLE_FTP_ACCESS_DENIED", CURLE_FTP_ACCESS_DENIED},
#else
     {"CURLE_FTP_ACCESS_DENIED", -1},
#endif
#if HAVE_DECL_CURLE_FTP_USER_PASSWORD_INCORRECT
     {"CURLE_FTP_USER_PASSWORD_INCORRECT", CURLE_FTP_USER_PASSWORD_INCORRECT},
#else
     {"CURLE_FTP_USER_PASSWORD_INCORRECT", -1},
#endif
#if HAVE_DECL_CURLE_FTP_WEIRD_PASS_REPLY
     {"CURLE_FTP_WEIRD_PASS_REPLY", CURLE_FTP_WEIRD_PASS_REPLY},
#else
     {"CURLE_FTP_WEIRD_PASS_REPLY", -1},
#endif
#if HAVE_DECL_CURLE_FTP_WEIRD_USER_REPLY
     {"CURLE_FTP_WEIRD_USER_REPLY", CURLE_FTP_WEIRD_USER_REPLY},
#else
     {"CURLE_FTP_WEIRD_USER_REPLY", -1},
#endif
#if HAVE_DECL_CURLE_FTP_WEIRD_PASV_REPLY
     {"CURLE_FTP_WEIRD_PASV_REPLY", CURLE_FTP_WEIRD_PASV_REPLY},
#else
     {"CURLE_FTP_WEIRD_PASV_REPLY", -1},
#endif
#if HAVE_DECL_CURLE_FTP_WEIRD_227_FORMAT
     {"CURLE_FTP_WEIRD_227_FORMAT", CURLE_FTP_WEIRD_227_FORMAT},
#else
     {"CURLE_FTP_WEIRD_227_FORMAT", -1},
#endif
#if HAVE_DECL_CURLE_FTP_CANT_GET_HOST
     {"CURLE_FTP_CANT_GET_HOST", CURLE_FTP_CANT_GET_HOST},
#else
     {"CURLE_FTP_CANT_GET_HOST", -1},
#endif
#if HAVE_DECL_CURLE_FTP_CANT_RECONNECT
     {"CURLE_FTP_CANT_RECONNECT", CURLE_FTP_CANT_RECONNECT},
#else
     {"CURLE_FTP_CANT_RECONNECT", -1},
#endif
#if HAVE_DECL_CURLE_FTP_COULDNT_SET_BINARY
     {"CURLE_FTP_COULDNT_SET_BINARY", CURLE_FTP_COULDNT_SET_BINARY},
#else
     {"CURLE_FTP_COULDNT_SET_BINARY", -1},
#endif
#if HAVE_DECL_CURLE_PARTIAL_FILE
     {"CURLE_PARTIAL_FILE", CURLE_PARTIAL_FILE},
#else
     {"CURLE_PARTIAL_FILE", -1},
#endif
#if HAVE_DECL_CURLE_FTP_COULDNT_RETR_FILE
     {"CURLE_FTP_COULDNT_RETR_FILE", CURLE_FTP_COULDNT_RETR_FILE},
#else
     {"CURLE_FTP_COULDNT_RETR_FILE", -1},
#endif
#if HAVE_DECL_CURLE_FTP_WRITE_ERROR
     {"CURLE_FTP_WRITE_ERROR", CURLE_FTP_WRITE_ERROR},
#else
     {"CURLE_FTP_WRITE_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_FTP_QUOTE_ERROR
     {"CURLE_FTP_QUOTE_ERROR", CURLE_FTP_QUOTE_ERROR},
#else
     {"CURLE_FTP_QUOTE_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_HTTP_NOT_FOUND
     {"CURLE_HTTP_NOT_FOUND", CURLE_HTTP_NOT_FOUND},
#else
     {"CURLE_HTTP_NOT_FOUND", -1},
#endif
#if HAVE_DECL_CURLE_WRITE_ERROR
     {"CURLE_WRITE_ERROR", CURLE_WRITE_ERROR},
#else
     {"CURLE_WRITE_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_MALFORMAT_USER
     {"CURLE_MALFORMAT_USER", CURLE_MALFORMAT_USER},
#else
     {"CURLE_MALFORMAT_USER", -1},
#endif
#if HAVE_DECL_CURLE_FTP_COULDNT_STOR_FILE
     {"CURLE_FTP_COULDNT_STOR_FILE", CURLE_FTP_COULDNT_STOR_FILE},
#else
     {"CURLE_FTP_COULDNT_STOR_FILE", -1},
#endif
#if HAVE_DECL_CURLE_READ_ERROR
     {"CURLE_READ_ERROR", CURLE_READ_ERROR},
#else
     {"CURLE_READ_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_OUT_OF_MEMORY
     {"CURLE_OUT_OF_MEMORY", CURLE_OUT_OF_MEMORY},
#else
     {"CURLE_OUT_OF_MEMORY", -1},
#endif
#if HAVE_DECL_CURLE_OPERATION_TIMEOUTED
     {"CURLE_OPERATION_TIMEOUTED", CURLE_OPERATION_TIMEOUTED},
#else
     {"CURLE_OPERATION_TIMEOUTED", -1},
#endif
#if HAVE_DECL_CURLE_FTP_COULDNT_SET_ASCII
     {"CURLE_FTP_COULDNT_SET_ASCII", CURLE_FTP_COULDNT_SET_ASCII},
#else
     {"CURLE_FTP_COULDNT_SET_ASCII", -1},
#endif
#if HAVE_DECL_CURLE_FTP_PORT_FAILED
     {"CURLE_FTP_PORT_FAILED", CURLE_FTP_PORT_FAILED},
#else
     {"CURLE_FTP_PORT_FAILED", -1},
#endif
#if HAVE_DECL_CURLE_FTP_COULDNT_USE_REST
     {"CURLE_FTP_COULDNT_USE_REST", CURLE_FTP_COULDNT_USE_REST},
#else
     {"CURLE_FTP_COULDNT_USE_REST", -1},
#endif
#if HAVE_DECL_CURLE_FTP_COULDNT_GET_SIZE
     {"CURLE_FTP_COULDNT_GET_SIZE", CURLE_FTP_COULDNT_GET_SIZE},
#else
     {"CURLE_FTP_COULDNT_GET_SIZE", -1},
#endif
#if HAVE_DECL_CURLE_HTTP_RANGE_ERROR
     {"CURLE_HTTP_RANGE_ERROR", CURLE_HTTP_RANGE_ERROR},
#else
     {"CURLE_HTTP_RANGE_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_HTTP_POST_ERROR
     {"CURLE_HTTP_POST_ERROR", CURLE_HTTP_POST_ERROR},
#else
     {"CURLE_HTTP_POST_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_SSL_CONNECT_ERROR
     {"CURLE_SSL_CONNECT_ERROR", CURLE_SSL_CONNECT_ERROR},
#else
     {"CURLE_SSL_CONNECT_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_FTP_BAD_DOWNLOAD_RESUME
     {"CURLE_FTP_BAD_DOWNLOAD_RESUME", CURLE_FTP_BAD_DOWNLOAD_RESUME},
#else
     {"CURLE_FTP_BAD_DOWNLOAD_RESUME", -1},
#endif
#if HAVE_DECL_CURLE_FILE_COULDNT_READ_FILE
     {"CURLE_FILE_COULDNT_READ_FILE", CURLE_FILE_COULDNT_READ_FILE},
#else
     {"CURLE_FILE_COULDNT_READ_FILE", -1},
#endif
#if HAVE_DECL_CURLE_LDAP_CANNOT_BIND
     {"CURLE_LDAP_CANNOT_BIND", CURLE_LDAP_CANNOT_BIND},
#else
     {"CURLE_LDAP_CANNOT_BIND", -1},
#endif
#if HAVE_DECL_CURLE_LDAP_SEARCH_FAILED
     {"CURLE_LDAP_SEARCH_FAILED", CURLE_LDAP_SEARCH_FAILED},
#else
     {"CURLE_LDAP_SEARCH_FAILED", -1},
#endif
#if HAVE_DECL_CURLE_LIBRARY_NOT_FOUND
     {"CURLE_LIBRARY_NOT_FOUND", CURLE_LIBRARY_NOT_FOUND},
#else
     {"CURLE_LIBRARY_NOT_FOUND", -1},
#endif
#if HAVE_DECL_CURLE_FUNCTION_NOT_FOUND
     {"CURLE_FUNCTION_NOT_FOUND", CURLE_FUNCTION_NOT_FOUND},
#else
     {"CURLE_FUNCTION_NOT_FOUND", -1},
#endif
#if HAVE_DECL_CURLE_ABORTED_BY_CALLBACK
     {"CURLE_ABORTED_BY_CALLBACK", CURLE_ABORTED_BY_CALLBACK},
#else
     {"CURLE_ABORTED_BY_CALLBACK", -1},
#endif
#if HAVE_DECL_CURLE_BAD_FUNCTION_ARGUMENT
     {"CURLE_BAD_FUNCTION_ARGUMENT", CURLE_BAD_FUNCTION_ARGUMENT},
#else
     {"CURLE_BAD_FUNCTION_ARGUMENT", -1},
#endif
#if HAVE_DECL_CURLE_BAD_CALLING_ORDER
     {"CURLE_BAD_CALLING_ORDER", CURLE_BAD_CALLING_ORDER},
#else
     {"CURLE_BAD_CALLING_ORDER", -1},
#endif
#if HAVE_DECL_CURLE_HTTP_PORT_FAILED
     {"CURLE_HTTP_PORT_FAILED", CURLE_HTTP_PORT_FAILED},
#else
     {"CURLE_HTTP_PORT_FAILED", -1},
#endif
#if HAVE_DECL_CURLE_BAD_PASSWORD_ENTERED
     {"CURLE_BAD_PASSWORD_ENTERED", CURLE_BAD_PASSWORD_ENTERED},
#else
     {"CURLE_BAD_PASSWORD_ENTERED", -1},
#endif
#if HAVE_DECL_CURLE_TOO_MANY_REDIRECTS
     {"CURLE_TOO_MANY_REDIRECTS", CURLE_TOO_MANY_REDIRECTS},
#else
     {"CURLE_TOO_MANY_REDIRECTS", -1},
#endif
#if HAVE_DECL_CURLE_UNKNOWN_TELNET_OPTION
     {"CURLE_UNKNOWN_TELNET_OPTION", CURLE_UNKNOWN_TELNET_OPTION},
#else
     {"CURLE_UNKNOWN_TELNET_OPTION", -1},
#endif
#if HAVE_DECL_CURLE_TELNET_OPTION_SYNTAX
     {"CURLE_TELNET_OPTION_SYNTAX", CURLE_TELNET_OPTION_SYNTAX},
#else
     {"CURLE_TELNET_OPTION_SYNTAX", -1},
#endif
#if HAVE_DECL_CURLE_SSL_PEER_CERTIFICATE
     {"CURLE_SSL_PEER_CERTIFICATE", CURLE_SSL_PEER_CERTIFICATE},
#else
     {"CURLE_SSL_PEER_CERTIFICATE", -1},
#endif
#if HAVE_DECL_CURLE_GOT_NOTHING
     {"CURLE_GOT_NOTHING", CURLE_GOT_NOTHING},
#else
     {"CURLE_GOT_NOTHING", -1},
#endif
#if HAVE_DECL_CURLE_SSL_ENGINE_NOT_FOUND
     {"CURLE_SSL_ENGINE_NOT_FOUND", CURLE_SSL_ENGINE_NOTFOUND},
#else
     {"CURLE_SSL_ENGINE_NOT_FOUND", -1},
#endif
#if HAVE_DECL_CURLE_SSL_ENGINE_SET_FAILED
     {"CURLE_SSL_ENGINE_SET_FAILED", CURLE_SSL_ENGINE_SETFAILED},
#else
     {"CURLE_SSL_ENGINE_SET_FAILED", -1},
#endif
#if HAVE_DECL_CURLE_SEND_ERROR
     {"CURLE_SEND_ERROR", CURLE_SEND_ERROR},
#else
     {"CURLE_SEND_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_RECV_ERROR
     {"CURLE_RECV_ERROR", CURLE_RECV_ERROR},
#else
     {"CURLE_RECV_ERROR", -1},
#endif
#if HAVE_DECL_CURLE_SHARE_IN_USE
     {"CURLE_SHARE_IN_USE", CURLE_SHARE_IN_USE},
#else
     {"CURLE_SHARE_IN_USE", -1},
#endif
#if HAVE_DECL_CURLE_SSL_CERTPROBLEM
     {"CURLE_SSL_CERTPROBLEN", CURLE_SSL_CERTPROBLEM},
#else
     {"CURLE_SSL_CERTPROBLEN", -1},
#endif
#if HAVE_DECL_CURLE_SSL_CIPHER
     {"CURLE_SSL_CIPHER", CURLE_SSL_CIPHER},
#else
     {"CURLE_SSL_CIPHER", -1},
#endif
#if HAVE_DECL_CURLE_SSL_CACERT
     {"CURLE_SSL_CACERT", CURLE_SSL_CACERT},
#else
     {"CURLE_SSL_CACERT", -1},
#endif
#if HAVE_DECL_CURLE_BAD_CONTENT_ENCODING
     {"CURLE_BAD_CONTENT_ENCODING", CURLE_BAD_CONTENT_ENCODING},
#else
     {"CURLE_BAD_CONTENT_ENCODING", -1},
#endif
#if HAVE_DECL_CURLE_LDAP_INVALID_URL
     {"CURLE_LDAP_INVALID_URL", CURLE_LDAP_INVALID_URL},
#else
     {"CURLE_LDAP_INVALID_URL", -1},
#endif
#if HAVE_DECL_CURLE_FILESIZE_EXCEEDED
     {"CURLE_FILESIZE_EXCEEDED", CURLE_FILESIZE_EXCEEDED},
#else
     {"CURLE_FILESIZE_EXCEEDED", -1},
#endif
#if HAVE_DECL_CURLE_FTP_SSL_FAILED
     {"CURLE_FTP_SSL_FAILED", CURLE_FTP_SSL_FAILED},
#else
     {"CURLE_FTP_SSL_FAILED", -1},
#endif
     {"CURLE_OK", CURLE_OK},
     {NULL, 0}
};

typedef struct CURLOptionMapping CURLOptionMapping;

struct CURLOptionMapping
{
     void (*optionHandler)(Connection *, value);
     char *name;
     CURLoption option;
};

CURLOptionMapping unimplementedOptionMap[] =
{
     {NULL, "CURLOPT_STDERR", CURLOPT_STDERR},
     {NULL, NULL, 0}
};

static void handleWriteFunction(Connection *, value);
static void handleReadFunction(Connection *, value);
static void handleInFileSize(Connection *, value);
static void handleURL(Connection *, value);
static void handleProxy(Connection *, value);
static void handleProxyPort(Connection *, value);
static void handleHTTPProxyTunnel(Connection *, value);
static void handleVerbose(Connection *, value);
static void handleHeader(Connection *, value);
static void handleNoProgress(Connection *, value);
static void handleNoSignal(Connection *, value);
static void handleNoBody(Connection *, value);
static void handleFailOnError(Connection *, value);
static void handleUpload(Connection *, value);
static void handlePost(Connection *, value);
static void handleFTPListOnly(Connection *, value);
static void handleFTPAppend(Connection *, value);
static void handleNETRC(Connection *, value);
static void handleEncoding(Connection *, value);
static void handleFollowLocation(Connection *, value);
static void handleTransferText(Connection *, value);
static void handlePut(Connection *, value);
static void handleUserPwd(Connection *, value);
static void handleProxyUserPwd(Connection *, value);
static void handleRange(Connection *, value);
static void handleErrorBuffer(Connection *, value);
static void handleTimeout(Connection *, value);
static void handlePostFields(Connection *, value);
static void handlePostFieldSize(Connection *, value);
static void handleReferer(Connection *, value);
static void handleUserAgent(Connection *, value);
static void handleFTPPort(Connection *, value);
static void handleLowSpeedLimit(Connection *, value);
static void handleLowSpeedTime(Connection *, value);
static void handleResumeFrom(Connection *, value);
static void handleCookie(Connection *, value);
static void handleHTTPHeader(Connection *, value);
static void handleHTTPPost(Connection *, value);
static void handleSSLCert(Connection *, value);
static void handleSSLCertType(Connection *, value);
static void handleSSLCertPasswd(Connection *, value);
static void handleSSLKey(Connection *, value);
static void handleSSLKeyType(Connection *, value);
static void handleSSLKeyPasswd(Connection *, value);
static void handleSSLEngine(Connection *, value);
static void handleSSLEngineDefault(Connection *, value);
static void handleCRLF(Connection *, value);
static void handleQuote(Connection *, value);
static void handlePostQuote(Connection *, value);
static void handleHeaderFunction(Connection *, value);
static void handleCookieFile(Connection *, value);
static void handleSSLVersion(Connection *, value);
static void handleTimeCondition(Connection *, value);
static void handleTimeValue(Connection *, value);
static void handleCustomRequest(Connection *, value);
static void handleInterface(Connection *, value);
static void handleKRB4Level(Connection *, value);
static void handleProgressFunction(Connection *, value);
static void handleSSLVerifyPeer(Connection *, value);
static void handleCAInfo(Connection *, value);
static void handleCAPath(Connection *, value);
static void handlePasswdFunction(Connection *, value);
static void handleFileTime(Connection *, value);
static void handleMaxRedirs(Connection *, value);
static void handleMaxConnects(Connection *, value);
static void handleClosePolicy(Connection *, value);
static void handleFreshConnect(Connection *, value);
static void handleForbidReuse(Connection *, value);
static void handleRandomFile(Connection *, value);
static void handleEGDSocket(Connection *, value);
static void handleConnectTimeout(Connection *, value);
static void handleHTTPGet(Connection *, value);
static void handleSSLVerifyHost(Connection *, value);
static void handleCookieJar(Connection *, value);
static void handleSSLCipherList(Connection *, value);
static void handleHTTPVersion(Connection *, value);
static void handleFTPUseEPSV(Connection *, value);
static void handleDNSCacheTimeout(Connection *, value);
static void handleDNSUseGlobalCache(Connection *, value);
static void handleDebugFunction(Connection *, value);
static void handlePrivate(Connection *, value);
static void handleHTTP200Aliases(Connection *, value);
static void handleUnrestrictedAuth(Connection *, value);
static void handleFTPUseEPRT(Connection *, value);
static void handleHTTPAuth(Connection *, value);
static void handleFTPCreateMissingDirs(Connection *, value);
static void handleProxyAuth(Connection *, value);
static void handleFTPResponseTimeout(Connection *, value);
static void handleIPResolve(Connection *, value);
static void handleMaxFileSize(Connection *, value);
static void handleInFileSizeLarge(Connection *, value);
static void handleResumeFromLarge(Connection *, value);
static void handleMaxFileSizeLarge(Connection *, value);
static void handleNETRCFile(Connection *, value);
static void handleFTPSSL(Connection *, value);
static void handlePostFieldSizeLarge(Connection *, value);
static void handleTCPNoDelay(Connection *, value);

CURLOptionMapping implementedOptionMap[] =
{
     {handleWriteFunction, "CURLOPT_WRITEFUNCTION", CURLOPT_WRITEFUNCTION},
     {handleReadFunction, "CURLOPT_READFUNCTION", CURLOPT_READFUNCTION},
     {handleInFileSize, "CURLOPT_INFILESIZE", CURLOPT_INFILESIZE},
     {handleURL, "CURLOPT_URL", CURLOPT_URL},
     {handleProxy, "CURLOPT_PROXY", CURLOPT_PROXY},
     {handleProxyPort, "CURLOPT_PROXYPORT", CURLOPT_PROXYPORT},
     {handleHTTPProxyTunnel, "CURLOPT_HTTPPROXYTUNNEL", CURLOPT_HTTPPROXYTUNNEL},
     {handleVerbose, "CURLOPT_VERBOSE", CURLOPT_VERBOSE},
     {handleHeader, "CURLOPT_HEADER", CURLOPT_HEADER},
     {handleNoProgress, "CURLOPT_NOPROGRESS", CURLOPT_NOPROGRESS},
#if HAVE_DECL_CURLOPT_NOSIGNAL
     {handleNoSignal, "CURLOPT_NOSIGNAL", CURLOPT_NOSIGNAL},
#else
     {handleNoSignal, "CURLOPT_NOSIGNAL", 0},
#endif
     {handleNoBody, "CURLOPT_NOBODY", CURLOPT_NOBODY},
     {handleFailOnError, "CURLOPT_FAILONERROR", CURLOPT_FAILONERROR},
     {handleUpload, "CURLOPT_UPLOAD", CURLOPT_UPLOAD},
     {handlePost, "CURLOPT_POST", CURLOPT_POST},
     {handleFTPListOnly, "CURLOPT_FTPLISTONLY", CURLOPT_FTPLISTONLY},
     {handleFTPAppend, "CURLOPT_FTPAPPEND", CURLOPT_FTPAPPEND},
     {handleNETRC, "CURLOPT_NETRC", CURLOPT_NETRC},
#if HAVE_DECL_CURLOPT_ENCODING
     {handleEncoding, "CURLOPT_ENCODING", CURLOPT_ENCODING},
#else
     {handleEncoding, "CURLOPT_ENCODING", 0},
#endif
     {handleFollowLocation, "CURLOPT_FOLLOWLOCATION", CURLOPT_FOLLOWLOCATION},
     {handleTransferText, "CURLOPT_TRANSFERTEXT", CURLOPT_TRANSFERTEXT},
     {handlePut, "CURLOPT_PUT", CURLOPT_PUT},
     {handleUserPwd, "CURLOPT_USERPWD", CURLOPT_USERPWD},
     {handleProxyUserPwd, "CURLOPT_PROXYUSERPWD", CURLOPT_PROXYUSERPWD},
     {handleRange, "CURLOPT_RANGE", CURLOPT_RANGE},
     {handleErrorBuffer, "CURLOPT_ERRORBUFFER", CURLOPT_ERRORBUFFER},
     {handleTimeout, "CURLOPT_TIMEOUT", CURLOPT_TIMEOUT},
     {handlePostFields, "CURLOPT_POSTFIELDS", CURLOPT_POSTFIELDS},
     {handlePostFieldSize, "CURLOPT_POSTFIELDSIZE", CURLOPT_POSTFIELDSIZE},
     {handleReferer, "CURLOPT_REFERER", CURLOPT_REFERER},
     {handleUserAgent, "CURLOPT_USERAGENT", CURLOPT_USERAGENT},
     {handleFTPPort, "CURLOPT_FTPPORT", CURLOPT_FTPPORT},
     {handleLowSpeedLimit, "CURLOPT_LOW_SPEED_LIMIT", CURLOPT_LOW_SPEED_LIMIT},
     {handleLowSpeedTime, "CURLOPT_LOW_SPEED_TIME", CURLOPT_LOW_SPEED_TIME},
     {handleResumeFrom, "CURLOPT_RESUME_FROM", CURLOPT_RESUME_FROM},
     {handleCookie, "CURLOPT_COOKIE", CURLOPT_COOKIE},
     {handleHTTPHeader, "CURLOPT_HTTPHEADER", CURLOPT_HTTPHEADER},
     {handleHTTPPost, "CURLOPT_HTTPPOST", CURLOPT_HTTPPOST},
     {handleSSLCert, "CURLOPT_SSLCERT", CURLOPT_SSLCERT},
     {handleSSLCertType, "CURLOPT_SSLCERTTYPE", CURLOPT_SSLCERTTYPE},
     {handleSSLCertPasswd, "CURLOPT_SSLCERTPASSWD", CURLOPT_SSLCERTPASSWD},
     {handleSSLKey, "CURLOPT_SSLKEY", CURLOPT_SSLKEY},
     {handleSSLKeyType, "CURLOPT_SSLKEYTYPE", CURLOPT_SSLKEYTYPE},
     {handleSSLKeyPasswd, "CURLOPT_SSLKEYPASSWD", CURLOPT_SSLKEYPASSWD},
     {handleSSLEngine, "CURLOPT_SSLENGINE", CURLOPT_SSLENGINE},
     {handleSSLEngineDefault, "CURLOPT_SSLENGINE_DEFAULT", CURLOPT_SSLENGINE_DEFAULT},
     {handleCRLF, "CURLOPT_CRLF", CURLOPT_CRLF},
     {handleQuote, "CURLOPT_QUOTE", CURLOPT_QUOTE},
     {handlePostQuote, "CURLOPT_POSTQUOTE", CURLOPT_POSTQUOTE},
     {handleHeaderFunction, "CURLOPT_HEADERFUNCTION", CURLOPT_HEADERFUNCTION},
     {handleCookieFile, "CURLOPT_COOKIEFILE", CURLOPT_COOKIEFILE},
     {handleSSLVersion, "CURLOPT_SSLVERSION", CURLOPT_SSLVERSION},
     {handleTimeCondition, "CURLOPT_TIMECONDITION", CURLOPT_TIMECONDITION},
     {handleTimeValue, "CURLOPT_TIMEVALUE", CURLOPT_TIMEVALUE},
     {handleCustomRequest, "CURLOPT_CUSTOMREQUEST", CURLOPT_CUSTOMREQUEST},
     {handleInterface, "CURLOPT_INTERFACE", CURLOPT_INTERFACE},
     {handleKRB4Level, "CURLOPT_KRB4LEVEL", CURLOPT_KRB4LEVEL},
     {handleProgressFunction, "CURLOPT_PROGRESSFUNCTION", CURLOPT_PROGRESSFUNCTION},
     {handleSSLVerifyPeer, "CURLOPT_SSLVERIFYPEER", CURLOPT_SSL_VERIFYPEER},
     {handleCAInfo, "CURLOPT_CAINFO", CURLOPT_CAINFO},
     {handleCAPath, "CURLOPT_CAPATH", CURLOPT_CAPATH},
     {handlePasswdFunction, "CURLOPT_PASSWDFUNCTION", CURLOPT_PASSWDFUNCTION},
     {handleFileTime, "CURLOPT_FILETIME", CURLOPT_FILETIME},
     {handleMaxRedirs, "CURLOPT_MAXREDIRS", CURLOPT_MAXREDIRS},
     {handleMaxConnects, "CURLOPT_MAXCONNECTS", CURLOPT_MAXCONNECTS},
     {handleClosePolicy, "CURLOPT_CLOSEPOLICY", CURLOPT_CLOSEPOLICY},
     {handleFreshConnect, "CURLOPT_FRESH_CONNECT", CURLOPT_FRESH_CONNECT},
     {handleForbidReuse, "CURLOPT_FORBID_REUSE", CURLOPT_FORBID_REUSE},
     {handleRandomFile, "CURLOPT_RANDOM_FILE", CURLOPT_RANDOM_FILE},
     {handleEGDSocket, "CURLOPT_EGDSOCKET", CURLOPT_EGDSOCKET},
     {handleConnectTimeout, "CURLOPT_CONNECTTIMEOUT", CURLOPT_CONNECTTIMEOUT},
     {handleHTTPGet, "CURLOPT_HTTPGET", CURLOPT_HTTPGET},
     {handleSSLVerifyHost, "CURLOPT_SSL_VERIFYHOST", CURLOPT_SSL_VERIFYHOST},
     {handleCookieJar, "CURLOPT_COOKIEJAR", CURLOPT_COOKIEJAR},
     {handleSSLCipherList, "CURLOPT_SSL_CIPHERLIST", CURLOPT_SSL_CIPHER_LIST},
     {handleHTTPVersion, "CURLOPT_HTTP_VERSION", CURLOPT_HTTP_VERSION},
     {handleFTPUseEPSV, "CURLOPT_FTP_USE_EPSV", CURLOPT_FTP_USE_EPSV},
     {handleDNSCacheTimeout, "CURLOPT_DNS_CACHE_TIMEOUT", CURLOPT_DNS_CACHE_TIMEOUT},
     {handleDNSUseGlobalCache, "CURLOPT_DNS_USE_GLOBAL_CACHE", CURLOPT_DNS_USE_GLOBAL_CACHE},
     {handleDebugFunction, "CURLOPT_DEBUGFUNCTION", CURLOPT_DEBUGFUNCTION},
#if HAVE_DECL_CURLOPT_PRIVATE
     {handlePrivate, "CURLOPT_PRIVATE", CURLOPT_PRIVATE},
#else
     {handlePrivate, "CURLOPT_PRIVATE", 0},
#endif
#if HAVE_DECL_CURLOPT_HTTP200ALIASES
     {handleHTTP200Aliases, "CURLOPT_HTTP200ALIASES", CURLOPT_HTTP200ALIASES},
#else
     {handleHTTP200Aliases, "CURLOPT_HTTP200ALIASES", 0},
#endif
#if HAVE_DECL_CURLOPT_UNRESTRICTED_AUTH
     {handleUnrestrictedAuth, "CURLOPT_UNRESTRICTED_AUTH", CURLOPT_UNRESTRICTED_AUTH},
#else
     {handleUnrestrictedAuth, "CURLOPT_UNRESTRICTED_AUTH", 0},
#endif
#if HAVE_DECL_CURLOPT_FTP_USE_EPRT
     {handleFTPUseEPRT, "CURLOPT_FTP_USE_EPRT", CURLOPT_FTP_USE_EPRT},
#else
     {handleFTPUseEPRT, "CURLOPT_FTP_USE_EPRT", 0},
#endif
#if HAVE_DECL_CURLOPT_HTTPAUTH
     {handleHTTPAuth, "CURLOPT_HTTPAUTH", CURLOPT_HTTPAUTH},
#else
     {handleHTTPAuth, "CURLOPT_HTTPAUTH", 0},
#endif
#if HAVE_DECL_CURLOPT_FTP_CREATE_MISSING_DIRS
     {handleFTPCreateMissingDirs, "CURLOPT_FTP_CREATE_MISSING_DIRS", CURLOPT_FTP_CREATE_MISSING_DIRS},
#else
     {handleFTPCreateMissingDirs, "CURLOPT_FTP_CREATE_MISSING_DIRS", 0},
#endif
#if HAVE_DECL_CURLOPT_PROXYAUTH
     {handleProxyAuth, "CURLOPT_PROXYAUTH", CURLOPT_PROXYAUTH},
#else
     {handleProxyAuth, "CURLOPT_PROXYAUTH", 0},
#endif
#if HAVE_DECL_CURLOPT_FTP_RESPONSE_TIMEOUT
     {handleFTPResponseTimeout, "CURLOPT_FTP_RESPONSE_TIMEOUT", CURLOPT_FTP_RESPONSE_TIMEOUT},
#else
     {handleFTPResponseTimeout, "CURLOPT_FTP_RESPONSE_TIMEOUT", 0},
#endif
#if HAVE_DECL_CURLOPT_IPRESOLVE
     {handleIPResolve, "CURLOPT_IPRESOLVE", CURLOPT_IPRESOLVE},
#else
     {handleIPResolve, "CURLOPT_IPRESOLVE", 0},
#endif
#if HAVE_DECL_CURLOPT_MAXFILESIZE
     {handleMaxFileSize, "CURLOPT_MAXFILESIZE", CURLOPT_MAXFILESIZE},
#else
     {handleMaxFileSize, "CURLOPT_MAXFILESIZE", 0},
#endif
#if HAVE_DECL_CURLOPT_INFILSIZE_LARGE
     {handleInFileSizeLarge, "CURLOPT_INFILESIZE_LARGE", CURLOPT_INFILESIZE_LARGE},
#else
     {handleInFileSizeLarge, "CURLOPT_INFILESIZE_LARGE", 0},
#endif
#if HAVE_DECL_CURLOPT_RESUME_FROM_LARGE
     {handleResumeFromLarge, "CURLOPT_RESUME_FROM_LARGE", CURLOPT_RESUME_FROM_LARGE},
#else
     {handleResumeFromLarge, "CURLOPT_RESUME_FROM_LARGE", 0},
#endif
#if HAVE_DECL_CURLOPT_MAXFILESIZE_LARGE
     {handleMaxFileSizeLarge, "CURLOPT_MAXFILESIZE_LARGE", CURLOPT_MAXFILESIZE_LARGE},
#else
     {handleMaxFileSizeLarge, "CURLOPT_MAXFILESIZE_LARGE", 0},
#endif
#if HAVE_DECL_CURLOPT_NETRC_FILE
     {handleNETRCFile, "CURLOPT_NETRC_FILE", CURLOPT_NETRC_FILE},
#else
     {handleNETRCFile, "CURLOPT_NETRC_FILE", 0},
#endif
#if HAVE_DECL_CURLOPT_FTP_SSL
     {handleFTPSSL, "CURLOPT_FTP_SSL", CURLOPT_FTP_SSL},
#else
     {handleFTPSSL, "CURLOPT_FTP_SSL", 0},
#endif
#if HAVE_DECL_CURLOPT_POSTFIELDSIZE_LARGE
     {handlePostFieldSizeLarge, "CURLOPT_POSTFIELDSIZE_LARGE", CURLOPT_POSTFIELDSIZE_LARGE},
#else
     {handlePostFieldSizeLarge, "CURLOPT_POSTFIELDSIZE_LARGE", 0},
#endif
#if HAVE_DECL_CURLOPT_TCP_NODELAY
     {handleTCPNoDelay, "CURLOPT_TCP_NODELAY", CURLOPT_TCP_NODELAY}
#else
     {handleTCPNoDelay, "CURLOPT_TCP_NODELAY", 0}
#endif
};

static char *findOption(CURLOptionMapping optionMap[],
                        CURLoption option)
{
     return optionMap[option].name;
}

static void free_curl_slist(struct curl_slist *slist)
{
     struct curl_slist *item;

     if (slist == NULL)
          return;

     item = slist;

     while (item != NULL)
     {
          free(item->data);
          item->data = NULL;
          item = item->next;
     }

     curl_slist_free_all(slist);
}

static void raiseError(Connection *conn, CURLcode code)
{
     value *exception;
     value exceptionData;
     char *errorString = "Unknown Error";
     int i;

     for (i = 0; errorMap[i].name != NULL; i++)
     {
          if (errorMap[i].error == code)
          {
               errorString = errorMap[i].name;
               break;
          }
     }

     exceptionData = alloc(3, 0);

     Field(exceptionData, 0) = Val_int(code);
     Field(exceptionData, 1) = Val_int(code);
     Field(exceptionData, 2) = copy_string(errorString);

     if (conn->errorBuffer != NULL)
     {
          Field(conn->ocamlErrorBuffer, 0) =
               copy_string(conn->errorBuffer);
     }

     exception = caml_named_value("CurlException");

     if (exception == NULL)
          failwith(errorString);

     raise_with_arg(*exception, exceptionData);
}

static Connection *newConnection(void)
{
     Connection *connection;

     connection = (Connection *)malloc(sizeof(Connection));
     
     enter_blocking_section();
     connection->connection = curl_easy_init();
     leave_blocking_section();

     connection->next = NULL;
     connection->prev = NULL;

     if (connectionList.tail == NULL)
     {
          connectionList.tail = connection;
          connectionList.head = connection;
     }
     else
     {
          connection->prev = connectionList.head;
          connectionList.head->next = connection;
          connectionList.head = connection;
     }

     connection->ocamlWriteCallback = Val_unit;
     connection->ocamlReadCallback = Val_unit;
     connection->ocamlErrorBuffer = Val_unit;
     connection->ocamlPostFields = Val_unit;
     connection->ocamlHTTPHeader = Val_unit;
     connection->ocamlHTTPPost = Val_unit;
     connection->ocamlQuote = Val_unit;
     connection->ocamlPostQuote = Val_unit;
     connection->ocamlHeaderCallback = Val_unit;
     connection->ocamlProgressCallback = Val_unit;
     connection->ocamlPasswdCallback = Val_unit;
     connection->ocamlDebugCallback = Val_unit;
     connection->ocamlHTTP200Aliases = Val_unit;
     connection->ocamlHTTPAuth = Val_unit;
     connection->ocamlProxyAuth = Val_unit;

     register_global_root(&connection->ocamlWriteCallback);
     register_global_root(&connection->ocamlReadCallback);
     register_global_root(&connection->ocamlErrorBuffer);
     register_global_root(&connection->ocamlPostFields);
     register_global_root(&connection->ocamlHTTPHeader);
     register_global_root(&connection->ocamlHTTPPost);
     register_global_root(&connection->ocamlQuote);
     register_global_root(&connection->ocamlPostQuote);
     register_global_root(&connection->ocamlHeaderCallback);
     register_global_root(&connection->ocamlProgressCallback);
     register_global_root(&connection->ocamlPasswdCallback);
     register_global_root(&connection->ocamlDebugCallback);
     register_global_root(&connection->ocamlHTTP200Aliases);
     register_global_root(&connection->ocamlHTTPAuth);
     register_global_root(&connection->ocamlProxyAuth);

     connection->url = NULL;
     connection->proxy = NULL;
     connection->userPwd = NULL;
     connection->proxyUserPwd = NULL;
     connection->range = NULL;
     connection->errorBuffer = NULL;
     connection->postFields = NULL;
     connection->postFieldSize = -1;
     connection->referer = NULL;
     connection->userAgent = NULL;
     connection->ftpPort = NULL;
     connection->cookie = NULL;
     connection->httpHeader = NULL;
     connection->httpPostFirst = NULL;
     connection->httpPostLast = NULL;
     connection->httpPostStrings = NULL;
     connection->sslCert = NULL;
     connection->sslCertType = NULL;
     connection->sslCertPasswd = NULL;
     connection->sslKey = NULL;
     connection->sslKeyType = NULL;
     connection->sslKeyPasswd = NULL;
     connection->sslEngine = NULL;
     connection->quote = NULL;
     connection->postQuote = NULL;
     connection->cookieFile = NULL;
     connection->customRequest = NULL;
     connection->caInfo = NULL;
     connection->caPath = NULL;
     connection->randomFile = NULL;
     connection->egdSocket = NULL;
     connection->cookieJar = NULL;
     connection->sslCipherList = NULL;
     connection->private = NULL;
     connection->http200Aliases = NULL;
     connection->netrcFile = NULL;

     return connection;
}

static Connection *duplicateConnection(Connection *original)
{
     Connection *connection;

     connection = (Connection *)malloc(sizeof(Connection));

     enter_blocking_section();
     connection->connection = curl_easy_duphandle(original->connection);
     leave_blocking_section();

     connection->next = NULL;
     connection->prev = NULL;

     if (connectionList.tail == NULL)
     {
          connectionList.tail = connection;
          connectionList.head = connection;
     }
     else
     {
          connection->prev = connectionList.head;
          connectionList.head->next = connection;
          connectionList.head = connection;
     }

     connection->ocamlWriteCallback = Val_unit;
     connection->ocamlReadCallback = Val_unit;
     connection->ocamlErrorBuffer = Val_unit;
     connection->ocamlPostFields = Val_unit;
     connection->ocamlHTTPHeader = Val_unit;
     connection->ocamlHTTPPost = Val_unit;
     connection->ocamlQuote = Val_unit;
     connection->ocamlPostQuote = Val_unit;
     connection->ocamlHeaderCallback = Val_unit;
     connection->ocamlProgressCallback = Val_unit;
     connection->ocamlPasswdCallback = Val_unit;
     connection->ocamlDebugCallback = Val_unit;
     connection->ocamlHTTP200Aliases = Val_unit;
     connection->ocamlHTTPAuth = Val_unit;
     connection->ocamlProxyAuth = Val_unit;

     register_global_root(&connection->ocamlWriteCallback);
     register_global_root(&connection->ocamlReadCallback);
     register_global_root(&connection->ocamlErrorBuffer);
     register_global_root(&connection->ocamlPostFields);
     register_global_root(&connection->ocamlHTTPHeader);
     register_global_root(&connection->ocamlHTTPPost);
     register_global_root(&connection->ocamlQuote);
     register_global_root(&connection->ocamlPostQuote);
     register_global_root(&connection->ocamlHeaderCallback);
     register_global_root(&connection->ocamlProgressCallback);
     register_global_root(&connection->ocamlPasswdCallback);
     register_global_root(&connection->ocamlDebugCallback);
     register_global_root(&connection->ocamlHTTP200Aliases);
     register_global_root(&connection->ocamlHTTPAuth);
     register_global_root(&connection->ocamlProxyAuth);

     connection->ocamlWriteCallback = original->ocamlWriteCallback;
     connection->ocamlReadCallback = original->ocamlReadCallback;
     connection->ocamlErrorBuffer = original->ocamlErrorBuffer;
     connection->ocamlPostFields = original->ocamlPostFields;
     connection->ocamlHTTPHeader = original->ocamlHTTPHeader;
     connection->ocamlQuote = original->ocamlQuote;
     connection->ocamlPostQuote = original->ocamlPostQuote;
     connection->ocamlHeaderCallback = original->ocamlHeaderCallback;
     connection->ocamlProgressCallback = original->ocamlProgressCallback;
     connection->ocamlPasswdCallback = original->ocamlPasswdCallback;
     connection->ocamlDebugCallback = original->ocamlDebugCallback;
     connection->ocamlHTTP200Aliases = original->ocamlHTTP200Aliases;
     connection->ocamlHTTPAuth = original->ocamlHTTPAuth;
     connection->ocamlProxyAuth = original->ocamlProxyAuth;

     connection->url = NULL;
     connection->proxy = NULL;
     connection->userPwd = NULL;
     connection->proxyUserPwd = NULL;
     connection->range = NULL;
     connection->errorBuffer = NULL;
     connection->postFields = NULL;
     connection->postFieldSize = -1;
     connection->referer = NULL;
     connection->userAgent = NULL;
     connection->ftpPort = NULL;
     connection->cookie = NULL;
     connection->httpHeader = NULL;
     connection->httpPostFirst = NULL;
     connection->httpPostLast = NULL;
     connection->httpPostStrings = NULL;
     connection->sslCert = NULL;
     connection->sslCertType = NULL;
     connection->sslCertPasswd = NULL;
     connection->sslKey = NULL;
     connection->sslKeyType = NULL;
     connection->sslKeyPasswd = NULL;
     connection->sslEngine = NULL;
     connection->quote = NULL;
     connection->postQuote = NULL;
     connection->cookieFile = NULL;
     connection->customRequest = NULL;
     connection->caInfo = NULL;
     connection->caPath = NULL;
     connection->randomFile = NULL;
     connection->egdSocket = NULL;
     connection->cookieJar = NULL;
     connection->sslCipherList = NULL;
     connection->private = NULL;
     connection->http200Aliases = NULL;
     connection->netrcFile = NULL;

     if (original->url != NULL)
          handleURL(connection, (value)original->url);
     if (original->proxy != NULL)
          handleProxy(connection, (value)original->proxy);
     if (original->userPwd != NULL)
          handleUserPwd(connection, (value)original->userPwd);
     if (original->proxyUserPwd != NULL)
          handleProxyUserPwd(connection, (value)original->proxyUserPwd);
     if (original->range != NULL)
          handleRange(connection, (value)original->range);
     if (original->errorBuffer != NULL)
          handleErrorBuffer(connection, (value)original->ocamlErrorBuffer);
     if (original->postFields != NULL)
          handlePostFields(connection, (value)original->ocamlPostFields);
     if (original->referer != NULL)
          handleReferer(connection, (value)original->referer);
     if (original->userAgent != NULL)
          handleUserAgent(connection, (value)original->userAgent);
     if (original->ftpPort != NULL)
          handleFTPPort(connection, (value)original->ftpPort);
     if (original->cookie != NULL)
          handleCookie(connection, (value)original->cookie);
     if (original->httpHeader != NULL)
          handleHTTPHeader(connection, original->ocamlHTTPHeader);
     if (original->httpPostFirst != NULL)
          handleHTTPPost(connection, original->ocamlHTTPPost);
     if (original->sslCert != NULL)
          handleSSLCert(connection, (value)original->sslCert);
     if (original->sslCertType != NULL)
          handleSSLCertType(connection, (value)original->sslCertType);
     if (original->sslCertPasswd != NULL)
          handleSSLCertPasswd(connection, (value)original->sslCertPasswd);
     if (original->sslKey != NULL)
          handleSSLKey(connection, (value)original->sslKey);
     if (original->sslKeyType != NULL)
          handleSSLKeyType(connection, (value)original->sslKeyType);
     if (original->sslKeyPasswd != NULL)
          handleSSLKeyPasswd(connection, (value)original->sslKeyPasswd);
     if (original->sslEngine != NULL)
          handleSSLEngine(connection, (value)original->sslEngine);
     if (original->quote != NULL)
          handleQuote(connection, original->ocamlQuote);
     if (original->postQuote != NULL)
          handlePostQuote(connection, original->ocamlPostQuote);
     if (original->cookieFile != NULL)
          handleCookieFile(connection, (value)original->cookieFile);
     if (original->customRequest != NULL)
          handleCustomRequest(connection, (value)original->customRequest);
     if (original->caInfo != NULL)
          handleCAInfo(connection, (value)original->caInfo);
     if (original->caPath != NULL)
          handleCAPath(connection, (value)original->caPath);
     if (original->randomFile != NULL)
          handleRandomFile(connection, (value)original->randomFile);
     if (original->egdSocket != NULL)
          handleEGDSocket(connection, (value)original->egdSocket);
     if (original->cookieJar != NULL)
          handleCookieJar(connection, (value)original->cookieJar);
     if (original->sslCipherList != NULL)
          handleSSLCipherList(connection, (value)original->sslCipherList);
     if (original->private != NULL)
          handlePrivate(connection, (value)original->private);
     if (original->http200Aliases != NULL)
          handleHTTP200Aliases(connection, original->ocamlHTTP200Aliases);
     if (original->netrcFile != NULL)
          handleNETRCFile(connection, (value)original->netrcFile);

     return connection;
}

static void removeConnection(Connection *connection)
{
     enter_blocking_section();
     curl_easy_cleanup(connection->connection);
     leave_blocking_section();

     if (connectionList.tail == connection)
          connectionList.tail = connectionList.tail->next;
     if (connectionList.head == connection)
          connectionList.head = connectionList.head->prev;

     if (connection->next != NULL)
          connection->next->prev = connection->prev;
     if (connection->prev != NULL)
          connection->prev->next = connection->next;

     remove_global_root(&connection->ocamlWriteCallback);
     remove_global_root(&connection->ocamlReadCallback);
     remove_global_root(&connection->ocamlErrorBuffer);
     remove_global_root(&connection->ocamlHTTPHeader);
     remove_global_root(&connection->ocamlHTTPPost);
     remove_global_root(&connection->ocamlQuote);
     remove_global_root(&connection->ocamlPostQuote);
     remove_global_root(&connection->ocamlHeaderCallback);
     remove_global_root(&connection->ocamlProgressCallback);
     remove_global_root(&connection->ocamlPasswdCallback);
     remove_global_root(&connection->ocamlDebugCallback);
     remove_global_root(&connection->ocamlHTTP200Aliases);
     remove_global_root(&connection->ocamlHTTPAuth);
     remove_global_root(&connection->ocamlProxyAuth);

     if (connection->url != NULL)
          free(connection->url);
     if (connection->proxy != NULL)
          free(connection->proxy);
     if (connection->userPwd != NULL)
          free(connection->userPwd);
     if (connection->proxyUserPwd != NULL)
          free(connection->proxyUserPwd);
     if (connection->range != NULL)
          free(connection->range);
     if (connection->errorBuffer != NULL)
          free(connection->range);
     if (connection->postFields != NULL)
          free(connection->postFields);
     if (connection->referer != NULL)
          free(connection->referer);
     if (connection->userAgent != NULL)
          free(connection->userAgent);
     if (connection->ftpPort != NULL)
          free(connection->ftpPort);
     if (connection->cookie != NULL)
          free(connection->cookie);
     if (connection->httpHeader != NULL)
          free_curl_slist(connection->httpHeader);
     if (connection->httpPostFirst != NULL)
          curl_formfree(connection->httpPostFirst);
     if (connection->httpPostStrings != NULL)
          free_curl_slist(connection->httpPostStrings);
     if (connection->sslCert != NULL)
          free(connection->sslCert);
     if (connection->sslCertType != NULL)
          free(connection->sslCertType);
     if (connection->sslCertPasswd != NULL)
          free(connection->sslCertPasswd);
     if (connection->sslKey != NULL)
          free(connection->sslKey);
     if (connection->sslKeyType != NULL)
          free(connection->sslKeyType);
     if (connection->sslKeyPasswd != NULL)
          free(connection->sslKeyPasswd);
     if (connection->sslEngine != NULL)
          free(connection->sslEngine);
     if (connection->quote != NULL)
          free_curl_slist(connection->quote);
     if (connection->postQuote != NULL)
          free_curl_slist(connection->postQuote);
     if (connection->cookieFile != NULL)
          free(connection->cookieFile);
     if (connection->customRequest != NULL)
          free(connection->customRequest);
     if (connection->caInfo != NULL)
          free(connection->caInfo);
     if (connection->caPath != NULL)
          free(connection->caPath);
     if (connection->randomFile != NULL)
          free(connection->randomFile);
     if (connection->egdSocket != NULL)
          free(connection->egdSocket);
     if (connection->cookieJar != NULL)
          free(connection->cookieJar);
     if (connection->sslCipherList != NULL)
          free(connection->sslCipherList);
     if (connection->private != NULL)
          free(connection->private);
     if (connection->http200Aliases != NULL)
          free_curl_slist(connection->http200Aliases);
     if (connection->netrcFile != NULL)
          free(connection->netrcFile);

     free(connection);
}

static void checkConnection(Connection *connection)
{
     Connection *listIter;

     listIter = connectionList.tail;

     while (listIter != NULL)
     {
          if (listIter == connection)
               return;

          listIter = listIter->next;
     }

     failwith("Invalid Connection");
}

static size_t writeFunction(char *ptr, size_t size, size_t nmemb, void *data)
{
     value str;
     Connection *conn = (Connection *)data;
     int i;

     leave_blocking_section();

     checkConnection(conn);

     str = alloc_string(size*nmemb);

     for (i = 0; i < size*nmemb; i++)
          Byte(str, i) = ptr[i];

     callback(conn->ocamlWriteCallback, str);

     enter_blocking_section();

     return nmemb;
}

static size_t readFunction(void *ptr, size_t size, size_t nmemb, void *data)
{
     value result;
     Connection *conn = (Connection *)data;
     int length;

     leave_blocking_section();

     checkConnection(conn);

     result = callback(conn->ocamlReadCallback, Val_int(size*nmemb));

     length = string_length(result);

     if (length >= size*nmemb)
          length = size*nmemb;

     memcpy(ptr, String_val(result), length);

     enter_blocking_section();

     return length;
}

static size_t headerFunction(char *ptr, size_t size, size_t nmemb, void *data)
{
     value str;
     Connection *conn = (Connection *)data;
     int i;

     leave_blocking_section();

     checkConnection(conn);

     str = alloc_string(size*nmemb);

     for (i = 0; i < size*nmemb; i++)
          Byte(str, i) = ptr[i];

     callback(conn->ocamlHeaderCallback, str);

     enter_blocking_section();

     return nmemb;
}

static int progressFunction(void *data,
                            double dlTotal,
                            double dlNow,
                            double ulTotal,
                            double ulNow)
{
     value callbackData[4];
     Connection *conn = (Connection *)data;

     leave_blocking_section();

     checkConnection(conn);

     callbackData[0] = copy_double(dlTotal);
     callbackData[1] = copy_double(dlNow);
     callbackData[2] = copy_double(ulTotal);
     callbackData[3] = copy_double(ulNow);

     callbackN(conn->ocamlProgressCallback, 4, callbackData);

     enter_blocking_section();

     return 0;
}

static int passwdFunction(void *data,
                          char *prompt,
                          char *buffer,
                          int bufferLength)
{
     value ocamlPasswd;
     value ocamlPrompt;
     int length;
     Connection *conn = (Connection *)data;

     leave_blocking_section();

     checkConnection(conn);

     ocamlPrompt = copy_string(prompt);

     ocamlPasswd = callback2(conn->ocamlPasswdCallback,
                             ocamlPrompt,
                             Val_long(bufferLength));

     if (Wosize_val(ocamlPasswd) != 2)
     {
          enter_blocking_section();

          return 1;
     }

     length = string_length(Field(ocamlPasswd, 1));
     if (length > bufferLength - 1)
          length = bufferLength - 1;

     memcpy(buffer, String_val(Field(ocamlPasswd, 0)), length);

     buffer[length] = 0;

     enter_blocking_section();

     return !(Bool_val(Field(ocamlPasswd, 0)));
}

static int debugFunction(CURL *debugConnection,
                         curl_infotype infoType,
                         char *buffer,
                         size_t bufferLength,
                         void *data)
{
     value camlDebugConnection;
     value camlInfoType;
     value camlMessage;
     int i;
     Connection *conn = (Connection *)data;

     leave_blocking_section();

     checkConnection(conn);

     camlDebugConnection = (value)conn;
     camlInfoType = Val_long(infoType);
     camlMessage = alloc_string(bufferLength);

     for (i = 0; i < bufferLength; i++)
          Byte(camlMessage, i) = buffer[i];

     callback3(conn->ocamlDebugCallback,
               camlDebugConnection,
               camlInfoType,
               camlMessage);

     enter_blocking_section();

     return 0;
}

/**
 **  curl_global_init helper function
 **/

CAMLprim value helper_curl_global_init(value initOption)
{
     CAMLparam1(initOption);

     switch (Long_val(initOption))
     {
     case 0: /* CURLINIT_GLOBALALL */
          CAMLreturn(Val_long(curl_global_init(CURL_GLOBAL_ALL)));
          break;

     case 1: /* CURLINIT_GLOBALSSL */
          CAMLreturn(Val_long(curl_global_init(CURL_GLOBAL_SSL)));
          break;

     case 2: /* CURLINIT_GLOBALWIN32 */
          CAMLreturn(Val_long(curl_global_init(CURL_GLOBAL_WIN32)));
          break;

     case 3: /* CURLINIT_GLOBALNOTHING */
          CAMLreturn(Val_long(curl_global_init(CURL_GLOBAL_NOTHING)));
          break;

     default:
          failwith("Invalid Initialization Option");
          break;
     }
}

/**
 **  curl_global_cleanup helper function
 **/

CAMLprim void helper_curl_global_cleanup(void)
{
     CAMLparam0();

     curl_global_cleanup();

     CAMLreturn0;
}

/**
 ** curl_easy_init helper function
 **/

CAMLprim value helper_curl_easy_init(void)
{
     CAMLparam0();
     CAMLlocal1(result);

     Connection *conn = newConnection();

     result = alloc(1, Abstract_tag);
     Field(result, 0) = (value)conn;

     CAMLreturn(result);
}

/**
 **  curl_easy_setopt helper utility functions
 **/

static void handleWriteFunction(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (Tag_val(option) == Closure_tag)
          conn->ocamlWriteCallback = option;
     else
          failwith("Not a proper closure");

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_WRITEFUNCTION,
                               writeFunction);

     if (result != CURLE_OK)
          raiseError(conn, result);

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_WRITEDATA,
                               conn);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleReadFunction(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (Tag_val(option) == Closure_tag)
          conn->ocamlReadCallback = option;
     else
          failwith("Not a proper closure");

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_READFUNCTION,
                               readFunction);

     if (result != CURLE_OK)
          raiseError(conn, result);

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_READDATA,
                               conn);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleURL(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->url != NULL)
          free(conn->url);

     conn->url = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_URL,
                               conn->url);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleInFileSize(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_INFILESIZE,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleProxy(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->proxy != NULL)
          free(conn->proxy);

     conn->proxy = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PROXY,
                               conn->proxy);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleProxyPort(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PROXYPORT,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleHTTPProxyTunnel(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_HTTPPROXYTUNNEL,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleVerbose(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_VERBOSE,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleHeader(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_HEADER,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleNoProgress(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_NOPROGRESS,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleNoSignal(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_NOSIGNAL
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_NOSIGNAL,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_NOSIGNAL");
#endif
}

static void handleNoBody(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_NOBODY,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleFailOnError(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FAILONERROR,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleUpload(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_UPLOAD,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handlePost(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_POST,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleFTPListOnly(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FTPLISTONLY,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleFTPAppend(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FTPAPPEND,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleNETRC(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;
     long netrc;

     switch (Tag_val(option))
     {
     case 0: /* CURL_NETRC_OPTIONAL */
          netrc = CURL_NETRC_OPTIONAL;
          break;

     case 1:/* CURL_NETRC_IGNORED */
          netrc = CURL_NETRC_IGNORED;
          break;

     case 2: /* CURL_NETRC_REQUIRED */
          netrc = CURL_NETRC_REQUIRED;
          break;

     default:
          failwith("Invalid NETRC Option");
          break;
     }

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_NETRC,
                               netrc);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleEncoding(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_ENCODING
     CURLcode result = CURLE_OK;

     switch (Tag_val(option))
     {
     case 0: /* CURL_ENCODING_NONE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_ENCODING,
                                    "identity");
          break;

     case 1: /* CURL_ENCODING_DEFLATE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_ENCODING,
                                    "deflate");
          break;

     default:
          failwith("Invalid Encoding Option");
          break;
     }

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_ENCODING");
#endif
}

static void handleFollowLocation(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FOLLOWLOCATION,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleTransferText(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_TRANSFERTEXT,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handlePut(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PUT,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleUserPwd(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->userPwd != NULL)
          free(conn->userPwd);

     conn->userPwd = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_USERPWD,
                               conn->userPwd);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleProxyUserPwd(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->proxyUserPwd != NULL)
          free(conn->proxyUserPwd);

     conn->proxyUserPwd = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PROXYUSERPWD,
                               conn->proxyUserPwd);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleRange(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->range != NULL)
          free(conn->range);

     conn->range = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_RANGE,
                               conn->range);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleErrorBuffer(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->errorBuffer != NULL)
          free(conn->errorBuffer);

     conn->errorBuffer = malloc(sizeof(char) * CURL_ERROR_SIZE);
     conn->ocamlErrorBuffer = option;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_ERRORBUFFER,
                               conn->errorBuffer);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleTimeout(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_TIMEOUT,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handlePostFields(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     conn->ocamlPostFields = option;

     if (conn->postFields != NULL)
          free(conn->postFields);

     conn->postFields = malloc(string_length(option)+1);
     memcpy(conn->postFields, String_val(option), string_length(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_POSTFIELDS,
                               conn->postFields);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handlePostFieldSize(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_POSTFIELDSIZE,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleReferer(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->referer != NULL)
          free(conn->referer);

     conn->referer = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_REFERER,
                               conn->referer);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleUserAgent(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->userAgent != NULL)
          free(conn->userAgent);

     conn->userAgent = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_USERAGENT,
                               conn->userAgent);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleFTPPort(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->ftpPort != NULL)
          free(conn->ftpPort);

     conn->ftpPort = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FTPPORT,
                               conn->ftpPort);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleLowSpeedLimit(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_LOW_SPEED_LIMIT,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleLowSpeedTime(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_LOW_SPEED_TIME,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleResumeFrom(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_RESUME_FROM,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleCookie(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->cookie != NULL)
          free(conn->cookie);

     conn->cookie = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_COOKIE,
                               conn->cookie);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleHTTPHeader(Connection *conn, value option)
{
     value listIter;
     CURLcode result = CURLE_OK;
     char *str;

     conn->ocamlHTTPHeader = option;

     if (conn->httpHeader != NULL)
          free_curl_slist(conn->httpHeader);

     conn->httpHeader = NULL;

     listIter = option;;

     while (!Is_long(listIter))
     {
          if (Tag_val(Field(listIter, 0)) != String_tag)
               failwith("Not a string");

          str = strdup(String_val(Field(listIter, 0)));

          conn->httpHeader = curl_slist_append(conn->httpHeader, str);

          listIter = Field(listIter, 1);
     }

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_HTTPHEADER,
                               conn->httpHeader);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleHTTPPost(Connection *conn, value option)
{
     value listIter;
     value formItem;
     value contentType;
     CURLcode result = CURLE_OK;
     char *str1, *str2, *str3, *str4;

     listIter = option;

     conn->ocamlHTTPPost = option;

     if (conn->httpPostFirst != NULL)
          curl_formfree(conn->httpPostFirst);

     conn->httpPostFirst = NULL;
     conn->httpPostLast = NULL;

     if (conn->httpPostStrings != NULL)
          free_curl_slist(conn->httpPostStrings);

     while (!Is_long(listIter))
     {
          formItem = Field(listIter, 0);

          switch (Tag_val(formItem))
          {
          case 0: /* CURLFORM_CONTENT */
               if (Wosize_val(formItem) < 3)
               {
                    failwith("Incorrect CURLFORM_CONTENT parameters");
               }

               if (Is_long(Field(formItem, 2)) &&
                   Long_val(Field(formItem, 2)) == 0)
               {
                    str1 = (char *)malloc(string_length(Field(formItem, 0))+1);
                    memcpy(str1,
                           String_val(Field(formItem, 0)),
                           string_length(Field(formItem, 0)));
                    str1[string_length(Field(formItem, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str1);

                    str2 = (char *)malloc(string_length(Field(formItem, 1))+1);
                    memcpy(str2,
                           String_val(Field(formItem, 1)),
                           string_length(Field(formItem, 1)));
                    str2[string_length(Field(formItem, 1))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str2);

                    curl_formadd(&conn->httpPostFirst,
                                 &conn->httpPostLast,
                                 CURLFORM_PTRNAME,
                                 str1,
                                 CURLFORM_NAMELENGTH,
                                 string_length(Field(formItem, 0)),
                                 CURLFORM_PTRCONTENTS,
                                 str2,
                                 CURLFORM_CONTENTSLENGTH,
                                 string_length(Field(formItem, 1)),
                                 CURLFORM_END);
               }
               else if (Is_block(Field(formItem, 2)))
               {
                    str1 = (char *)malloc(string_length(Field(formItem, 0))+1);
                    memcpy(str1,
                           String_val(Field(formItem, 0)),
                           string_length(Field(formItem, 0)));
                    str1[string_length(Field(formItem, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str1);

                    str2 = (char *)malloc(string_length(Field(formItem, 1))+1);
                    memcpy(str2,
                           String_val(Field(formItem, 1)),
                           string_length(Field(formItem, 1)));
                    str2[string_length(Field(formItem, 1))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str2);

                    contentType = Field(formItem, 2);

                    str3 = (char *)malloc(string_length(Field(contentType, 0))+1);
                    memcpy(str3,
                           String_val(Field(contentType, 0)),
                           string_length(Field(contentType, 0)));
                    str3[string_length(Field(contentType, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str3);

                    curl_formadd(&conn->httpPostFirst,
                                 &conn->httpPostLast,
                                 CURLFORM_PTRNAME,
                                 str1,
                                 CURLFORM_NAMELENGTH,
                                 string_length(Field(formItem, 0)),
                                 CURLFORM_PTRCONTENTS,
                                 str2,
                                 CURLFORM_CONTENTSLENGTH,
                                 string_length(Field(formItem, 1)),
                                 CURLFORM_CONTENTTYPE,
                                 str3,
                                 CURLFORM_END);
               }
               else
               {
                    failwith("Incorrect CURLFORM_CONTENT parameters");
               }
               break;

          case 1: /* CURLFORM_FILECONTENT */
               if (Wosize_val(formItem) < 3)
               {
                    failwith("Incorrect CURLFORM_FILECONTENT parameters");
               }

               if (Is_long(Field(formItem, 2)) &&
                   Long_val(Field(formItem, 2)) == 0)
               {
                    str1 = (char *)malloc(string_length(Field(formItem, 0))+1);
                    memcpy(str1,
                           String_val(Field(formItem, 0)),
                           string_length(Field(formItem, 0)));
                    str1[string_length(Field(formItem, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str1);

                    str2 = (char *)malloc(string_length(Field(formItem, 1))+1);
                    memcpy(str2,
                           String_val(Field(formItem, 1)),
                           string_length(Field(formItem, 1)));
                    str2[string_length(Field(formItem, 1))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str2);

                    curl_formadd(&conn->httpPostFirst,
                                 &conn->httpPostLast,
                                 CURLFORM_PTRNAME,
                                 str1,
                                 CURLFORM_NAMELENGTH,
                                 string_length(Field(formItem, 0)),
                                 CURLFORM_FILECONTENT,
                                 str2,
                                 CURLFORM_END);
               }
               else if (Is_block(Field(formItem, 2)))
               {
                    str1 = (char *)malloc(string_length(Field(formItem, 0))+1);
                    memcpy(str1,
                           String_val(Field(formItem, 0)),
                           string_length(Field(formItem, 0)));
                    str1[string_length(Field(formItem, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str1);

                    str2 = (char *)malloc(string_length(Field(formItem, 1))+1);
                    memcpy(str2,
                           String_val(Field(formItem, 1)),
                           string_length(Field(formItem, 1)));
                    str2[string_length(Field(formItem, 1))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str2);

                    contentType = Field(formItem, 2);

                    str3 = (char *)malloc(string_length(Field(contentType, 0))+1);
                    memcpy(str3,
                           String_val(Field(contentType, 0)),
                           string_length(Field(contentType, 0)));
                    str3[string_length(Field(contentType, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str3);

                    curl_formadd(&conn->httpPostFirst,
                                 &conn->httpPostLast,
                                 CURLFORM_PTRNAME,
                                 str1,
                                 CURLFORM_NAMELENGTH,
                                 string_length(Field(formItem, 0)),
                                 CURLFORM_FILECONTENT,
                                 str2,
                                 CURLFORM_CONTENTTYPE,
                                 str3,
                                 CURLFORM_END);
               }
               else
               {
                    failwith("Incorrect CURLFORM_FILECONTENT parameters");
               }
               break;

          case 2: /* CURLFORM_FILE */
               if (Wosize_val(formItem) < 3)
               {
                    failwith("Incorrect CURLFORM_FILE parameters");
               }

               if (Is_long(Field(formItem, 2)) &&
                   Long_val(Field(formItem, 2)) == 0)
               {
                    str1 = (char *)malloc(string_length(Field(formItem, 0))+1);
                    memcpy(str1,
                           String_val(Field(formItem, 0)),
                           string_length(Field(formItem, 0)));
                    str1[string_length(Field(formItem, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str1);

                    str2 = (char *)malloc(string_length(Field(formItem, 1))+1);
                    memcpy(str2,
                           String_val(Field(formItem, 1)),
                           string_length(Field(formItem, 1)));
                    str2[string_length(Field(formItem, 1))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str2);

                    curl_formadd(&conn->httpPostFirst,
                                 &conn->httpPostLast,
                                 CURLFORM_PTRNAME,
                                 str1,
                                 CURLFORM_NAMELENGTH,
                                 string_length(Field(formItem, 0)),
                                 CURLFORM_FILE,
                                 str2,
                                 CURLFORM_END);
               }
               else if (Is_block(Field(formItem, 2)))
               {
                    str1 = (char *)malloc(string_length(Field(formItem, 0))+1);
                    memcpy(str1,
                           String_val(Field(formItem, 0)),
                           string_length(Field(formItem, 0)));
                    str1[string_length(Field(formItem, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str1);

                    str2 = (char *)malloc(string_length(Field(formItem, 1))+1);
                    memcpy(str2,
                           String_val(Field(formItem, 1)),
                           string_length(Field(formItem, 1)));
                    str2[string_length(Field(formItem, 1))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str2);

                    contentType = Field(formItem, 2);

                    str3 = (char *)malloc(string_length(Field(contentType, 0))+1);
                    memcpy(str3,
                           String_val(Field(contentType, 0)),
                           string_length(Field(contentType, 0)));
                    str3[string_length(Field(contentType, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str3);

                    curl_formadd(&conn->httpPostFirst,
                                 &conn->httpPostLast,
                                 CURLFORM_PTRNAME,
                                 str1,
                                 CURLFORM_NAMELENGTH,
                                 string_length(Field(formItem, 0)),
                                 CURLFORM_FILE,
                                 str2,
                                 CURLFORM_CONTENTTYPE,
                                 str3,
                                 CURLFORM_END);
               }
               else
               {
                    failwith("Incorrect CURLFORM_FILE parameters");
               }
               break;

          case 3: /* CURLFORM_BUFFER */
               if (Wosize_val(formItem) < 4)
               {
                    failwith("Incorrect CURLFORM_BUFFER parameters");
               }

               if (Is_long(Field(formItem, 3)) &&
                   Long_val(Field(formItem, 3)) == 0)
               {
                    str1 = (char *)malloc(string_length(Field(formItem, 0))+1);
                    memcpy(str1,
                           String_val(Field(formItem, 0)),
                           string_length(Field(formItem, 0)));
                    str1[string_length(Field(formItem, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str1);

                    str2 = (char *)malloc(string_length(Field(formItem, 1))+1);
                    memcpy(str2,
                           String_val(Field(formItem, 1)),
                           string_length(Field(formItem, 1)));
                    str2[string_length(Field(formItem, 1))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str2);

                    str3 = (char *)malloc(string_length(Field(formItem, 2))+1);
                    memcpy(str3,
                           String_val(Field(formItem, 2)),
                           string_length(Field(formItem, 2)));
                    str3[string_length(Field(formItem, 2))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str3);

                    curl_formadd(&conn->httpPostFirst,
                                 &conn->httpPostLast,
                                 CURLFORM_PTRNAME,
                                 str1,
                                 CURLFORM_NAMELENGTH,
                                 string_length(Field(formItem, 0)),
                                 CURLFORM_BUFFER,
                                 str2,
                                 CURLFORM_BUFFERPTR,
                                 str3,
                                 CURLFORM_BUFFERLENGTH,
                                 string_length(Field(formItem, 2)),
                                 CURLFORM_END);
               }
               else if (Is_block(Field(formItem, 3)))
               {
                    str1 = (char *)malloc(string_length(Field(formItem, 0))+1);
                    memcpy(str1,
                           String_val(Field(formItem, 0)),
                           string_length(Field(formItem, 0)));
                    str1[string_length(Field(formItem, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str1);

                    str2 = (char *)malloc(string_length(Field(formItem, 1))+1);
                    memcpy(str2,
                           String_val(Field(formItem, 1)),
                           string_length(Field(formItem, 1)));
                    str2[string_length(Field(formItem, 1))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str2);

                    str3 = (char *)malloc(string_length(Field(formItem, 2))+1);
                    memcpy(str3,
                           String_val(Field(formItem, 2)),
                           string_length(Field(formItem, 2)));
                    str3[string_length(Field(formItem, 2))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str3);

                    contentType = Field(formItem, 3);

                    str4 = (char *)malloc(string_length(Field(contentType, 0))+1);
                    memcpy(str4,
                           String_val(Field(contentType, 0)),
                           string_length(Field(contentType, 0)));
                    str4[string_length(Field(contentType, 0))] = 0;
                    conn->httpPostStrings =
                         curl_slist_append(conn->httpPostStrings, str4);

                    curl_formadd(&conn->httpPostFirst,
                                 &conn->httpPostLast,
                                 CURLFORM_PTRNAME,
                                 str1,
                                 CURLFORM_NAMELENGTH,
                                 string_length(Field(formItem, 0)),
                                 CURLFORM_BUFFER,
                                 str2,
                                 CURLFORM_BUFFERPTR,
                                 str3,
                                 CURLFORM_BUFFERLENGTH,
                                 string_length(Field(formItem, 2)),
                                 CURLFORM_CONTENTTYPE,
                                 str4,
                                 CURLFORM_END);
               }
               else
               {
                    failwith("Incorrect CURLFORM_BUFFER parameters");
               }
               break;
          }

          listIter = Field(listIter, 1);
     }

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_HTTPPOST,
                               conn->httpPostFirst);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLCert(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->sslCert != NULL)
          free(conn->sslCert);

     conn->sslCert = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLCERT,
                               conn->sslCert);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLCertType(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->sslCertType != NULL)
          free(conn->sslCertType);

     conn->sslCertType = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLCERTTYPE,
                               conn->sslCertType);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLCertPasswd(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->sslCertPasswd != NULL)
          free(conn->sslCertPasswd);

     conn->sslCertPasswd = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLCERTPASSWD,
                               conn->sslCertPasswd);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLKey(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->sslKey != NULL)
          free(conn->sslKey);

     conn->sslKey = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLKEY,
                               conn->sslKey);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLKeyType(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->sslKeyType != NULL)
          free(conn->sslKeyType);

     conn->sslKeyType = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLKEYTYPE,
                               conn->sslKeyType);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLKeyPasswd(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->sslKeyPasswd != NULL)
          free(conn->sslKeyPasswd);

     conn->sslKeyPasswd = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLKEYPASSWD,
                               conn->sslKeyPasswd);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLEngine(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->sslEngine != NULL)
          free(conn->sslEngine);

     conn->sslEngine = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLENGINE,
                               conn->sslEngine);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLEngineDefault(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLENGINE_DEFAULT,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleCRLF(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_CRLF,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleQuote(Connection *conn, value option)
{
     value listIter;
     CURLcode result = CURLE_OK;
     char *str;

     conn->ocamlQuote = option;

     if (conn->quote != NULL)
          free_curl_slist(conn->quote);

     conn->quote = NULL;

     listIter = option;;

     while (!Is_long(listIter))
     {
          if (Tag_val(Field(listIter, 0)) != String_tag)
               failwith("Not a string");

          str = strdup(String_val(Field(listIter, 0)));

          conn->quote = curl_slist_append(conn->quote, str);

          listIter = Field(listIter, 1);
     }

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_QUOTE,
                               conn->quote);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handlePostQuote(Connection *conn, value option)
{
     value listIter;
     CURLcode result = CURLE_OK;
     char *str;

     conn->ocamlPostQuote = option;

     if (conn->postQuote != NULL)
          free_curl_slist(conn->postQuote);

     conn->postQuote = NULL;

     listIter = option;;

     while (!Is_long(listIter))
     {
          if (Tag_val(Field(listIter, 0)) != String_tag)
               failwith("Not a string");

          str = strdup(String_val(Field(listIter, 0)));

          conn->postQuote = curl_slist_append(conn->postQuote, str);

          listIter = Field(listIter, 1);
     }

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_POSTQUOTE,
                               conn->postQuote);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleHeaderFunction(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (Tag_val(option) == Closure_tag)
          conn->ocamlHeaderCallback = option;
     else
          failwith("Not a proper closure");

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_HEADERFUNCTION,
                               headerFunction);

     if (result != CURLE_OK)
          raiseError(conn, result);

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_WRITEHEADER,
                               conn);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleCookieFile(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->cookieFile != NULL)
          free(conn->cookieFile);

     conn->cookieFile = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_COOKIEFILE,
                               conn->cookieFile);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLVersion(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSLVERSION,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleTimeCondition(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     switch (Long_val(option))
     {
     case 0: /* TIMECOND_IFMODSINCE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_TIMECONDITION,
                                    CURL_TIMECOND_IFMODSINCE);
          break;

     case 1: /* TIMECOND_IFUNMODSINCE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_TIMECONDITION,
                                    CURL_TIMECOND_IFUNMODSINCE);
          break;

     default:
          failwith("Invalid TIMECOND Option");
          break;
     }

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleTimeValue(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_TIMEVALUE,
                               Int32_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleCustomRequest(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->customRequest != NULL)
          free(conn->customRequest);

     conn->customRequest = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_CUSTOMREQUEST,
                               conn->customRequest);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleInterface(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->interface != NULL)
          free(conn->interface);

     conn->interface = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_INTERFACE,
                               conn->interface);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleKRB4Level(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     switch (Tag_val(option))
     {
     case 0: /* KRB4_NONE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_KRB4LEVEL,
                                    NULL);
          break;

     case 1: /* KRB4_CLEAR */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_KRB4LEVEL,
                                    "clear");
          break;

     case 2: /* KRB4_SAFE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_KRB4LEVEL,
                                    "safe");
          break;

     case 3: /* KRB4_CONFIDENTIAL */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_KRB4LEVEL,
                                    "confidential");
          break;

     case 4: /* KRB4_PRIVATE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_KRB4LEVEL,
                                    "private");
          break;

     default:
          failwith("Invalid KRB4 Option");
          break;
     }

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleProgressFunction(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (Tag_val(option) == Closure_tag)
          conn->ocamlProgressCallback = option;
     else
          failwith("Not a proper closure");

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PROGRESSFUNCTION,
                               progressFunction);
     if (result != CURLE_OK)
          raiseError(conn, result);

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PROGRESSDATA,
                               conn);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLVerifyPeer(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSL_VERIFYPEER,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleCAInfo(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->caInfo != NULL)
          free(conn->caInfo);

     conn->caInfo = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_CAINFO,
                               conn->caInfo);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleCAPath(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->caPath != NULL)
          free(conn->caPath);

     conn->caPath = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_CAPATH,
                               conn->caPath);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handlePasswdFunction(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (Tag_val(option) == Closure_tag)
          conn->ocamlPasswdCallback = option;
     else
          failwith("Not a proper closure");

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PASSWDFUNCTION,
                               passwdFunction);
     if (result != CURLE_OK)
          raiseError(conn, result);

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PASSWDDATA,
                               conn);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleFileTime(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FILETIME,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleMaxRedirs(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_MAXREDIRS,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleMaxConnects(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_MAXCONNECTS,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleClosePolicy(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     switch (Tag_val(option))
     {
     case 0: /* CLOSEPOLICY_OLDEST */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_CLOSEPOLICY,
                                    CURLCLOSEPOLICY_OLDEST);
          break;

     case 1: /* CLOSEPOLICY_LEAST_RECENTLY_USED */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_CLOSEPOLICY,
                                    CURLCLOSEPOLICY_LEAST_RECENTLY_USED);
          break;

     default:
          failwith("Invalid CLOSEPOLICY Option");
          break;
     }

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleFreshConnect(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FRESH_CONNECT,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleForbidReuse(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FORBID_REUSE,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleRandomFile(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->randomFile != NULL)
          free(conn->randomFile);

     conn->randomFile = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_RANDOM_FILE,
                               conn->randomFile);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleEGDSocket(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->egdSocket != NULL)
          free(conn->egdSocket);

     conn->egdSocket = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_EGDSOCKET,
                               conn->egdSocket);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleConnectTimeout(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_CONNECTTIMEOUT,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleHTTPGet(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_HTTPGET,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLVerifyHost(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     switch (Long_val(option))
     {
     case 0: /* SSLVERIFYHOST_EXISTENCE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_SSL_VERIFYHOST,
                                    1);
          break;

     case 1: /* SSLVERIFYHOST_HOSTNAME */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_SSL_VERIFYHOST,
                                    2);
          break;

     default:
          failwith("Invalid SSLVERIFYHOST Option");
          break;
     }

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleCookieJar(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->cookieJar != NULL)
          free(conn->cookieJar);

     conn->cookieJar = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_COOKIEJAR,
                               conn->cookieJar);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleSSLCipherList(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (conn->sslCipherList != NULL)
          free(conn->sslCipherList);

     conn->sslCipherList = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_SSL_CIPHER_LIST,
                               conn->sslCipherList);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleHTTPVersion(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     switch (Tag_val(option))
     {
     case 0: /* HTTP_VERSION_NONE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_HTTP_VERSION,
                                    CURL_HTTP_VERSION_NONE);
          break;

     case 1: /* HTTP_VERSION_1_0 */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_HTTP_VERSION,
                                    CURL_HTTP_VERSION_1_0);
          break;

     case 2: /* HTTP_VERSION_1_1 */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_HTTP_VERSION,
                                    CURL_HTTP_VERSION_1_1);
          break;

     default:
          failwith("Invalid HTTP_VERSION Option");
          break;
     }

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleFTPUseEPSV(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FTP_USE_EPSV,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleDNSCacheTimeout(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_DNS_CACHE_TIMEOUT,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleDNSUseGlobalCache(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_DNS_USE_GLOBAL_CACHE,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handleDebugFunction(Connection *conn, value option)
{
     CURLcode result = CURLE_OK;

     if (Tag_val(option) == Closure_tag)
          conn->ocamlDebugCallback = option;
     else
          failwith("Not a proper closure");

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_DEBUGFUNCTION,
                               debugFunction);
     if (result != CURLE_OK)
          raiseError(conn, result);

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_DEBUGDATA,
                               conn);

     if (result != CURLE_OK)
          raiseError(conn, result);
}

static void handlePrivate(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_PRIVATE
     CURLcode result = CURLE_OK;

     if (conn->private != NULL)
          free(conn->private);

     conn->private = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PRIVATE,
                               conn->private);

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_PRIVATE");
#endif
}

static void handleHTTP200Aliases(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_HTTP200ALIASES
     value listIter;
     CURLcode result = CURLE_OK;
     char *str;

     conn->ocamlHTTP200Aliases = option;

     if (conn->http200Aliases != NULL)
          free_curl_slist(conn->http200Aliases);

     conn->http200Aliases = NULL;

     listIter = option;;

     while (!Is_long(listIter))
     {
          if (Tag_val(Field(listIter, 0)) != String_tag)
               failwith("Not a string");

          str = strdup(String_val(Field(listIter, 0)));

          conn->http200Aliases = curl_slist_append(conn->http200Aliases, str);

          listIter = Field(listIter, 1);
     }

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_HTTP200ALIASES,
                               conn->http200Aliases);

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_HTTP200ALIASES");
#endif
}

static void handleUnrestrictedAuth(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_UNRESTRICTED_AUTH
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_UNRESTRICTED_AUTH,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_UNRESTRICTED_AUTH");
#endif
}

static void handleFTPUseEPRT(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_FTP_USE_EPRT
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FTP_USE_EPRT,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_FTP_USE_EPRT");
#endif
}

static void handleHTTPAuth(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_HTTPAUTH
     value listIter;
     CURLcode result = CURLE_OK;
     long auth = CURLAUTH_NONE;

     listIter = option;;

     while (!Is_long(listIter))
     {
          switch (Tag_val(Field(listIter, 0)))
          {
          case 0: /* CURLAUTH_BASIC */
               auth |= CURLAUTH_BASIC;
               break;

          case 1: /* CURLAUTH_DIGEST */
               auth |= CURLAUTH_DIGEST;
               break;

          case 2: /* CURLAUTH_GSSNEGOTIATE */
               auth |= CURLAUTH_GSSNEGOTIATE;
               break;

          case 3: /* CURLAUTH_NTLM */
               auth |= CURLAUTH_NTLM;
               break;

          case 4: /* CURLAUTH_ANY */
               auth |= CURLAUTH_ANY;
               break;

          case 5: /* CURLAUTH_ANYSAFE */
               auth |= CURLAUTH_ANYSAFE;
               break;

          default:
               failwith("Invalid HTTPAUTH Value");
               break;
          }

          listIter = Field(listIter, 1);
     }

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_HTTPAUTH,
                               auth);

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_HTTPAUTH");
#endif
}

static void handleFTPCreateMissingDirs(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_FTP_CREATE_MISSING_DIRS
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FTP_CREATE_MISSING_DIRS,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_FTP_CREATE_MISSING_DIRS");
#endif
}

static void handleProxyAuth(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_PROXYAUTH
     value listIter;
     CURLcode result = CURLE_OK;
     long auth = CURLAUTH_NONE;

     listIter = option;;

     while (!Is_long(listIter))
     {
          switch (Tag_val(Field(listIter, 0)))
          {
          case 0: /* CURLAUTH_BASIC */
               auth |= CURLAUTH_BASIC;
               break;

          case 1: /* CURLAUTH_DIGEST */
               auth |= CURLAUTH_DIGEST;
               break;

          case 2: /* CURLAUTH_GSSNEGOTIATE */
               auth |= CURLAUTH_GSSNEGOTIATE;
               break;

          case 3: /* CURLAUTH_NTLM */
               auth |= CURLAUTH_NTLM;
               break;

          case 4: /* CURLAUTH_ANY */
               auth |= CURLAUTH_ANY;
               break;

          case 5: /* CURLAUTH_ANYSAFE */
               auth |= CURLAUTH_ANYSAFE;
               break;

          default:
               failwith("Invalid HTTPAUTH Value");
               break;
          }

          listIter = Field(listIter, 1);
     }

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_PROXYAUTH,
                               auth);

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_PROXYAUTH");
#endif
}

static void handleFTPResponseTimeout(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_FTP_RESPONSE_TIMEOUT
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_FTP_RESPONSE_TIMEOUT,
                               Long_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_FTP_RESPONSE_TIMEOUT");
#endif
}

static void handleIPResolve(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_IPRESOLVE
     CURLcode result = CURLE_OK;

     switch (Tag_val(option))
     {
     case 0: /* CURL_IPRESOLVE_WHATEVER */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_IPRESOLVE,
                                    CURL_IPRESOLVE_WHATEVER);
          break;

     case 1: /* CURL_IPRESOLVE_V4 */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_IPRESOLVE,
                                    CURL_IPRESOLVE_V4);
          break;

     case 2: /* CURL_IPRESOLVE_V6 */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_IPRESOLVE,
                                    CURL_IPRESOLVE_V6);
          break;

     default:
          failwith("Invalid IPRESOLVE Value");
          break;
     }

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_IPRESOLVE");
#endif
}

static void handleMaxFileSize(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_MAXFILESIZE
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_MAXFILESIZE,
                               Int32_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_MAXFILESIZE");
#endif
}

static void handleInFileSizeLarge(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_INFILSIZE_LARGE
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_INFILESIZE_LARGE,
                               Int64_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_INFILESIZE_LARGE");
#endif
}

static void handleResumeFromLarge(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_RESUME_FROM_LARGEa
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_RESUME_FROM_LARGE,
                               Int64_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_RESUME_FROM_LARGE");
#endif
}

static void handleMaxFileSizeLarge(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_MAXFILESIZE_LARGE
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_MAXFILESIZE_LARGE,
                               Int64_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_MAXFILESIZE_LARGE");
#endif
}

static void handleNETRCFile(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_NETRC_FILE
     CURLcode result = CURLE_OK;

     if (conn->netrcFile != NULL)
          free(conn->netrcFile);

     conn->netrcFile = strdup(String_val(option));

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_NETRC_FILE,
                               conn->netrcFile);

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_NETRC_FILE");
#endif
}

static void handleFTPSSL(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_FTP_SSL
     CURLcode result = CURLE_OK;

     switch (Tag_val(option))
     {
     case 0: /* CURLFTPSSL_NONE */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_FTP_SSL,
                                    CURLFTPSSL_NONE);
          break;

     case 1: /* CURLFTPSSL_TRY */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_FTP_SSL,
                                    CURLFTPSSL_TRY);
          break;

     case 2: /* CURLFTPSSL_CONTROL */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_FTP_SSL,
                                    CURLFTPSSL_CONTROL);
          break;

     case 3: /* CURLFTPSSL_ALL */
          result = curl_easy_setopt(conn->connection,
                                    CURLOPT_FTP_SSL,
                                    CURLFTPSSL_ALL);
          break;

     default:
          failwith("Invalid FTP_SSL Value");
          break;
     }

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_FTP_SSL");
#endif
}

static void handlePostFieldSizeLarge(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_POSTFIELDSIZE_LARGE
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_POSTFIELDSIZE_LARGE,
                               Int64_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_POSTFIELDSIZE_LARGE");
#endif
}

static void handleTCPNoDelay(Connection *conn, value option)
{
#if HAVE_DECL_CURLOPT_TCP_NODELAY
     CURLcode result = CURLE_OK;

     result = curl_easy_setopt(conn->connection,
                               CURLOPT_TCP_NODELAY,
                               Bool_val(option));

     if (result != CURLE_OK)
          raiseError(conn, result);
#else
     failwith("libcurl does not implement CURLOPT_TCP_NODELAY");
#endif
}

/**
 **  curl_easy_setopt helper function
 **/

CAMLprim void helper_curl_easy_setopt(value conn, value option)
{
     CAMLparam2(conn, option);
     CAMLlocal1(data);
     Connection *connection = (Connection *)Field(conn, 0);

     checkConnection(connection);

     if (Is_long(option))
     {
          char error[128];

          sprintf(error, "Unimplemented Option: %s",
                  findOption(unimplementedOptionMap,
                             (CURLoption)(Long_val(option))));

          failwith(error);
     }

     if (!Is_block(option))
          failwith("Not a block");

     if (Wosize_val(option) < 1)
          failwith("Insufficient data in block");

     data = Field(option, 0);

     if (Tag_val(option) < sizeof(implementedOptionMap)/sizeof(CURLOptionMapping))
          (*implementedOptionMap[Tag_val(option)].optionHandler)(connection,
                                                                 data);
     else
          failwith("Invalid CURLOPT Option");

     CAMLreturn0;
}

/**
 **  curl_easy_perform helper function
 **/

CAMLprim void helper_curl_easy_perform(value conn)
{
     CAMLparam1(conn);
     CURLcode result = CURLE_OK;
     Connection *connection = (Connection *)Field(conn, 0);

     checkConnection(connection);

     enter_blocking_section();
     result = curl_easy_perform(connection->connection);
     leave_blocking_section();

     if (result != CURLE_OK)
          raiseError(connection, result);

     CAMLreturn0;
}

/**
 **  curl_easy_cleanup helper function
 **/

CAMLprim void helper_curl_easy_cleanup(value conn)
{
     CAMLparam1(conn);
     Connection *connection = (Connection *)Field(conn, 0);

     checkConnection(connection);

     removeConnection(connection);

     CAMLreturn0;
}

/**
 **  curl_easy_duphandle helper function
 **/

CAMLprim value helper_curl_easy_duphandle(value conn)
{
     CAMLparam1(conn);
     CAMLlocal1(result);
     Connection *connection = (Connection *)Field(conn, 0);

     checkConnection(connection);

     result = alloc(1, Abstract_tag);
     Field(result, 0) = (value)duplicateConnection(connection);

     CAMLreturn(result);
}

/**
 **  curl_easy_getinfo helper function
 **/

enum GetInfoResultType { StringValue, LongValue, DoubleValue };

CAMLprim value helper_curl_easy_getinfo(value conn, value option)
{
     CAMLparam2(conn, option);
     CAMLlocal1(result);
     CURLcode curlResult;
     Connection *connection = (Connection *)Field(conn, 0);
     enum GetInfoResultType resultType;
     char *strValue;
     double doubleValue;
     long longValue;

     checkConnection(connection);

     switch(Long_val(option))
     {
#if HAVE_DECL_CURLINFO_EFFECTIVE_URL
     case 0: /* CURLINFO_EFFECTIVE_URL */
          resultType = StringValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_EFFECTIVE_URL,
                                         &strValue);
          break;
#else
#endif

#if HAVE_DECL_CURLINFO_RESPONSE_CODE || HAVE_DECL_CURLINFO_HTTP_CODE
     case 1: /* CURLINFO_HTTP_CODE */
     case 2: /* CURLINFO_RESPONSE_CODE */
#if HAVE_DECL_CURLINFO_RESPONSE_CODE
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_RESPONSE_CODE,
                                         &longValue);
#else
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_HTTP_CODE,
                                         &longValue);
#endif
          break;
#endif

#if HAVE_DECL_CURLINFO_TOTAL_TIME
     case 3: /* CURLINFO_TOTAL_TIME */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_TOTAL_TIME,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_NAMELOOKUP_TIME
     case 4: /* CURLINFO_NAMELOOKUP_TIME */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_NAMELOOKUP_TIME,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_CONNECT_TIME
     case 5: /* CURLINFO_CONNECT_TIME */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_CONNECT_TIME,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_PRETRANSFER_TIME
     case 6: /* CURLINFO_PRETRANSFER_TIME */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_PRETRANSFER_TIME,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_SIZE_UPLOAD
     case 7: /* CURLINFO_SIZE_UPLOAD */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_SIZE_UPLOAD,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_SIZE_DOWNLOAD
     case 8: /* CURLINFO_SIZE_DOWNLOAD */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_SIZE_DOWNLOAD,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_SPEED_DOWNLOAD
     case 9: /* CURLINFO_SPEED_DOWNLOAD */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_SPEED_DOWNLOAD,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_SPEED_UPLOAD
     case 10: /* CURLINFO_SPEED_UPLOAD */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_SPEED_UPLOAD,
                                         &doubleValue);
          break;

#endif

#if HAVE_DECL_CURLINFO_HEADER_SIZE
     case 11: /* CURLINFO_HEADER_SIZE */
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_HEADER_SIZE,
                                         &longValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_REQUEST_SIZE
     case 12: /* CURLINFO_REQUEST_SIZE */
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_REQUEST_SIZE,
                                         &longValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_SSL_VERIFYRESULT
     case 13: /* CURLINFO_SSL_VERIFYRESULT */
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_SSL_VERIFYRESULT,
                                         &longValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_FILETIME
     case 14: /* CURLINFO_FILETIME */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_FILETIME,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_CONTENT_LENGTH_DOWNLOAD
     case 15: /* CURLINFO_CONTENT_LENGTH_DOWNLOAD */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_CONTENT_LENGTH_DOWNLOAD,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_CONTENT_LENGTH_UPLOAD
     case 16: /* CURLINFO_CONTENT_LENGTH_UPLOAD */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_CONTENT_LENGTH_UPLOAD,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_STARTTRANSFER_TIME
     case 17: /* CURLINFO_STARTTRANSFER_TIME */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_STARTTRANSFER_TIME,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_CONTENT_TYPE
     case 18: /* CURLINFO_CONTENT_TYPE */
          resultType = StringValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_CONTENT_TYPE,
                                         &strValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_REDIRECT_TIME
     case 19: /* CURLINFO_REDIRECT_TIME */
          resultType = DoubleValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_REDIRECT_TIME,
                                         &doubleValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_REDIRECT_COUNT
     case 20: /* CURLINFO_REDIRECT_COUNT */
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_REDIRECT_COUNT,
                                         &longValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_PRIVATE
     case 21: /* CURLINFO_PRIVATE */
          resultType = StringValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_PRIVATE,
                                         &strValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_HTTP_CONNECT_CODE
     case 22: /* CURLINFO_HTTP_CONNECT_CODE */
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_HTTP_CONNECT_CODE,
                                         &longValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_HTTPAUTH_AVAIL
     case 23: /* CURLINFO_HTTPAUTH_AVAIL */
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_HTTPAUTH_AVAIL,
                                         &longValue);
          break;
#endif

#if HAVE_DECL_CURLINFO_PROXYAUTH_AVAIL
     case 24: /* CURLINFO_PROXYAUTH_AVAIL */
          resultType = LongValue;

          curlResult = curl_easy_getinfo(connection->connection,
                                         CURLINFO_PROXYAUTH_AVAIL,
                                         &longValue);
          break;
#endif

     default:
          failwith("Invalid CURLINFO Option");
          break;
     }

     if (curlResult != CURLE_OK)
          raiseError(connection, curlResult);

     switch (resultType)
     {
     case StringValue:
          result = alloc(1, StringValue);
          Field(result, 0) = copy_string(strValue);
          break;

     case LongValue:
          result = alloc(1, LongValue);
          Field(result, 0) = Val_long(longValue);
          break;

     case DoubleValue:
          result = alloc(1, DoubleValue);
          Field(result, 0) = copy_double(doubleValue);
          break;
     }

     CAMLreturn(result);
}

/**
 **  curl_escape helper function
 **/

CAMLprim value helper_curl_escape(value str)
{
     CAMLparam1(str);
     value result;
     char *curlResult;
     
     curlResult = curl_escape(String_val(str), string_length(str));
     result = copy_string(curlResult);
     free(curlResult);

     CAMLreturn(result);
}

/**
 **  curl_unescape helper function
 **/

CAMLprim value helper_curl_unescape(value str)
{
     CAMLparam1(str);
     CAMLlocal1(result);
     char *curlResult;
     
     curlResult = curl_unescape(String_val(str), string_length(str));
     result = copy_string(curlResult);
     free(curlResult);

     CAMLreturn(result);
}

/**
 **  curl_getdate helper function
 **/

CAMLprim value helper_curl_getdate(value str, value now)
{
     CAMLparam2(str, now);
     CAMLlocal1(result);
     time_t curlResult;
     time_t curlNow;

     curlNow = (time_t)Double_val(now);
     curlResult = curl_getdate(String_val(str), &curlNow);
     result = copy_double((double)curlResult);

     CAMLreturn(result);
}

/**
 **  curl_version helper function
 **/

CAMLprim value helper_curl_version(void)
{
     CAMLparam0();
     CAMLlocal1(result);
     char *str;

     str = curl_version();
     result = copy_string(str);

     CAMLreturn(result);
}
