#include "fileio.h"

//////////////////////////////////////////////////////////////

struct AsciiFormat : public FileFormat {
  STD_string description() const {return "ASCII";}
  svector suffix() const  {
    svector result; result.resize(1);
    result[0]="asc";
    return result;
  }
  svector dialects() const {return svector();}
  int read(Data<float,4>& data, const STD_string& filename, const FileReadOpts& opts, Protocol& prot) {
    STD_string ascfile;
    ::load(ascfile,filename);
    unsigned int nelements=tokens(ascfile).size();
    data.resize(1,nelements,1,1);
    if(data.read_asc_file(filename)<0) return -1;
    return nelements;
  }
  int write(const Data<float,4>& data, const STD_string& filename, const FileWriteOpts&, const Protocol& prot) {
    return data.write_asc_file(filename);
  }
};

//////////////////////////////////////////////////////////////
struct MatlabAsciiFormat : public FileFormat {
  STD_string description() const {return "Matlab ascii 2D data matrix";}
  svector suffix() const  {
    svector result; result.resize(1);
    result[0]="dat";
    return result;
  }
  svector dialects() const {return svector();}

  int read(Data<float,4>& data, const STD_string& filename, const FileReadOpts& opts, Protocol& prot) {
    STD_string str;
    if(::load(str, filename)<0) return -1;
    sarray table(parse_table(str));
    int nrows=table.size(0);
    int ncols=table.size(1);
    data.resize(1,1,nrows,ncols);
    for(int irow=0; irow<nrows; irow++) {
      for(int icol=0; icol<ncols; icol++) {
        data(0,0,irow,icol)=atof(table(irow,icol).c_str());
      }
    }
    return 1;
  }

  int write(const Data<float,4>& data, const STD_string& filename, const FileWriteOpts&, const Protocol& prot) {
    int nrows=data.extent(2);
    int ncols=data.extent(3);
    sarray table(nrows,ncols);
    for(int irow=0; irow<nrows; irow++) {
      for(int icol=0; icol<ncols; icol++) {
        table(irow,icol)=ftos(data(0,0,irow,icol));
      }
    }
    if(::write(print_table(table), filename)<0) return -1;
    return 1;
  }

};

//////////////////////////////////////////////////////////////

struct PosFormat : public FileFormat {
  STD_string description() const {return "x-y positions of non-zeroes in ASCII";}
  svector suffix() const  {
    svector result; result.resize(1);
    result[0]="pos";
    return result;
  }
  svector dialects() const {return svector();}

  int read(Data<float,4>& data, const STD_string& filename, const FileReadOpts& opts, Protocol& prot) {
   Log<FileIO> odinlog("PosFormat","read");
   ODINLOG(odinlog,errorLog) << "Implement me" << STD_endl;
    return -1;
  }
  int write(const Data<float,4>& data, const STD_string& filename, const FileWriteOpts&, const Protocol& prot) {
    int nx=data.extent(3);
    int ny=data.extent(2);
    STD_string str;
    for(int i=0; i<data.numElements(); i++) {
      TinyVector<int,4> index=data.create_index(i);
      if(data(index)>0.0) {
        int ix=index(3);
        int iy=index(2);
        str+=ftos(float(ix)/float(nx)-0.5)+" "+ftos(float(iy)/float(ny)-0.5)+"\n";
      }
    }
    return ::write(str,filename);
  }
};

//////////////////////////////////////////////////////////////

void register_asc_format() {
  static AsciiFormat af;
  static PosFormat pf;
  static MatlabAsciiFormat mf;
  af.register_format();
  pf.register_format();
  mf.register_format();
}

