/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "AudioInputNode.h"
#include "InputPort.h"
#include "OSCSender.h"
#include "OutputPort.h"
#include "Plugin.h"
#include "Patch.h"
#include "Connection.h"
#include "Om.h"
#include "JackDriver.h"
#include <cassert>

namespace Om {


AudioInputNode::AudioInputNode(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size)
: InputNode(path, poly, parent, srate, buffer_size),
  m_jack_port(NULL)
{
	OutputPort* oport = new OutputPort(this, "in", 0, m_poly, new PortInfo("in", AUDIO, OUTPUT), m_buffer_size);
	m_external_port = new InputPort(parent, m_name, 0, m_poly, new PortInfo(m_name, AUDIO, INPUT), m_buffer_size);
	m_external_port->join(oport);

	m_num_ports = 1;
	m_ports.alloc(m_num_ports);
	m_ports.at(0) = oport;
	
	m_plugin.type(Plugin::Internal);
	m_plugin.lib_path("");
	m_plugin.lib_name("");
	m_plugin.plug_label("audio_input");
	m_plugin.name("Om patch audio input node");

	m_listnode = new ListNode<InputNode*>(this);

	m_ports.at(0)->set_value(0, 0);
}


AudioInputNode::~AudioInputNode()
{
	delete m_external_port;
	delete m_listnode;
	delete m_jack_port;
}


void
AudioInputNode::activate()
{
	assert(parent_patch() != NULL);
	
	InternalNode::activate();
	
	if (parent_patch()->parent() == NULL && om != NULL)
		m_jack_port = om->jack_driver()->create_input(parent_patch(), this);
}


void
AudioInputNode::deactivate()
{
	if (m_is_added)
		remove_from_patch();
	
	InternalNode::deactivate();
	
	if (parent_patch() != NULL && parent_patch()->parent() == NULL && om != NULL && m_jack_port != NULL)
		om->jack_driver()->unregister_port(m_jack_port->elem());
}

	
void
AudioInputNode::add_to_patch()
{
	assert(parent_patch() != NULL);
	
	parent_patch()->input_nodes().push_back(m_listnode);
	InternalNode::add_to_patch();
	
	// Activate jack port now in the audio thread (not before when created, to avoid race issues)
	if (parent_patch()->parent() == NULL) {
		assert(m_jack_port != NULL);
		om->jack_driver()->add_input(m_jack_port);
	}
}


void
AudioInputNode::remove_from_patch()
{
	assert(parent_patch() != NULL);
	
	if (m_is_added) {
		if (parent_patch()->parent() == NULL) {
			assert(m_jack_port != NULL);
			m_jack_port = om->jack_driver()->remove_input(m_jack_port->elem());
		}
		ListNode<InputNode*>* ln = parent_patch()->input_nodes().remove(this);
		om->maid()->push(ln);
		m_listnode = NULL;
		InternalNode::remove_from_patch();
	}
}


void
AudioInputNode::send_creation_messages(lo_address addr) const
{
	om->osc_sender()->send_new_port_to(addr, m_external_port);
	InternalNode::send_creation_messages(addr);
}


void
AudioInputNode::send_deletion_messages(lo_address addr) const
{
	InternalNode::send_deletion_messages(addr);
	om->osc_sender()->send_port_removal_to(addr, m_external_port);
}


void
AudioInputNode::set_name(const string& new_name)
{
	InputNode::set_name(new_name);

	if (m_jack_port != NULL)
		jack_port_set_name(m_jack_port->elem()->jack_port(), path().c_str());
}

} // namespace Om

