/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "NodeFactory.h"
#include "config.h"
#include <cstdlib>
#include <pthread.h>
#include <dirent.h>
#include <float.h>
#include <cmath>
#include <dlfcn.h>
#include "JackDriver.h"
#include "MidiNoteNode.h"
#include "MidiTriggerNode.h"
#include "MidiControlNode.h"
#include "AudioInputNode.h"
#include "ControlInputNode.h"
#include "AudioOutputNode.h"
#include "ControlOutputNode.h"
#include "TransportNode.h"
#include "PluginLibrary.h"
#include "Plugin.h"
#include "Patch.h"
#include "Om.h"
#include "OmApp.h"
#ifdef HAVE_LADSPA
#include "LADSPAPlugin.h"
#endif
#ifdef HAVE_DSSI
#include "DSSIPlugin.h"
#endif

using std::string;
using std::cerr; using std::cout; using std::endl;


namespace Om {


/* I am perfectly aware that the vast majority of this class is a 
 * vomit inducing nightmare at the moment ;)
 */



NodeFactory::NodeFactory()
: m_has_loaded(false)
{
	pthread_mutex_init(&m_plugin_list_mutex, NULL);
	
	// Add builtin plugin types to m_internal_plugins list
	// FIXME: ewwww, definitely a better way to do this!
	//Plugin* pi = NULL;

	Patch* parent = new Patch("dummy", 1, NULL, 1, 1, 1);

	Node* n = NULL;
	n = new AudioInputNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(new Plugin(n->plugin()));
	delete n;
	n = new ControlInputNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(new Plugin(n->plugin()));
	delete n;
	n = new AudioOutputNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(new Plugin(n->plugin()));
	delete n;
	n = new ControlOutputNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(new Plugin(n->plugin()));
	delete n;
	n = new MidiNoteNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(new Plugin(n->plugin()));
	delete n;
	n = new MidiTriggerNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(new Plugin(n->plugin()));
	delete n;
	n = new MidiControlNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(new Plugin(n->plugin()));
	delete n;
	n = new TransportNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(new Plugin(n->plugin()));
	delete n;

	delete parent;
}


NodeFactory::~NodeFactory()
{
	for (list<Plugin*>::iterator i = m_plugins.begin(); i != m_plugins.end(); ++i)
		delete (*i);
	
	for (list<PluginLibrary*>::iterator i = m_libraries.begin(); i != m_libraries.end(); ++i) {
		(*i)->close();
		delete (*i);
	}
}


/** Loads a plugin.
 *
 * Calls the load_*_plugin functions to actually do things, just a wrapper.
 */
Node*
NodeFactory::load_plugin(const Plugin* info, const string& name, uint poly, Patch* parent)
{
	pthread_mutex_lock(&m_plugin_list_mutex);
	
	Node* r = NULL;
	
	switch (info->type()) {
#if HAVE_LADSPA
	case Plugin::LADSPA:
		r = load_ladspa_plugin(info->lib_name(), info->plug_label(), name, poly, parent);
		break;
#endif
#if HAVE_DSSI
	case Plugin::DSSI:
		r = load_dssi_plugin(info->lib_name(), info->plug_label(), name, poly, parent);
		break;
#endif
	case Plugin::Internal:
		r = load_internal_plugin(info->plug_label(), name, poly, parent);
		break;
	default:
		cerr << "[NodeFactory] WARNING: Unknown plugin type." << endl;
	}

	pthread_mutex_unlock(&m_plugin_list_mutex);

	return r;
}


/** Loads an internal plugin.
 */
Node*
NodeFactory::load_internal_plugin(const string& plug_label, const string& name, uint poly, Patch* parent)
{
	// FIXME: this hardcoded plugin label stuff is nasty.  remove
	
	assert(poly == 1 || poly == parent->internal_poly());

	// Poly to use for input/output nodes
	// A port on a patch (represented by an InputNode or OutputNode) will only
	// appear polyphonic if the patch's polyphony is the same as it's patch's parent's
	// polyphony
	uint io_poly = 1;
	if (parent->parent() != NULL
			&& parent->parent_patch()->internal_poly() == parent->internal_poly())
		io_poly = parent->internal_poly();
	
	if (plug_label == "input" || plug_label == "audio_input") {  // FIXME: remove temp compat fix(es)
		AudioInputNode* in = new AudioInputNode(name, io_poly, parent,
			om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return in;
	} else if (plug_label == "control_input") {
		ControlInputNode* in = new ControlInputNode(name, io_poly, parent,
			om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return in;
	} else if (plug_label == "output" || plug_label == "audio_output") {
		AudioOutputNode* on = new AudioOutputNode(name, io_poly, parent,
			om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return on;
	} else if (plug_label == "control_output") {
		ControlOutputNode* on = new ControlOutputNode(name, io_poly, parent,
			om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return on;
	} else if (plug_label == "note_in" || plug_label == "midi_note_in") {
		MidiNoteNode* mn = new MidiNoteNode(name, poly, parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return mn;
	} else if (plug_label == "trigger_in" || plug_label == "midi_trigger_in") {
		MidiTriggerNode* mn = new MidiTriggerNode(name, 1, parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return mn;
	} else if (plug_label == "midi_control_in") {
		MidiControlNode* mn = new MidiControlNode(name, 1, parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return mn;
	} else if (plug_label == "transport") {
		TransportNode* tn = new TransportNode(name, 1, parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return tn;
	} else {
		cerr << "Unknown internal plugin type '" << plug_label << "'" << endl;
		exit(EXIT_FAILURE);
	}

	return NULL;
}


#if HAVE_DSSI


/** Loads information about all DSSI plugins into internal plugin database.
 */
void
NodeFactory::load_dssi_plugins()
{
	// FIXME: this is awful code duplication!
	
	char* env_dssi_path = getenv("DSSI_PATH");
	string dssi_path;
	if (!env_dssi_path) {
	 	cerr << "[NodeFactory] DSSI_PATH is empty.  Assuming /usr/lib/dssi:/usr/local/lib/dssi:~/.dssi" << endl;
		dssi_path = string("/usr/lib/dssi:/usr/local/lib/dssi:").append(
			getenv("HOME")).append("/.dssi");
	} else {
		dssi_path = env_dssi_path;
	}

	DSSI_Descriptor_Function df         = NULL;
	DSSI_Descriptor*         descriptor = NULL;
	
	string dir;
	string full_lib_name;
	
	// Yep, this should use an sstream alright..
	while (dssi_path != "") {
		dir = dssi_path.substr(0, dssi_path.find(':'));
		if (dssi_path.find(':') != string::npos)
			dssi_path = dssi_path.substr(dssi_path.find(':')+1);
		else
			dssi_path = "";
	
		DIR* pdir = opendir(dir.c_str());
		if (pdir == NULL) {
			//cerr << "[NodeFactory] Unreadable directory in DSSI_PATH: " << dir.c_str() << endl;
			continue;
		}

		struct dirent* pfile;
		while ((pfile = readdir(pdir))) {
			
			if (!strcmp(pfile->d_name, ".") || !strcmp(pfile->d_name, ".."))
				continue;
			
			full_lib_name = dir +"/"+ pfile->d_name;
			
			// Load descriptor function
			// Loaded with LAZY here, will be loaded with NOW on node loading
			void* handle = dlopen(full_lib_name.c_str(), RTLD_LAZY);
			if (handle == NULL)
				continue;
			
			df = (DSSI_Descriptor_Function)dlsym(handle, "dssi_descriptor");
			if (df == NULL) {
				// Not a DSSI plugin library
				dlclose(handle);
				continue;
			}

			PluginLibrary* plugin_library = new PluginLibrary(full_lib_name);
			m_libraries.push_back(plugin_library);

			const LADSPA_Descriptor* ld = NULL;
			
			for (uint i=0; (descriptor = (DSSI_Descriptor*)df(i)) != NULL; ++i) {
				ld = descriptor->LADSPA_Plugin;
				assert(ld != NULL);
				Plugin* info = new Plugin();
				assert(plugin_library != NULL);
				info->library(plugin_library);
				info->lib_path(dir);
				info->lib_name(pfile->d_name);
				info->plug_label(ld->Label);
				info->name(ld->Name);
				info->type(Plugin::DSSI);

				bool found = false;
				for (list<Plugin*>::const_iterator i = m_plugins.begin(); i != m_plugins.end(); ++i) {
					if ((*i)->lib_name() == info->lib_name() && (*i)->plug_label() == info->plug_label()) {
						cerr << "Warning: Duplicate DSSI plugin (" << info->lib_name() << ":"
							<< info->plug_label() << ")" << " found.  Using " << (*i)->lib_path()
							<< " version." << endl;
						found = true;
						break;
					}
				}
				if (!found)
					m_plugins.push_back(info);
				else
					delete info;
			}
			
			df = NULL;
			descriptor = NULL;
			dlclose(handle);
		}
		closedir(pdir);
	}
}


/** Creates a Node by instancing a DSSI plugin.
 */
Node*
NodeFactory::load_dssi_plugin(const string& lib_name, const string& plug_name, const string& name, uint poly, Patch* parent)
{
	// FIXME: awful code duplication here
	
	assert(plug_name != "");
	assert(lib_name != "");
	assert(name != "");
	assert(poly > 0);

	DSSI_Descriptor_Function df = NULL;
	const Plugin* plugin = NULL;
	Node* n = NULL;
	void* handle = NULL;
	
	// Attempt to find the lib
	list<Plugin*>::iterator i;
	for (i = m_plugins.begin(); i != m_plugins.end(); ++i) {
		plugin = (*i);
		if (plugin->lib_name() == lib_name && plugin->plug_label() == plug_name) break;
	}

	if (i == m_plugins.end()) {
		cerr << "Did not find DSSI plugin " << lib_name << ":" << plug_name << " in database." << endl;
		return NULL;
	} else {
		assert(plugin != NULL);
		plugin->library()->open();
		handle = plugin->library()->handle();
		assert(handle != NULL);	
		
		// Load descriptor function
		dlerror();
		df = (DSSI_Descriptor_Function)dlsym(handle, "dssi_descriptor");
		if (df == NULL || dlerror() != NULL) {
			cerr << "Looks like this isn't a DSSI plugin." << endl;
			return NULL;
		}
	}

	// Attempt to find the plugin in lib
	DSSI_Descriptor* descriptor = NULL;
	for (uint i=0; (descriptor = (DSSI_Descriptor*)df(i)) != NULL; ++i) {
		if (descriptor->LADSPA_Plugin != NULL && descriptor->LADSPA_Plugin->Label == plug_name) {
			break;
		}
	}
	
	if (descriptor == NULL) {
		cerr << "Could not find plugin \"" << plug_name << "\" in " << lib_name << endl;
		return NULL;
	}

	n = new DSSIPlugin(name, poly, parent, descriptor,
		om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
	bool success = ((DSSIPlugin*)n)->instantiate();
	if (!success)
		return NULL;
	
	n->plugin(plugin);

	return n;
}
#endif // HAVE_DSSI


void
NodeFactory::load_plugins()
{
	// Only load if we havn't already, so every client connecting doesn't cause
	// this (expensive!) stuff to happen.  Not the best solution - would be nice
	// if clients could refresh plugins list for whatever reason :/
	if (!m_has_loaded) {
		pthread_mutex_lock(&m_plugin_list_mutex);
		
		m_plugins.clear();
		m_plugins = m_internal_plugins;
	
#if HAVE_DSSI
		load_dssi_plugins();
#endif
#if HAVE_LADSPA
		load_ladspa_plugins();
#endif
		
		m_has_loaded = true;
		
		pthread_mutex_unlock(&m_plugin_list_mutex);
	}
}


#ifdef HAVE_LADSPA
/** Loads information about all LADSPA plugins into internal plugin database.
 */
void
NodeFactory::load_ladspa_plugins()
{
	char* env_ladspa_path = getenv("LADSPA_PATH");
	string ladspa_path;
	if (!env_ladspa_path) {
	 	cerr << "[NodeFactory] LADSPA_PATH is empty.  Assuming /usr/lib/ladspa:/usr/local/lib/ladspa:~/.ladspa" << endl;
		ladspa_path = string("/usr/lib/ladspa:/usr/local/lib/ladspa:").append(
			getenv("HOME")).append("/.ladspa");
	} else {
		ladspa_path = env_ladspa_path;
	}

	LADSPA_Descriptor_Function df         = NULL;
	LADSPA_Descriptor*         descriptor = NULL;
	
	string dir;
	string full_lib_name;
	
	// Yep, this should use an sstream alright..
	while (ladspa_path != "") {
		dir = ladspa_path.substr(0, ladspa_path.find(':'));
		if (ladspa_path.find(':') != string::npos)
			ladspa_path = ladspa_path.substr(ladspa_path.find(':')+1);
		else
			ladspa_path = "";
	
		DIR* pdir = opendir(dir.c_str());
		if (pdir == NULL) {
			//cerr << "[NodeFactory] Unreadable directory in LADSPA_PATH: " << dir.c_str() << endl;
			continue;
		}

		struct dirent* pfile;
		while ((pfile = readdir(pdir))) {
			
			if (!strcmp(pfile->d_name, ".") || !strcmp(pfile->d_name, ".."))
				continue;
			
			full_lib_name = dir +"/"+ pfile->d_name;
			
			// Load descriptor function
			// Loaded with LAZY here, will be loaded with NOW on node loading
			void* handle = dlopen(full_lib_name.c_str(), RTLD_LAZY);
			if (handle == NULL)
				continue;
			
			df = (LADSPA_Descriptor_Function)dlsym(handle, "ladspa_descriptor");
			if (df == NULL) {
				dlclose(handle);
				continue;
			}	

			PluginLibrary* plugin_library = new PluginLibrary(full_lib_name);
			m_libraries.push_back(plugin_library);

			for (uint i=0; (descriptor = (LADSPA_Descriptor*)df(i)) != NULL; ++i) {
				Plugin* info = new Plugin();
				assert(plugin_library != NULL);
				info->library(plugin_library);
				info->lib_path(dir);
				info->lib_name(pfile->d_name);
				info->plug_label(descriptor->Label);
				info->name(descriptor->Name);
				info->type(Plugin::LADSPA);
				
				bool found = false;
				for (list<Plugin*>::const_iterator i = m_plugins.begin(); i != m_plugins.end(); ++i) {
					if ((*i)->lib_name() == info->lib_name() && (*i)->plug_label() == info->plug_label()) {
						cerr << "Warning: Duplicate LADSPA plugin (" << info->lib_name() << ":"
							<< info->plug_label() << ")" << " found.  Using " << (*i)->lib_path()
							<< " version." << endl;
						found = true;
						break;
					}
				}
				if (!found)
					m_plugins.push_back(info);
				else
					delete info;
			}
			
			df = NULL;
			descriptor = NULL;
			dlclose(handle);
		}
		closedir(pdir);
	}
}


/** Loads a LADSPA plugin.
 * Returns 'poly' independant plugins as a Node*
 */
// FIXME: make this take a Plugin as a param
Node*
NodeFactory::load_ladspa_plugin(const string& lib_name, const string& plug_name, const string& name, uint poly, Patch* parent)
{
	assert(plug_name != "");
	assert(lib_name != "");
	assert(name != "");
	assert(poly > 0);

	LADSPA_Descriptor_Function df = NULL;
	Plugin* plugin = NULL;
	Node* n = NULL;
	void* plugin_lib = NULL;
	
	// Attempt to find the lib
	list<Plugin*>::iterator i;
	for (i = m_plugins.begin(); i != m_plugins.end(); ++i) {
		plugin = (*i);
		if (plugin->lib_name() == lib_name && plugin->plug_label() == plug_name) break;
	}

	if (i == m_plugins.end()) {
		cerr << "Did not find LADSPA plugin " << lib_name << ":" << plug_name << " in database." << endl;
		return NULL;
	} else {
		assert(plugin != NULL);
		plugin->library()->open();
		plugin_lib = plugin->library()->handle();
		assert(plugin_lib != NULL);	
		
		// Load descriptor function
		dlerror();
		df = (LADSPA_Descriptor_Function)dlsym(plugin_lib, "ladspa_descriptor");
		if (df == NULL || dlerror() != NULL) {
			cerr << "Looks like this isn't a LADSPA plugin." << endl;
			return NULL;
		}
	}

	// Attempt to find the plugin in lib
	LADSPA_Descriptor* descriptor = NULL;
	for (uint i=0; (descriptor = (LADSPA_Descriptor*)df(i)) != NULL; ++i) {
		if (descriptor->Label == plug_name) {
			break;
		}
	}
	
	if (descriptor == NULL) {
		cerr << "Could not find plugin \"" << plug_name << "\" in " << lib_name << endl;
		return NULL;
	}

	n = new LADSPAPlugin(name, poly, parent, descriptor,
		om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
	bool success = ((LADSPAPlugin*)n)->instantiate();
	if (!success)
		return NULL;
	
	n->plugin(plugin);

	return n;
}


#endif // HAVE_LADSPA


} // namespace Om
