# vcc3-api.tcl --
#
#       All the API functions that can be called by the RPC client to control
#       the Vcc3.  Contains a few helper functions.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

source vcc3-lib.tcl

# returns 1 if successful, 0 if not
proc vcc3_init {{filename "/dev/cuac01"}} {
    return [vcc3_setupControl $filename]
}

proc vcc3_startMove {dir} {
    switch -exact -- $dir {
	left {
	    return [vcc3_startLeft]
	}
	right {
	    return [vcc3_startRight]
	}
	up {
	    return [vcc3_startUp]
	}
	down {
	    return [vcc3_startDown]
	}
	in {
	    return [vcc3_startIn]
	}
	out {
	    return [vcc3_startOut]
	}
	default {
	    return -code error "dir must be \[left, right, up, down\]"
	}
    }
}

proc vcc3_startLeft {} {
#    puts stdout "startLeft called"
    set str [binary format "ccccc" 0x05 0x12 0x01 0x02 0x00]
#    return [vcc3_doCommand $str]
    return [vcc3_doCommand $str]
}

proc vcc3_startRight {} {
#    puts stdout "startRight called"
    set str [binary format "ccccc" 0x05 0x12 0x01 0x01 0x00]
    return [vcc3_doCommand $str]
}

proc vcc3_startUp {} {
    set str [binary format "ccccc" 0x05 0x12 0x01 0x00 0x01]
    return [vcc3_doCommand $str]
}

proc vcc3_startDown {} {
    set str [binary format "ccccc" 0x05 0x12 0x01 0x00 0x02]
    return [vcc3_doCommand $str]
}

# FIXME - if I am in the process of fading out, then I want to fade in, I
#   should be allowed to
#
# this has to do with the problem of serializability I introduced
proc vcc3_fadeIn {} {
    set str [binary format "cccc" 0x08 0x11 0x02 0x00]
    # FIXME - technically, we should read the fade speed, then figure out how
    #   long before timing out on the fade
    return [vcc3_doCommand $str "fadeInOut"]
}

proc vcc3_fadeOut {} {
    set str [binary format "cccc" 0x08 0x11 0x02 0x01]
    return [vcc3_doCommand $str "fadeInOut"]
}

proc vcc3_setFadeSpeed {speed} {
    global g_sync
    
    set loSpeed [expr $speed & 0xFF]
    set hiSpeed [expr ($speed & 0xFF00) >> 8]
    set str [binary format "cccccc" 0x08 0x11 0x04 0x02 $hiSpeed $loSpeed]
    return [vcc3_doCommand $str]
}

proc vcc3_setFadeGainLevel {gain level} {
    global g_sync

    if {$gain < 0 || $gain > 255} {
	return -code error "fade gain must be in the range \[0,255\]"
    }
    if {$level < 0 || $level > 255} {
	return -code error "fade level must be in the range \[0,255\]"
    }
    set str [binary format "cccccc" 0x08 0x11 0x01 0x02 $gain $level]
    return [vcc3_doCommand $str]
}

proc vcc3_setFadeGain {gain} {
    global g_sync

    if {$gain < 0 || $gain > 255} {
	return -code error "fade gain must be in the range \[0,255\]"
    }
    set level [vcc3_getFadeLevel]
    if {$level == -9999} {
	puts stdout "vcc3_setFadeGain: error getting fade level"
	return 0
    }
    return [vcc3_setFadeGainLevel $gain $level]
}

proc vcc3_setFadeLevel {level} {
    global g_sync

    if {$level < 0 || $level > 255} {
	return -code error "fade level must be in the range \[0,255\]"
    }
    set gain [vcc3_getFadeGain]
    if {$gain == -9999} {
	puts stdout "vcc3_setFadeLevel: error getting fade gain"
	return 0
    }
    return [vcc3_setFadeGainLevel $gain $level]
}

proc vcc3_getFadeSpeed {} {
    global g_fadeSpeed
    
    set retList [vcc3_getFadeInfo]
    if {[llength $retList] == 0} {
	return -9999
    }
    return $g_fadeSpeed
}
proc vcc3_getFadeGain {} {
    global g_fadeSpeed g_fadeGain g_fadeLevel
    
    set retList [vcc3_getFadeInfo]
    if {[llength $retList] == 0} {
	return -9999
    }
    return $g_fadeGain
}

proc vcc3_getFadeLevel {} {
    global g_fadeSpeed g_fadeGain g_fadeLevel
    
    set retList [vcc3_getFadeInfo]
    if {[llength $retList] == 0} {
	return -9999
    }
    return $g_fadeLevel
}

proc vcc3_getFadeInfo {} {
    global g_sync
    global g_fadeSpeed g_fadeGain g_fadeLevel
    
    set str [binary format "ccc" 0x08 0x04 0x02]
    set result [vcc3_doCommand $str "readFadeStatus"]
    if {!$result} {
	set retVal [list]
	return $retVal
    }
    set retVal [list $g_fadeSpeed $g_fadeGain $g_fadeLevel]
    return $retVal
}

proc vcc3_stop {} {

#    puts stdout "vcc3_stop called"
    
    set timeout 500
    set sleepTime 50
    set timeSlept 0
    set result [vcc3_stopMove]
    while {!$result && ($timeSlept < $timeout)} {
	sleep $sleepTime
	set timeSlept [expr $timeSlept + $sleepTime]
	set result [vcc3_stopMove]
    }
    if {!$result} {
#	puts stdout "vcc3_stop timed out"
	return 0
    }
    return [vcc3_stopZoom]
}

proc vcc3_stopMove {} {
    set str [binary format "ccc" 0x05 0x12 0x02]
    return [vcc3_doCommand $str]
}

# waits for the home position to be reached before returning
proc vcc3_home {} {
    set str [binary format "cc" 0x05 0x11]
    return [vcc3_doCommand $str "home"]
}

# not sure what this guy does
proc vcc3_setup {} {
    set str [binary format "cc" 0x05 0x10]
    return [vcc3_doCommand $str "panTiltSetup"]
}

proc vcc3_resetSoftware {} {
    set str [binary format "cc" 0x08 0x01]
    return [vcc3_doCommand $str]
}

proc vcc3_disablePCControl {} {
    set str [binary format "cccc" 0x08 0x17 0x01 0x01]
    return [vcc3_doCommand $str]
}

proc vcc3_enablePCControl {} {
    set str [binary format "cccc" 0x08 0x17 0x01 0x00]
    return [vcc3_doCommand $str]
}

proc vcc3_startIn {} {
    set str [binary format "cccc" 0x01 0x12 0x01 0x00]
    return [vcc3_doCommand $str]
}

proc vcc3_startOut {} {
    set str [binary format "cccc" 0x01 0x12 0x01 0x01]
    return [vcc3_doCommand $str]
}

proc vcc3_stopZoom {} {
    set str [binary format "ccc" 0x01 0x12 0x03]
    return [vcc3_doCommand $str]
}

# returns -9999 if not available
proc vcc3_getPanSpeed {} {
    global g_sync g_panSpeed g_tiltSpeed g_panPosition g_tiltPosition

    set retList [vcc3_getPanTiltInfo]
    if {[llength $retList] > 0} {
	return $g_panSpeed
    } else {
	return -9999
    }
}

# returns -9999 if not available
proc vcc3_getTiltSpeed {} {
    global g_sync g_panSpeed g_tiltSpeed g_panPosition g_tiltPosition

    set retList [vcc3_getPanTiltInfo]
    if {[llength $retList] > 0} {
	return $g_tiltSpeed
    } else {
	return -9999
    }
}

# returns -9999 if not available
proc vcc3_getPanPosition {} {
    global g_sync g_panSpeed g_tiltSpeed g_panPosition g_tiltPosition

    set retList [vcc3_getPanTiltInfo]
    if {[llength $retList] > 0} {
	return $g_panPosition
    } else {
	return -9999
    }
}

# returns -9999 if not available
proc vcc3_getTiltPosition {} {
    global g_sync g_panSpeed g_tiltSpeed g_panPosition g_tiltPosition
    
    set retList [vcc3_getPanTiltInfo]
    if {[llength $retList] > 0} {
	return $g_tiltPosition
    } else {
	return -9999
    }
}

#
# 1 <= panSpeed <= 76
#
proc vcc3_setPanSpeed {panSpeed} {
    if {$panSpeed < 1 || $panSpeed > 76} {
	return -code error "pan speed must be in the range \[1,76\]"
    }
    set tiltSpeed [vcc3_getTiltSpeed]
    if {$tiltSpeed == -9999} {
	return 0
    }
    return [vcc3_setPanTiltSpeed $panSpeed $tiltSpeed]
}

#
# 1 <= tiltSpeed <= 70
#
proc vcc3_setTiltSpeed {tiltSpeed} {
    if {$tiltSpeed < 1 || $tiltSpeed > 70} {
	return -code error "tilt speed must be in the range \[1,70\]"
    }
    set panSpeed [vcc3_getPanSpeed]
    if {$panSpeed == -9999} {
	return 0
    }
    return [vcc3_setPanTiltSpeed $panSpeed $tiltSpeed]
}

#
# 1 <= panSpeed <= 76
# 1 <= tiltSpeed <= 70
#
proc vcc3_setPanTiltSpeed {panSpeed tiltSpeed} {
    if {$panSpeed < 1 || $panSpeed > 76} {
	return -code error "pan speed must be in the range \[1,76\]"
    }
    if {$tiltSpeed < 1 || $tiltSpeed > 70} {
	return -code error "tilt speed must be in the range \[1,70\]"
    }
    set str [binary format "cccccc" 0x05 0x12 0x03 0x02 $panSpeed $tiltSpeed]
    return [vcc3_doCommand $str]
}

# get the current speed and position and store it into g_ variables
#
# returns a list of [panSpeed tiltSpeed panPosition tiltPosition]
#
# returns an empty list if unsuccessful
#
proc vcc3_getPanTiltInfo {} {
    global g_sync g_panSpeed g_tiltSpeed g_panPosition g_tiltPosition
    
    set str [binary format "ccc" 0x05 0x04 0x02]
    set result [vcc3_doCommand $str "readPanTiltStatus"]
    if {$result} {
	set retVal [list $g_panSpeed $g_tiltSpeed $g_panPosition $g_tiltPosition]
    } else {
	set retVal [list]
    }
    return $retVal
}

proc vcc3_setPanPosition {panPosition} {
    set tiltPos [vcc3_getTiltPosition]
    if {$tiltPos == -9999} {
	puts stdout "vcc3_setPanPosition: error getting tiltPosition"
	return 0
    }
    return [vcc3_setPanTiltPosition $panPosition $tiltPos]
}

proc vcc3_setTiltPosition {tiltPosition} {
    set panPos [vcc3_getPanPosition]
    if {$panPos == -9999} {
	return 0
    }
    return [vcc3_setPanTiltPosition $panPos $tiltPosition]
}

proc vcc3_getZoomPosition {} {
    global g_zoomSpeed g_zoomPosition g_maxZoom

    set retList [vcc3_getZoomInfo]
    if {[llength $retList] == 0} {
	return -9999
    } 
    return $g_zoomPosition
}

proc vcc3_getZoomSpeed {} {
    global g_zoomSpeed g_zoomPosition g_maxZoom

    set retList [vcc3_getZoomInfo]
    if {[llength $retList] == 0} {
	return -9999
    }
    return $g_zoomSpeed
}

proc vcc3_getZoomInfo {} {
    global g_sync g_panSpeed g_tiltSpeed g_panPosition g_tiltPosition
    global g_zoomSpeed g_zoomPosition g_maxZoom
    
    set str [binary format "ccc" 0x01 0x04 0x02]
    set result [vcc3_doCommand $str "readZoomStatus"]
    if {!$result} {
	set retVal [list]
	return $retVal
    }
    # get the max zoom info
    set str [binary format "cccc" 0x01 0x12 0x02 0x03]
    set result [vcc3_doCommand $str "readMaxZoom"]
    if {!$result} {
	set retVal [list]
	return $retVal
    }
    set retVal [list $g_zoomSpeed $g_zoomPosition $g_maxZoom]
    return $retVal
}

proc vcc3_getMaxZoom {} {
    global g_zoomSpeed g_zoomPosition g_maxZoom

    set retList [vcc3_getZoomInfo]
    if {[llength $retList] == 0} {
	return -9999
    }
    return $g_maxZoom
}

proc vcc3_setZoomSpeed {speed} {
    if {$speed < 0 || $speed > 7} {
	return -code error "zoom speed must be in the range \[0,7\]"
    }
    set str [binary format "ccccc" 0x01 0x12 0x04 0x02 $speed]
    return [vcc3_doCommand $str]
}

#
# waits for the move to complete before returning
#
# 0 <= zoomPosition <= $g_maxZoom
proc vcc3_setZoomPosition {zoomPosition} {
    global g_sync g_maxZoom

    if {$zoomPosition < 0 || $zoomPosition > $g_maxZoom} {
	return -code error "zoom position must be in the range \[0,$g_maxZoom\]"
    }
    set lo [expr $zoomPosition & 0xFF]
    set hi [expr ($zoomPosition & 0xFF00) >> 8]
    set str [binary format "cccccc" 0x01 0x12 0x02 0x02 $hi $lo]
    return [vcc3_doCommand $str "zoomMove"]
}



# FIXME - are these absolutes, or depend on the camera?  do we need to
#     detect the max range on startup?
#
# waits for the move to complete before returning
#
# -800 <= panPosition <= 800
# -266 <= tiltPosition <= 222
proc vcc3_setPanTiltPosition {panPosition tiltPosition} {
    global g_sync
    
    set pan_pos [expr 0x8000 - $panPosition]
    set tilt_pos [expr 0x8000 + $tiltPosition]
    set loPan [expr $pan_pos & 0xFF]
    set hiPan [expr ($pan_pos & 0xFF00) >> 8]
    set loTilt [expr $tilt_pos & 0xFF]
    set hiTilt [expr ($tilt_pos & 0xFF00) >> 8]
    set str [binary format "cccccccc" 0x05 0x12 0x05 0x02 $hiPan $loPan $hiTilt $loTilt]
    return [vcc3_doCommand $str "absoluteMove"]
}

#
# waits for the move to complete before returning
#
proc vcc3_moveRelative {dir {amount 40}} {
    global g_sync
    
    set loPan 0x0
    set hiPan 0x0
    set loTilt 0x0
    set hiTilt 0x0
    
    switch -exact -- $dir {
	left {
	    set loPan [expr $amount & 0xFF]
	    set hiPan [expr ($amount & 0xFF00) >> 8]	    
	}
	right {
	    set loPan [expr $amount & 0xFF]
	    set hiPan [expr ($amount & 0xFF00) >> 8]
	    # since right (this is opposite what the manual says!)
	    set hiPan [expr $hiPan | 0x80]	}
	up {
	    set loTilt [expr $amount & 0xFF]
	    set hiTilt [expr ($amount & 0xFF00) >> 8]	    
	}
	down {
	    set loTilt [expr $amount & 0xFF]
	    set hiTilt [expr ($amount & 0xFF00) >> 8]	    
	    # since down (this is opposite what the manual says!)
	    set hiTilt [expr $hiPan | 0x80]
	}
	default {
	    return -code error "dir must be \[left, right, up, down\]"
	}
    }
    set str [binary format "cccccccc" 0x05 0x12 0x04 0x02 $hiPan $loPan $hiTilt $loTilt]
    return [vcc3_doCommand $str "relativeMove"]
}

# this is not the most efficient way, since we serialize all commands, but it
#   is easier
#
# a more efficient way would be to send all the commands to the camera, then
#   sync on each response
proc vcc3_goPreset {num} {
    global g_sync g_presets

    if {$num < 1 || $num > 6} {
	return -code error "num must be in the range \[1,6\]"
    }

    set info [vcc3_getPresetInfo $num]
    if {[llength $info] == 0} {
	return 0
    }
    set aeRef [lindex $info 0]
    set panPos [lindex $info 1]
    set tiltPos [lindex $info 2]
    set zoomPos [lindex $info 3]
    set result [vcc3_setPanTiltPosition $panPos $tiltPos]
    if {!$result} {
	return 0
    }
    set result [vcc3_setZoomPosition $zoomPos]
    if {!$result} {
	return 0
    }
    return [vcc3_setAERef $aeRef]
}

proc vcc3_getPresetInfo {num} {
    global g_sync g_presets

    if {$num < 1 || $num > 6} {
	return -code error "num must be in the range \[1,6\]"
    }
    set str [binary format "cccc" 0x08 0x18 0x01 $num]
    set result [vcc3_doCommand $str "readPreset"]
    if {!$result} {
	set retVal [list]
    } else {
	set retVal [list $g_presets($num,aeRef) $g_presets($num,panPosition) $g_presets($num,tiltPosition) $g_presets($num,zoomPosition)]
    }
    return $retVal
}

# listIfo is a list of the form: [aeRef panPosition tiltPosition zoomPosition]
proc vcc3_setPreset {num {listInfo "current"}} {
    if {$num < 1 || $num > 6} {
	return -code error "num must be in the range \[1,6\]"
    }

    if {$listInfo == "current"} {
	set info(aeRef) [vcc3_getAERef]
	if {$info(aeRef) == -9999} {
	    return 0
	}
	set temp [vcc3_getPanTiltInfo]
	if {[llength $temp] == 0} {
	    return 0
	}
	set info(panPosition) [lindex $temp 2]
	set info(tiltPosition) [lindex $temp 3]	
	set temp [vcc3_getZoomInfo]
	if {[llength $temp] == 0} {
	    return 0
	}
	set info(zoomPosition) [lindex $temp 1]
    } else {
	set info(aeRef) [lindex $listInfo 0]
	set info(panPosition) [lindex $listInfo 1]
	set info(tiltPosition) [lindex $listInfo 2]
	set info(zoomPosition) [lindex $listInfo 3]
    }
    
    set str [binary format "ccccc" 0x08 0x18 0x02 $num $num]
    set info(panPosition) [expr 0x8000 - $info(panPosition)]
    set info(tiltPosition) [expr 0x8000 + $info(tiltPosition)]
    set loPan [expr $info(panPosition) & 0xFF]
    set hiPan [expr ($info(panPosition) & 0xFF00) >> 8]
    set loTilt [expr $info(tiltPosition) & 0xFF]
    set hiTilt [expr ($info(tiltPosition) & 0xFF00) >> 8]
    set loZoom [expr $info(zoomPosition) & 0xFF]
    set hiZoom [expr ($info(zoomPosition) & 0xFF00) >> 8]
    set str [binary format "a*ccccccc" $str $info(aeRef) $hiPan $loPan $hiTilt $loTilt $hiZoom $loZoom]
    return [vcc3_doCommand $str]
}

# mode can be manual or auto
proc vcc3_setFocusMode {mode} {
    switch -exact -- $mode {
	auto {
	    set str [binary format "cccc" 0x01 0x10 0x01 0x00]
	}
	manual {
	    set str [binary format "cccc" 0x01 0x10 0x01 0x01]
	}
	default {
	    return -code error "mode must be \[auto,manual\]"
	}
    }
    return [vcc3_doCommand $str]
}

proc vcc3_startFocus {dir} {
    switch -exact -- $dir {
	far {
	    set str [binary format "cccc" 0x01 0x10 0x02 0x00]
	}
	near {
	    set str [binary format "cccc" 0x01 0x10 0x02 0x01]
	}
	default {
	    return -code error "dir must be \[far,near\]"
	}
    }
    return [vcc3_doCommand $str]
}

proc vcc3_stopFocus {} {
    set str [binary format "ccc" 0x01 0x10 0x04]

    # need to retry in case of quick button pushes
    set timeout 500
    set sleepTime 50
    set timeSlept 0
    set result [vcc3_doCommand $str]
    while {!$result && ($timeSlept < $timeout)} {
	sleep $sleepTime
	set timeSlept [expr $timeSlept + $sleepTime]
	set result [vcc3_doCommand $str]
    }
    return $result
}

proc vcc3_getFocusPosition {} {
    global g_focusSpeed g_focusPosition

    set retList [vcc3_getFocusInfo]
    if {[llength $retList] == 0} {
	return -9999
    }
    return $g_focusPosition
}

proc vcc3_getFocusSpeed {} {
    global g_focusSpeed g_focusPosition

    set retList [vcc3_getFocusInfo]
    if {[llength $retList] == 0} {
	return -9999
    }
    return $g_focusSpeed
}

proc vcc3_setFocusSpeed {focusSpeed} {
    if {$focusSpeed < 0 || $focusSpeed > 7} {
	return -code error "focusSpeed must be in the range \[0,7\]"
    }
    set str [binary format "ccccc" 0x01 0x10 0x05 0x02 $focusSpeed]
    return [vcc3_doCommand $str]
}

proc vcc3_getFocusInfo {} {
    global g_sync 
    global g_focusSpeed g_focusPosition
    
    set str [binary format "ccc" 0x01 0x04 0x01]
    set result [vcc3_doCommand $str "readFocusStatus"]
    if {!$result} {
	set retVal [list]
    } else {
	set retVal [list $g_focusSpeed $g_focusPosition]
    }
    return $retVal
}

#default is 72
proc vcc3_setAERef {aeRef} {
    if {$aeRef < 0 || $aeRef > 255} {
	return -code error "aeRef must be in the range \[0,255\]"
    }
    set str [binary format "cccc" 0x01 0x14 0x04 $aeRef]
    return [vcc3_doCommand $str]
}

proc vcc3_getAERef {} {
    global g_aeRef g_exposureSpeed g_iris g_shutterSpeed g_gain

    set retList [vcc3_getExposureInfo]
    if {[llength $retList] == 0} {
	return -9999
    }
    return $g_aeRef
}

proc vcc3_getExposureInfo {} {
    global g_sync
    global g_aeRef g_exposureSpeed g_iris g_shutterSpeed g_gain 
    
    set str [binary format "ccc" 0x01 0x04 0x03]
    set result [vcc3_doCommand $str "readExposureStatus"]
    if {!$result} {
	set retVal [list]
    } else {
	set retVal [list $g_exposureSpeed $g_aeRef $g_iris $g_shutterSpeed $g_gain]
    }
    return $retVal
}

#
# this sends the command str and synchronizes on syncVar if specified
#
proc vcc3_doCommand {str {syncVar ""} {timeout -1}} {
    global g_sync

    if {$syncVar != ""} { 
	set g_sync($syncVar) "WAIT"
    }
    set result [vcc3_sendCommand $str]
    if {!$result} {
	if {$syncVar != ""} {
	    set g_sync($syncVar) "NO_WAIT"
	}
#	puts stdout "vcc3_doCommand: sendCommand failed"
	return 0
    }
    if {$syncVar != ""} {
	# wait for the response to come back
	if {$timeout != -1} {
	    return [vcc3_waitSync $syncVar $timeout]
	} else {
	    return [vcc3_waitSync $syncVar]
	}
    }
    return 1
}

proc vcc3_sweep {speed zoom tilt pan1 pan2 delay {restore 0}} {
    set oSpeed [vcc3_getPanSpeed]
    set info [vcc3_getPanTiltInfo]
    set oTilt [lindex $info 3]
    set oPan [lindex $info 2]
    set oZoom [vcc3_getZoomPosition]
    
    vcc3_setPanSpeed $speed
    vcc3_setZoomPosition $zoom
    vcc3_setPanTiltPosition $pan1 $tilt
    # pause for a little bit
    after $delay
    vcc3_setPanTiltPosition $pan2 $tilt

    # always restore the speed
    vcc3_setPanSpeed $oSpeed
    if {$restore} {
	vcc3_setZoomPosition $oZoom
	vcc3_setPanTiltPosition $oPan $oTilt
    }
}

# FIXME - the error checking here isn't very good
proc vcc3_fastMove {pan {tilt ""} {zoom ""}} {
    set info [vcc3_getPanTiltInfo]
    if {[llength $info] == 0} {
	# we failed to get the info, the camera must already be busy
	# give up
	return "-1"
    }
    set oPanSpeed [lindex $info 0]
    set oTiltSpeed [lindex $info 1]
#    set oPan [lindex $info 2]
    set oTilt [lindex $info 3]

    vcc3_setPanTiltSpeed 76 70

    if {$zoom != ""} {
	set info [vcc3_getZoomInfo]
	if {[llength $info] == 0} {
	    # we failed to get the info, the camera must already be busy
	    return "-1"
	}
	set oZoom [lindex $info 1]
	set oZoomSpeed [lindex $info 0]
	vcc3_setZoomSpeed 7
	vcc3_setZoomPosition $zoom
	vcc3_setZoomSpeed $oZoomSpeed
    }

    if {$tilt == ""} {
	set tilt $oTilt
    }
    vcc3_setPanTiltPosition $pan $tilt

    vcc3_setPanTiltSpeed $oPanSpeed $oTiltSpeed
 
    return ""
}

#
# Internal functions
#

proc vcc3_test {} {
    puts stdout "test called"
    set result [vcc3_moveRelative left]
    puts stdout "result is $result"
    set result [vcc3_moveRelative right]
    puts stdout "result is $result"
    set result [vcc3_moveRelative up]
    puts stdout "result is $result"
    set result [vcc3_moveRelative down]
    puts stdout "result is $result"

    #  for {set x 0} {$x < 5} {incr x 1} {
#  	set result [vcc3_startLeft]
#  	puts stdout "start result is $result"
#  	sleep 50
#  	set result [vcc3_stop]
#  	puts stdout "stop result is $result"
#      }
    
    puts stdout "vcc3_test: finished stepping"
    
    #  set result [vcc3_setPanTiltPosition 400 0]
#      puts stdout "result is $result"
    
#      set result [vcc3_setPanTiltPosition -400 0]
#      puts stdout "result is $result"
    
#      set result [vcc3_setPanTiltPosition 600 100]
#      puts stdout "result is $result"

    #       illegal position
    #       vcc3_setPanTiltPosition -800 -400

   set result [vcc3_setPanPosition -800]
#      puts stdout "result is $result"
#      set result [vcc3_setTiltPosition -150]
#      puts stdout "result is $result"
    
    set result [vcc3_setZoomSpeed 4]
    puts stdout "result is $result"

    set info [list 72 400 -100 500]
    set result [vcc3_setPreset 2 $info]
    puts stdout "result is $result"
    
   #   set result [vcc3_goPreset 1]
#      puts stdout "result is $result"
#      set result [vcc3_setPreset 1]
#      puts stdout "result is $result"
    
#      set result [vcc3_setZoomPosition 500]
#      puts stdout "result is $result"
    
    #       vcc3_setup
    set result [vcc3_setFocusSpeed 7]
    puts stdout "result is $result"
    set result [vcc3_setAERef 72]
    puts stdout "result is $result"

    puts stdout "test completed"
}
