/*
 * transcoder-h261.cc --
 *
 *      H.261 Transcoder
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/video/transcoder-h261.cc,v 1.10 2002/02/03 03:13:15 lim Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef WIN32
#include <sys/param.h>
#include <netinet/in.h>
#endif
#include "inet.h"
#include "rtp.h"
#include "transcoder.h"
#include "transmitter.h"
#include "p64/p64.h"
#include "encoder.h"
#include "crdef.h"

class H261Transcoder : public VideoTranscoder {
public:
	H261Transcoder();
	~H261Transcoder();
protected:
	virtual void configure() = 0;
	P64Decoder* decoder_;
	int ibit_;
	int h261_rtp_bug_;
};

#ifdef notyet
class H261DCTTranscoder : public H261Transcoder {
public:
	virtual void configure();
	virtual void recv_data(pktbuf* pb);
};
#endif

class H261PixelTranscoder : public H261Transcoder {
public:
	virtual void configure();
	virtual void recv_data(pktbuf* pb);
};

#ifdef notyet
static class H261DCTTranscoderClass : public TclClass {
public:
	H261DCTTranscoderClass() : TclClass("Transcoder/H261/DCT") {}
	TclObject* create(int, const char*const*) {
		    return (new H261DCTTranscoder);
	}
} h261_dct_transcoder_class;
#endif

static class H261PixelTranscoderClass : public TclClass {
public:
	H261PixelTranscoderClass() : TclClass("Transcoder/H261/Pixel") {}
	TclObject* create(int, const char*const*) {
		    return (new H261PixelTranscoder);
	}
} h261_pixel_transcoder_class;

H261Transcoder::H261Transcoder()
	       :VideoTranscoder(4), decoder_(0),
		 ibit_(-1), h261_rtp_bug_(0)
{}

H261Transcoder::~H261Transcoder()
{
	delete decoder_;
}

#ifdef notyet
void H261DCTTranscoder::configure()
{
	delete decoder_;
	if (ibit_ != 0)
		decoder_ = new IntraP64DCTDecoder();
	else
		decoder_ = new FullP64Decoder();

	decoder_->mark(CR_SEND|CR_MOTION);
}
#endif

void H261PixelTranscoder::configure()
{
	delete decoder_;
	if (ibit_ != 0)
		decoder_ = new IntraP64Decoder();
	else
		decoder_ = new FullP64Decoder();

	decoder_->mark(CR_MOTION_BIT | CR_LQ);
}

void H261PixelTranscoder::recv_data(pktbuf* pb)
{
	rtphdr* rh = (rtphdr*)(pb->dp);
	int cc = pb->len - sizeof(rtphdr);
	const u_char* bp = (const u_char*)(rh + 1);

	u_int v = ntohl(*(u_int*)(rh + 1));
	int sbit = v >> 29;
	int ebit = (v >> 26) & 7;
	int quant = (v >> 10) & 0x1f;
	int mvdh = (v >> 5) & 0x1f;
	int mvdv = v & 0x1f;
        int mba, gob;
	/*
	 * vic-2.7 swapped the GOB and MBA fields in the RTP packet header
	 * with respect to the spec.  To maintain backward compat, whenever
	 * we see an out of range gob, we change our assumption about the
	 * stream and continue.
	 */
	if (!h261_rtp_bug_) {
		mba = (v >> 15) & 0x1f;
		gob = (v >> 20) & 0xf;
		if (gob > 12) {
			h261_rtp_bug_ = 1;
			mba = (v >> 19) & 0x1f;
			gob = (v >> 15) & 0xf;
		}
	} else {
		mba = (v >> 19) & 0x1f;
		gob = (v >> 15) & 0xf;
		if (gob > 12) {
			h261_rtp_bug_ = 0;
			mba = (v >> 15) & 0x1f;
			gob = (v >> 20) & 0xf;
		}
	}

	if (gob > 12) {
		pb->release();
		return;
	}

	int ibit = (v >> 25) & 1;
	if (ibit != ibit_) {
		ibit_ = ibit;
		configure();
	}

	bp += 4;;
	cc -= 4;

	decoder_->marks(crvec_);
	(void)decoder_->decode(bp, cc, sbit, ebit, mba, gob, quant, mvdh,
			       mvdv);
	pb->release();
	/*
	 * If the stream changes format, issue a resize.
	 */
	if (decoder_->width() != inw_ || decoder_->height() != inh_) {
		inw_ = decoder_->width();
		inh_ = decoder_->height();
		Tcl& tcl = Tcl::instance();
		tcl.evalf("%s frame_width %d", encoder_->name(), inw_);
		outw_ = atoi(tcl.result());
		tcl.evalf("%s frame_height %d", encoder_->name(), inh_);
		outh_ = atoi(tcl.result());
		pixel_crinit(outw_, outh_);
		decoder_->marks(crvec_);
	}

	if (!(ntohs(rh->rh_flags) & RTP_M))
		return;

	/*
	 * Have a frame.
	 */
	decoder_->sync();

	double now;
	if (!txonly_) {
		if (bps_ == 0)
			return;
		now = gettimeofday();
	}

	if (fc_ <= now || txonly_) {
		/* If we have fallen behind (>200ms), re-sync. */
		if (now - fc_ > 200000.)
			fc_ = now;
		blk_to_mb_cr();
		YuvFrame vf(ntohl(rh->rh_ts),
			    ((P64Decoder*)decoder_)->frame(),
			    mb_crvec_, outw_, outh_);
		int nb = encoder_->nb();
		encoder_->recv(&vf);
		obytes_ = encoder_->nb();
		nb = obytes_ - nb;
		double bits = 8 * nb;
		lastfc_ = fc_;
		fc_ += 1e6 * bits / bps_;
		ofrms_++;

		decoder_->mark(age_blocks() | CR_MOTION_BIT | CR_LQ);
	}
}

#ifdef notyet
void H261DCTTranscoder::recv_data(pktbuf* pb)
{
	rtphdr* rh = (rtphdr*)(pb->dp);
	int cc = pb->len - sizeof(rtphdr);
	const u_char* bp = (const u_char*)(rh + 1);

	u_int v = ntohl(*(u_int*)(rh + 1));
	int sbit = v >> 29;
	int ebit = (v >> 26) & 7;
	int quant = (v >> 10) & 0x1f;
	int mvdh = (v >> 5) & 0x1f;
	int mvdv = v & 0x1f;
        int mba, gob;
	/*
	 * vic-2.7 swapped the GOB and MBA fields in the RTP packet header
	 * with respect to the spec.  To maintain backward compat, whenever
	 * we see an out of range gob, we change our assumption about the
	 * stream and continue.
	 */
	if (!h261_rtp_bug_) {
		mba = (v >> 15) & 0x1f;
		gob = (v >> 20) & 0xf;
		if (gob > 12) {
			h261_rtp_bug_ = 1;
			mba = (v >> 19) & 0x1f;
			gob = (v >> 15) & 0xf;
		}
	} else {
		mba = (v >> 19) & 0x1f;
		gob = (v >> 15) & 0xf;
		if (gob > 12) {
			h261_rtp_bug_ = 0;
			mba = (v >> 15) & 0x1f;
			gob = (v >> 20) & 0xf;
		}
	}

	int ibit = (v >> 25) & 1;
	if (ibit != ibit_) {
		ibit_ = ibit;
		configure();
	}

	bp += sizeof(h261hdr);
	cc -= sizeof(h261hdr);

	(void)decoder_->decode(bp, cc, sbit, ebit, mba, gob, quant, mvdh,
			       mvdv);
	pb->release();
	/*
	 * If the stream changes format, issue a resize.
	 */
	if (decoder_->width() != inw_ || decoder_->height() != inh_) {
		inw_ = decoder_->width();
		inh_ = decoder_->height();
		Tcl& tcl = Tcl::instance();
		tcl.evalf("%s frame_width %d", encoder_->name(), inw);
		outw_ = atoi(tcl.result());
		tcl.evalf("%s frame_height %d", encoder_->name(), inh);
		outh_ = atoi(tcl.result());
		crinit(outw_, outh_);
		decoder_->marks(crvec_);
	}

	if (!(ntohs(rh->rh_flags) & RTP_M))
		return;

	/*
	 * Have a frame.
	 */
	decoder_->sync();

	if (bps_ == 0)
		return;

	double now = gettimeofday();
	if (fc_ <= now) {
		/* If we have fallen behind (>200ms), re-sync. */
		if (now - fc_ > 200000.)
			fc_ = now;
		DCTFrame df(ntohl(rh->rh_ts),
			    (IntraP64DCTDecoder*)decoder_->frame(), crvec_,
			    outw_, outh_);
		int nb = encoder_->nb();
		encoder_->recv(&df);
		obytes_ = encoder_->nb();
		nb = obytes_ - nb;
		double bits = 8 * nb;
		lastfc_ = fc_;
		fc_ += 1e6 * bits / bps_;
		ofrms_++;

		mark_ = age_blocks() | CR_MOTION_BIT | CR_LQ;
	}
}
#endif
