/*
 * mb-time.h --
 *
 *      main header file for MediaBoard
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-time.h,v 1.7 2002/02/03 03:16:30 lim Exp $
 */

#ifndef MASH_MB_TIME_H
#define MASH_MB_TIME_H

#ifndef WIN32
#include <sys/time.h>
#endif

#include "assert.h"

/* for now, we use gettimeofday() values in 10 ms units */
typedef u_int32_t MBTime;

/* this value mean to display 'event's in any time
 * it should be a value that does not correspond to any real physical time */
const MBTime cMBTimeAny = 0UL;
const MBTime cMBTimePosInf = 0xFFFFFFFFUL;
const MBTime cMBTimeNegInf = 1UL;


/* Mediaboard time in seconds since UTC 1/1/70,
 *   in cycles of approx 0xffffffff/100 secs.
 *
 * Note that it will wrap around and the nearby dates are approx.:
 * 1)   Fri Mar 21 17:17:20 1997
 * 2)   Fri Jul 31 20:45:31 1998
 */

//const MBTime cMBTimeSecOffset = 865000000;

/* for debugging the conversion routines, either TEST_TIME of
 * TEST_TIME1 but not both (leads to infinite loop */

//#define TEST_TIME1

void tod2mb(const timeval& tv, MBTime& t);
void tod2mb(const long& sec, MBTime& t);

inline void mb2tod_(const MBTime& t, long& sec)
{
	sec = t/100;
}

inline void mb2tod(const MBTime& t, long& sec)
{
	mb2tod_(t, sec);
#ifdef TEST_TIME
	assert(t!=cMBTimeAny && t!=cMBTimeNegInf && t!=cMBTimePosInf);
	MBTime t1;
	tod2mb(sec, t1);
	/* we lost the portion after the decimal point, hence the /100's */
	assert(t/100 == t1/100 && "mb2tod is not reversible");
#endif /* TEST_TIME */
}

inline void mb2tod(const MBTime& t, timeval& tv)
{
	mb2tod_(t, tv.tv_sec);
	tv.tv_usec = (t % 100) * 10000;
#ifdef TEST_TIME
	assert(t!=cMBTimeAny && t!=cMBTimeNegInf && t!=cMBTimePosInf);
	MBTime t1;
	tod2mb(tv, t1);
	assert(t == t1 && "tod2mb is not reversible");
#endif /* TEST_TIME */
}

inline void mb2tod(const MBTime& t, double& d)
{
	timeval tv;
	mb2tod(t, tv);
	d = tv.tv_sec + tv.tv_usec*1e-6;
}

inline void tod2mb_sec_(const u_int32_t& sec, MBTime& t)
{
	t = sec*100;
}

inline void tod2mb_usec_(const u_int32_t& usec, MBTime& t)
{
	t = usec/10000;
}

inline void tod2mb(const long& sec, MBTime& t)
{
	tod2mb_sec_(sec, t);
	if (t<=cMBTimeNegInf) t++; /* avoid using special value */
	if (t==cMBTimePosInf) t--;
#ifdef TEST_TIME1
	timeval tv;
	mb2tod(t, tv);
	assert(tv.tv_sec == tv.tv_sec && "tod2mb is not reversible");
#endif  /* TEST_TIME */
}


inline void tod2mb(const timeval& tv, MBTime& t)
{
	MBTime t1;
	tod2mb_usec_(tv.tv_usec, t1);
	tod2mb_sec_(tv.tv_sec, t);
	t+=t1;
	if (t<=cMBTimeNegInf) t++; /* avoid using special value */
	if (t==cMBTimePosInf) t--;
#ifdef TEST_TIME1
	timeval tv1;
	mb2tod(t, tv1);
	assert(abs(tv.tv_usec/10000 - tv1.tv_usec/10000) < 2
		&& tv.tv_sec == tv1.tv_sec &&
	       "tod2mb is not reversible");
#endif  /* TEST_TIME */
}

inline MBTime tod2mb(const timeval& tv)
{
	MBTime t;
	tod2mb(tv, t);
	return t;
}

inline void tod2mb(const double d, MBTime& t)
{
	timeval tv;
	tv.tv_sec = (long)d;
	tv.tv_usec = (long)((d - tv.tv_sec)*1000000);
	tod2mb(tv, t);
}

#endif /* MASH_MB_TIME_H */
