/*
 * mbv2-item.h --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *  @(#) $Header: /usr/mash/src/repository/mash/mash-1/mbv2/mbv2-item.h,v 1.7 2002/02/03 03:17:08 lim Exp $
 */

#ifndef MASH_MBV2_ITEM_H
#define MASH_MBV2_ITEM_H

#include "mbv2.h"
#include "mbv2-adu.h"


struct MBv2ItemSpec {
	void (*function)(void *dest, const char *value);
	char *option;
	int offset;
};


class MBv2Item {
public:
	MBv2Item(MBv2ItemType type) : itemType_(type), numPoints_(0),
		points_(NULL) { }
	virtual ~MBv2Item();
	inline MBv2ItemType type() { return itemType_; }
	inline int num_points() { return numPoints_; }
	inline MBv2Point *points() { return points_; }
	inline void points(MBv2Point *points, int numPoints) {
		if (points_) delete [] points_;
		points_ = points;
		numPoints_ = numPoints;
	}

	Tcl_Obj *get_properties(Tcl_Interp *interp) {
		Tcl_Obj *props=Tcl_NewStringObj("", -1);
		if (!get_properties(interp, props)) {
			Tcl_DecrRefCount(props);
			return NULL;
		} else {
			return props;
		}
	}
	virtual Bool set_properties(int propsc, const char * const *propsv);

	virtual Bool extract(const Byte *pb, int len)=0;
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size()
	{ return (sizeof(ADU_Coords) + numPoints_*sizeof(MBv2Point)); }
	virtual MBv2Item *make_copy()=0;

	static MBv2Item *create(MBv2ItemType type, const Byte *pb, int len);
	static MBv2Item *create(MBv2ItemType type);

protected:
	static void set_int    (void *dest, const char *value);
	static void set_arrow  (void *dest, const char *value);
	static void set_string (void *dest, const char *value);
	static void set_color  (void *dest, const char *value);
	static void set_justify(void *dest, const char *value);

	void copy_points(MBv2Item *item);
	virtual MBv2ItemSpec *get_property_specs()=0;
	virtual Bool get_properties(Tcl_Interp *interp, Tcl_Obj *props)=0;
	Bool append_property(Tcl_Interp *interp, Tcl_Obj *props,
			     const char *propname, Tcl_Obj *propvalue) {
		if (Tcl_ListObjAppendElement(interp, props, Tcl_NewStringObj(
			(char*)propname, -1))==TCL_ERROR) return FALSE;
		if (Tcl_ListObjAppendElement(interp, props, propvalue)
		    ==TCL_ERROR) return FALSE;
		return TRUE;
	}

	Bool append_color(Tcl_Interp *interp, Tcl_Obj *props,
			  const char *propname, const MBv2Color &color) {
		char colorStr[16];
		if (color.a) {
			// transparent
			// don't append anything
			return TRUE;
		} else {
			sprintf(colorStr, "#%04x%04x%04x", color.r, color.g,
				color.b);
			return append_property(interp, props, propname,
					       Tcl_NewStringObj(colorStr, -1));
		}
	}

private:
	MBv2ItemType itemType_;
	int numPoints_;
	MBv2Point *points_;
};


class MBv2LineItem : public MBv2Item {
public:
	MBv2LineItem() : MBv2Item(MBv2Line), width_(1),
		arrowType_(MBv2ArrowNone) { }

private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size()
	{ return (MBv2Item::adu_size() + sizeof(ADU_LineItem)); }

	virtual MBv2Item *make_copy() {
		MBv2LineItem *item = new MBv2LineItem;
		*item = *this;
		item->copy_points(this);
		return item;
	}

	virtual MBv2ItemSpec *get_property_specs() { return specs_; }
	virtual Bool get_properties(Tcl_Interp *interp, Tcl_Obj *props);

	int width_;
	int arrowType_; /* this should be MBv2ArrowType, but I want to
			 * ensure that this is the size of an int */
	MBv2Color color_;

	static MBv2ItemSpec specs_[];
};


class MBv2PolyItem : public MBv2Item {
public:
	MBv2PolyItem(MBv2ItemType type) : MBv2Item(type), width_(1) { }

private:
	virtual Bool set_properties(int propsc, const char * const *propsv) {
		// assume the default fill is none
		fill_.a = 1;
		fill_.r = fill_.g = fill_.b = 0;
		return MBv2Item::set_properties(propsc, propsv);
	}

	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size()
	{ return (MBv2Item::adu_size() + sizeof(ADU_PolyItem)); }

	virtual MBv2Item *make_copy() {
		MBv2PolyItem *item = new MBv2PolyItem(type());
		*item = *this;
		item->copy_points(this);
		return item;
	}

	virtual MBv2ItemSpec *get_property_specs() { return specs_; }
	virtual Bool get_properties(Tcl_Interp *interp, Tcl_Obj *props);

	int width_;
	MBv2Color fill_, outline_;

	static MBv2ItemSpec specs_[];
};


class MBv2TextItem : public MBv2Item {
public:
	MBv2TextItem() : MBv2Item(MBv2Text), justify_(MBv2JustifyLeft),
		text_(NULL), font_(NULL), max_chars_(0) { }
	virtual ~MBv2TextItem() {
		if (text_) delete [] text_;
		if (font_) delete [] font_;
	}
	inline void set_text(char *t) { if (text_) delete [] text_; text_ = t;}
	inline const char *get_text() { return text_; }
	inline u_int16_t max_chars() { return max_chars_; }
	inline void max_chars(u_int16_t m) { max_chars_ = m; }

private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size()
	{ return (MBv2Item::adu_size() + sizeof(ADU_TextItem) +
		  strlen(font_) + 1); }

	virtual MBv2Item *make_copy();

	virtual MBv2ItemSpec *get_property_specs() { return specs_; }
	virtual Bool get_properties(Tcl_Interp *interp, Tcl_Obj *props);

	MBv2Color color_;
	MBv2JustifyType justify_;
	char *text_;
	char *font_;
	u_int16_t max_chars_;

	static MBv2ItemSpec specs_[];
};


class MBv2FileItem : public MBv2Item {
public:
	MBv2FileItem(MBv2ItemType type)
		: MBv2Item(type), filename_(NULL), filesize_(-1) { }
	virtual ~MBv2FileItem() { if (filename_) delete [] filename_; }
	void filename(const char *name) {
		if (filename_) delete [] filename_;
		filename_ = new char [strlen(name)+1];
		strcpy(filename_, name);
	}
	const char *filename() { return filename_; }

protected:
	char *filename_;
	int filesize_;

private:
	virtual Bool extract(const Byte *pb, int len);
	virtual Byte *packetize(Byte *pb);
	virtual int adu_size();
};


class MBv2ImageItem : public MBv2FileItem {
public:
	MBv2ImageItem() : MBv2FileItem(MBv2Image), imagename_(NULL) { }
	virtual ~MBv2ImageItem() { delete_image(); }
	void imagename(const char *name) {
		if (imagename_) delete [] imagename_;
		imagename_ = new char [strlen(name)+1];
		strcpy(imagename_, name);
	}
	const char *imagename() { return imagename_; }

private:
	virtual MBv2Item *make_copy();
	virtual MBv2ItemSpec *get_property_specs() { return specs_; }
	virtual Bool get_properties(Tcl_Interp *interp, Tcl_Obj *props);
	Bool delete_image();
	Bool create_image();

	char *imagename_;

	static MBv2ItemSpec specs_[];
};



#endif /* #ifdef MASH_MBV2_ITEM_H */
