/*
 * module-pkt_q.cc --
 *
 *      This file contains the definition and methods of the class
 *      PacketQModule.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
#include "tclcl.h"
#include "pktbuf.h"
#include "codec/module.h"
#include "rtp/media-timer.h"

/*
 * The PacketQModule can be used to store a stream of
 * packet buffers into a FIFO queue which can then be
 * dumped to another PacketModule derived object on command.
 * This class corresponds to the OTcl class "Module/PacketQ"
 */

class PacketQModule : public PacketModule {
public:
  PacketQModule() : PacketModule(), head_(0), tail_(0) {};
  void recv(pktbuf *pb);
  virtual int command(int argc, const char*const* argv);

protected:
  pktbuf *head_;
  pktbuf *tail_;
};

/*
 * Recv entry method of incoming packet buffers.
 * Packet buffers are simply stored in a queue until the
 * dump command is given on the OTcl interface at which
 * point the packets are forwarded to the target_'s recv
 * method.
 */

void
PacketQModule::recv(pktbuf *pb)
{
  if (head_ == 0) {
    head_ = tail_ = pb;
    pb->next = 0;
    return;
  }
  tail_->next = pb;
  pb->next = 0;
  tail_ = pb;
}

/*
 * OTcl interface to Module/PacketQ. Provides the
 * "dump" command which takes no arguments and causes
 * all currently queued packets to be forwarded to the
 * target.
 */

int
PacketQModule::command(int argc, const char*const* argv)
{
  if (argc == 2) {
    if (strcmp(argv[1], "dump") == 0) {
      if (target_ != 0) {
	while(head_ != 0) {
	  pktbuf *pb;

	  pb = head_;
	  head_ = head_->next;
	  pb->next = 0;

	  target_->recv(pb);
	}
	tail_ = head_ = 0;
      } else {
	while(head_ != 0) {
	  pktbuf *pb;

	  pb = head_;
	  head_ = head_->next;
	  pb->next = 0;

	  pb->release();
	}
	tail_ = head_ = 0;
      }
      return TCL_OK;
    }
  }
  return (PacketModule::command(argc, argv));
}

/*
 * Binding between the C++ class PacketQModule and
 * the OTcl class Module/PacketQ.
 */

static class PacketQModuleClass : public TclClass {
public:
  PacketQModuleClass() : TclClass("Module/PacketQ") {}
  TclObject *create(int argc, const char*const* argv) {
    return (new PacketQModule());
  }
} sc_exp_pkt_q_;

