# ui-progress.tcl --
#
#       a message box for displaying progress of some action
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#   @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/ui_tools/ui-progress.tcl,v 1.8 2002/02/03 04:30:23 lim Exp $


import MessageBox


WidgetClass ProgressBar -configspec {
	{ -min min Min 0 config_min cget_min }
	{ -max max Max 100 config_max cget_max }
	{ -value val Val 0 config_value cget_value }
	{ -width width Width 200 config_width cget_width }
	{ -height height Height 20 config_height cget_height }
}


ProgressBar private build_widget { path } {
	$self set min_ 0
	$self set max_ 0
	$self set value_ 0

	canvas $path.canvas
	bind $path.canvas <Configure> "$self refresh"
	pack $path.canvas -side top -fill both -expand true
}


ProgressBar private config_min { option value } {
	$self instvar min_ max_
	set min_ $value
	if { $min_ > $max_ } { set max_ $min_ }

	$self do_when_idle "$self refresh"
}


ProgressBar private cget_min { option } {
	return [$self set min_]
}


ProgressBar private config_max { option value } {
	$self instvar min_ max_
	set max_ $value
	if { $min_ > $max_ } { set min_ $max_ }

	$self do_when_idle "$self refresh"
}


ProgressBar private cget_max { option } {
	return [$self set max_]
}


ProgressBar private config_value { option v } {
	$self instvar min_ max_
	if { $v < $min_ } { set v $min_ }
	if { $v > $max_ } { set v $max_ }

	$self set value_ $v
	$self do_when_idle "$self refresh"
}


ProgressBar private cget_value { option } {
	return [$self set value_]
}


ProgressBar instproc config_height { option height } {
	$self subwidget canvas configure -height $height
	$self do_when_idle "$self refresh"
}


ProgressBar instproc cget_height { option } {
	return [$self subwidget canvas cget -height]
}


ProgressBar instproc config_width { option width } {
	$self subwidget canvas configure -width $width
	$self do_when_idle "$self refresh"
}


ProgressBar instproc cget_width { option } {
	return [$self subwidget canvas cget -width]
}


ProgressBar instproc refresh { } {
	$self instvar min_ max_ value_

	set canv [$self subwidget canvas]
	set width  [winfo width $canv]
	set height [winfo height $canv]

	if {""==[$canv find withtag border]} {
		set t [$canv create rectangle 2 2 2 2]
		$canv addtag border withtag $t
	}
	$canv coords border 2 2 [expr $width - 2] [expr $height - 2]

	if {""==[$canv find withtag progress]} {
		set t [$canv create rectangle 2 2 2 2 -fill blue \
				-outline blue]
		$canv addtag progress withtag $t
	}

	if { $max_==$min_ } {
		set w 4
	} else {
		set w [expr 4 + ($width - 8)*($value_ - $min_)/($max_ - $min_)]
	}
	$canv coords progress 4 4 $w [expr $height - 4]
}




# FIXME: ProgressBox should use ProgressBar as a building block

WidgetClass ProgressBox -superclass MessageBox -configspec {
	{ -min min Min 0 config_min cget_min }
	{ -max max Max 100 config_max cget_max }
	{ -text text Text {} config_text cget_text }
	{ -value val Val 0 config_value cget_value }
	{ -width width Width 200 config_width cget_width }
	{ -height height Height 20 config_height cget_height }
	{ -type type Type none config_type cget_type }
}

ProgressBox instproc build_widget { path } {
	$self next $path
	canvas $path.progcanv
	pack $path.progcanv -side top -fill both -expand true
}

ProgressBox instproc config_text {args} {
	eval $self next $args
#	set path [$self info path]
#	wm geom $path [winfo reqwidth $path]x[winfo reqheight $path]
}

ProgressBox instproc config_min { option value } {
	$self set min_ $value
}

ProgressBox instproc cget_min { option } {
	return [$self set min_]
}

ProgressBox instproc config_max { option value } {
#	puts "setting max to $value"
	$self set max_ $value
}

ProgressBox instproc cget_max { option } {
	return [$self set max_]
}

ProgressBox instproc config_value { option v } {
	$self set value_ $v
	$self do_when_idle "$self refresh"
}

ProgressBox instproc cget_value { option } {
	return [$self set value_]
}

ProgressBox instproc config_height { option height } {
	$self set height_ $height
	[$self subwidget progcanv] configure -height $height
	$self do_when_idle "$self refresh"
}

ProgressBox instproc cget_height { option } {
	return [$self set height_]
}

ProgressBox instproc config_width { option width } {
	$self set width_ $width
	[$self subwidget progcanv] configure -width $width
	$self do_when_idle "$self refresh"
}

ProgressBox instproc cget_width { option } {
	return [$self set width_]
}

ProgressBox instproc refresh { } {
#	puts "in refresh"
	$self instvar min_ max_ value_ width_ height_
	if {!( [info exists min_] && [info exists max_] \
			&& [info exists value_] && [info exists width_] \
			&& [info exists height_] ) } {
		return
	}
	set canv [$self subwidget progcanv]
	$canv configure -width $width_ -height $height_
	if {""==[$canv find withtag border]} {
		set t [$canv create rectangle 2 2 2 2]
		$canv addtag border withtag $t
	}
	$canv coords border 2 2 [expr $width_ - 2] [expr $height_ - 2]

	if {""==[$canv find withtag progress]} {
		set t [$canv create rectangle 2 2 2 2 -fill blue \
				-outline blue]
		$canv addtag progress withtag $t
	}
	if { $max_==$min_ } {
		set w 4
	} else {
		set w [expr 4 + ($width_ - 8)*($value_ - $min_)/($max_-$min_)]
	}
#	puts "$w=w width_=$width_ min_=$min_ max=$max_ v=$value_"
	$canv coords progress 4 4 $w [expr $height_ - 4]
}

