# ui-srclist.tcl --
#
#       A toplevel window for displaying a list of the sources currently
#       participating in the session.
#
# Copyright (c) 1993-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#
# A toplevel window for displaying a list of the sources currently participating in the session.
#
Class UISrcListWindow -superclass TopLevelWindow

#
# Instantiate, but do not yet display or iconify, a dismissable,
# scrollable toplevel titled "Video Participants" using the provided
# widgetpath, <i>w</i>.  To map this window, toggle it.  In this window,
# list the sources currently participating in the session.
#FIXME allow all/or non-senders to be shown
#
UISrcListWindow public init {w videoAgent} {
	$self instvar nSRCLIST_
	set nSRCLIST_ 0
	$self next $w\X$nSRCLIST_

	#
	# create at startup since we manipulate the
	# tk windows on the fly
	#
	set win $w\X$nSRCLIST_
	$self create-window $win "Video Participants"
	wm geometry $win 300x320
	wm minsize $win 0 0
	new UISrcList $win $videoAgent

	incr nSRCLIST_
}

#
# A display of a list of the sources currently participating in the session.
#
Class UISrcList -superclass {TkWindow Observer}

#
# Within the provided widget, <i>w</i>, create and pack a frame in that
# lists the sources currently participating in the session.
#
UISrcList public init {w videoAgent} {
	$self instvar bottom_
	set bottom_ 2

	$self instvar srclist_
	frame $w.b -borderwidth 2 -relief sunken
	scrollbar $w.b.scroll -relief groove -borderwidth 2 \
			-command "$w.b.list yview"
	#
	# Note that we make the canvas dimensions small so that
	# if the user resizes the window, the dismiss button
	# doesn't disappear.  The default size is set by the
	# wm geometry command above.
	#
	canvas $w.b.list -relief groove -borderwidth 0 \
			-height 10 -width 10 -yscrollcommand "$w.b.scroll set"

	set srclist_ $w.b.list

	button $w.ok -text " Dismiss " -borderwidth 0 -relief raised \
			-command "wm withdraw $w" -font [$self get_option medfont]

	pack $w.b -fill both -expand 1
	pack $w.b.scroll -side left -fill y
	pack $w.b.list -side left -expand 1 -fill both
	pack $w.ok -fill x

	wm protocol $w WM_DELETE_WINDOW "wm withdraw $w"

	# let this be an observer of the VideoAgent so it can catch "register" and 
	# "unregister"
	$videoAgent attach $self
}

#
# Register a src in the src display.  This happens for all the
# srcs participating in the conferencing (e.g., including
# those not actually sending video).
#
UISrcList instproc register src {
	$self next
	$self instvar nametag_ srclist_ bottom_ srcstate_
	set srcstate_($src) 1
	set f [$self get_option medfont]
	set nametag_($src) [$srclist_ create text 5 $bottom_  \
			-font $f -text [$src addr] -anchor nw ]
	set bottom_ [lindex [$srclist_ bbox $nametag_($src)] 3]
	incr bottom_ 2
	$srclist_ config -scrollregion "0 0 2.5i $bottom_"
}

#
# Change the name of the source, <i>src</i> in the list if it exists.
#
UISrcList instproc change_name src {
	$self instvar srclist_ nametag_
	# change name in source list if it exists
	if [info exists nametag_($src)] {
		$srclist_ itemconfigure $nametag_($src) -text [$src getid]
	}
}

#
#
#
UISrcList instproc adjustNames { thresh h } {
	$self instvar nametag_ srclist_ bottom_
	foreach s [array names nametag_] {
		set y [lindex [$srclist_ coords $nametag_($s)] 1]
		if { $y > $thresh } {
			$srclist_ move $nametag_($s) 0 -$h
		}
	}
	incr bottom_ -$h
	$srclist_ config -scrollregion "0 0 2.5i $bottom_"
}

#
# Remove source, <i>src</i>, from the display.  This happens when
# a party quits or has been dead for a sufficiently long time.
# We assume the source has already been deactivated.
#
UISrcList instproc unregister src {
	$self instvar nametag_ srclist_
	#FIXME
	global name_line info_line

	#FIXME
	destroy_rtp_stats $src

	#
	# Delete name_line and info_line if they exist.
	# They might not be created until the first NAME
	# report (and they aren't used until a source is activated).
	#
	if [info exists name_line($src)] {
		unset name_line($src)
		unset info_line($src)
	}

	set thresh [lindex [$srclist_ coords $nametag_($src)] 1]
	set bb [$srclist_ bbox $nametag_($src)]
	set height [expr [lindex $bb 3] - [lindex $bb 1]]
	incr height 2
	$srclist_ delete $nametag_($src)
	unset nametag_($src)
	$self adjustNames $thresh $height
}

#
#
#
UISrcList instproc trigger_idle src {
	$self instvar nametag_ srclist_ srcstate_
	if [info exists nametag_($src)] {
		if [$src lost] {
			$srclist_ itemconfigure $nametag_($src) -stipple gray50
			set srcstate_($src) 2
		} else {
			$srclist_ itemconfigure $nametag_($src) -stipple {}
			set srcstate_($src) 1
		}
	}
}
