/*
 * video-directx.cc --
 *
 *      This file implements a DirectX screen capture device for Windows.
 *
 * Copyright (c) 2001-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <ddraw.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

#include "video-device.h"
#include "Tcl.h"
#include "module.h"
#include "rgb-converter.h"

#define PAL	    0
#define NTSC	    1
#define NTSC_WIDTH  640
#define NTSC_HEIGHT 480
#define PAL_WIDTH   768
#define PAL_HEIGHT  576
#define CIF_WIDTH   352
#define CIF_HEIGHT  288

#define CF_422 0
#define CF_411 1
#define CF_CIF 2

#define VIDEO_PALETTE_YUV422	1
#define VIDEO_PALETTE_YUV422P	2
#define VIDEO_PALETTE_UYVY		4
#define VIDEO_PALETTE_YUYV		8

class VideoCaptureDirectx : public VideoCapture {
public:
    VideoCaptureDirectx(const char * cformat, const char *dev);
    virtual ~VideoCaptureDirectx();
    void dealloc(); // used by destructor, and to dealloc objects on error
    
    virtual int  command(int argc, const char*const* argv);
    virtual void start();
    virtual void stop();
    virtual void grab();
protected:
    int capture();
    void format();
    void setsize();
    
    LPDIRECTDRAW lpDD;
    LPDIRECTDRAWSURFACE lpDDSPrime;
    DDSURFACEDESC DDSdesc;
    
    RGB_Converter * converter_;
    
    int format_;
    int cformat_;
    int width_;
    int height_;
    int screen_width_;
    int screen_height_;
    int screen_pitch_;
    int decimate_;
    char *tempBuffer;
};

class DirectxDevice : public VideoDevice {
public:
    DirectxDevice(const char* clsname, const char* nickname,
	const char* devname, const char* attr, int free);
    TclObject* create(int argc, const char*const* argv) {
	if (argc != 5)
	    abort();/*FIXME*/
	if (strcmp(argv[4], "422") == 0)
	    return (new VideoCaptureDirectx("422",name_));
	else if (strcmp(argv[4], "411") == 0)
	    return (new VideoCaptureDirectx("411",name_));
	else if (strcmp(argv[4], "cif") == 0)
	    return (new VideoCaptureDirectx("cif",name_));
	return (0);
    }
protected:
    const char* name_;
};

DirectxDevice::DirectxDevice(const char* clsname, const char* nickname,
			     const char *devname, const char* attr, int free)
			     : VideoDevice(clsname, nickname), name_(devname)
{
    if (free)
	attributes_ = attr;
    else
	attributes_ = "disabled";
}

class DirectxScanner {
public:
    DirectxScanner(const char **dev);
};

static DirectxScanner find_directx_devices(NULL);

DirectxScanner::DirectxScanner(const char **dev)
{
    const char *directxclassnames[] = {"VideoCapture/DirX"};
    
    char * attr = new char[512];
    strcpy(attr,"format { 411 422 cif } ");
    strcat(attr,"size { small cif large } ");
    strcat(attr,"port { screen0 } norm { auto } ");
    
    char * nick = new char[strlen("Screen0")+10];
    sprintf(nick,"DirectX: %s","Screen0");
    new DirectxDevice(directxclassnames[0],nick,"devname",attr,1);
}

/* ----------------------------------------------------------------- */

VideoCaptureDirectx::VideoCaptureDirectx(const char *cformat, const char *dev)
{
    int i=0;

    lpDD = NULL;
    lpDDSPrime = NULL;
    memset(&DDSdesc, 0, sizeof(DDSURFACEDESC) );

    width_=0;
    height_=0;
    screen_width_ = 0;
    screen_height_ = 0;
    screen_pitch_ = 0;
    converter_ = NULL;
    decimate_ = 2;
    tempBuffer = NULL;
    
    HRESULT ddrval = DirectDrawCreate(0, &lpDD, 0);
    if (ddrval != DD_OK) {
	dealloc(); // unalloc objects
	return;
    }
    
    ddrval = lpDD->SetCooperativeLevel(NULL, DDSCL_NORMAL);
    if (ddrval != DD_OK) {
	dealloc(); // unalloc objects
	return;
    }
    
    ZeroMemory(&DDSdesc, sizeof(DDSdesc));
    DDSdesc.dwSize = sizeof(DDSdesc);
    DDSdesc.dwFlags = DDSD_CAPS;
    DDSdesc.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
    ddrval = lpDD->CreateSurface(&DDSdesc, &lpDDSPrime, 0);
    if (ddrval != DD_OK) {
	dealloc(); // unalloc objects
	return;
    }
    
    ZeroMemory(&DDSdesc, sizeof(DDSdesc));
    DDSdesc.dwSize = sizeof(DDSdesc);
    DDSdesc.dwFlags = DDSD_ALL;
    ddrval = lpDDSPrime->GetSurfaceDesc(&DDSdesc);
    if (ddrval != DD_OK) {
	dealloc(); // unalloc objects
	return;
    }
    
    screen_pitch_ = DDSdesc.lPitch;
    screen_width_ = DDSdesc.dwWidth;
    screen_height_ = DDSdesc.dwHeight;
    
    // fill in defaults
    if(!strcmp(cformat, "411"))
	cformat_ = CF_411;
    else if(!strcmp(cformat, "422"))
	cformat_ = CF_422;
    else if(!strcmp(cformat, "cif"))
	cformat_ = CF_CIF;
    else {
	cformat_ = 0; // this is an error case that shouldn't occur
	static bool warnShowed = false;
	if( !warnShowed ) {
	    printf("warning: directx capture has unrecognized cformat: %s\n", cformat);
	    warnShowed = true;
	}
    }
    
    tempBuffer = new char[screen_width_*screen_height_*(DDSdesc.ddpfPixelFormat.dwRGBBitCount>>3)];
    if( NULL==tempBuffer ) {
	printf("DirectX capture buffer allocation failed\n");
	dealloc(); // unalloc objects
	return;
    }
    
    if( DDSdesc.ddpfPixelFormat.dwFlags&DDPF_RGB) {
	printf("Direct X capture RGB bitmask is r=%08x, g=%08x, b=%08x\n", 
	    DDSdesc.ddpfPixelFormat.dwRBitMask,
	    DDSdesc.ddpfPixelFormat.dwGBitMask,
	    DDSdesc.ddpfPixelFormat.dwBBitMask);

	int useBGR = -1;
	if( DDSdesc.ddpfPixelFormat.dwBBitMask&0x1 ) {
	    printf("DirectX capture using RGB\n");
	    useBGR = 0;
	}
	if( DDSdesc.ddpfPixelFormat.dwRBitMask&0x1 ) {
	    printf("DirectX capture using BGR\n");
	    useBGR = 1;
	}
	if( -1==useBGR ) {
	    printf("DirectX capture using RGB (unrecognized)\n");
	    useBGR = 0;
	}
	switch(cformat_) {
	case CF_411:
	    converter_ = new RGB_Converter_411(DDSdesc.ddpfPixelFormat.dwRGBBitCount, 
		NULL, 0, useBGR);
	    break;
	case CF_422:
	    converter_ = new RGB_Converter_422(DDSdesc.ddpfPixelFormat.dwRGBBitCount, 
		NULL, 0, useBGR);
	    break;
	case CF_CIF:
	    converter_ = new RGB_Converter_411(DDSdesc.ddpfPixelFormat.dwRGBBitCount, 
		NULL, 0, useBGR);
	    break;
	default:
	    converter_ = NULL;
	    break;
	}
    }
    else {
	printf("DirectX capture buffer got a non-RGB buffer\n");
	dealloc(); // unalloc objects
    }    
}

void VideoCaptureDirectx::dealloc()
{
    if(tempBuffer!=NULL) {
	delete tempBuffer;
	tempBuffer=NULL;
    }
    if (lpDDSPrime != NULL)
    {
	lpDDSPrime->Release();
	lpDDSPrime = NULL;
    }
    if (lpDD != NULL)
    {
	lpDD->Release();
	lpDD = NULL;
    }
}

VideoCaptureDirectx::~VideoCaptureDirectx()
{
    dealloc();
}

int VideoCaptureDirectx::command(int argc, const char*const* argv)
{
    if (argc == 3) {
	if (strcmp(argv[1], "decimate") == 0) {
	    decimate_ = atoi(argv[2]);
	    if (running_)
		format();
	}
	if (strcmp(argv[1], "norm") == 0) {
	    return (TCL_OK);
	}
    }
    return (VideoCapture::command(argc, argv));
}

void VideoCaptureDirectx::start()
{
    format();
    VideoCapture::start();
}

void VideoCaptureDirectx::stop()
{
    VideoCapture::stop();
}

#include "rgb-converter.h"

int VideoCaptureDirectx::capture()
{
    // don't capture if the initialization process was not successful
    if ( !lpDDSPrime ) {
	return 0;
    }

    // lock the surface so can get a pointer to the memory

    DDSURFACEDESC surfdesc;
    ZeroMemory(&surfdesc, sizeof(surfdesc)); 
    surfdesc.dwSize = sizeof(surfdesc);
#ifdef DDLOCK_DONOTWAIT
    HRESULT ddrval = lpDDSPrime->Lock(NULL, &surfdesc, DDLOCK_READONLY | //DDLOCK_WAIT, 0);
	DDLOCK_DONOTWAIT, 0);
#else
    HRESULT ddrval = lpDDSPrime->Lock(NULL, &surfdesc, DDLOCK_READONLY, 0);
#endif
    if (ddrval != DD_OK) {
	return 0;
    }

    // copy memory to a local buffer (to minimize time that the lock is held)
    if( 0==screen_pitch_ ) {
	memcpy(tempBuffer, (char*)surfdesc.lpSurface, 
	    screen_width_*screen_height_*(DDSdesc.ddpfPixelFormat.dwRGBBitCount>>3));
    }
    else { // need to skip over pitch
	for(int i=0; i<screen_height_; ++i) {
	    memcpy( tempBuffer+(i*screen_width_*(DDSdesc.ddpfPixelFormat.dwRGBBitCount>>3)),
		(char*)surfdesc.lpSurface+(i*screen_pitch_),
		screen_width_*(DDSdesc.ddpfPixelFormat.dwRGBBitCount>>3));
	}
    }
    
    // And unlock the primary surface!
    lpDDSPrime->Unlock(surfdesc.lpSurface);
    
    converter_->convert((u_int8_t *)tempBuffer, screen_width_, screen_height_,
	frame_, width_, height_, 0);
    
    return (1);
}

void VideoCaptureDirectx::grab()
{
    if (capture() == 0)
	return;
    
    suppress(frame_);
    saveblks(frame_);
    YuvFrame f(media_ts(), frame_, crvec_, outw_, outh_);
    target_->recv(&f);
}

void VideoCaptureDirectx::format()
{
    width_  = CIF_WIDTH  *2  / decimate_;
    height_ = CIF_HEIGHT *2  / decimate_;
    switch (cformat_) {
    case CF_CIF:
	set_size_cif(width_, height_);
	//DEBUG(fprintf(stderr," cif"));
	break;
    case CF_411:
	set_size_411(width_, height_);
	//DEBUG(fprintf(stderr," 411"));
	break;
    case CF_422:
	set_size_422(width_, height_);
	//DEBUG(fprintf(stderr," 422"));
	break;
    }
    allocref();
}
