;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;



; Miscellany.

(defun memq (item list)
  (do* ((tail list (%cdr tail)))
       ((null tail))
    (if (eq item (car tail))
      (return tail))))

(defun assq (item list)
  (dolist (pair list)
    (when (and pair (eq item (car pair)))
      (return pair))))


(defun append-2 (y z)
  (if (null y)
    z
    (let* ((new (cons (car y) nil))
           (tail new))
      (declare (list new tail))
      (dolist (head (cdr y))
        (setq tail (cdr (rplacd tail (cons head nil)))))
      (rplacd tail z)
      new)))



(defun make-point (vh &optional v)
  (if v
    (macrolet ((signed-byte-16-p (x)
                 `(and (fixnump ,x)
                       (locally (declare (fixnum ,x))
                         (and (<= ,x #x7fff)
                              (>= ,x #x-8000))))))
      (if (and (signed-byte-16-p vh)
               (signed-byte-16-p v))
        (locally (declare (fixnum vh v))
          (if (and (< v (ash (1+ most-positive-fixnum) -16))
                   (>= v (ash most-negative-fixnum -16)))
            (logior (the fixnum (logand #xffff vh))
                    (the fixnum (ash v 16)))
            (logior (the fixnum (logand #xffff vh))
                    (ash v 16))))
        (cons (require-type vh 'integer) (require-type v 'integer))))
    (if (consp vh)
      (progn
        (require-type (%car vh) 'integer)
        (require-type (%cdr vh) 'integer)
        vh)
      (require-type vh 'integer))))


(defun point-h (p)
  (if (consp p)
    (require-type (%car p) 'integer)
    (integer-point-h p)))



(defun point-v (p)
  (if (consp p)
    (require-type (%cdr p) 'integer)
    (integer-point-v p)))

(defun add-points (pt1 pt2)
  (make-point (+ (point-h pt1) (point-h pt2))
              (+ (point-v pt1) (point-v pt2))))

(defun subtract-points (pt1 pt2)
  (make-point (- (point-h pt1) (point-h pt2))
              (- (point-v pt1) (point-v pt2))))



(defun dbg (&optional arg)
  (dbg arg))


; This takes a simple-base-string and passes a C string into
; the kernel "Bug" routine.  Not too fancy, but neither is #_DebugStr,
; and there's a better chance that users would see this message.
(defun bug (arg)
  (if (typep arg 'simple-base-string)
    (let* ((len (length arg)))
      (%stack-block ((buf (1+ len)))
        (%copy-ivector-to-ptr arg 0 buf 0 len)
        (setf (%get-byte buf len) 0)
        (ff-call 
         (%kernel-import arch::kernel-import-lisp-bug)
         :address buf
         :void)))
    (bug "Bug called with non-simple-base-string.")))

(defun total-bytes-allocated ()
  (+ (unsignedwide->integer *total-bytes-freed*)
     (%heap-bytes-allocated)))

(defun %freebytes ()
  (%normalize-areas)
  (let ((res 0))
    (do-consing-areas (area)
      (let ((bytes (ash (- (%fixnum-ref area arch::area.high)
                           (%fixnum-ref area arch::area.active))
                        arch::fixnum-shift)))
        (incf res bytes)))
    res))

(defun %reservedbytes ()
  (let* ((reserved-area (%get-kernel-global 'arch::all-areas)))
    (ash (- (logand #xffffffff (%fixnum-ref reserved-area arch::area.high))
	    (logand #xffffffff (%fixnum-ref reserved-area arch::area.low)))
	 arch::fixnum-shift)))

(defun object-in-application-heap-p (address)
  (declare (ignore address))
  t)


(defun %usedbytes ()
  (%normalize-areas)
  (let ((static 0)
        (dynamic 0)
        (library 0))
    (do-consing-areas (area)
      (let* ((active (%fixnum-ref area arch::area.active))
             (bytes (ash (- active (%fixnum-ref area arch::area.low))
                         arch::fixnum-shift))
             (code (%fixnum-ref area arch::area.code)))
        (when (object-in-application-heap-p active)
          (if (eql code arch::area-dynamic)
            (incf dynamic bytes)
            (if (eql code arch::area-staticlib)
              (incf library bytes)
              (incf static bytes))))))
    (values dynamic static library)))



(defun %stack-space ()
  (%normalize-areas)
  (let ((free 0)
        (used 0))
    (do-gc-areas (area)
      (when (member (%fixnum-ref area arch::area.code)
                    '(#.arch::area-vstack
                      #.arch::area-cstack
                      #.arch::area-tstack))
        (let ((active (%fixnum-ref area arch::area.active))
              (high (%fixnum-ref area arch::area.high))
              (low (%fixnum-ref area arch::area.low)))
          (declare (fixnum active high low))
          (incf used (ash (abs (- high active)) arch::fixnum-shift))
          (incf free (ash (abs (- active low)) arch::fixnum-shift)))))
    (values (+ free used) used free)))



; Returns an alist of the form:
; ((stack-group cstack-free cstack-used vstack-free vstack-used tstack-free tstack-used)
;  ...)
(defun %stack-space-by-stack-group ()
  (let* ((res nil))
    (without-interrupts
     (do-unexhausted-stack-groups (sg)
       (push (cons sg (multiple-value-list (%stack-group-stack-space sg))) res)))
    res))



; Returns six values.
;   sp free
;   sp used
;   vsp free
;   vsp used
;   tsp free
;   tsp used
(defun %stack-group-stack-space (&optional (sg *current-stack-group*))
  (when (eq sg *current-stack-group*)
    (%normalize-areas))
  (labels ((free-and-used (area)
             (let* ((low (%fixnum-ref area arch::area.low))
                    (high (%fixnum-ref area arch::area.high))
                    (active (%fixnum-ref area arch::area.active))
                    (free (ash (- active low) arch::fixnum-shift))
                    (used (ash (- high active) arch::fixnum-shift)))
               (declare (fixnum low high active))
               (loop
                 (setq area (%fixnum-ref area arch::area.older))
                 (when (eql area 0) (return))
                 (let ((low (%fixnum-ref area arch::area.low))
                       (high (%fixnum-ref area arch::area.high)))
                   (declare (fixnum low high))
                   (incf used (ash (- high low) arch::fixnum-shift))))
               (values free used))))
    (multiple-value-bind (cf cu) (free-and-used (sg.cs-area sg))
      (multiple-value-bind (vf vu) (free-and-used (sg.vs-area sg))
        (multiple-value-bind (tf tu) (free-and-used (sg.ts-area sg))
          (values cf cu vf vu tf tu))))))


(defun room (&optional (verbose :default))
  (let* ((freebytes (%freebytes)))
    (format t "~&There are at least ~:D bytes of available RAM.~%"
            (+ freebytes))
    (when verbose
      (multiple-value-bind (usedbytes static-used staticlib-used) (%usedbytes)
        (let* ((lispheap  (+ freebytes usedbytes))
	       (reserved (%reservedbytes))
               (static (+ static-used staticlib-used)))
          (flet ((k (n) (round n 1024)))
            (princ "
                  Total Size             Free                 Used")
            (format t "~&Lisp Heap: ~12T~10D (~DK)  ~33T~10D (~DK)  ~54T~10D (~DK)"
                    lispheap (k lispheap)
                    freebytes (k freebytes)
                    usedbytes (k usedbytes))
            (multiple-value-bind (stack-total stack-used stack-free)
                                 (%stack-space)
              (format t "~&Stacks:    ~12T~10D (~DK)  ~33T~10D (~DK)  ~54T~10D (~DK)"
                      stack-total (k stack-total)
                      stack-free (k stack-free)
                      stack-used (k stack-used)))
            (format t "~&Static: ~12T~10D (~DK)  ~33T~10D (~DK) ~54T~10D (~DK)"
                    static (k static)
                    0 0
                    static (k static))
	    (format t "~&~,3f MB reserved for heap expansion."
		    (/ reserved (float (ash 1 20))))
            (unless (eq verbose :default)
              (terpri)
              (dolist (sg-info (%stack-space-by-stack-group))
                (destructuring-bind (sg sp-free sp-used vsp-free vsp-used tsp-free tsp-used)
                                    sg-info
                  (let ((sp-total (+ sp-used sp-free))
                        (vsp-total (+ vsp-used vsp-free))
                        (tsp-total (+ tsp-used tsp-free)))
                    (format t "~%~a~%  cstack:~12T~10D (~DK)  ~33T~10D (~DK)  ~54T~10D (~DK)~
                               ~%  vstack:~12T~10D (~DK)  ~33T~10D (~DK)  ~54T~10D (~DK)~
                               ~%  tstack:~12T~10D (~DK)  ~33T~10D (~DK)  ~54T~10D (~DK)"
                            (sg.name sg)
                            sp-total (k sp-total) sp-free (k sp-free) sp-used (k sp-used)
                            vsp-total (k vsp-total) vsp-free (k vsp-free) vsp-used (k vsp-used)
                            tsp-total (k tsp-total) tsp-free (k tsp-free) tsp-used (k tsp-used)))))))))))
  (values))


(defun list-length (l)
  (do* ((n 0 (+ n 2))
        (fast l (cddr fast))
        (slow l (cdr slow)))
       ((null fast) n)
    (declare (fixnum n))
    (if (null (cdr fast))
      (return (the fixnum (1+ n)))
      (if (and (eq fast slow)
               (> n 0))
        (return nil)))))



(defun length (seq)
  (let* ((typecode (typecode seq)))
    (declare (fixnum typecode))
    (if (= typecode arch::tag-list)
      (or (list-length seq)
          (%err-disp $XIMPROPERLIST seq))
      (if (= typecode arch::subtag-vectorH)
        (%svref seq arch::vectorH.logsize-cell)
        (if (> typecode arch::subtag-vectorH)
          (uvsize seq)
          (report-bad-arg seq 'sequence))))))
(defun %str-from-ptr (pointer len)
  (%copy-ptr-to-ivector pointer 0 (make-string len :element-type 'base-char) 0 len))

(defun %get-cstring (pointer &optional (offset 0) (end offset))
  (with-macptrs ((p pointer))
    (loop (if (%izerop (%get-byte pointer end))
            (return)
            (setq end (%i+ end 1))))
    (%str-from-ptr (%incf-ptr p offset) (%i- end offset))))

;;; This is mostly here so we can bootstrap shared libs without
;;; having to bootstrap #_strcmp.
;;; Return true if the cstrings are equal, false otherwise.
(defun %cstrcmp (x y)
  (do* ((i 0 (1+ i))
	(bx (%get-byte x i) (%get-byte x i))
	(by (%get-byte y i) (%get-byte y i)))
       ((not (= bx by)))
    (declare (fixnum i bx by))
    (when (zerop bx)
      (return t))))
