/*
   Copyright (C) 1994-2001 Digitool, Inc
   This file is part of Opensourced MCL.

   Opensourced MCL is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   Opensourced MCL is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "lisp-exceptions.h"
#include "lisp_globals.h"
#include "area.h"
#include <ctype.h>
#include <stdio.h>
#include <stddef.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <stdio.h>
#ifdef LINUX
#include <strings.h>
#include <sys/mman.h>
#include <fpu_control.h>
#endif

#ifdef DARWIN
#include <sys/mman.h>
#define _FPU_RESERVED 0xffffff00
#ifndef SA_NODEFER
#define SA_NODEFER 0
#endif
#endif

#include "Threads.h"


/*
  Handle exceptions.

  On System 7.x (for known values of "x"), it's necessary to hook into
  the system exception handling mechanisms; by default, those mechanisms
  deliver the exception to a debugger -before- we get a chance to see
  it.  That's no fun.

  Supposedly, later OS versions will provide some means whereby applications
  can inform the system of whether or not they want things to behave in a
  useful manner.
*/

extern LispObj lisp_nil;

extern unsigned lisp_heap_gc_threshold;
extern Boolean grow_dynamic_area(unsigned);


/* top 6 bits will be zero, subtag will be subtag_code_vector */
#define CV_HDR_MASK     (OP_MASK | subtagmask)
#define CV_HDR_VALUE    subtag_code_vector
#define codevec_hdr_p(value)	(((value) & CV_HDR_MASK) == CV_HDR_VALUE)



void
allocation_failure(Boolean pointerp, unsigned size)
{
  char buf[64];
  sprintf(buf, "Can't allocate %s of size %d bytes.", pointerp ? "pointer" : "handle", size);
  Fatal(":   Kernel memory allocation failure.  ", buf);
}

void
fatal_oserr(StringPtr param, OSErr err)
{
  char buf[64];
  sprintf(buf," - operating system error %d.", err);
  Fatal(param, buf);
}

#ifdef VXWORKS
Ptr
allocate(unsigned size)
{
  Ptr p;
  extern int lisp_partition_exists;
  if (lisp_partition_exists == 0) {
    fprintf(stderr, "Warning: allocate_memory called before lisp_partition_exists\n");
  }

  p = (Ptr) zone_malloc(size);
  if (p == NULL) {
    fprintf(stderr, "Warning: can't allocate %d bytes\n", size);
  }
  return p;
}

#else

Ptr
allocate(unsigned size)
{
  return (Ptr) malloc(size);
}

#endif

#ifdef VXWORKS
void
deallocate(Ptr p)
{
  zone_free(p);
}
#else
void
deallocate(Ptr p)
{
  free((void *)p);
}
#endif

Ptr
zalloc(unsigned size)
{
  Ptr p = allocate(size);
  if (p != NULL) {
    memset(p, 0, size);
  }
  return p;
}

int
ProtectMemory(LogicalAddress addr, int nbytes)
{
  return mprotect(addr, nbytes, PROT_READ);
}

int
UnProtectMemory(LogicalAddress addr, int nbytes)
{
  return mprotect(addr, nbytes, PROT_READ|PROT_WRITE);
}

void
unprotect_area(protected_area_ptr p)
{
  BytePtr start = p->start;
  unsigned nprot = p->nprot;
  
  if (nprot) {
    UnProtectMemory(start, nprot);
    p->nprot = 0;
  }
}



int
page_size = 4096;

extern void
zero_cache_lines(BytePtr, size_t, size_t);

void
zero_page(BytePtr start)
{
  zero_cache_lines(start, (page_size/KG->cache_block_size), KG->cache_block_size);
}

void
zero_heap_segment(BytePtr start)
{
#if defined(LINUX) || defined(DARWIN)
  /* We can use mmap() to establish a new mapping with read/write access */
  if (start != mmap(start,
		    heap_segment_size,
		    PROT_READ | PROT_WRITE,
		    MAP_PRIVATE | MAP_FIXED | MAP_ANON,
		    -1,
		    0)) {
    perror("mmap:");
    exit(1);
  }
#else
  zero_cache_lines(start, (heap_segment_size/KG->cache_block_size), KG->cache_block_size);
#endif
}

protected_area_ptr
AllProtectedAreas = NULL;

/* 
  This does a linear search.  Areas aren't created all that often;
  if there get to be very many of them, some sort of tree search
  might be justified.
*/

protected_area_ptr
find_protected_area(BytePtr addr)
{
  protected_area* p;
  
  for(p = AllProtectedAreas; p; p=p->next) {
    if ((p->start <= addr) && (p->end > addr)) {
      return p;
    }
  }
  return NULL;
}

void
signal_stack_soft_overflow(ExceptionInformation *xp, unsigned reg)
{
  /* The cstack just overflowed.  Force the current thread's
     control stack to do so until all stacks are well under their overflow
     limits. */

  lisp_global(CS_OVERFLOW_LIMIT) = 0xfffffff8; /* force unsigned traps to fail */
  handle_error(xp, error_stack_overflow, reg, 0, (unsigned) xpPC(xp));
}

extern LispObj (*start_lisp)(LispObj, LispObj);

void
reset_lisp_process(ExceptionInformation *xp)
{
  catch_frame *last_catch = (catch_frame *) (untag(lisp_global(CATCH_TOP)));

  lisp_global(SAVE_MEMO) = xpGPR(xp, memo);
  lisp_global(SAVE_FREEPTR) = xpGPR(xp, freeptr);

  lisp_global(SAVE_VSP) = ((lisp_frame *)last_catch->csp)->savevsp;
  lisp_global(SAVE_TSP) = ((LispObj) last_catch) - 8; /* account for TSP header */

  start_lisp(lisp_nil, 1);
}
  


/* Ensure that there's a (soft or hard) guard segment at the end of the heap.
   Zero everything between the freepointer and the guard segment.
   If we can't leave a soft guard segment, we're out of memory.  So:
     signal an OUT-OF-MEMORY condition.  If there's some "spare"
     room in the hard area, steal it so that the condition handler can
     cons some.  If we've already stolen the spare bytes and have 
     gotten here five times in a row, we're wedged and have to exit
     via Fatal().
   If we can leave a soft guard segment, we're not in (severe) trouble.
   If we've never stolen the spare segments, or if we've stolen them
   and can give them back, do so.
*/
void
reprotect_dynamic_heap(ExceptionInformation *xp, 
		       BytePtr newfree, 
		       unsigned free_space_size)
{
  unsigned protbytes;
  area *a = active_dynamic_area;
  protected_area_ptr  
    heap_soft_area  =  a->softprot,
    heap_hard_area  =  a->hardprot;
  BytePtr protptr, newlimit;
  
  unprotect_area(heap_soft_area);
  heap_soft_area->start = heap_soft_area->end;
  unprotect_area(heap_hard_area);
  if (free_space_size) {
    BytePtr lowptr = (BytePtr)(lisp_global(HEAP_START));
    newlimit = lowptr + align_to_power_of_2(newfree-lowptr+free_space_size,
					    log2_heap_segment_size);
    if (newlimit > heap_hard_area->end) {
      grow_dynamic_area(newlimit-heap_hard_area->end);
    } else if ((newlimit + free_space_size) < heap_hard_area->end) {
      shrink_dynamic_area(heap_hard_area->end-newlimit);
    }
  }

  protbytes = (((heap_soft_area->start)-newfree)&~(heap_segment_size-1));
  if (protbytes) {
    protect_area_prefix(heap_soft_area, protbytes);
  }
  protptr = heap_soft_area->start;
  protbytes = (protptr-newfree);
  while (protbytes >= page_size) {
    protptr -= page_size;
    protbytes -= page_size;
    zero_page(protptr);
  }
  memset(newfree,0,protbytes);
}


 
void
update_area_active (int global, BytePtr value)
{
  area *a = (area *) lisp_global(global);
  for (; a; a = a->older) {
    if ((a->low <= value) && (a->high >= value)) break;
  };
  if (a == NULL) Bug(NULL, "Can't find active area");
  a->active = value;
  lisp_global(global) = (LispObj) a;

  for (a = a->younger; a; a = a->younger) {
    a->active = a->high;
  }
}


/* Returns #bytes freed by invoking GC */

int
gc_from_xp(ExceptionInformation *xp)
{
  area *a;
  xframe_list xf;
  BytePtr oldfree, newfree;
  BytePtr oldend, newend;

  xf.prev = (xframe_list *) lisp_global(XFRAME);
  xf.curr = xp;

  a = (area *) lisp_global(CURRENT_CS);
  a->active = (BytePtr) xpGPR(xp, sp);
  
  update_area_active(CURRENT_VS, (BytePtr) xpGPR(xp, vsp));
  update_area_active(CURRENT_TS, (BytePtr) xpGPR(xp, tsp));

  a = active_dynamic_area;
  oldend = a->high;
  oldfree = a->active = (BytePtr) xpGPR(xp, freeptr);
  gc(&xf);

  newfree = a->active;
  xpGPR(xp,freeptr) = xpGPR(xp, initptr) = (unsigned) newfree;

  newend = a->high;
  return ((oldfree-newfree)+(newend-oldend));
}

extern BytePtr
current_stack_pointer(void);

void
protect_area(protected_area_ptr p)
{
  BytePtr start = p->start;
  unsigned n = p->protsize;

  if (n && ! p->nprot) {
    ProtectMemory(start, n);
    p->nprot = n;
  }
}


/* This is called by do_vsp_overflow & do_tsp_overflow.
   Allocating a stack area may cause a GC due to growzone.
   Since we get to the stack allocation code via a trap, not a ppc-ff-call,
   we need to store the area info where gc_from_c expects to find it.
 */

area *
allocate_area_with_c_gc_context(ExceptionInformation *xp, area *allocator(unsigned), unsigned size)
{
  area *new_area;

  (((area *) lisp_global(CURRENT_CS)) -> active) = (BytePtr) xpGPR(xp, sp);
  lisp_global(SAVE_VSP) = (LispObj) xpGPR(xp, vsp);
  lisp_global(SAVE_TSP) = (LispObj) xpGPR(xp, tsp);
  lisp_global(SAVE_FREEPTR) = (LispObj) xpGPR(xp, freeptr);
  lisp_global(SAVE_MEMO) = (LispObj) xpGPR(xp, memo);
  
  new_area = allocator(size);

  set_xpGPR(xp, freeptr, lisp_global(SAVE_FREEPTR));
  set_xpGPR(xp, memo, lisp_global(SAVE_MEMO));

  return new_area;
}


/* 
   Call this to GC from C code that was entered via a ppc-ff-call from lisp.
   That sounds like an extraordinarily bad idea.
*/

int
gc_from_c()
{
  ExceptionInformation xp_record, *xp = &xp_record;
#ifdef LINUX
  struct pt_regs regs;
#endif
  int bytes_freed;

  memset(&xp_record, 0, sizeof(ExceptionInformation));
#ifdef LINUX
  memset(&regs, 0, sizeof(struct pt_regs));

  xp_record.regs = &regs;
#endif
  set_xpGPR(xp, sp, ((area *) lisp_global(CURRENT_CS)) ->active);
  set_xpGPR(xp, vsp, lisp_global(SAVE_VSP));
  set_xpGPR(xp, tsp, lisp_global(SAVE_TSP));
  set_xpGPR(xp, freeptr, lisp_global(SAVE_FREEPTR));
  set_xpGPR(xp, memo, lisp_global(SAVE_MEMO));
  
  bytes_freed = gc_from_xp(xp);

  lisp_global(SAVE_FREEPTR) = (LispObj) xpGPR(xp, freeptr);
  lisp_global(SAVE_MEMO) = (LispObj) xpGPR(xp, memo);


  return bytes_freed;
}


long long saved_bytes_freed;
LispObj saved_freeptr = (LispObj)NULL;
long saved_freemem = 0;




protection_handler
 * protection_handlers[] = {
   do_spurious_wp_fault,
   do_soft_stack_overflow,
   do_soft_stack_overflow,
   do_soft_stack_overflow,
   do_hard_stack_overflow,    
   do_hard_stack_overflow,
   do_hard_stack_overflow,
   do_memo_reset,
   do_heap_soft_probe,
   do_heap_hard_probe
   };


Boolean
is_write_fault(ExceptionInformation *xp)
{
#ifndef VXWORKS
  return((xp->regs->trap == 0x0300) &&
	 ((xp->regs->dsisr & (1 << 25)) != 0));
#else
  /* Some OSes lose track of the DSISR and DSR SPRs, or don't provide
     valid values of those SPRs in the context they provide to
     exception handlers.  Look at the opcode of the offending
     instruction & recognize 32-bit store operations */
  opcode instr = *(xpPC(xp));

  if (xp->regs->trap != 0x300) {
    return 0;
  }
  switch (instr >> 26) {
  case 47:			/* STMW */
  case 36:			/* STW */
  case 37:			/* STWU */
    return 1;
  case 31:
    switch ((instr >> 1) & 1023) {
    case 151:			/* STWX */
    case 183:			/* STWUX */
      return 1;
    default:
      return 0;
    }
  default:
    return 0;
  }
#endif
}

OSStatus
handle_protection_violation(ExceptionInformation *xp)
{
  BytePtr addr;
  protected_area_ptr area;
  protection_handler *handler;

  if (! is_write_fault(xp)) {
    return -1;
  }

  addr = (BytePtr) ((int) (xp->regs->dar));
  area = find_protected_area(addr);

  if (area == NULL) {		/* Don't know why this fault happened. */
    return -1;
  }
  
  handler = protection_handlers[area->why];

  return handler(xp, area, addr);
}


protected_area_ptr
new_protected_area(BytePtr start, BytePtr end, lisp_protection_kind reason, unsigned protsize, Boolean now)
{
  protected_area_ptr p = (protected_area_ptr) allocate(sizeof(protected_area));
  
  if (p == NULL) return NULL;
  p->protsize = protsize;
  p->nprot = 0;
  p->start = start;
  p->end = end;
  p->why = reason;
  p->next = AllProtectedAreas;

  AllProtectedAreas = p;
  if (now) {
    protect_area(p);
  }
  
  return p;
}

/*
  Un-protect the first nbytes bytes in specified area.
  Note that this may cause the area to be empty.
*/
void
unprotect_area_prefix(protected_area_ptr area, size_t delta)
{
  unprotect_area(area);
  area->start += delta;
  if ((area->start + area->protsize) <= area->end) {
    protect_area(area);
  }
}


/*
  Extend the protected area, causing the preceding nbytes bytes
  to be included and protected.
*/
void
protect_area_prefix(protected_area_ptr area, size_t delta)
{
  unprotect_area(area);
  area->start -= delta;
  protect_area(area);
}


/*
  If the address being written to isn't the highest address in the area
  (4 bytes below the end of the area), something unexpected occurred.
  Lose in this case, which "shouldn't happen."
  
  call note_memoized_references() to reset the memo stack pointer.
  
  Reset the memo register to memo_base, and win (return to lisp.)
  */

OSStatus 
do_memo_reset(ExceptionInformation *xp, protected_area_ptr area, BytePtr addr)
{
  
  if (addr != (area->end - 4)) {
    return -1;
  }

  note_memoized_references(xp, area->end,  memo_base, NULL, NULL);

  return 0;
}


OSStatus
do_heap_soft_probe(ExceptionInformation *xp, protected_area_ptr p, BytePtr addr)
{
#ifdef SUPPORT_PRAGMA_UNUSED
#pragma unused(xp)
#endif
  area *a = active_dynamic_area;
  BytePtr old_start = p->start;

  /* Sanity check ... */
  if ((old_start + heap_segment_size) >= addr) {
    /* Maybe do an egc here. */
    if (a->older && lisp_global(OLDEST_EPHEMERAL)) {
      if (((BytePtr)(xpGPR(xp,freeptr))-(a->low)) >= a->threshold) {
        gc_from_xp(xp);
        return 0;               /* May fault again real soon ...*/
      }
    }
    unprotect_area_prefix(p, heap_segment_size);
    zero_heap_segment(old_start);
    /* If the soft area's now empty, protect the hard area */
    if (p->nprot == 0) {
      protect_area(active_dynamic_area->hardprot);
    }
    return 0;
  }
  return -1;
}


OSStatus
do_heap_hard_probe(ExceptionInformation *xp, protected_area_ptr p, BytePtr addr)
{
  int want, have, got, freed;
  LispObj curfreeptr = (LispObj) xpGPR(xp, freeptr);
  area *a = active_dynamic_area;

  a->active = (BytePtr)curfreeptr;

  want = (LispObj) addr - curfreeptr;
  have = (LispObj) (p->start) - curfreeptr;
  
  
  untenure_from_area(KG->tenured_area); /* force a full GC */
  freed = gc_from_xp(xp);
  got = have + freed;

  
  /* We only win (without making another heap or something)
     if we've "got" more than we "want", since the free pointer
     has to wind up before the hard guard segment. */
  if (got > want) {
    if (active_dynamic_area->softprot->nprot) {
      /* Soft area is protected, so unprotect the hard area */
      unprotect_area(p);
    }
  } else {
    /* We failed to allocate something less than 32K bytes.  It'd probably
       be wise to panic & try to grab another heap at this point.
       Calling lisp at this point ... um, lets us test stack overflow detection.
       */
    /* I don't think that this can happen anymore. */
    handle_error(xp, error_alloc_failed, rnil, 0, (unsigned) xpPC(xp));
  }
  return 0;
}

OSStatus
do_hard_stack_overflow(ExceptionInformation *xp, protected_area_ptr area, BytePtr addr)
{
#ifdef SUPPORT_PRAGMA_UNUSED
#pragma unused(area,addr)
#endif
  reset_lisp_process(xp);
  return -1;
}

extern area*
allocate_vstack(unsigned useable);       /* This is in "pmcl-kernel.c" */

extern area*
allocate_tstack(unsigned useable);       /* This is in "pmcl-kernel.c" */

Boolean
catch_frame_p(lisp_frame *spPtr)
{
  catch_frame* catch = (catch_frame *) untag(lisp_global(CATCH_TOP));

  for (; catch; catch = (catch_frame *) untag(catch->link)) {
    if (spPtr == ((lisp_frame *) catch->csp)) {
      return true;
    }
  }
  return false;
}

Boolean
unwind_protect_cleanup_frame_p(lisp_frame *spPtr)
{
  if ((spPtr->savevsp == (LispObj)NULL) ||  /* The frame to where the unwind-protect will return */
      (((spPtr->backlink)->savevsp) == (LispObj)NULL)) {  /* The frame that returns to the kernel  from the cleanup form */
    return true;
  } else {
    return false;
  }
}

Boolean
lexpr_entry_frame_p(lisp_frame *spPtr)
{
  LispObj savelr = spPtr->savelr;
  LispObj lexpr_return = (LispObj) lisp_global(LEXPR_RETURN);
  LispObj lexpr_return1v = (LispObj) lisp_global(LEXPR_RETURN1V);
  LispObj ret1valn = (LispObj) lisp_global(RET1VALN);

  return
    (savelr == lexpr_return1v) ||
    (savelr == lexpr_return) ||
    ((savelr == ret1valn) &&
     (((spPtr->backlink)->savelr) == lexpr_return));
}

Boolean
lisp_frame_p(lisp_frame *spPtr)
{
  LispObj savefn;
  /* We can't just look at the size of the stack frame under the EABI
     calling sequence, but that's the first thing to check. */
  if (((lisp_frame *) spPtr->backlink) != (spPtr+1)) {
    return false;
  }
  savefn = spPtr->savefn;
  return (savefn == 0) || (fulltag_of(savefn) == fulltag_misc);
  
}


int ffcall_overflow_count = 0;

/* Find a frame that is neither a catch frame nor one of the
   lexpr_entry frames We don't check for non-lisp frames here because
   we'll always stop before we get there due to a dummy lisp frame
   pushed by .SPcallback that masks out the foreign frames.  The one
   exception is that there is a non-lisp frame without a valid VSP
   while in the process of ppc-ff-call. We recognize that because its
   savelr is NIL.  If the saved VSP itself is 0 or the savevsp in the
   next frame is 0, then we're executing an unwind-protect cleanup
   form, and the top stack frame belongs to its (no longer extant)
   catch frame.  */
lisp_frame *
find_non_catch_frame_from_xp (ExceptionInformation *xp)
{
  lisp_frame *spPtr = (lisp_frame *) xpGPR(xp, sp);
  if ((((unsigned) spPtr) + sizeof(lisp_frame)) != ((unsigned) (spPtr->backlink))) {
    ffcall_overflow_count++;          /* This is mostly so I can breakpoint here */
  }
  for (; !lisp_frame_p(spPtr)  || /* In the process of ppc-ff-call */
         unwind_protect_cleanup_frame_p(spPtr) ||
         catch_frame_p(spPtr) ||
         lexpr_entry_frame_p(spPtr) ; ) {
     spPtr = spPtr->backlink;
     };
  return spPtr;
}

Boolean
db_link_chain_in_area_p (area *a)
{
  LispObj *db = (LispObj *) lisp_global(DB_LINK),
          *high = (LispObj *) a->high,
          *low = (LispObj *) a->low;
  for (; db; db = (LispObj *) *db) {
    if ((db >= low) && (db < high)) return true;
  };
  return false;
}

typedef struct preserved_registers {
  LispObj the_pc;
  LispObj the_lr;
  LispObj sp_r1;
  LispObj rnil_r2;
  LispObj tsp_r12;
  LispObj vsp_r13;
  LispObj save7_r24;
  LispObj save6_r25;
  LispObj save5_r26;
  LispObj save4_r27;
  LispObj save3_r28;
  LispObj save2_r29;
  LispObj save1_r30;
  LispObj save0_r31;
} preserved_registers;

/* This is for debugging so that you can easily see in the Power Mac
   Debugger a window containing the preserved state */
void
sample_preserved_registers (ExceptionInformation *xp, preserved_registers *pr)
{
  pr->the_pc = (LispObj) xpPC(xp);
  pr->the_lr = (LispObj) xpLR(xp);
  pr->sp_r1 = xpGPR(xp, sp);
  pr->rnil_r2 = xpGPR(xp, rnil);
  pr->tsp_r12 = xpGPR(xp, tsp);
  pr->vsp_r13 = xpGPR(xp, vsp);
  pr->save7_r24 = xpGPR(xp, save7);
  pr->save6_r25 = xpGPR(xp, save6);
  pr->save5_r26 = xpGPR(xp, save5);
  pr->save4_r27 = xpGPR(xp, save4);
  pr->save3_r28 = xpGPR(xp, save3);
  pr->save2_r29 = xpGPR(xp, save2);
  pr->save1_r30 = xpGPR(xp, save1);
  pr->save0_r31 = xpGPR(xp, save0);
}


/* Note: CURRENT_VS (CURRENT_TS) is always either the area containing
  the current value of VSP (TSP) or an older area.  */

/* Find or create a new VSP segment. Copy the top-most stack frame to
  it */

OSStatus
do_vsp_overflow (ExceptionInformation *xp, BytePtr addr)
{
#ifdef SUPPORT_PRAGMA_UNUSED
#pragma unused(addr)
#endif
  area *first_a = (area *) lisp_global(CURRENT_VS);
  area *a, *new_a = NULL, *maybe_new_a = NULL;
  pc where = xpPC(xp);
  opcode instruction = *where;
  unsigned vsp_reg = RA_field(instruction);   /* the "VSP" register for the instruction causing the trap */
  BytePtr vspPtr = (BytePtr) xpGPR(xp, vsp_reg);
  BytePtr real_vsp = (BytePtr) xpGPR(xp, vsp);
  BytePtr new_vsp;
  lisp_frame *spPtr = find_non_catch_frame_from_xp(xp), *cf_ptr;
  BytePtr savevsp = (BytePtr) spPtr->savevsp;
  int size, frame_size;
  preserved_registers pr;

  sample_preserved_registers(xp, &pr);

  if (xpGPR(xp, rnil) != lisp_nil) {
    Bug(xp, "VSP overflow in non-lisp context");
  };

  for (a = first_a; a && ((a->low > savevsp) || (a->high < savevsp)); a = a->older) {
  };
  if (!a) Bug(xp, "Can't find savevsp area");

  if (vsp_reg != vsp) {
    for (a = first_a; a && ((a->low > real_vsp) || (a->high < real_vsp)); a = a->older) {
    };
    /* Might have overflowed once already while sliding the stack.
       This could leave the real_vsp pointing at a younger stack
       frame: (let ((x (make-list 5000))) (time (apply 'foo x))) then
       proceed through the stack overflows */
    if (!a) {
      for (a = first_a->younger; a && ((a->low > real_vsp) || (a->high < real_vsp)); a = a->younger) {
      };
    };
    if (!a) {
      Bug(xp, "Can't find real_vsp area");
    };
  };

  for (a = first_a; a && ((a->low > vspPtr) || (a->high < vspPtr)); a = a->older) {
  };
  if (!a) Bug(xp, "Can't find VSP area");

  /* May have already extended the stack once, in which case the saved
     VSP is in another segment */
  if ((a->low > savevsp) || (a->high < savevsp))
    savevsp = a->high;

  frame_size = savevsp - vspPtr;
  size = align_to_power_of_2(512 + frame_size, 12);      /* Leave some space in the new segment */
  if (STACK_SEGMENT_SIZE > size)
    size = STACK_SEGMENT_SIZE;

  /* Look for an existing segment that's big enough.  If the register
     being stored through is not VSP, then we're in one of the
     tXXXslide subprims, copying from another segment, so we have to
     be careful not to use the VSP segment.  Also need to be careful
     not to use a segment that contains part of the db_link chain (may
     be left from a previous tXXXslide).  */
  for (maybe_new_a = a->younger; maybe_new_a != NULL; maybe_new_a = maybe_new_a->younger) {
    int a_size = maybe_new_a->high - maybe_new_a->softlimit;
    if ((size <= a_size) &&
	((vsp_reg == vsp) || (real_vsp < maybe_new_a->low) || (real_vsp > maybe_new_a->high)) &&
	!db_link_chain_in_area_p(maybe_new_a)) {
      area *prev_a = maybe_new_a->younger;
      area *next_a = maybe_new_a->older;
      new_a = maybe_new_a;
      if (next_a != a) {
        if (prev_a)
          prev_a->older = next_a;
        if (next_a)
          next_a->younger = prev_a;
        prev_a = a->younger;
        prev_a->older = new_a;
        new_a->younger = prev_a;
        new_a->older = a;
        a->younger = new_a;
      };
      break;
    }
  };

  /* If no existing segment, allocate a new one */
  if (! new_a) {
    area *prev_a = a->younger;
    new_a = allocate_area_with_c_gc_context(xp, allocate_vstack, size);
    if (! new_a)
      return do_hard_stack_overflow(xp, NULL, NULL);
    if (prev_a) {
      prev_a->older = new_a;
      new_a->younger = prev_a;
    }
    new_a->older = a;
    a->younger = new_a;
  };

  lisp_global(CURRENT_VS) = (LispObj) new_a;
  a->active = vspPtr;
  new_vsp = new_a->high - frame_size;
  memcpy(new_vsp, vspPtr, frame_size);
  xpGPR(xp, vsp_reg) = (LispObj) new_vsp;
  new_a->active = new_vsp;              /* Not really necessary */
  
  /* If current instruction's an STMW, there would have been a preceding
     (la vsp -n vsp), which is really an (addi vsp -n vsp) instruction ...
     "emulate" the STMW so that if a GC happens during handle_error, this
     part of the stack is initialized. */
  if ((major_opcode_p(instruction, 47)) &&
      (D_field(instruction) == 0)) {
    /* (stmw rs 0 vsp) */
    LispObj *vspP = (LispObj *) new_vsp;
    unsigned reg = RS_field(instruction);

    a->active = vspPtr + (4 * (32 - reg));         /* Ignore the (la vsp -n vsp) in the old segment */
    
    for (; reg < 32; reg++) {
      *vspP++ = xpGPR(xp, reg);
    }
  };

  /* Update the VSP in any moved catch frames so that throwing will
     stay in the new stack segment.  This is important since otherwise
     after the throw we'll see old values of variables.  Sometimes a
     catch frame is already in the new_vsp segment (if an overflow has
     already happenned).  */
  {
    unsigned diff = new_vsp - vspPtr;
    unsigned high = (unsigned) new_a->high;
    for (cf_ptr = (lisp_frame *) xpGPR(xp, sp); cf_ptr != spPtr; cf_ptr = cf_ptr->backlink) {
      if (cf_ptr->savefn != lisp_nil) {		/*  ppc-ff-call frame if NULL (invalid savevsp) */
        unsigned catch_vsp = ((unsigned) cf_ptr->savevsp);
        if (catch_vsp && ((catch_vsp < ((unsigned) new_vsp)) || (catch_vsp > high))) {
          catch_vsp += diff;
          if ((catch_vsp < ((unsigned) new_vsp)) || (catch_vsp > high)) {
            Bug(xp, "Can't relocate catch or lexpr frame VSP");
          } else {
            cf_ptr->savevsp = (LispObj) catch_vsp;
          }
        }
      }
    }
  }
  
  /* Don't signal an error if we found an existing segment.
     This is important for stack group switch.
     Otherwise we might signal an error during the call to YieldToThread
     at a time when Lisp code cannot run.
     If vsp_reg is not vsp, then we're sliding from a segment that is
     about to be replaced by the one that we just allocated. In this case,
     Lisp has already approved the stack extension. Calling lisp there is
     also dangerous since a GC might free the stack frame being copied from.
     */
  if ((maybe_new_a == NULL) && (vsp_reg == vsp) &&
      ((((unsigned) nrs_GC_EVENT_STATUS_BITS.vcell) & gc_allow_stack_overflows_bit) == 0)) {
    handle_error(xp, error_stack_overflow, vsp, 0, (unsigned) xpPC(xp));
  };
  
  return 0;                     /* if we ever return. */
}

OSStatus
do_tsp_overflow (ExceptionInformation *xp, BytePtr addr)
{
  area *first_a = (area *) lisp_global(CURRENT_TS);
  area *a = first_a, *new_a, *maybe_new_a;
  BytePtr tspPtr = (BytePtr) xpGPR(xp, tsp);
  LispObj rnil_value = xpGPR(xp, rnil);
  int frame_size = tspPtr - addr;
  BytePtr new_tsp;
  pc where = xpPC(xp);
  opcode instruction = *where;

  preserved_registers pr;

  sample_preserved_registers(xp, &pr);

  if ((rnil_value != lisp_nil) && ((unsigned) rnil_value) != (((unsigned) lisp_nil) - 1)) {
    Bug(xp, "TSP overflow in non-lisp context");
  }

  if ((frame_size > 4096) || (frame_size < 0)) {
    Bug(xp, "TSP frame size out of range");
  };

  for (; a && ((a->low > tspPtr) || (a->high < tspPtr)); a = a->older) {
  };
  if (!a)
    Bug(xp, "Can't find TSP area");

  new_a = maybe_new_a = a->younger;

  /* If no existing segment, allocate a new one */
  if (! new_a) {
    new_a = allocate_area_with_c_gc_context(xp, allocate_tstack, STACK_SEGMENT_SIZE);
    if (! new_a)
      return do_hard_stack_overflow(xp, NULL, NULL);
    new_a->older = a;
    a->younger = new_a;
  };

  lisp_global(CURRENT_TS) = (LispObj) new_a;
  a->active = tspPtr;
  new_tsp = new_a->high - frame_size;
  
  /* Emulate current instruction, which must be a stwu or stwux */
  if ((major_opcode_p(instruction, 37) ||                /* stwu */
       X_opcode_p(instruction, 31, 183)) &&              /* stwux */
      (RA_field(instruction) == tsp)) {
    /* (stwu rs D tsp) or (stwux rs tsp rb) */
    LispObj *tspP = (LispObj *) new_tsp;
    unsigned reg = RS_field(instruction);

    *tspP = xpGPR(xp, reg);
    xpPC(xp) += 1;
  } else {
    Bug(xp, "Instruction causing TSP overflow not stwu or stwux");
  };

  /* Update the tsp. This must happen after the stwu has been emulated, so the
     link will work correctly
     */
  xpGPR(xp, tsp) = (LispObj) new_tsp;
  new_a->active = new_tsp;              /* Not really necessary */

  /* Don't signal an error if we found an existing segment.
     Unlike the VSP, this is not functionally important, but
     it has a noticeable performance impact.
     Also, don't signal an error if nilreg is not NIL. This happens
     when .SPcallback allocates a TSP frame to save the C saved registers.
     */
  if ((maybe_new_a == NULL) && (xpGPR(xp, rnil) == lisp_nil) &&
      ((((unsigned) nrs_GC_EVENT_STATUS_BITS.vcell) & gc_allow_stack_overflows_bit) == 0)) {
    handle_error(xp, error_stack_overflow, tsp, 0, (unsigned) xpPC(xp));
  };
  
  return 0;                     /* if we ever return. */
}

OSStatus
do_soft_stack_overflow(ExceptionInformation *xp, protected_area_ptr prot_area, BytePtr addr)
{
  /* Trying to write into a guard page on the vstack or tstack.
     Allocate a new stack segment, emulate stwu and stwux for the TSP, and
     signal an error_stack_overflow condition.
      */
  lisp_protection_kind which = prot_area->why;
  Boolean on_TSP = (which == kTSPsoftguard);

  if (on_TSP) {
    return do_tsp_overflow(xp, addr);
   } else {
    return do_vsp_overflow(xp, addr);
   }
}

OSStatus
do_spurious_wp_fault(ExceptionInformation *xp, protected_area_ptr area, BytePtr addr)
{
#ifdef SUPPORT_PRAGMA_UNUSED
#pragma unused(xp,area,addr)
#endif
  return -1;
}

OSStatus
handle_sigfpe(ExceptionInformation *xp)
{
#ifdef PROXY_SCHEDULER
  extern pid_t lisp_task_pid;
  
  /* This is annoying: the SIGFPE gets propagated to other threads,
     at least until FPSCR[FEX] clear.  Ignore the SIGFPE if it
     happens in some thread other than the primary one.
  */
  if (lisp_task_pid != getpid()) {
    return 0;
  }
#endif
  (void) zero_fpscr();

  lisp_global(FPSCR_SAVE) = xp->regs->gpr[PT_FPSCR] & ~_FPU_RESERVED;
  /* 'handle_fpux_binop' scans back from the specified PC until it finds an FPU
     operation; there's an FPU operation right at the PC, so tell it to start
     looking one word beyond */
  return handle_fpux_binop(xp, (unsigned)(xpPC(xp))+4);
}

int
altivec_present = 1;
extern void altivec_probe(void);

OSStatus
PMCL_exception_handler(int xnum, ExceptionInformation *xp)
{
  unsigned oldMQ = xp->regs->gpr[PT_MQ];
  OSStatus status = -1;
  xp->regs->gpr[PT_MQ] = 0;


  if ((xnum == SIGSEGV) ||
      xnum == SIGBUS) {
    status = handle_protection_violation(xp);
  } else if (xnum == SIGFPE) {
    status = handle_sigfpe(xp);
  } else if ((xnum == SIGILL) || (xnum == SIGTRAP)) {
    pc program_counter = xpPC(xp);
    opcode instruction = *program_counter;

    if (IS_UUO(instruction)) {
      status = handle_uuo(xp, instruction, program_counter);
    } else if (is_conditional_trap(instruction)) {
      status = handle_trap(xp, instruction, program_counter);
    } else if (program_counter == (pc)altivec_probe) {
      altivec_present = 0;
      adjust_exception_pc(xp, 4);
      status = 0;
    }
  }
  xp->regs->gpr[PT_MQ] = oldMQ;    
  return status;
}

void
adjust_exception_pc(ExceptionInformation *xp, int delta)
{
  xpPC(xp) += (delta >> 2);
}

OSStatus
handle_xalloc(ExceptionInformation *xp, unsigned rt, unsigned ra, unsigned rb)
{
  /* If rt is rzero, this means "do a GC or purify now."
     If ra is rzero, do a GC, otherwise call "purify", then:
       if rb is rzero, return; otherwise, write a PEF image
       to the (open) file descriptor whose boxed value is in rb.
     If rt is rnil, this means "get or set lisp_heap_gc_threshold".
     Otherwise, we're trying to allocate something >32KB; we may
     or may not be able to do so and may or may not have to GC
     to do so. */
  
  area *a = active_dynamic_area;
  Boolean egc_was_enabled = (a->older != NULL);
  

  if (rt == rzero) {
    if (egc_was_enabled) {
      egc_control(false, (BytePtr) xpGPR(xp, freeptr));
    }
    gc_from_xp(xp);
    if (ra != rzero) {
      if (ra == vsp) {          /* %save-library */
        extern unsigned next_libnum;
        OSErr err = -50;
        char libname[64], *src;
        int i, tag, subtag, elements, refnum = (int) (xpGPR(xp, rb));
        LispObj 
          *firstobj = (LispObj *) xpGPR(xp, arg_y), 
          *lastobj = (LispObj *) xpGPR(xp, arg_z),
          namestr = xpGPR(xp, arg_x),
          header;
        unsigned libnum = next_libnum;

        tag = fulltag_of(namestr);
        if (tag == fulltag_misc) {
          header = deref(namestr, 0);
          subtag = header_subtag(header);
          elements = header_element_count(header);
          if ((subtag == subtag_simple_base_string) &&
              (elements < 64)) {
            src = ((char *) namestr)+misc_data_offset;
            for (i = 0; i < elements; i++) {
              libname[i] = src[i];
            }
            libname[elements] = 0;
            err = noErr;
            if (firstobj == (LispObj *) lisp_nil) {
              firstobj = (LispObj *) a->low;
            } else {
              firstobj = (LispObj *) untag((LispObj)firstobj);
              if (firstobj < ((LispObj *) (a->low))) {
                err = -50;
              }
            }
            if (lastobj == (LispObj *)lisp_nil) {
              lastobj = (LispObj *) a->active;
            } else {
              lastobj = (LispObj *) untag((LispObj)lastobj);
              if (lastobj > ((LispObj *) (a->active))) {
                err = -50;
              }
            }
          }
          if (err == noErr) {
            err = lib_purify(xp, libname, firstobj, lastobj);
            gc_from_xp(xp);
            if (err == noErr) {
              extern OSErr save_library(unsigned, unsigned, unsigned *);
              unsigned version;

              err = save_library(refnum, libnum, &version);
              if (err == noErr) {
                xpGPR(xp, imm1) = version;
              }
            }
          }
        }
        xpGPR(xp, imm0) = (err << fixnumshift); 
      } else {
        purify(xp);
        gc_from_xp(xp);
        if (rb != 0) {
          unsigned refnum = ((unsigned) (xpGPR(xp, rb))) >> fixnumshift;
          OSErr err;
          extern OSErr save_application(unsigned);
          
          err = save_application(refnum);
          if (err == noErr) {
#ifdef VXWORKS
	    fprintf(stderr, "lisp exiting\n");
	    fflush(stderr);
#endif
            lisp_exit(0);
          }
          fatal_oserr(": save_application", err);
        }
      }
    }
    if (egc_was_enabled) {
      egc_control(true, NULL);
    }
  } else if (rt == rnil) {
    /* If both ra & rb are rnil, try to put the current threshold
       in effect.  This may need to disable/reenable the EGC. */
    if ((ra == rnil) && (rb == rnil)) {
      a->active = (BytePtr)(xpGPR(xp, freeptr));
      untenure_from_area(KG->tenured_area);
      reprotect_dynamic_heap(xp, a->active, lisp_heap_gc_threshold);
      if (egc_was_enabled) {
	if ((a->high - a->active) >= a->threshold) {
	  tenure_to_area(KG->tenured_area);
	}
      }
    }
    if (ra != rnil) {
      /* Set the threshold, based on the value of ra. */
      LispObj a_raw = xpGPR(xp, ra);
      if (tag_of(a_raw) == tag_fixnum) {
	int a_val = unbox_fixnum(a_raw);
	
	if (a_val > 0) {
	  lisp_heap_gc_threshold = align_to_power_of_2(a_val +
						       (heap_segment_size - 1),
						       log2_heap_segment_size);
	}
      }
    }
    if (rb != rnil) {
      xpGPR(xp, rb) = box_fixnum(lisp_heap_gc_threshold);
    }
  } else {
    BytePtr curfree = (BytePtr) xpGPR(xp,freeptr);
    unsigned 
      have = (a->high - curfree),
      total = (a->high - a->low),
      need,
      subtag =  xpGPR(xp,rb) >> fixnumshift, /* Typically arg_z */
      elements = xpGPR(xp,ra) >> fixnumshift, /* Typically arg_y */
      tag_n = tag_of(subtag);

    if (KG->tenured_area) {
      total = (a->high - KG->tenured_area->low);
    }
    if ((tag_n == fulltag_nodeheader) ||
        (subtag <= max_32_bit_ivector_subtag)) {
      need = 4 + (elements<<2);
    } else if (subtag <= max_8_bit_ivector_subtag) {
      need = 4 + elements;
    } else if (subtag <= max_16_bit_ivector_subtag) {
      need = 4 + (elements<<1);
    } else if (subtag == subtag_double_float_vector) {
      need = 8 + (elements<<3);
    } else {
      need = 4 + ((elements+7)>>3);
    }
    need = (need+7)&~7;
    if ((need >= have) && (need <= total)) {
      /* If we need to gc, force a full gc */
      a->active = curfree;
      untenure_from_area(KG->tenured_area);
      gc_from_xp(xp);           /* may retenure everything */
      curfree = a->active;
      have = a->high-curfree;
    }
    if (need >= have) {
      area *reserved = reserved_area;
      unsigned avail = reserved->high - reserved->low;

      if (avail > need) {
	/* If EGC is active, deactivate it. */
	untenure_from_area(KG->tenured_area);
	reprotect_dynamic_heap(xp, curfree, have+need);
	have = a->high-curfree;
	if (a->older) {
	  tenure_to_area(KG->tenured_area);
	}
      }
    }
    if (need < have) {
      /* We win, but have to do the allocation manually.
         That may involve unprotecting some protected pages.
         The EGC status isn't affected by this; the large
         object just becomes "the pig in the python" */
      protected_area_ptr soft = a->softprot, hard = a->hardprot;
      BytePtr 
        protlow = soft->start,
        newfree = (BytePtr)(curfree+need);

      while (protlow < newfree) {
        unprotect_area_prefix(soft, heap_segment_size);
        zero_heap_segment(protlow);
        protlow += heap_segment_size;
      }
      if (soft->nprot == 0) {
        protect_area(hard);
      }
      *((unsigned*)curfree) = make_header(subtag,elements);
      xpGPR(xp,freeptr) = xpGPR(xp,initptr) = (LispObj)newfree;
      xpGPR(xp,rt) = (LispObj)(curfree+fulltag_misc);
    } else {
      handle_error(xp, error_alloc_failed, rnil, 0, (unsigned) xpPC(xp));
    }
  }
  adjust_exception_pc(xp,4);
  return noErr;
}


/* 
  This wants to scan backwards until "where" points to an instruction
   whose major opcode is either 63 (double-float) or 59 (single-float)
*/

OSStatus
handle_fpux_binop(ExceptionInformation *xp, unsigned where)
{
  OSStatus err;
  unsigned *there = (unsigned *) where, instr, errnum;
  int i = TRAP_LOOKUP_TRIES, delta = 0;
  
  while (i--) {
    instr = *--there;
    delta -= 4;
    if (codevec_hdr_p(instr)) {
      return -1;
    }
    if (major_opcode_p(instr, major_opcode_FPU_DOUBLE)) {
      errnum = error_FPU_exception_double;
      break;
    }

    if (major_opcode_p(instr, major_opcode_FPU_SINGLE)) {
      errnum = error_FPU_exception_short;
      break;
    }
  }
  
  err = handle_error(xp, errnum, rnil, 0, (unsigned) there);
  /* Yeah, we said "non-continuable".  In case we ever change that ... */
  
  adjust_exception_pc(xp, delta);
  ((unsigned *)(xp->regs))[PT_FPSCR]  &=  0x03fff;
  
  return err;

}

OSStatus
handle_uuo(ExceptionInformation *xp, opcode the_uuo, pc where) 
{
#ifdef SUPPORT_PRAGMA_UNUSED
#pragma unused(where)
#endif
  unsigned 
    minor = UUO_MINOR(the_uuo),
    rt = 0x1f & (the_uuo >> 21),
    ra = 0x1f & (the_uuo >> 16),
    rb = 0x1f & (the_uuo >> 11),
    errnum = 0x3ff & (the_uuo >> 16);

  OSStatus status = -1;

  int bump = 4;

  switch (minor) {
  case UUO_XALLOC:
    bump = 0;
    status = handle_xalloc(xp, rt, ra, rb);
    break;

  case UUO_ZERO_FPSCR:
    status = 0;
    xp->regs->gpr[PT_FPSCR] = 0;
    break;

  case UUO_FIXNUM_OVERFLOW:
    status = fix_fixnum_overflow(xp, rt, ra);
    break;

  case UUO_BOX_SIGNED:
    status = box_signed_integer(xp, rt, ra);
    break;
    
  case UUO_BOX_UNSIGNED:
    status = box_unsigned_integer(xp, rt, ra);
    break;
    
  case UUO_ADD_FIXNUMS:
    status = add_fixnums(xp, rt, ra, rb);
    break;
    
  case UUO_SUB_FIXNUMS:
    status = sub_fixnums(xp, rt, ra, rb);
    break;

  case UUO_INTERR:
    status = handle_error(xp, errnum, rb, 0, (unsigned) where);
    break;

  case UUO_INTCERR:
    status = handle_error(xp, errnum, rb, 1, (unsigned) where);
    break;

  case UUO_FPUX_BINOP:
    status = handle_fpux_binop(xp, (unsigned)where);
    bump = 0;
    break;

  default:
    status = -1;
    bump = 0;
  }
  
  if ((!status) && bump) {
    adjust_exception_pc(xp, bump);
  }
  return status;
}

unsigned
register_codevector_contains_pc (unsigned lisp_function, unsigned where)
{
  unsigned code_vector, size;

  if ((fulltag_of(lisp_function) == fulltag_misc) &&
      (header_subtag(header_of(lisp_function)) == subtag_function)) {
    code_vector = deref(lisp_function, 1);
    size = header_element_count(header_of(code_vector)) << 2;
    if ((untag(code_vector) < where) && (where < (code_vector + size)))
      return(code_vector);
  }

  return(0);
}

/* Callback to lisp to handle a trap. Need to translate the
   PC (where) into one of two forms of pairs:

   1. If PC is in fn or nfn's code vector, use the register number
      of fn or nfn and the index into that function's code vector.
   2. Otherwise use 0 and the pc itself
*/
void
callback_for_trap (LispObj callback_macptr, ExceptionInformation *xp, unsigned where,
                   unsigned arg1, unsigned arg2, unsigned arg3)
{
  unsigned code_vector = register_codevector_contains_pc(xpGPR(xp, fn), where);
  unsigned register_number = fn;
  unsigned index = where;

  if (code_vector == 0) {
    register_number = nfn;
    code_vector = register_codevector_contains_pc(xpGPR(xp, nfn), where);
  }
  if (code_vector == 0)
    register_number = 0;
  else
    index = (where - (code_vector + misc_data_offset)) >> 2;
  callback_to_lisp(callback_macptr, xp, register_number, index, arg1, arg2, arg3);
}

void
callback_to_lisp (LispObj callback_macptr, ExceptionInformation *xp,
                  unsigned arg1, unsigned arg2, unsigned arg3, unsigned arg4, unsigned arg5)
{
  xframe_list xframe_link;
  unsigned  callback_ptr, i;
  LispObj saved_registers[8];
  area *a;
  char raw_vectors[34*16];
  vector_buf vectors = (vector_buf) ((((int) raw_vectors) + 0xf) & ~0xf);

  /* Put the active stack pointer where .SPcallback expects it */
  a = (area *) lisp_global(CURRENT_CS);
  a->active = (BytePtr) xpGPR(xp, sp);

  /* Copy globals from the exception frame to nilreg globals */
  lisp_global(SAVE_FREEPTR) = (LispObj) xpGPR(xp, freeptr);
  lisp_global(SAVE_MEMO) = (LispObj) xpGPR(xp, memo);
  lisp_global(SAVE_VSP) = (LispObj) xpGPR(xp, vsp);
  lisp_global(SAVE_TSP) = (LispObj) xpGPR(xp, tsp);

  /* Link the exception frame into the xframe list */
  xframe_link.curr = xp;
  xframe_link.prev = (xframe_list *) lisp_global(XFRAME);
  lisp_global(XFRAME) = (LispObj) &xframe_link;

  /* pass the saved registers to .SPcallback 
     It would be nice to just push these on the VSP, but how would we handle overflow? */
  for (i = 0; i < 8; i++) {
    saved_registers[i] = (LispObj) xpGPR(xp, save0-i);
  }
  lisp_global(EXCEPTION_SAVED_REGISTERS) = (LispObj) saved_registers;

  if (altivec_available) {
    put_altivec_registers(vectors);
    xp->regs->gpr[PT_MQ] = (LispObj) vectors;
  }

  /* Call back.
     Lisp will handle trampolining through some code that
     will push lr/fn & pc/nfn stack frames for backtrace.
  */
  callback_ptr = ((macptr *)(untag(callback_macptr)))->address;
  ((ProcPtr)callback_ptr) (xp, arg1, arg2, arg3, arg4, arg5);

  if (altivec_available) {
    get_altivec_registers(vectors);
  }

  /* Unlink the exception frame from the xframe list */
  lisp_global(XFRAME) = (LispObj) xframe_link.prev;

  /* Copy GC registers back into exception frame */
  xpGPR(xp, initptr) = xpGPR(xp, freeptr) = lisp_global(SAVE_FREEPTR);
  xpGPR(xp, memo) = lisp_global(SAVE_MEMO);
}

area *
allocate_no_stack (unsigned size)
{
#ifdef SUPPORT_PRAGMA_UNUSED
#pragma unused(size)
#endif

  return (area *) NULL;
}


extern OSErr NewThread (ThreadStyle, ThreadEntryProcPtr, void *, Size, ThreadOptions, void **, ThreadID *);
extern OSErr DisposeThread (ThreadID, void *, Boolean);
extern OSErr YieldToThread (ThreadID);

/* Here is where we start up on the new thread after a control stack overflow. */
voidPtr *cs_stack_switch_startup(void *threadParam)
{
#ifdef SUPPORT_PRAGMA_UNUSED
#pragma unused(threadParam)
#endif

  LispObj callback_macptr;
  unsigned callback_ptr, i;
  int interrupt_level;
  area *a;
  xframe_list *xf;
  ExceptionInformation *xp;
  LispObj saved_registers[8];
  char raw_vectors[34*16];
  vector_buf vectors = (vector_buf) ((((int) raw_vectors) + 0xf) & ~0xf);

  for (;;) {

    callback_macptr = nrs_CS_OVERFLOW_CALLBACK.vcell;
    callback_ptr = ((macptr *)(untag(callback_macptr)))->address;
    a = (area *) lisp_global(CURRENT_CS);

    /* The LAP code (cs-overflow-callback) will install the new cs area.
       We need to keep the old one for now so that the callback code will
       store the correct frame pointer to the old stack frame */
    lisp_global(CS_OVERFLOW_LIMIT) = (LispObj) 0;

    /* pass the saved registers to .SPcallback  */
    xf = (xframe_list *) lisp_global(XFRAME);
    xp = xf->curr;
    for (i = 0; i < 8; i++) {
      saved_registers[i] = (LispObj) xpGPR(xp, save0-i);
    }
    lisp_global(EXCEPTION_SAVED_REGISTERS) = (LispObj) saved_registers;

    /* Decrement *interrupt-level* so that callback will initially operate
       without-interrupts. %cs-overflow-callback will increment it again. */
    interrupt_level = ((int) nrs_INTERRUPT_LEVEL.vcell) - (2 << fixnumshift);
    if (interrupt_level >= 0) interrupt_level = (-1 << fixnumshift);
    nrs_INTERRUPT_LEVEL.vcell = (LispObj) interrupt_level;

    /* Don't signal stack overflow errors for VSP & TSP overflows
       until the Lisp code clears this bit */
    ((unsigned) nrs_GC_EVENT_STATUS_BITS.vcell) |= gc_allow_stack_overflows_bit;

    /* Lisp code does most of the work */
    ((ProcPtr)callback_ptr)();

    /* The lisp code has returned. Just need to reinstall the cs-area and
       set the active area of the new cs_area to empty and we can switch back
       to the other thread */
    lisp_global(CURRENT_CS) = (LispObj) a;
    lisp_global(CS_OVERFLOW_LIMIT) = (LispObj) a->softlimit;
    a->younger->active = a->younger->high;
    nrs_INTERRUPT_LEVEL.vcell = (LispObj) (interrupt_level + (2 << fixnumshift));
    YieldToThread (area_ptr_threadID(a)); 

  }
}

area *
allocate_cstack (unsigned size)
{
  ThreadID threadMade;
  area *new_area;
  extern area *register_cstack(BytePtr, unsigned);

  size = (size+4095)&(~4095);           /* round up to multiple of 4096 */
  if (noErr == NewThread(kCooperativeThread,          /* threadStyle */
			 (ThreadEntryProcPtr) cs_stack_switch_startup,     /* threadEntry */
			 NULL,                        /* threadParam */
			 size,                        /* stackSize */
			 kCreateIfNeeded,             /* options */
			 NULL,                        /* threadResult */
			 &threadMade))                /* threadMade */
    {
    new_area = register_cstack(NULL, 0);             
    if (new_area == NULL) {
      DisposeThread(threadMade, NULL, false);
      return(NULL);
    };
    area_ptr_threadID(new_area) = threadMade;
    return new_area;
  } else {
    return NULL;
  };
}

/* Cons up a new thread and a new control stack area.
   If successful, continue execution on the new thread,
   and when that thread switches back to this one, return true.
   If not successful, return false */
Boolean
continue_on_new_thread (ExceptionInformation *xp)
{
  xframe_list xf;
  area *cs = (area *) lisp_global(CURRENT_CS);
  area *new_cs = cs->younger;
  LispObj callback_macptr;

  callback_macptr = nrs_CS_OVERFLOW_CALLBACK.vcell;
  if (!((fulltag_of(callback_macptr) == fulltag_misc) &&
        (header_subtag(header_of(callback_macptr)) == subtag_macptr))) {
    /* No cs-overflow-callback in the Lisp, must signal an error */
    return(false);
  } else {
    /* Link the exception frame into the xframe list */
    xf.prev = (xframe_list *) lisp_global(XFRAME);
    xf.curr = xp;
    lisp_global(XFRAME) = (LispObj) &xf;

    if (new_cs == NULL) {
      int cs_segment_size = unbox_fixnum(nrs_CS_SEGMENT_SIZE.vcell),
          cs_hard_overflow_size = unbox_fixnum(nrs_CS_HARD_OVERFLOW_SIZE.vcell),
          cs_soft_overflow_size = unbox_fixnum(nrs_CS_SOFT_OVERFLOW_SIZE.vcell);
      int stack_size = cs_segment_size + cs_hard_overflow_size + cs_soft_overflow_size;
      new_cs = allocate_area_with_c_gc_context(xp, allocate_cstack, (unsigned) stack_size);
      if (new_cs != NULL) {
        cs->younger = new_cs;
        new_cs->older = cs;
      }
    } else {
      /* Copy registers from xp to lisp globals */
      allocate_area_with_c_gc_context(xp, allocate_no_stack, 0);
    };

    /* Switch to the new thread */
    if (new_cs != NULL) {
      YieldToThread(area_ptr_threadID(new_cs));
    }

    /* Unlink the exception frame from the xframe list */
    lisp_global(XFRAME) = (LispObj) xf.prev;

    return (new_cs != NULL);
  }
}

/* callback to (symbol-value cmain) if it is a macptr, 
   otherwise report cause and function name to console.
   Returns noErr if exception handled OK */
OSStatus
handle_trap(ExceptionInformation *xp, opcode the_trap, pc where)
{
  unsigned  instr, err_arg1 = 0, err_arg2 = 0, err_arg3 = 0;
  int       ra, rs, fn_reg = 0;
  char *    error_msg = NULL;
  char      name[kNameBufLen];
  LispObj   cmain = nrs_CMAIN.vcell;
  Boolean   event_poll_p = false;
  int old_interrupt_level = 0;

  /* If we got here, "the_trap" is either a TWI or a TW instruction.
     It's a TWI instruction iff its major opcode is major_opcode_TWI. */

  /* If it's a "twllt" instruction where RA == sp, it's a failed 
     control stack overflow check.  In that case:
     
     a) We're in "yellow zone" mode if the value of the lisp_global(CS_OVERFLOW_LIMIT)
     is 0xfffffff8.  If we're not already in yellow zone mode, attempt to create
     a new thread and continue execution on its stack. If that fails, call
     signal_stack_soft_overflow to enter yellow zone mode and signal the condition to lisp.
     
     b) If we're already in "yellow zone" mode, then:
     
     1) if the SP is past the current control-stack area's hard overflow limit,
     signal a "hard" stack overflow error (e.g., throw to toplevel as quickly as
     possible. If we aren't in "yellow zone" mode, attempt to continue on another
     thread first.
     
     2) if SP is "well" (> 4K) below its soft overflow limit, set lisp_global(CS_OVERFLOW_LIMIT)
     to its "real" value.  We're out of "yellow zone mode" in this case.
     
     3) Otherwise, do nothing.  We'll continue to trap every time something gets pushed
     on the control stack, so we should try to detect and handle all of these cases
     fairly quickly.  Of course, the trap overhead is going to slow things down quite a bit.
     */

  if (X_opcode_p(the_trap,major_opcode_X31,minor_opcode_TW) &&
      (RA_field(the_trap) == sp) &&
      (TO_field(the_trap) == TO_LO)) {
    area 
      *CS_area = (area *) lisp_global(CURRENT_CS);
      
    unsigned 
      current_SP = xpGPR(xp,sp);

    if (current_SP  < (unsigned) (CS_area->hardlimit)) {
      /* If we're not in soft overflow mode yet, assume that the
         user has set the soft overflow size very small and try to
         continue on another thread before throwing to toplevel */
      if (lisp_global(CS_OVERFLOW_LIMIT == 0xfffffff8) ||
          !continue_on_new_thread(xp)) {
        reset_lisp_process(xp);
      }
    } else {
      if (lisp_global(CS_OVERFLOW_LIMIT) == 0xfffffff8) {
        /* If the control stack is at least 4K away from its soft limit,
           stop trapping.  Else keep trapping. */
        if (current_SP > (unsigned) ((CS_area->softlimit)+4096)) {
          lisp_global(CS_OVERFLOW_LIMIT) = (LispObj) (CS_area->softlimit);
        }
      } else {
        if (!continue_on_new_thread(xp)) {
          signal_stack_soft_overflow(xp, sp);
        }
      }
    }
    
    adjust_exception_pc(xp, 4);
    return noErr;
  } else {
    if (the_trap == LISP_BREAK_INSTRUCTION) {
      char *message =  (char *) xpGPR(xp,3);
      set_xpPC(xp, xpLR(xp));
      if (message == NULL) {
	message = "Lisp Breakpoint";
      }
      Debugger(xp, message);
      return noErr;
    }
    if ((fulltag_of(cmain) == fulltag_misc) &&
        (header_subtag(header_of(cmain)) == subtag_macptr)) {
      /* cmain is a macptr, we can call back to lisp */
      if (the_trap == TWI_instruction(TO_GT,nargs,0)) {
        /* Disable interrupts if we're about to process one */
        event_poll_p = true;    /* remember to turn interrupts back on */
	old_interrupt_level = nrs_INTERRUPT_LEVEL.vcell;
        nrs_INTERRUPT_LEVEL.vcell = (-1 << fixnumshift);
      }
      callback_for_trap(cmain, xp, (unsigned) where, (unsigned) the_trap, (event_poll_p && (old_interrupt_level == (2<<fixnumshift))) ? 1 : 0, 0);
      if (event_poll_p) {
        nrs_INTERRUPT_LEVEL.vcell = 0;
      }
      adjust_exception_pc(xp, 4);
      return(noErr);
    }

    if ((the_trap & OP_MASK) == OP(major_opcode_TWI)) {
      /* TWI.  If the RA field is "nargs", that means that the
         instruction is either a number-of-args check or an
         event-poll.  Otherwise, the trap is some sort of
         typecheck. */

      if (RA_field(the_trap) == nargs) {
        fn_reg = nfn;
        switch (TO_field(the_trap)) {
        case TO_NE:
          error_msg = ( (xpGPR(xp, nargs) < D_field(the_trap))
                       ? "Too few arguments (no opt/rest)"
                       : "Too many arguments (no opt/rest)" );
          break;
       
        case TO_GT:
          error_msg = "Event poll !";
          break;

        case TO_HI:
          error_msg = "Too many arguments (with opt)";
          break;
	
        case TO_LT:
          error_msg = "Too few arguments (with opt/rest/key)";
          break;
	
        default:                /* some weird trap, not ours. */
          fn_reg = 0;
          break;
        }
      } else {
        /* A type or boundp trap of some sort. */
        switch (TO_field(the_trap)) {
        case TO_EQ:
          /* Boundp traps are of the form:
             tweqi rX,unbound
             where some preceding instruction is of the form:
             lwz rX,symbol.value(rY).
             The error message should try to say that rY is unbound. */

          if (D_field(the_trap) == unbound) {
            instr = scan_for_instr(LWZ_instruction(RA_field(the_trap),
                                                   unmasked_register,
                                                   offsetof(lispsymbol,vcell)-fulltag_misc),
                                   D_RT_IMM_MASK,
                                   where);
            if (instr) {
              ra = RA_field(instr);
              if (lisp_reg_p(ra)) {
                error_msg = "Unbound variable: %s";
                (void)symbol_name( xpGPR(xp, ra), name, kNameBufLen );
                err_arg1 = (unsigned)name;
                fn_reg = fn;
              }
            }
          }
          break;

        case TO_NE:
          /* A type check.  If the type (the immediate field of the trap instruction)
             is a header type, an "lbz rX,misc_header_offset(rY)" should precede it,
             in which case we say that "rY is not of header type <type>."  If the
             type is not a header type, then rX should have been set by a preceding
             "clrlwi rX,rY,29/30".  In that case, scan backwards for an RLWINM instruction
             that set rX and report that rY isn't of the indicated type. */
          err_arg2 = D_field(the_trap);
          if (((err_arg2 & fulltagmask) == fulltag_nodeheader) ||
              ((err_arg2 & fulltagmask) == fulltag_immheader)) {
            instr = scan_for_instr(LBZ_instruction(RA_field(the_trap),
                                                   unmasked_register,
                                                   misc_subtag_offset),
                                   D_RT_IMM_MASK,
                                   where);
            if (instr) {
              ra = RA_field(instr);
              if (lisp_reg_p(ra)) {
                error_msg = "value %08X is not of the expected header type %02X";
                err_arg1 = xpGPR(xp, ra);
                fn_reg = fn;
              }
            }
          } else {		
            /* Not a header type, look for rlwinm whose RA field matches the_trap's */
            instr = scan_for_instr((OP(major_opcode_RLWINM) | (the_trap & RA_MASK)),
                                   (OP_MASK | RA_MASK),
                                   where);
            if (instr) {
              rs = RS_field(instr);
              if (lisp_reg_p(rs)) {
                error_msg = "value %08X is not of the expected type %02X";
                err_arg1 = xpGPR(xp, rs);
                fn_reg = fn;
              }
            }
          }
          break;
        }
      }
    } else {
      /* a "TW <to>,ra,rb" instruction."
         twltu sp,rN is stack-overflow on SP.
         twgeu rX,rY is subscript out-of-bounds, which was preceded
         by an "lwz rM,misc_header_offset(rN)" instruction.
         rM may or may not be the same as rY, but no other header
         would have been loaded before the trap. */
      switch (TO_field(the_trap)) {
      case TO_LO:
        if (RA_field(the_trap) == sp) {
          fn_reg = fn;
          error_msg = "Stack overflow! Run away! Run away!";
        }
        break;

      case (TO_HI|TO_EQ):
        instr = scan_for_instr(OP(major_opcode_LWZ) | (D_MASK & misc_header_offset),
                               (OP_MASK | D_MASK),
                               where);
        if (instr) {
          ra = RA_field(instr);
          if (lisp_reg_p(ra)) {
            error_msg = "Bad index %d for vector %08X length %d";
            err_arg1 = unbox_fixnum(xpGPR(xp, RA_field(the_trap)));
            err_arg2 = xpGPR(xp, ra);
            err_arg3 = unbox_fixnum(xpGPR(xp, RB_field(the_trap)));
            fn_reg = fn;
          }
        }
        break;
      }
    }
  

    if (!error_msg) {
      return -1;
    }

    fprintf( stderr, "\nError: ");
    fprintf( stderr, error_msg, err_arg1, err_arg2, err_arg3 );
    fprintf( stderr, "\n");
    if (fn_reg && exception_fn_name( xp, fn_reg, name, kNameBufLen )) {
      fprintf( stderr, "While executing: %s.\n", name );
    }
    fflush( stderr );
    switch( error_action() ) {
    case kDebugger:
      return(-1);
      break;
    case kContinue:
      adjust_exception_pc(xp, 4);
      return(noErr);
      break;
    case kExit:
    default:
      lisp_exit(1);
      break;
    }
    return -1;
  }
}


/* Look at up to TRAP_LOOKUP_TRIES instrs before trap instr for a pattern.
   Stop if subtag_code_vector is encountered. */
unsigned
scan_for_instr( unsigned target, unsigned mask, pc where )
{
  int i = TRAP_LOOKUP_TRIES;

  while( i-- ) {
    unsigned instr = *(--where);
    if ( codevec_hdr_p(instr) ) {
      return 0;
    } else if ( match_instr(instr, mask, target) ) {
      return instr;
    }
  }
  return 0;
}

/* Copy function name to name & terminate, return # chars */
size_t
exception_fn_name( ExceptionInformation *xp, int fn_reg, char *name, size_t name_len )
{
  unsigned the_fn = xpGPR(xp, fn_reg);

  if ((fulltag_of(the_fn) != fulltag_misc) ||
      (header_subtag(header_of(the_fn)) != subtag_function)) {
    non_fatal_error( "exception_fn_name: bogus fn" );
    name[0] = 0;
    return 0;
  }

  if (named_function_p(the_fn)) {
    unsigned the_sym = named_function_name(the_fn);
    /* trust it - if ((fulltag_of(the_sym) == fulltag_misc) &&
       (header_subtag(header_of(the_sym)) == subtag_symbol)) */

    return symbol_name( the_sym, name, name_len );

  } else {			/* unnamed function */
    strcpy( name, "<Anonymous Function>" );
    return strlen(name);
  }
}

/* Make name a c-string of symbol's pname, return length */
size_t
symbol_name( unsigned the_sym, char *name, size_t name_len )
{
  unsigned the_pname = ((lispsymbol *)(untag(the_sym)))->pname;
  /* trust it - if (the_pname) */

  size_t length = header_element_count(header_of(the_pname));
  if (length >= name_len) length = name_len - 1;
  memcpy( (void *)name,
	 (const void *)(the_pname + misc_data_offset),
	 length );
  name[length] = 0;
  return length;
}

void non_fatal_error( char *msg )
{
  fprintf( stderr, "Non-fatal error: %s.\n", msg );
  fflush( stderr );
}

/* The main opcode.  */

int 
is_conditional_trap(opcode instr)
{
  unsigned to = TO_field(instr);
  int is_tw = X_opcode_p(instr,major_opcode_X31,minor_opcode_TW);

#ifndef MACOS
  if (instr == LISP_BREAK_INSTRUCTION) {
    return 1;
  }
#endif
  if (is_tw || major_opcode_p(instr,major_opcode_TWI)) {
    /* A "tw" or "twi" instruction.  To be unconditional, the EQ bit must be set
       in the TO mask and either the register operands (if "tw") are the same or
       either both of the signed or both of the unsigned inequality bits
       must be set. */
    if (! (to & TO_EQ)) {
      return 1;			/* Won't trap on EQ: conditional */
    }
    if (is_tw && (RA_field(instr) == RB_field(instr))) {
      return 0;			/* Will trap on EQ, same regs: unconditional */
    }
    if (((to & (TO_LO|TO_HI)) == (TO_LO|TO_HI)) || 
	((to & (TO_LT|TO_GT)) == (TO_LT|TO_GT))) {
      return 0;			/* Will trap on EQ and either (LT|GT) or (LO|HI) : unconditional */
    }
    return 1;			/* must be conditional */
  }
  return 0;			/* Not "tw" or "twi".  Let debugger have it */
}

OSStatus
handle_error(ExceptionInformation *xp, unsigned errnum, unsigned rb, unsigned continuable, unsigned where)
{
  LispObj   pname;
  LispObj   errdisp = nrs_ERRDISP.vcell;

  if ((fulltag_of(errdisp) == fulltag_misc) &&
      (header_subtag(header_of(errdisp)) == subtag_macptr)) {
    /* errdisp is a macptr, we can call back to lisp */
    callback_for_trap(errdisp, xp, where, errnum, rb, continuable);
    return(0);
    }

  switch (errnum) {
  case error_udf_call:
    rb = xpGPR(xp, fname);
    if ((fulltag_of(rb) == fulltag_misc) &&
	(header_subtag(header_of(rb)) == subtag_symbol)) {
      pname = ((lispsymbol *)(untag(rb)))->pname;
    } else {
      pname = (LispObj)NULL;
    }
      
    fprintf(stderr, "\nERROR: undefined function call: ");
    if (pname) {
      fwrite((const void *)(pname+misc_data_offset),
	     1,
	     header_element_count(header_of(pname)),
	     stderr);
    } else {
      fprintf(stderr, "[can't determine symbol name.]");
    }
    putc('\n',stderr);
    fflush(stderr);
    switch( error_action() ) {
    case kExit:
      lisp_exit(1);
      break;
    case kDebugger:
    default:
      return(-1);
      break;
    }
    break;
  default:
    break;
  }
  return(-1);
}
	       
ErrAction
error_action( void )
{
  /* getchar reads from line start, so end message with \n */
  fprintf( stderr, "\nContinue/Debugger/eXit <enter>?\n" );
  fflush( stderr );

  do {
    int c = toupper( getchar() );
    switch( c ) {
    case 'X':
      return( kExit );
      break;
    case 'D':
      return( kDebugger );
      break;
    case 'C':
      return( kContinue );
      break;
    }
  } while( true );
}



void
signal_handler(int signum, ExceptionInformationPowerPC  *context)
{

  setr2(0);
  if (noErr != PMCL_exception_handler(signum, context)) {
    Bug(context, "Unhandled exception %d at 0x%08x, context->regs at #x%08x", signum, xpPC(context), (int)(context->regs));
  }
}

#ifdef VXWORKS
typedef void (*__sighandler_t) (int);
#define signal vx_signal
extern __sighandler_t (*vx_signal)(int, __sighandler_t);
#endif


void
install_signal_handler(int signo, __sighandler_t handler)
{
#ifdef VXWORKS
  signal (signo, handler);
#else
  struct sigaction sa;
  
#ifdef DARWIN
  sa.sa_handler = (void *)handler;
#else
  sa.sa_sigaction = (void *)handler;
#endif
  sigemptyset(&sa.sa_mask);
  sa.sa_flags = SA_NODEFER | SA_RESTART;

  sigaction(signo, &sa, NULL);
#endif
}

void
install_pmcl_exception_handlers()
{
#ifdef DARWIN
  extern void darwin_exception_init(void);
  
  darwin_exception_init();
#else
  extern int no_sigtrap;
  install_signal_handler(SIGILL, (__sighandler_t)signal_handler);
  if (no_sigtrap != 1) {
    install_signal_handler(SIGTRAP, (__sighandler_t)signal_handler);
  }
  install_signal_handler(SIGBUS,  (__sighandler_t)signal_handler);
  install_signal_handler(SIGSEGV, (__sighandler_t)signal_handler);
  install_signal_handler(SIGFPE, (__sighandler_t)signal_handler);
#endif
  signal(SIGPIPE, SIG_IGN);
}


void
unprotect_all_areas()
{
  protected_area_ptr p;

  for(p = AllProtectedAreas, AllProtectedAreas = NULL; p; p = p->next) {
    unprotect_area(p);
  }
}

Boolean
exception_filter_installed_p()
{
  return true;
#if 0
  return installed_exception_filter == PMCL_exception_filter;
#endif
}

void
exception_cleanup()
{
#ifdef VXWORKS
  extern void (*vxlow_cleanup)(void);
  vxlow_cleanup();
#endif
#ifdef LINUX
  signal(SIGALRM, SIG_IGN);
#endif
  unprotect_all_areas();
}

void
exception_init()
{
#ifdef VXWORKS
  extern void (*init_vxlow)(void);
  init_vxlow();
#endif
  install_pmcl_exception_handlers();
}


void
Bug(ExceptionInformation *xp, const char *format, ...)
{
  va_list args;
  
  va_start(args, format);
  vfprintf(stderr, format, args);

  fflush(stderr);
  switch( error_action() ) {
  case kExit:
    lisp_exit(1);
    break;
  case kDebugger:
    Debugger(xp, "Bug() called");
    break;
  default:
    break;
  }
}

void
lisp_bug(char *string)
{
  Bug(NULL, "Bug in MCL-PPC system code:\n%s", string);
}

#ifdef VXWORKS


#if 0
#define SYNC() __asm__ volatile("sync");
#define TLBSYNC() __asm__ volatile("tlbsync");
#define EIEIO() __asm__ volatile("eieio");

PTE *
find_pte_in_pteg(u_int match, PTE *pteg)
{
  int i;

  for (i = 0; i < MMU_PTE_BY_PTEG; i++, pteg++) {
    if (pteg->bytes.word0 == match) {
      return pteg;
    }
  }
  return (PTE *) 0;
}


extern u_int mmuPpcSdr1Get(void);
extern void mmuPpcTlbie(void *);

PTE *
find_pte(u_int addr)
{
  u_int segment_id, primary_hash, secondary_hash;
  u_int primary_match, secondary_match;
  u_int sdr1, hash_table_base, hash_function_mask;
  PTE *pte;

  sdr1 = mmuPpcSdr1Get();
  hash_table_base = sdr1 & MMU_SDR1_HTABORG_MASK;
  hash_function_mask = 0xffc0 | ((sdr1 & MMU_SDR1_HTABMASK_MASK) 
				 << MMU_SDR1_HTABMASK_SHIFT);
  segment_id = addr >> 28;

  primary_hash = segment_id ^ ((addr >> 12) & 0xffff);
  primary_hash = hash_table_base +
    ((primary_hash << 6) & hash_function_mask);

  primary_match = MMU_STATE_VALID | (segment_id << 7) | (0x3f & (addr >> 22));

  pte = find_pte_in_pteg(primary_match, (PTE *)primary_hash);
  if (pte == NULL) {
    secondary_match = primary_match | (1 << 6);
    secondary_hash = primary_hash ^ hash_function_mask;
    pte = find_pte_in_pteg(secondary_match, (PTE *) secondary_hash);
  }
  return pte;
}

void
set_pte_prot(void *addr, PTE *pte, int prot)
{
  int oldMSR = intLock();
  vxMsrSet(oldMSR & (~_PPC_MSR_EE) & (~_PPC_MSR_DR));
  pte->field.v = 0;
  SYNC();
  mmuPpcTlbie(addr);
  TLBSYNC();
  SYNC();
  pte->field.pp = prot;
  EIEIO();
  pte->field.v = 1;
  SYNC();
  vxMsrSet(oldMSR);
}


/* Mprotect. */
int 
mprotect(void *addr, size_t len, int prot)
{
  unsigned
    base = (unsigned) addr,
    low = base & ~4095,
    limit = (base + len + 4095) & (~4095),
    page,
    ppc_prot;

  switch (prot) {
  case (PROT_READ | PROT_WRITE):
    ppc_prot = 2;
    break;
  case PROT_READ:
    ppc_prot = 3;
    break;
  default:
    return -1;
  }

  for (page = low; page < limit; page += 4096) {
    PTE *pte = find_pte(page);
    if (pte == NULL) {
      /* Undo whatever we've done so far, then fail */
      limit = page;
      ppc_prot ^=1;
      for (page = low; page < limit; page += 4096) {
	pte = find_pte(page);
	if (pte != NULL) {
	  set_pte_prot((void *)page, pte, ppc_prot);
	}
      }
      return -1;
    }
    set_pte_prot(addr, pte, ppc_prot);
  }
  return 0;
}
#else
#include <vmLib.h>
int 
mprotect(void *addr, size_t len, int prot)
{
  extern void setHID0(int);
  extern int getHID0(void);
  int saveHID0 = getHID0();
  int vxprot = (prot == (PROT_READ | PROT_WRITE)) ? 
    VM_STATE_WRITABLE : VM_STATE_WRITABLE_NOT;
  int result = vmBaseStateSet(NULL, addr, len, VM_STATE_MASK_WRITABLE, vxprot);
  /*
  setHID0(saveHID0 | 0x4400);
  setHID0(saveHID0 | 0x4000);
  */
  setHID0(saveHID0);
  return result;
}
#endif
#endif

#ifdef DARWIN
/*
  Mach's exception mechanism works a little better than its signal
  mechanism (and, not incidentally, it gets along with GDB a lot
  better.

  Initially, we install an exception handler to handle each native
  thread's exceptions.  (Since there's only one native thread, that's
  not too much overhead.)  Installing a Mach exception handler basically
  involves creating a thread to listen for exception messages from the
  Mach kernel.  If/when we have multiple native threads, we may want
  to make the exception handler global (task-specific instead of
  thread-specific.)

  A few exceptions can be handled directly in the handler thread;
  others require that we resume the user thread (and that the
  exception thread resumes listening for exceptions.)  The user
  thread might eventually want to return to the original context
  (possibly modified somewhat.)

  As it turns out, the simplest way to force the faulting user
  thread to handle its own exceptions is to do pretty much what
  signal() does: the exception handlng thread sets up a sigcontext
  on the user thread's stack and forces the user thread to resume
  execution as if a signal handler had been called with that
  context as an argument.  We can use a distinguished UUO at a
  distinguished address to do something like sigreturn(); that'll
  have the effect of resuming the user thread's execution in
  the (pseudo-) signal context.

  Since:
    a) we have miles of code in C and in Lisp that knows how to
    deal with Linux sigcontexts
    b) Linux sigcontexts contain a little more useful information
    (the DAR, DSISR, etc.) than their Darwin counterparts
    c) we have to create a sigcontext ourselves when calling out
    to the user thread: we aren't really generating a signal, just
    leveraging existing signal-handling code.

  we create a Linux sigcontext struct.

  Simple ?  Hopefully from the outside it is ...
*/

#define	C_REDZONE_LEN		224
#define	C_STK_ALIGN			16
#define C_PARAMSAVE_LEN		64
#define	C_LINKAGE_LEN		48
#define TRUNC_DOWN(a,b,c)  (((((unsigned)a)-(b))/(c)) * (c))
#include <mach/mach.h>
#include <mach/mach_error.h>
#include <mach/machine/thread_state.h>
#include <mach/machine/thread_status.h>

#define MACH_CHECK_ERROR(x) if (x != KERN_SUCCESS) {abort();}

/* a distinguished UUO at a distinguished address */
extern void pseudo_sigreturn(struct linux_sigcontext_struct *);


/* This code runs in the exception handling thread, in response
   to an attempt to execute the UU0 at "pseudo_sigreturn" (e.g.,
   in response to a call to pseudo_sigreturn() from the specified
   user thread.
   Find that context (the user thread's R3 points to it), then
   use that context to set the user thread's state.  When this
   function's caller returns, the Mach kernel will resume the
   user thread.
*/

kern_return_t
do_pseudo_sigreturn(mach_port_t thread)
{
  ppc_thread_state_t ts;
  mach_msg_type_number_t thread_state_count;
  struct linux_sigcontext_struct *lss;
  struct pt_regs *ptr;
  int i, j;

  thread_state_count = MACHINE_THREAD_STATE_COUNT;
  thread_get_state(thread, 
		   MACHINE_THREAD_STATE,	/* GPRs, some SPRs  */
		   (thread_state_t)&ts,
		   &thread_state_count);
  lss = (struct linux_sigcontext_struct *) ts.r3;
  ptr = lss->regs;
  /* Set the thread's FP state from the lss */
  thread_set_state(thread,
		   PPC_FLOAT_STATE,
		   (thread_state_t)&(ptr->gpr[PT_FPR0]),
		   PPC_FLOAT_STATE_COUNT);

  /* Copy the GPRs */
  for (i = 0, j = 2; i < 32; i++, j++) {
    ((unsigned *)&ts)[j] = ptr->gpr[i] ;
  }
  /* and the PC & other SPRs */
  ts.srr0 =  ptr->nip;
  ts.srr1 =  ptr->msr;
  ts.ctr =  ptr->ctr;
  ts.lr = ptr->link;
  ts.xer = ptr->xer;
  ts.cr = ptr->ccr;

  /* The thread'll be as good as new ... */
  thread_set_state(thread, 
		   MACHINE_THREAD_STATE,
		   (thread_state_t)&ts,
		   MACHINE_THREAD_STATE_COUNT);

  return KERN_SUCCESS;

}  

/*
  This code sets up the user thread so that it executes a "pseudo-signal
  handler" function when it resumes.  Create a linux sigcontext struct
  on the thread's stack and pass it as an argument to the pseudo-signal
  handler.

  If the handler wants to "return to" the context, it has to explicitly
  call pseudo_sigreturn().  If we were a little more clever here, we
  could set things up so that pseudo_sigreturn() was "returned to" when
  the handler returned.

  If the handler invokes code that throws (or otherwise never sigreturn()'s
  to the context), that's fine.
*/

void
setup_signal_frame(mach_port_t thread,
		   void *handler_address,
		   int signum,
		   int code)
{
  ppc_thread_state_t ts;
  ppc_exception_state_t xs;
  mach_msg_type_number_t thread_state_count;
  struct linux_sigcontext_struct *lss;
  struct pt_regs *ptr;
  unsigned stackp, backlink;
  int i, j;

  thread_state_count = MACHINE_THREAD_STATE_COUNT;
  thread_get_state(thread, 
		   MACHINE_THREAD_STATE,	/* GPRs, some SPRs  */
		   (thread_state_t)&ts,
		   &thread_state_count);

  stackp = ts.r1;
  backlink = stackp;
  stackp = TRUNC_DOWN(stackp, C_REDZONE_LEN, C_STK_ALIGN);
  stackp -= sizeof(*lss);
  lss = (struct linux_sigcontext_struct *) stackp;
  /* pt_regs struct should be aligned to 8-byte boundary */
  stackp = TRUNC_DOWN(stackp, (PT_FPSCR+1)*sizeof(unsigned), 8);
  ptr = (struct pt_regs *) stackp;

  thread_state_count = PPC_FLOAT_STATE_COUNT;
  thread_get_state(thread,
		   PPC_FLOAT_STATE,
		   (thread_state_t)&(ptr->gpr[PT_FPR0]),
		   &thread_state_count);

  for (i = 0, j = 2; i < 32; i++, j++) {
    ptr->gpr[i] = ((unsigned *)&ts)[j];
  }
  ptr->nip = ts.srr0;
  ptr->msr = ts.srr1;
  ptr->ctr = ts.ctr;
  ptr->link = ts.lr;
  ptr->xer = ts.xer;
  ptr->ccr = ts.cr;

  thread_state_count = PPC_EXCEPTION_STATE_COUNT;
  thread_get_state(thread,
		   PPC_EXCEPTION_STATE,
		   (thread_state_t)(&xs),
		   &thread_state_count);

  ptr->dar = xs.dar;
  ptr->dsisr = xs.dsisr;
  ptr->trap = (xs.exception)<<8;

  lss->regs = ptr;
  lss->signal = signum;
  lss->handler = (unsigned long) handler_address;
  lss->oldmask = 0;

  stackp = TRUNC_DOWN(stackp, C_PARAMSAVE_LEN, C_STK_ALIGN);
  stackp -= C_LINKAGE_LEN;
  *(unsigned *)stackp = backlink;

  /* 
     It seems like we've created a (Linux) sigcontext on the thread's
     stack.  Set things up so that we call the handler (with appropriate
     args) when the thread's resumed.
  */

  ts.srr0 = (int) handler_address;
  ts.r1 = stackp;
  ts.r3 = signum;
  ts.r4 = code;
  ts.r5 = (int)lss;

  /* Fix this: set the LR to point to something which returns to the
     (possibly modified) lss.
  */

  thread_set_state(thread, 
		   MACHINE_THREAD_STATE,
		   (thread_state_t)&ts,
		   MACHINE_THREAD_STATE_COUNT);
}


void
pseudo_signal_handler(int signum,
		      int code,
		      struct linux_sigcontext_struct *context)
{
  signal_handler(signum, context);
  pseudo_sigreturn(context);	/* Don't fall off the end  */
}


/*
  This function runs in the exception handling thread.  It's
  called (by this precise name) from the library function "exc_server()"
  when the thread's exception ports are set up.  (exc_server() is called
  via mach_msg_server(), which is a function that waits for and dispatches
  on exception messages from the Mach kernel.)

  This checks to see if the exception was caused by a pseudo_sigreturn()
  UUO; if so, it arranges for the thread to have its state restored
  from the specified context.

  Otherwise, it tries to map the exception to a signal number and
  arranges that the thread run a "pseudo signal handler" to handle
  the exception.

  Some exceptions could and should be handled here directly.
*/

kern_return_t
catch_exception_raise(mach_port_t exception_port,
		      mach_port_t thread,
		      mach_port_t task, 
		      exception_type_t exception,
		      exception_data_t code_vector,
		      mach_msg_type_number_t code_count)
{
  int signum = 0, code = *code_vector;

  if ((exception == EXC_BAD_INSTRUCTION) &&
      (code_vector[0] == EXC_PPC_UNIPL_INST) &&
      (code_vector[1] == (int)pseudo_sigreturn)) {
    return do_pseudo_sigreturn(thread);
  }
  switch (exception) {
  case EXC_BAD_ACCESS:
    signum = SIGSEGV;
    break;

  case EXC_BAD_INSTRUCTION:
    signum = SIGILL;
    break;

  case EXC_SOFTWARE:
    if (code == EXC_PPC_TRAP) {
      signum = SIGTRAP;
    }
    break;
  default:
    break;
  }
  if (signum) {
    setup_signal_frame(thread,
		       (void *)pseudo_signal_handler,
		       signum,
		       code);
      return KERN_SUCCESS;
  }
  return 17;
}


/*
  Create a Mach thread to run a message-dispatching loop that'll
  ultimately call 'catch_exception_raise()' in response to an
  exception.
*/

thread_act_t
create_exception_thread(void (*func)(void *), void *arg, unsigned stacksize)
{
  kern_return_t kret;
  char *stackbuf, *stack_bottom;
  ppc_thread_state_t initial_state;
  thread_act_t new_thread;

  kret = vm_allocate(mach_task_self(),
		     (vm_address_t *)&stackbuf,
		     stacksize,
		     TRUE);

  if (kret != KERN_SUCCESS) {
    abort();
  }
  stack_bottom = stackbuf + stacksize;
  initial_state.r1 = (int) stack_bottom - 100;
  initial_state.r3 = (int) arg;
  initial_state.srr0 = (int)func;
  initial_state.srr1 = 0;
  kret = thread_create_running(mach_task_self(),
			       PPC_THREAD_STATE,
			       (thread_state_t)&initial_state,
			       PPC_THREAD_STATE_COUNT,
			       &new_thread);
  if (kret != KERN_SUCCESS) {
    abort();
  }
  return new_thread;
}

/*
  The initial function for an exception-handling thread.
*/

void
exception_handler_proc(void *arg)
{
  extern boolean_t exc_server();
  mach_port_t p = (mach_port_t) arg;

  mach_msg_server(exc_server, 256, p, 0);
  /* Should never return. */
  abort();
}

/*
  Setup a new thread to handle those exceptions specified by
  the mask "which".  This involves creating a special Mach
  message port, telling the Mach kernel to send exception
  messages for the calling thread to that port, and setting
  up a handler thread which listens for and responds to
  those messages.
*/

kern_return_t
setup_mach_exception_handling(exception_mask_t which)
{
  mach_port_t 
    thread_exception_port,
    thread_self = mach_thread_self(),
    task_self = mach_task_self();
  kern_return_t kret;

  kret = mach_port_allocate(task_self,
			    MACH_PORT_RIGHT_RECEIVE,
			    &thread_exception_port);
  MACH_CHECK_ERROR(kret);

  kret = mach_port_insert_right(task_self,
				thread_exception_port,
				thread_exception_port,
				MACH_MSG_TYPE_MAKE_SEND);
  MACH_CHECK_ERROR(kret);

  kret = thread_set_exception_ports(thread_self,
				    which,
				    thread_exception_port,
				    EXCEPTION_DEFAULT,
				    THREAD_STATE_NONE);
  if (kret == KERN_SUCCESS) {
    create_exception_thread(exception_handler_proc, 
			    (void *)thread_exception_port,
			    8192);
  }
  return kret;
}

/*
  This might need to be done for each native thread.
*/
void
darwin_exception_init()
{
  kern_return_t kret;

  /* Should also handle FPU exceptions, others ? */
  if ((kret = setup_mach_exception_handling(EXC_MASK_BAD_ACCESS | 
					    EXC_MASK_SOFTWARE |
					    EXC_MASK_BAD_INSTRUCTION))
      != KERN_SUCCESS) {
    fprintf(stderr, "Couldn't setup exception handler - error = %d\n", kret);
    exit(-1);
  }
}
#endif

