/*
   Copyright (C) 1994-2001 Digitool, Inc
   This file is part of Opensourced MCL.

   Opensourced MCL is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   Opensourced MCL is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdlib.h>
#include <unistd.h>
#include <sys/mman.h>
#include <stdio.h>
#include "Threads.h"
#include "lisp.h"
#include "qt.h"

/* Our notion of a Mac thread.  The next and prev pointers
   are just to there to make insertion and deletion of 
   mac_thread structures easier; there's no implied ordering,
   aside from the fact that "current_mac_thread" points to
   the currently active mac_thread.
*/

typedef struct mac_thread
{
  struct mac_thread *prev;
  struct mac_thread *next;
  ThreadID id;
  void *stack_buf;		/* what was allocated, so we can free it */
  qt_t *q;
} mac_thread;

mac_thread  main_thread, *current_mac_thread;
int next_mac_thread_id = 1000;



Ptr
create_stack(int size)
{
#if defined(LINUX) || defined(DARWIN)
  Ptr p = (Ptr) mmap(NULL,
		     (size_t)size,
		     PROT_READ | PROT_WRITE | PROT_EXEC,
		     MAP_PRIVATE | MAP_ANON,
		     -1,	/* Darwin insists on this when not mmap()ing
				 a real fd */
		     0);
  if (p != (Ptr)(-1)) {
    *((size_t *)p) = size;
    return p;
  }
  allocation_failure(true, size);

#endif
#ifdef VXWORKS
  return (Ptr)allocate(size);
#endif
}
  
void *
allocate_stack(unsigned size)
{
  return create_stack(size);
}

void
free_stack(void *s)
{
  size_t size = *((size_t *)s);
  munmap(s, size);
}

void
init_mac_threads(void * stack_base)
{
  current_mac_thread = &main_thread;
  current_mac_thread->next = 
    current_mac_thread->prev =
    current_mac_thread;
  current_mac_thread->id = (ThreadID) kApplicationThreadID;
  current_mac_thread->stack_buf = stack_base;
  current_mac_thread->q = NULL;
}

mac_thread *
find_mac_thread(ThreadID id) 
{
  mac_thread *p = current_mac_thread, *q = p;

  if (id == (ThreadID)kCurrentThreadID) {
    return p;
  }
  do {
    if (p->id == id) {
      return p;
    }
    p = p->next;
  } while (p != q);
  return NULL;
}
void
qt_error (void)
{
  extern void abort(void);

  abort();
}
   
void
mac_thread_only(void *userparam, void *m, qt_userf_t *f)
{
  current_mac_thread = (mac_thread *)m;
  (*f)(userparam);
  qt_error();
}

void
mac_thread_insert(mac_thread *m)
{
  mac_thread *next = current_mac_thread,
    *prev = next->prev;

  prev->next = m;
  next->prev = m;
  m->next = next;
  m->prev = prev;
}

mac_thread *
mac_thread_create(Size stacksize, qt_userf_t *f, void *param)
{
  mac_thread *m = (mac_thread *) allocate(sizeof(mac_thread));
  void *s = allocate_stack(stacksize);
#if 0
  fprintf(stderr, "<<<Note: created new stack segment of size %d>>>\n", stacksize);
#endif
  if (m) {
    if (s == NULL) {
      deallocate(m);
      return NULL;
    }
    m->id = (ThreadID) next_mac_thread_id++;
    m->stack_buf = s;
    m->q = QT_SP(s, stacksize);
    m->q = QT_ARGS(m->q, param, m, f, mac_thread_only);
    mac_thread_insert(m);
  }
  return m;
}

void *
mac_thread_yieldhelp(qt_t *q, void *old, void *param)
{
  ((mac_thread *)old)->q = q;
  return param;
}

void
mac_thread_yield(mac_thread *new)
{
  mac_thread *old = current_mac_thread;
  current_mac_thread = new;
  QT_BLOCK(mac_thread_yieldhelp, old, NULL, new->q);
}

/* 
   This code pretends to be as much of the MacOS Thread
   Manager as the lisp uses: it handles thread creation,
   destruction, and context switch.
*/

OSErr
NewThread(ThreadStyle style, 
	  ThreadEntryProcPtr entry,
	  void *param, 
	  Size stacksize, 
	  ThreadOptions options,
	  void **resultP,
	  ThreadID *idP)
{
  OSErr err = noErr;
  mac_thread *m;
  if (style != kCooperativeThread) {
    return -50;
  }
  m = mac_thread_create(stacksize, (qt_userf_t *)entry, param);
  if (m != NULL) {
    *idP = m->id;
    return 0;
  }
  return -50;
}

OSErr
DisposeThread(ThreadID id, void *result, Boolean recycle)
{
  mac_thread *m = find_mac_thread(id),
    *prev,
    *next;

  if ((m == NULL) ||
      (m == current_mac_thread) ||
      (m->id == kApplicationThreadID)) {
    return -50;
  }
  next = m->next;
  prev = m->prev;
  
  prev->next = next;
  next->prev = prev;
  free_stack(m->stack_buf);
  deallocate((void *) m);
  return noErr;
}

OSErr
YieldToThread(ThreadID id)
{
  mac_thread *m = find_mac_thread(id);
  if ((m != NULL) && (m != current_mac_thread)) {
    mac_thread_yield(m);
    return noErr;
  }
  return -50;
}
  
OSErr
ThreadCurrentStackSpace(ThreadID id, unsigned *resultP)
{
  mac_thread *m = find_mac_thread(id);
  unsigned size = 0;
  OSErr result = noErr;

  if (m == NULL) {
    result = -50;
  } else {
    if (m == current_mac_thread) {
      size = (unsigned) ((char *)(&m) - (char *) m->stack_buf);
    } else {
      size = (unsigned) ((char *) m->q - (char *) m->stack_buf);
    }
  }
  if (resultP != NULL) {
    *resultP = size;
  }
  return result;
}

void
qt_null (void)
{
}


