/*
 * Copyright (c) 2000, Amnon BARAK (amnon@cs.huji.ac.il). All rights reserved.
 *
 *	MOSIX $Id: mon.c,v 1.13 2001/03/06 01:32:40 amnons Exp $
 *
 * Permission to use, copy and distribute this software is hereby granted 
 * under the terms of version 2 or any later version of the GNU General Public
 * License, as published by the Free Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED IN ITS "AS IS" CONDITION, WITH NO WARRANTY
 * WHATSOEVER. NO LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING
 * FROM THE USE OF THIS SOFTWARE WILL BE ACCEPTED.
 */
/*
 * Author(s): Amnon Shiloh
 */

#include <stdio.h>
#include <stdlib.h>
#include <termio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <linux/mosctl.h>
#include <sys/time.h>
#include <curses.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>

#define KEY_DOWN	0402		/* Sent by terminal down arrow key */
#define KEY_UP		0403		/* Sent by terminal up arrow key */
#define KEY_LEFT	0404		/* Sent by terminal left arrow key */
#define KEY_RIGHT	0405		/* Sent by terminal right arrow key */
#define KEY_HOME	0406		/* Sent by home key. */
#define KEY_NPAGE	0522		/* Sent by next-page key */
#define KEY_PPAGE	0523		/* Sent by previous-page key */
#define KEY_ENTER	0527		/* Enter/send (unreliable) */
#define KEY_A1		0534		/* Upper left of keypad */
#define KEY_A3		0535		/* Upper right of keypad */
#define KEY_B2		0536		/* Center of keypad */
#define KEY_C1		0537		/* Lower left of keypad */
#define KEY_C3		0540		/* Lower right of keypad */
#define KEY_END		0550		/* end key */


#define HELP_STR1 "MOSIX Load Monitor Version 1.3, Copyright (c) 1995-2000 MOSIX GROUP HUJI.\n\n \
Each load unit represents about one running process on a standard processor.\n \
(slower processors show more load-units per process, while faster processors\n \
 and SMP-nodes show proportinally less)\n\n \
For larger systems (with 10 or more nodes), node-numbers can be shown\n \
either horizontally (better looking) or vertically (to see more at a time).\n \
To force horizontal/vertical numbering run 'mon -w', 'mon -v' or 'mon -V'\n \
respectively.\n \
To display the total number of operational nodes, run 'mon -t'.\n \
To begin the display at a particular node-number, run 'mon +{node_number}'.\n \
To display also dead (configured but not responding) nodes, run 'mon -d'.\n\n \
While displaying the loads, you may use the following keys:\n \
q or Q\t\tQuit program\n \
w/v/V/a\thorizontal (wide), vertical, super-vertical (tight) or\n \
\t\tautomatic selection of numbering\n \
s\t\tshow processor speeds, rather than loads\n \
m\t\tshow memory (used out of total) rather than loads\n \
r\t\tshow memory (raw used/free out of total) rather than loads\n \
u\t\tshow utilizability percentage rather than loads\n \
l\t\tshow loads again\n"

#define	HELP_STR2 "\
Additional keys:\n\n \
d\t\tshow also dead (configured but not-responding) nodes\n \
D\t\tstop showing dead nodes\n \
t\t\ttoggle display of the number of operational nodes\n \
\t\t(not recommended on very large clusters - will be very slow)\n \
y\t\tshow the yardstick in use (eg. speed of a standard processor)\n \
Enter\t\tredraw the screen\n\n \
When not all nodes fit on the screen, press:\n\n \
Left/Right Arrow\tmove one node to the left or right\n \
n or p\t\t\tmove one screen to the left or right\n"

void find_npe();
void adjust_first();
void ch_alarm();
void not_yet_ch(int ch);
void help();
void yardstick();
void sleep_or_input(int secs);
void cb(char* msg, int y, int x);
int readc_half_second();
int is_input();
int my_getch();


void onint(i)
{
	clear();
	echo();
	nocbreak();
	system("stty -cbreak echo");
	move(LINES-1, 0);
	refresh();
	endwin();
	ioctl(0, TCIOFLUSH, 0);
	exit(i);
}

void
not_mosix()
{
	fprintf(stderr, "mon: This is NOT a MOSIX system!\n");
	exit(1);
}

#define COL_FROM_0          7
#define MAX_SILENT		(-5)
#define	VERY_DEAD		(MAX_SILENT-1)

int screen_width;
struct mosix_info info[MOSIX_MAX+1];
float load[MOSIX_MAX+1], other[MOSIX_MAX+1];
char valid[MOSIX_MAX+1];

#define	RECSZ	sizeof(struct mosix_info)

enum
{
	D_GETLOAD,
	D_GETSPEED,
	D_GETUTIL,
	D_GETMEM,
	D_GETBMEM
} item = D_GETLOAD;

int ifd;	/* input file */
int npe;
int first = 1;

int main(int na, char** argv)
{
	int l, col;
	register int i, j, k;
	float max, curmax = 0;
	int cool = 0;
	int dead;
	struct winsize ws;
	int vflg = 0, wflg = 0, tflg = 0;
	int dflg = 0;
	int last;
	int base;	/* baseline location */
	int ver = 0;
	int wid;	/* width of processor numbers */
	char need_count = 1;
	int tot=0, tot_display=0, cpus_display=0;
	char *fmt;
	unsigned int turns = 0;
	char space, ospace;
	int rev;
	char inp[100];
	int first_read, nread;

	if((ifd = open("/proc/mosix/info/infos", 0)) == -1)
		not_mosix();
	while(na > 1 && (argv[1][0] == '-' || argv[1][0] == '+'))
	{
		if(argv[1][0] == '+')
		{
			first = atoi(&argv[1][1]);
			if(first < 1 || first > MOSIX_MAX)
				goto help;
			for(j = 1 ; argv[1][j] >= '0' && argv[1][j] <= '9' ; j++)
				;
			if(argv[1][j])
				goto help;
		}
		else
		for(j = 1 ; argv[1][j] ; j++)
		switch(argv[1][j])
		{
			default:
			help:
				printf("%s\n", HELP_STR1);
				if(isatty(0))
				{
					printf("Press <Enter> to continue...");
					fflush(stdout);
					fgets(inp, 100, stdin);
					printf("\n");
				}
				printf("%s", HELP_STR2);
				exit(0);
			case 'v':
				vflg = 1;
				wflg = 0;
				break;
			case 'V':
				vflg = 2;
				wflg = 0;
				break;
			case 'w':
			case 'W':
				wflg = 1;
				vflg = 0;
				break;
			case 't':
			case 'T':
				tflg = 1;
				break;
			case 'd':
				dflg = 1;
				break;
		}
		na--;
		argv++;
	}
	if(na > 1)
		goto help;

	find_npe();

	signal(SIGINT, (sig_t)onint);
	signal(SIGQUIT, (sig_t)onint);
	initscr();
	noecho();
	cbreak();
	system("stty cbreak -echo");

	for(i=0 ; i <= npe ; i++)
		load[i] = 0;

	while(1)
	{
		first_read = nread = 0;
		/* get the new (if changed) win size  */
		if(ioctl(0,TIOCGWINSZ,(char *)&ws) == -1)
			perror("ioctl"),onint(1);

		if((COLS != ws.ws_col) || (LINES != ws.ws_row)) 
		{
			initscr();
			need_count = 1;
		}
		if(tflg)
		{
			lseek(ifd, 0l, 0);
			j = read(ifd, &info[1], npe * RECSZ) / RECSZ;
			if(j < npe)
			{
				npe = j;
				adjust_first();
			}
			first_read = 1;
			nread = npe;
			for(tot_display = 0,cpus_display = 0,i = 1 ; i <= npe ; i++)
			if(info[i].status & DS_MOSIX_UP)
			{
				tot_display++;
				cpus_display += info[i].ncpus;
				valid[i] = 1;
				load[i] = -1;
			}
			else if((info[i].status & DS_MOSIX_DEF) &&
				valid[i] > (turns >= -MAX_SILENT ? MAX_SILENT :
					-(long long)turns))
			{
				tot_display++;
			        cpus_display += info[i].ncpus;
			}
		}
		if(need_count && !vflg && !wflg)
		{
			if(need_count && !vflg && !wflg)
				find_npe();
			k = (COLS - COL_FROM_0 - 1) + 1;
			tot = 0;
			/* we only need to answer whether there are at least k
			 * nodes up ahead or not:  once we have the answer,
			 * stop reading.  Of course, with "tflg" we have already
			 * read everything */
			first_read = first;
			if(!tflg)
				nread = 0;
			for(i = first ; i <= npe && tot < k ; i += k)
			{
				last = i + k - 1;
				if(last > npe)
					last = npe;
				if(!tflg)
				{
					lseek(ifd, i-1, 0);
					j = read(ifd, &info[i],
						(last - i + 1) * RECSZ) / RECSZ;
					if(j < (last - i + 1))
						last = npe = i + j - 1;
					nread = last + 1 - first;
				}
				for(j = i ; j <= last && tot < k ; j++)
				if(info[j].status &
					(dflg ? DS_MOSIX_DEF : DS_MOSIX_UP))
					tot++;
			}
			need_count = 0;
		}

		/* calculate the screen_width */
		wid = 1 + (npe > 9) + (npe > 99) + (npe > 999) + (npe > 9999);
		k = (COLS - COL_FROM_0 - 1) / (wid+1);
		if(vflg)
		{
			vert:
			base = LINES - 2 - wid;
			ver = 1;
			if(vflg == 2)
			{
				wid = 0;
				screen_width = (COLS - COL_FROM_0 - 1);
			}
			else
			{
				wid = 1;
				screen_width = (COLS - COL_FROM_0 - 1) / 2;
			}
			adjust_first();
		}
		else if(wflg || k >= tot || (COLS - COL_FROM_0 - 1) < tot)
		{
			ver = 0;
			screen_width = k;
			base = LINES - 3;
		}
		else
			goto vert;

		move(0,0);
		clrtobot();
		if(item == D_GETMEM || item == D_GETBMEM)
		{
			if(item == D_GETBMEM)
			{
				fmt = "Raw Used Memory";
				i = base / 2 - 8;
				if(i < 1)
					i = 1;
			}
			else
			{
				fmt = "Used Memory";
				i = base / 2 - 6;
				j = 1;
			}
			for( ; *fmt ; i++)
			{
				move(i, 0);
				addch(*fmt++);
			}
			while(i == base/4 || i == base/2 || i == base*3/4)
				i++;
			if(i < base)
			{
				move(i, 2);
				addstr("(MB)");
			}
		}
		else if(item == D_GETLOAD)
		{
			fmt = "LOAD";
			for(i = base / 2 - 3 ; *fmt ; i += 2)
			{
				move(i, 0);
				addch(*fmt++);
			}
		}
		else if(item == D_GETSPEED)
		{
			fmt = "SPEED";
			for(i = base / 2 - 4 ; *fmt ; i += 2)
			{
				move(i, 0);
				addch(*fmt++);
			}
		}
		else
		{
			fmt = "Utilizability";
			for(i = base/2 - 7 ; *fmt ; i++)
			{
				move(i, 0);
				addch(*fmt++);
			}
		}
		for(i = 0 ; i < base ; i++)
		{
			move(i, COL_FROM_0);
			addch('|');
		}
		move(base, COL_FROM_0);
		for(i = screen_width*(wid+1)+1 ; i > 0 ; i--)
			addch('-'); 
		j = (COLS - COL_FROM_0 - 8) / 2;
		if(j < COL_FROM_0)
			j = COL_FROM_0;
		else if(j > 27)
			j = 27;
		move(LINES-2, 0);
		addstr("Node #");
		if(tflg)
		{
			move(LINES-1, 0);
			printw(tot_display?"[Total %d] [CPUs %d]": "[Mosix Not Configured]", tot_display, cpus_display);
		}
		max = 0;
		dead = 0;
		turns++;
		if(!tflg)
			lseek(ifd, first-1, 0);
		for(i = first ; (i<(first+screen_width+dead)) && (i<=npe) ; i++)
		{
		    /* if we receive 0 we don't remove the load of
		       the process completely: only divide it by two */

			if(i >= first_read && i < first_read + nread)
				;	/* already there, no need to read */
			else
			{
				/*try to compute optimal read-ahead (not easy)*/
				if(!nread)
					first_read = i;
				/* guess same # of dead as already found */
				j = first+screen_width+dead+dead - i;
				if(j > npe - i + 1)
					j = npe - i + 1;
				j = read(ifd, &info[i], j * RECSZ) / RECSZ;
				if(j <= 0)
				{
					npe = i - 1;
					break;
				}
				nread += j;
			}
			if(info[i].status & DS_MOSIX_UP)
			{
			valid[i] = 1;
			switch(item)
			{
			case D_GETLOAD:
				load[i] = info[i].load / 100.0;
				other[i] = info[i].ncpus;
				break;
			case D_GETMEM:
				load[i] = info[i].tmem / 1048576.0;
				other[i] = load[i] - info[i].mem / 1048576.0;
				break;
			case D_GETBMEM:
				load[i] = info[i].tmem / 1048576.0;
				other[i] = load[i] - info[i].rmem / 1048576.0;
				break;
			case D_GETSPEED:
				load[i] = info[i].speed;
				other[i] = info[i].ncpus;
				break;
			case D_GETUTIL:
				load[i] = info[i].util / info[i].ncpus;
				break;
			}
			}
			else if(!(info[i].status & DS_MOSIX_DEF))
				load[i] = valid[i] = VERY_DEAD;
			else if(valid[i] <= MAX_SILENT+1)
				load[i] = valid[i] = MAX_SILENT;
			else
			{
				valid[i]--;
				if(load[i] < 0)
					load[i] = 0;
			}

			if(load[i] < 0)
			{
				load[i] = (valid[i] < 0) ? valid[i] : -1;
				if(load[i] <= (dflg ? VERY_DEAD : MAX_SILENT))
					dead++;
			}
			if(load[i] > max)
				max = load[i];
		}
		if(tflg)
		{
			for(; i <= npe ; i++)
			if(!(info[i].status & DS_MOSIX_DEF))
				valid[i] = VERY_DEAD;
			else if(valid[i] > MAX_SILENT &&
				!(info[i].status & DS_MOSIX_UP))
				valid[i]--;
			for(i = first-1 ; i > 0 ; i--)
			if(!(info[i].status & DS_MOSIX_DEF))
				valid[i] = VERY_DEAD;
			else if(valid[i] > MAX_SILENT &&
				!(info[i].status & DS_MOSIX_UP))
				valid[i]--;
		}
		if (max < 1) {
			if(max == 0 && item == D_GETLOAD) { /* idle */
				standout();
				move(base-1,2);
				addstr("IDLE");
				standend();
			}
			max = 1;
		}
		if(max >= curmax)
		{
			curmax = max;
			cool = 0;
		}
		else 
		{
			if(cool++ >= 3)
				curmax = max;
			max = curmax;
		}
		if(item == D_GETMEM)
		{
			/* typical values are very close to 1MB multiples,
			 * but not quite, causing distortions:
			 */
		}
		switch(item)
		{
			case D_GETLOAD:
				fmt = "%5.2f";
				break;
			case D_GETSPEED:
				fmt = "%5.0f";
				break;
			case D_GETUTIL:
				fmt = "%4.0f%%";
				break;
			case D_GETMEM:
			case D_GETBMEM:
				fmt = (max >= 999.0) ? "%5.0f" : "%5.3g";
				break;
		}
		if(max > 0)
		{
			move(0,1);
			printw(fmt, max);
		}
		/*if(max >= 3)*/
		{
			move(base/4,1);
			printw(fmt, (max*3)/4);
			move(base/2,1);
			printw(fmt, max/2);
			move(base*3/4,1);
			printw(fmt, max/4);
		}
		move(base,5);
		addch('0');

		last = first + screen_width + dead - 1;
		if(npe < last)
			last = npe;
		if(wid == 0 && 2 * (last - first + 1 - dead) <= screen_width)
			wid = 1;
		if(wid == 0)
		{
			space = '|';
			rev = 0;
		}
		else
		{
			space = ' ';
			rev = 1;
		}
		if(rev)
			standout();
		dead = 0; /* number of not responding machines */
		if(max)
		for(i = first ; i <= last ; i++)
		{
		if(load[i] > 0)
		{
			col = COL_FROM_0 + 1 + (wid != 0) + wid/2 +
						(wid+1) * (i-first-dead);
			l = base - (load[i] * base) / max + 0.5;
			if(item == D_GETMEM || item == D_GETBMEM)
			{
				k = base - (other[i] * base) / max + 0.5;
				if(rev)
					standend();
				for(; l < k ; l++)
				{
					move(l, col);
					addch(valid[i] > 0 ? '+' : '?');
				}
				if(rev)
					standout();
			}
			ospace = space;
			if(item == D_GETSPEED && other[i] > 1)
				space = '0' + other[i];
			for(;l < base ; l++)
			{
				move(l, col);
				addch(valid[i] > 0 ? space : '?');
			}
			space = ospace;
		}
		else if(load[i] < 0)
		{
			if(load[i] <= (dflg ? VERY_DEAD : MAX_SILENT))
			{
				dead++;
				continue;
			}
			col = COL_FROM_0 + 1 + (wid != 0) + wid/2 +
						(wid+1) * (i-first-dead);
			if(rev)
				standend();
			move(base-4, col);
			addch('D');
			move(base-3, col);
			addch('E');
			move(base-2, col);
			addch('A');
			move(base-1, col);
			addch('D');
			if(rev)
				standout();
		}}
		if(rev)
			standend();

		if(ver)
		for(j = 10 , k = LINES-2 ; k > base ; k-- , j *= 10)
		{
			move(k, COL_FROM_0 + 1);
			for(i = first ;
				i < first+screen_width+dead && i <= npe ; i++)
			if(load[i] > (dflg ? VERY_DEAD : MAX_SILENT))
			{
				if(wid)
					addch(' ');
				addch('0' + i % j / (j / 10));
			}
		}
		else
		{
			move(base+1,COL_FROM_0 + 1);
			for(i = first ;
				i < (first+screen_width+dead) && i <= npe ; i++)
			if(load[i] > (dflg ? VERY_DEAD : MAX_SILENT))
			{
				if(i <= 9 && wid % 2 == 0)
					printw("%*s%d%*s", 1+wid/2, "", i, wid/2-1, "");
				else if(i >= 100 && i <= 999 && wid % 2 == 0)
					printw("%*s%d%*s", wid/2, "", i, wid/2-2, "");
				else
				{
					j = wid - (1 + (i > 9) + (i > 99) + (i > 999) + (i > 9999));
					printw("%*s%d%*s", j/2+1, "", i, (j+1)/2, "");
				}
			}
		}

		move(base,79);
		refresh();
		if(turns % 60 == 0)
			find_npe();
		sleep_or_input(1);
		if(is_input())
		switch(my_getch())
		{
			case 'q':
			case 'Q':
				onint(0);
			case '\14':
			case '\r':
			case '\n':
				clear();
				refresh();
				break;
			case '+':
			case KEY_RIGHT:
				if(first + screen_width - 1 < npe)
					first++;
				break;
			case '-':
			case KEY_LEFT:
				if(first > 1)
				{
					need_count = 1;
					first--;
				}
				break;
			case 'n':
			case 'N':
				first += screen_width;
				if(first + screen_width - 1 > npe)
				{
					first = npe + 1 - screen_width;
					if(first < 1)
						first = 1;
				}
				break;
			case 'h':
			case 'H':
				help();
				break;
			case 'v':
				vflg = 1;
				wflg = 0;
				break;
			case 'V':
				vflg = 2;
				wflg = 0;
				break;
			case 'w':
			case 'W':
				wflg = 1;
				vflg = 0;
				break;
			case 't':
			case 'T':
				tflg = !tflg;
				break;
			case 'd':
				if(dflg)
					break;
				dflg = 1;
				break;
			case 'D':
				if(!dflg)
					break;
				dflg = 0;
				break;
			case 'a':
			case 'A':
				wflg = vflg = 0;
				need_count = 1;
				break;
			case 's':
			case 'S':
				item = D_GETSPEED;
				clear_mode:
				curmax = 0;
				for(i = 1 ; i <= npe ; i++) 
					load[i] = MAX_SILENT-1;
				break;
			case 'm':
			case 'M':
				item = D_GETMEM;
				goto clear_mode;
			case 'l':
			case 'L':
				item = D_GETLOAD;
				goto clear_mode;
			case 'u':
			case 'U':
				item = D_GETUTIL;
				goto clear_mode;
			case 'r':
			case 'R':
				item = D_GETBMEM;
				goto clear_mode;
			case 'y':
			case 'Y':
				yardstick();
				break;
			case '\2':
				cb("The MOSIX Groups", 4, 9);
				break;
			case 'p':
			case 'P':
			case 'b':
			case 'B':
				if(first)
					need_count = 1;
				if(first > screen_width)
					first -= screen_width;
				else
					first = 1;
				break;
			default:
				write(1,"\7",1);
		}
	}
}

void find_npe()
{
	register int i;
	struct mosix_info x;
	int onpe = npe;

	for(npe = i = 1 ; npe <= MOSIX_MAX ; i = npe , npe += npe)
	{
		lseek(ifd, npe-1, 0);
		if(read(ifd, &x, RECSZ) != RECSZ)
			break;
	}
	while(npe > i+1)
	{
		lseek(ifd, (i+npe)/2 - 1, 0);
		if(read(ifd, &x, RECSZ) != RECSZ)
			npe = (i+npe)/2;
		else
			i = (i+npe)/2;
	}
	npe = i;
	for(i = onpe+1 ; i <= npe ; i++)
		load[i] = valid[i] = 0;
	adjust_first();
}

void adjust_first()
{
	if(first >= npe)
		first = npe;
	if(first < 1)
		first = 1;
}

/* my_getch() - input stuff: */

char esc_no_wait;
int un_char, las_char;

void ch_alarm()
{
	signal(SIGALRM, (sig_t)ch_alarm);
}

void not_yet_ch(int ch)
{
	if(ch)
		un_char = ch;
	else if(las_char && las_char != ERR)
		un_char = las_char;
}

int readc_half_second()
{
	char r = ERR;
	int a;
	struct itimerval t, tt;

	if(ioctl(0,FIONREAD,&a) >= 0 && a > 0)
		read(0, &r, 1);
	else if(esc_no_wait)
		return(ERR);
	else
	{
		signal(SIGALRM, (sig_t)ch_alarm);
		t.it_interval.tv_sec = 0;
		t.it_interval.tv_usec = 500000;
		t.it_value = t.it_interval;
		setitimer(ITIMER_REAL, &t, &tt);
		read(0, &r, 1);
		t.it_interval.tv_usec = t.it_interval.tv_sec = 0;
		t.it_value = t.it_interval;
		setitimer(ITIMER_REAL, &t, &tt);
	}
	return(r);
}

int my_getch()
{
	char r = ERR;

	if(un_char && un_char != ERR)
	{
		las_char = un_char;
		un_char = 0;
		return(las_char);
	}

	read(0, &r, 1);
	if(r == '\33')
	{
		r = readc_half_second();
		if(r == ERR)
			return(las_char = '\33');
		if(r == '[' || r == 'O')
		{
			switch(r = readc_half_second())
			{
				case 'A':
					return(las_char = KEY_UP);
				case 'B':
					return(las_char = KEY_DOWN);
				case 'C':
					return(las_char = KEY_RIGHT);
				case 'D':
					return(las_char = KEY_LEFT);
				case 'M':
					return(las_char = KEY_ENTER);
				case 'q':
				case 'F':
					return(las_char = KEY_C1);
				case 'r':
					return(las_char = KEY_DOWN);
				case 's':
				case 'G':
					return(las_char = KEY_C3);
				case 't':
					return(las_char = KEY_LEFT);
				case 'v':
					return(las_char = KEY_RIGHT);
				case 'w':
				case 'H':
					return(las_char = KEY_A1);
				case 'x':
					return(las_char = KEY_UP);
				case 'y':
				case 'I':
					return(las_char = KEY_A3);
				case '5':
				case '6':
					if(readc_half_second() == '~')
						return(las_char = (r == '5' ? KEY_A3 : KEY_C3));
					/* fall thru */
				default:
					return(las_char = r);
			}
		}
		else
			return(las_char = r);
	}
	else
		return(las_char = r);
}

int is_input()
{
	int r;

	return((un_char && un_char != ERR) || (ioctl(0, FIONREAD, &r) >= 0 && r > 0));
}

void help()
{
	static WINDOW *w1, *w2;
	int c;

	w1 = newwin(0,0,0,0);	
	if(w1 == NULL)
		printf("error creating help.exiting..\n"),exit(1);
	waddstr(w1, HELP_STR1);
	waddstr(w1, "\nPress Any Key to Continue...");
	w2 = newwin(0,0,0,0);	
	if(w2 == NULL)
		printf("error creating help.exiting..\n"),exit(1);
	waddstr(w2,HELP_STR2);
	waddstr(w2, "\nPress Any Key to Continue...");
	/* run */
	alarm(0);
	clearok(w1,TRUE);
	wrefresh(w1);
	refresh();
	if((c = wgetch(w1)) == 'q' || c == 'Q')
		onint(0);
	clearok(w2,TRUE);
	wrefresh(w2);
	refresh();
	if((c = wgetch(w2)) == 'q' || c == 'Q')
		onint(0);
	delwin(w1);
	delwin(w2);
	clearok(stdscr,TRUE);
	refresh();
}

void yardstick()
{
	static WINDOW *w;
	char buf[1025];
	FILE *yard = fopen("/proc/mosix/admin/sspeed", "r");
	int n;

	if(!yard || fscanf(yard, "%d", &n) != 1)
		sprintf(buf, "Sorry, Failed obtaining yardstick: %s\n",
			sys_errlist[errno]);
	else
		sprintf(buf, "Yardstick speed currently set to %d\n", n);
	if(yard)
		fclose(yard);
	w = newwin(0,0,0,0);	
	if(w == NULL)
		printf("error creating yardwindow.exiting..\n"),exit(1);
	waddstr(w,buf);
	waddstr(w, "\nPress Any Key to Continue...");
	/* run */
	alarm(0);
	clearok(w,TRUE);
	wrefresh(w);
	wgetch(w);
	delwin(w);
	clearok(stdscr,TRUE);
	refresh();
}

void cb(char* msg, int y, int x)
{
	static int mode = 0;
	static WINDOW *wn;

	if(mode == 0) {/* create */
		wn = newwin(LINES/3,COLS*4/9,LINES/3,COLS/3);	
		if(wn == NULL)
			printf("very bad\n"),exit(1);
		mode = 1;
	}
	else
		wclear(wn);
	
	alarm(0);
	clearok(wn,TRUE);
	box(wn,'*','*');
	wmove(wn,y,x);
	waddstr(wn,msg);
	wrefresh(wn);
	sleep(1);
	clearok(stdscr,TRUE);
	refresh();
}

struct sigaction act;
void onio()
{
	act.sa_handler = (sig_t)onio;
	sigaction(SIGALRM, &act, 0);
	alarm(1);
}


void sleep_or_input(int secs)
{
	if(is_input())
		return;
	act.sa_handler = (sig_t)onio;
	sigaction(SIGALRM, &act, 0);
	alarm(secs);
	if(my_getch() != ERR)
		not_yet_ch(0);
	alarm(0);
}
