'encoding UTF-8  Do not remove or change this line!
'*******************************************************************************
'*
'*  OpenOffice.org - a multi-platform office productivity suite
'*
'*  $RCSfile: extension_manager_tools.inc,v $
'*
'*  $Revision: 1.5 $
'*
'*  last change: $Author: jsk $ $Date: 2007/12/18 11:58:04 $
'*
'*  The Contents of this file are made available subject to
'*  the terms of GNU Lesser General Public License Version 2.1.
'*
'*
'*    GNU Lesser General Public License Version 2.1
'*    =============================================
'*    Copyright 2005 by Sun Microsystems, Inc.
'*    901 San Antonio Road, Palo Alto, CA 94303, USA
'*
'*    This library is free software; you can redistribute it and/or
'*    modify it under the terms of the GNU Lesser General Public
'*    License version 2.1, as published by the Free Software Foundation.
'*
'*    This library is distributed in the hope that it will be useful,
'*    but WITHOUT ANY WARRANTY; without even the implied warranty of
'*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
'*    Lesser General Public License for more details.
'*
'*    You should have received a copy of the GNU Lesser General Public
'*    License along with this library; if not, write to the Free Software
'*    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
'*    MA  02111-1307  USA
'*
'/******************************************************************************
'*
'*  owner : joerg.skottke@sun.com
'*
'*  short description : Tools to ease working with the extension manager
'*
'*******************************************************************************
'**
' #1 hExtensionAddGUI               ' Add an extension via Extension Manager GUI
' #1 hExtensionRemoveGUI            ' Remove an extension vi Extension Manager GUI
' #1 sExtensionCLI                  ' Add/Remove an extension with command line tool unopkg
' #1 hExtensionGetItemList          ' Get the names of all items in the extensions list
' #1 hSelectExtensionName           ' Select an extension by name
' #1 hSelectExtensionID             ' Select an extension by index
' #1 hSelectOptionsItem             ' Select an options item by name and index
' #1 hIsExtensionAlreadyInstalled   ' Check if an extemsion is allready installed
' #1 hDisableUserExtensionGUI       ' Disable a userspace extension by name via GUI
' #1 hExpandExtensionTreeGUI        ' Expand Shared extension and my extensions nodes
'**
'\******************************************************************************

function hExtensionAddGUI( sPathToExtension as string, _
                           bExtensionHasLicense as boolean ) as integer


    '///<h3>Add an extension via Extension Manager</h3>
    '///<i>This function adds an extension to the office using the
    '///+ Extension Manager GUI (formerly known as Package Manager).<br>
    '///+ It notifies about unexpected messageboxes, progressbars, missing
    '///+ depency warnings as well as license agreements. The function
    '///+ will exit on any error, print a message and return an errorcode.<br>
    '///+ The errorcode <b>must</b> be evaluated, some errors are fatal. Cleanup must be 
    '///+ done outside of this function.<br>
    '///+ All new extensions are installed below My Packages (to the userlayer).<br>
    '///+ Starting point is any document or the backing window, the function
    '///+ will return to the starting document if possible</i><br><br>
    '///<u>Input value(s):</u><br>
    '///<ol>

    '///+<li>Path to extension (string)</li>
    '///<ul>
    '///+<li>The path must be absolute</li>
    '///</ul>

    '///+<li>Expect a license text (boolean)</li>
    '///<ul>
    '///+<li>TRUE: Extension shows license text during installation</li>
    '///+<li>FALSE: Custom package without license</li>
    '///</ul>

    '///</ol>


    '///<u>Return Value:</u><br>

    '///<ol>
    '///+<li>Errorcode (Integer)</li>
    '///<ul>
    '///+<li>0 = No errors, extension was added correctly</li>
    '///+<li>1 = Extension Manager could not be opened (fatal)</li>
    '///+<li>2 = Add-Button is not enabled (recover)</li>
    '///+<li>3 = Error while opening file (fatal)</li>
    '///+<li>4 = Unexpected messagebox was displayed (recover)</li>
    '///+<li>5 = Unsatisfied dependencies dialog was displayed (fatal)</li>
    '///+<li>6 = Expected license dialog is missing (recover)</li>
    '///+<li>7 = Unexpected license dialog is displayed (fatal)</li>   
    '///+<li>8 = Unknown error / did not return to extension manager (fatal)</li>    
    '///+<li>9 = Extension not found (test on filesystem level)</li>
    '///</ul>
    '///</ol>

    const CFN = "hExtensionAddGUI::"
    
    dim brc as boolean 'a multi purpose boolean returnvalue
    dim iWait as integer
    dim cMsg as string
    dim iUnsatisfiedDeps as integer
    dim bLicenseExists as boolean
    dim cSelectAll as string
    
    printlog( CFN & "Enter with option (FileSpec)  : " & sPathToExtension )
    printlog( CFN & "Enter with option (HasLicense): " & bExtensionHasLicense )

    '///<u>Description:</u>
    '///<ul>

    '///+<li>Verify that the path to the extension is correct on filesystem level</li>
    if ( dir( sPathToExtension ) = "" ) then
        printlog( CFN & "The extension does not exist." )
        hExtensionAddGUI() = 9
        exit function
    endif

    '///+<li>Open the Extension Manager</li>
    ToolsPackageManager
    kontext "PackageManager"
    if ( not PackageManager.exists( 2 ) ) then
        printlog( CFN & "Could not open Extension Manager" )
        hExtensionAddGUI() = 1
        exit function
    endif
        
    '///+<li>Click the &quot;Add&quot; button</li>
    BrowsePackages.typeKeys( "<HOME>" )
    if ( not Add.isEnabled() ) then
        printlog( CFN & "Add-button is not enabled" )
        hExtensionAddGUI() = 2
        exit function
    endif
    
    Add.click()
    
    '///+<li>Enter the path to the extension, click &quot;Open&quot;</li>
    kontext "OeffnenDlg"
    if ( OeffnenDlg.exists() ) then
        DateiName.setText( sPathToExtension )
        Oeffnen.click()
    else
        printlog( CFN & "Adding package: FileOpen failed" )
        hExtensionAddGUI() = 3
        exit function
    endif
    
    Kontext "AddingPackages"
    if ( AddingPackages.exists( 1 ) ) then
        printlog( CFN & "AddingPackages dialog exists" )
    endif
    
    '///+<li>Check for errormessages</li>
    kontext "Active"
    if ( active.exists() ) then
        cMsg = active.getText()
        printlog( CFN & "Unexpected Messagebox: " & cMsg )
        hExtensionAddGUI() = 4
        
        ' Recover if this is the FileNotFound warning otherwise fail without grace
        ' This e.g. happens when the extension already exists
        try
            printlog( CFN & "Recover: Trying to close msgbox with (using ok)" )
            active.ok()

            printlog( CFN & "Recover: Trying to close FileOpen dialog (using cancel)" )
            kontext "OeffnenDlg"
            OeffnenDlg.cancel()
            
            printlog( CFN & "Recover: Trying to close Extension Manager (using ok)" )
            kontext "PackageManager"
            PackageManager.ok()
        catch
        
            kontext "PackageManager"
            if ( PackageManager.exists() ) then
                qaerrorlog( CFN & "Extension Manager is still open, recovery failed. Trying again." )
                PackageManager.ok()
            else
                warnlog( CFN & "Could not recover from previous errors." )
            endif
        endcatch
            
        exit function
    endif
    
    '///+<li>Check for unsatisfied dependencies</li>
    kontext "UnsatisfiedDependencies"
    if ( UnsatisfiedDependencies.exists( 10 ) ) then
        iUnsatisfiedDeps = UnsatisfiedDependencyList.getItemCount()
        printlog( CFN & "Unsatisfied deps: " & iUnsatisfiedDeps )
        hExtensionAddGUI() = 5        
        exit function
    endif
    
    '///+<li>Check for software license agreement</li>
    '///<ul>
    
    '///+<li>Expected license exists (good)</li>   
    kontext "ExtensionSoftwareLicenseAgreement"
    bLicenseExists = ExtensionSoftwareLicenseAgreement.exists()
    if ( bExtensionHasLicense and bLicenseExists ) then
        while ( not Accept.isEnabled() )
            ScrollDown.click()
            sleep( 1 )
        wend
        accept.click()
    endif
     
    '///+<li>Expected license does not exist (bad)</li>
    if ( bExtensionHasLicense and not bLicenseExists ) then
        printlog( CFN & "Expected license is missing" )

        kontext "PackageManager"
        PackageManager.ok()
        
        hExtensionAddGUI() = 6
        exit function
    endif
    
    '///+<li>License exists but was not expected (bad)</li>
    if ( bLicenseExists and not bExtensionHasLicense ) then
        printlog( CFN & "License is displayed but was not expected" )
        while ( not Accept.isEnabled() )
            ScrollDown.click()
            sleep( 1 )
        wend
        
        accept.click()
        
        kontext "PackageManager"
        PackageManager.ok()
        
        hExtensionAddGUI() = 7
        exit function
    endif
        
    '///+<li>License does not exist and was not expected (good)</li>
    if ( not bExtensionHasLicense and not bLicenseExists ) then
        ' do nothing
    endif
    '///</ul>
    
    '///+<li>The progressbar is still open, wait until it is closed</li>
    kontext "AddingPackages"
    do while( AddingPackages.exists() )
        printlog( CFN & "Adding packages ..." )
        sleep( 1 )
        iWait = iWait + 1
        if ( iWait = 20 ) then
            warnlog( CFN & "It takes very long time to add the extension" )
            exit do
        endif
    loop

    '///+<li>Verify that we are back on the Extension Manager GUI</li>
    kontext "PackageManager"
    if ( not PackageManager.exists() ) then
        printlog( CFN & "We are not back to the Extension Manager" )
        hExtensionAddGUI() = 8
        exit function
    endif
    
    '///+<li>Close the Extension Manager</li>
    kontext "PackageManager"
    PackageManager.ok()
    
    '///</ul>

    printlog( CFN & "Successfully added extension in " & iWait & " seconds." )
    hExtensionAddGUI() = 0
    
    printlog( CFN & "Exit" )
    
end function


'*******************************************************************************

function hExtensionRemoveGUI( cExtensionName as string ) as integer


    '///<h3>Remove an extension via Extension Manager </h3>
    '///<i>This function removes a package/an extension by name.<br>
    '///+ It will notify about errorconditions such as non existing packages, 
    '///+ errors during removal etc.<br>
    '///+ Some errors are considered fatal, others are recovered. If the error
    '///+ is non-fatal, the function returns to the document it started from<br>
    '///+ The function will do printlogs but no warnings so you <b>must</b> evaluate
    '///+ the returncode and you must clean up manually<br>
    '///+ Starting point is any document or backing window, the function will return
    '///+ to the starting point if possible</i></br><br>
    

    '///<u>Input value(s):</u><br>
    '///<ol>
    '///+<li>UI Name of the extension (string)</li>
    '///</ol>


    '///<u>Return Value:</u><br>

    '///<ol>
    '///+<li>Errorcode (integer)</li>
    '///<ul>
    '///+<li>0 = No errors, extension was removed</li>
    '///+<li>1 = Failure to open Extension Manager (fatal)</li>
    '///+<li>2 = Cannot delete found extension, remove-button is disabled (fatal)</li>
    '///+<li>3 = The extension was not found (non-fatal)</li>
    '///</ul>
    '///</ol>

    const CFN = "hExtensionRemoveGUI::"

    dim iExtensionCount as integer
    dim iExtensionRemoved as integer
    dim iCurrentExtension as integer
    dim cCurrentExtension as string
    
    printlog( CFN & "Enter" )

    '///<u>Description:</u>
    '///<ul>
    '///+<li>Open the Extension Manager</li>
    ToolsPackageManager
    kontext "PackageManager"
    if ( not PackageManager.exists( 2 ) ) then
        printlog( CFN & "Could not open Extension Manager" )
        hExtensionRemoveGUI() = 1
        exit function
    endif   

    '///+<li>Expand &quot;My packages&quot;</li>
    sleep( 1 )
    BrowsePackages.typeKeys( "<RIGHT>" )
    sleep( 1 )
    
    '///+<li>Find out how many extensions are listed below &quot;My packages&quot;</li>    
    iExtensionCount = BrowsePackages.getItemCount()
    printlog( CFN & "Number of items in extension manager: " & iExtensionCount )
    sleep( 2 )
    
    '///+<li>Locate the extension and delete it if found</li>
    iCurrentExtension = hSelectExtensionName( cExtensionName )
    if ( iCurrentExtension > 0 ) then
        if ( remove.isEnabled() ) then
            remove.click()
        else
            printlog( CFN & "Unable to remove package, button is disabled" )
            hExtensionRemoveGUI() = 2
            kontext "PackageManager"
            if ( PackageManager.exists() ) then
                PackageManager.ok()
            endif
            exit function
        endif
    else 
        printlog( CFN & "Could not find extension with given name" )
        hExtensionRemoveGUI() = 3
        kontext "PackageManager"
        if ( PackageManager.exists() ) then
            PackageManager.ok()
        endif
        exit function
    endif
        
        
    '///+<li>Close the Extension Manager</li>
    kontext "PackageManager"
    if ( PackageManager.exists() ) then
        sleep( 5 )        
        PackageManager.ok()
    else
        warnlog( "Extension Manager appears to be blocked." )
    endif
    
    '///</ul>

    printlog( CFN & "Exit: Successfully removed extension" )
    hExtensionRemoveGUI() = 0


end function

'*******************************************************************************

function sExtensionCLI(sCommand as string, sExtensionName as string, optional sExtensionPath as string) as string
    '/// Add/remove an extension with the command line tool 'unopkg'///'
    '/// INPUT: sCommand: string of command from "add remove list reinstall" ///'
   '/// INPUT: sExtensionName: name of the extension ///'
   '/// INPUT: optional sExtensionPath: path to the extension ///'
   '/// RETURN: currently nothing ///'
    dim sLokalExtensionPath as string
    dim sCommands as string
    dim sUnoPkg as string
    dim i, a, b as integer
    dim args as string
    
    if isMissing(sExtensionPath) then
        sLokalExtensionPath = ""
    else
        sLokalExtensionPath = sExtensionPath
    endif
    
    sCommands = "add remove list reinstall"
    
    a = len(sAppExe)
    if a > 12 then
        b = inStr(a-12, sAppExe, "soffice")
        sUnoPkg = left(sAppExe, b-1) + "unopkg" + mid(sAppExe, b+len("soffice"))
        'printlog sUnoPkg
        sUnoPkg = convertToUrl(sUnoPkg)
        'printlog sUnoPkg
    else
        qaErrorLog ("Need to think about another solution..." + sAppExe)
    endif
    args = sCommand+" "+sLokalExtensionPath + sExtensionName
    printlog "Executing: "+sUnopkg+" "+args
    shell(sUnoPkg,2,args)
end function

'*******************************************************************************

function hExtensionGetItemList( cItemList() as string ) as integer

    '///<h3>Get the list of all items in the extensions list</h3>
    '///<i>The array contains the list of all items in the extension manager GUI,
    '///+ including all components of the extensions. Consider this when defining
    '///+ the size of the array to be passed to this function as problems here are
    '///+ hard to debug.<br>
    '///+ Starting point is any document, the function will return to the 
    '///+ calling document on completion</i><br><br>
    
    '///<u>Input:</u>
    '///<ol>
    '///+<li>Array for the list items (string)</li>
    '///</ol>
    
    
    '///<u>Return Value:</u><br>
    '///<ol>
    '///+<li>Number of items (integer)</li>
    '///<ul>
    '///+<li>0 on any error</li>
    '///+<li>2 if no extensions exist (My Macros/OpenOffice.org macros nodes present</li>
    '///+<li>&gt; 2 if any changes to the default exist</li>
    '///</ul>
    '///</ol>

    const CFN = "hExtensionGetItemList::"
    dim iItemCount as integer
    dim iCurrentItem as integer
    
    printlog( CFN & "Enter" )

    '///<u>Description:</u>
    '///<ul>
    '///+<li>Open the Extension Manager via UNO slot</li>
    ToolsPackageManager
    
    '///+<li>Check that it is open, if not: Abort</li>
    kontext "PackageManager"
    if ( not packageManager.exists() ) then
        printlog( CFN & "Extension Manager is not open" )
        hExtensionGetItemList() = 0
        cItemList( 0 ) = 0
        exit function
    endif
    
    kontext "PackageManager"
    
    iCurrentItem = 0
    iItemCount   = 1000
    BrowsePackages.typeKeys( "<HOME>" )
    sleep( 2 )
    do 
    
        iCurrentItem = iCurrentItem + 1
        BrowsePackages.typeKeys( "<RIGHT>" )
        iItemCount = Browsepackages.getItemCount()
        BrowsePackages.typeKeys( "<DOWN>" ) 
        
        'printlog( CFN & "DEBUG - MIN: " & iCurrentItem )
        'printlog( CFN & "DEBUG - MAX: " & iItemCount   )
        
    loop until ( iCurrentItem = iItemCount )
    
    'printlog( CFN & "Itemcount = " & iItemCount )
        
    '///+<li>Get the number of items in the extension manager treelist</li>
    cItemList( 0 ) = iItemCount

    kontext "PackageManager"
    BrowsePackages.typeKeys( "<HOME>" )

    '///+<li>Retrieve the item names</li>
    for iCurrentItem = 1 to iItemCount
    
        kontext "PackageManager"
        BrowsePackages.typeKeys( "<SPACE>" )
        'sleep( 1 )
    
        cItemList( iCurrentItem ) = BrowsePackages.getText()
        BrowsePackages.typeKeys( "<DOWN>" )    
        sleep( 1 )    
        
    next iCurrentItem

    'hListPrint( cItemList(), "List of nodes in Extension Manager" )
    
    '///+<li>Close Extension Manager</li>
    kontext "PackageManager"
    PackageManager.close()
    
    '///</ul>
    
    hExtensionGetItemList() = iItemCount
    printlog( CFN & "Exit with " & iItemCount & " items in the list" )

end function

'*******************************************************************************

function hSelectExtensionID( iPos as integer ) as string

    '///<h3>Select an item by index in the list of available extensions</h3>
    '///<i>This function needs the Extension Manager to be open. It will only
    '///+ select extensions but not their components.</i><br><br>
        
    '///<u>Input:</u>
    '///<ol>
    '///+<li>Absolute position of the extension to be selected (Integer)</li>
    '///</ol>
    
    '///<u>Return Value:</u><br>
    '///<ol>
    '///+<li>Name of the selected extension (String)</li>
    '///+<li>Empty string on error (index out of range)</li>
    '///</ol>

    dim cCurrentExtension as string
    dim iCurrentExtension as integer
    dim irc as integer

    const CFN = "hSelectExtensionID::"
    printlog( CFN & "Enter with option: " & iPos )

    '///<u>Description:</u>
    '///<ul>

    '///+<li>Expand the shared extensions node</li>
    '///+<li>Expand &quot;My Extensions&quot;</li>
    irc = hExpandExtensionTreeGUI()
    if ( ipos > irc ) then
        printlog( CFN & "Specified index out of range. Exiting" )
        hSelectExtensionID() = ""
        exit function
    endif

    '///+<li>Step through the treelist until the desired item has been found</li>
    for iCurrentExtension = 1 to iPos
 
        if ( iCurrentExtension > 1 ) then
            BrowsePackages.typeKeys( "<DOWN>" )
            sleep( 1 )
        endif
        BrowsePackages.typeKeys( "<SPACE>" )
        sleep( 1 )

    next iCurrentExtension

    '///+<li>Retrieve the name of the selected extension</li>
    cCurrentExtension = BrowsePackages.getSelText()
    printlog( CFN & "Exit with result: " & cCurrentExtension )
    hSelectExtensionID() = cCurrentExtension
    '///</ul>

end function

'*******************************************************************************

function hSelectExtensionName( cName as string ) as integer

    '///<h3>Select an item by name in the list of available extensions</h3>
    '///<i>This function needs the Extension Manager to be open. It will only
    '///+ select extensions but not their components.</i><br><br>
    
    '///<u>Input:</u>
    '///<ol>
    '///+<li>Name of the extension (String)</li>
    '///<ul>
    '///+<li>Name of any item in the treelist, even those of top nodes</li>
    '///</ul>
    '///</ol>
    
    '///<u>Return Value:</u><br>
    '///<ol>
    '///+<li>Absolute position of the selected extension (Integer)</li>
    '///<ul>
    '///+<li>0 = Extension was not found</li>
    '///+<li>&gt; 0 = Absolute position of the extension/node</li>
    '///</ul>
    '///</ol>

    dim cCurrentExtension as string
    dim iCurrentExtension as integer
    dim iExtensionCount as integer
    dim iHit as integer 
        iHit = 0

    const CFN = "hSelectExtensionName::"
    printlog( CFN & "Enter with option: " & cName )

    '///<u>Description:</u>
    '///<ul>
    
    '///+<li>Expand the shared extensions node</li>
    '///+<li>Expand &quot;My Extensions&quot;</li>
    '///+<li>Retrieve the number of listed nodes</li>
    iExtensionCount = hExpandExtensionTreeGUI()

    '///+<li>Step through the treelist until the desired item has been found</li>
    '///+<li>Retrieve the absolute position of the selected extension</li>
    for iCurrentExtension = 1 to iExtensionCount
 
        if ( iCurrentExtension > 1 ) then
            BrowsePackages.typeKeys( "<DOWN>" )
            sleep( 1 )
        endif
        
        BrowsePackages.typeKeys( "<SPACE>" )
        sleep( 1 )

        cCurrentExtension = BrowsePackages.getSelText()
        if ( cCurrentExtension = cName ) then
            iHit = iCurrentExtension
            exit for
        endif 

    next iCurrentExtension
    
    printlog( CFN & "Exit with result: " & iHit )
    hSelectExtensionName() = iHit
    '///</ul>

end function


'*******************************************************************************

function hSelectOptionsItem( cName as string, iIndex as integer ) as integer

    '///<h3>Select an item in Tools/Options and verify</h3>
     '///<i>You need to open the Tools/Options dialog before using this function,
     '///+ it will not close the dialog either. No warnlogs are printed so evaluation
     '///+ of the return value is mandatory</i><br><br>

    '///<u>Parameter(s):</u><br>
    '///<ol>

    '///+<li>Name of the extension node (string)</li>
    '///<ul>
    '///+<li>Name of the module</li>
    '///+<li>Name of the leaf</li>
    '///+<li>Node must be valid</li>
    '///</ul>

    '///+<li>Position of the node (absolute) (integer)</li>
    '///<ul>
    '///+<li>All nodes are expanded</li>
    '///+<li>Position must be valid (&gt; 0 and &le; number of nodes in list)</li>
    '///</ul>

    '///</ol>


    '///<u>Returns:</u><br>
    '///<ol>
    '///+<li>Errorcondition (integer)</li>
    '///<ul>
    '///+<li>0 = Success</li>
    '///+<li>1 = Partial success - node is at wrong position</li>
    '///+<li>2 = Node does not exist</li>    
    '///+<li>3 = Dialog not open</li>
    '///+<li>4 = Index out of range (incorrect call to function)</li>
    '///</ul>
    '///</ol>

    const CFN = "hSelectOptionsItem::"
    printlog( CFN & "Enter with option (Name).: " & cName  )
    printlog( CFN & "Enter with option (Index): " & iIndex )
        
    dim brc as boolean ' a multi purpose boolean returnvalue
    dim irc as integer ' a multi purpose integer returnvalue
    dim crc as string  ' a multi purpose string  returnvalue

    '///<u>Description:</u>
    '///<ul>
    '///+<li>Make sure we are on the Tools/Options dialog</li>
    kontext "OptionenDlg"
    if ( not OptionenDlg.exists( 2 ) ) then
        warnlog( CFN & "Tools/Options is not open, aborting" )
        hSelectOptionsItem() = 3
        exit function
    endif
    
    '///+<li>Expand all nodes on &quot;OptionsListe&quot;</li>
    irc = hExpandAllNodes( OptionsListe )
    if ( irc < iIndex ) then
        warnlog( CFN & "Index out of range, quitting" )
        hSelectOptionsItem() = 4
        exit function
    endif        
    
    '///+<li>Search for the node at the given index, handle errors, exit function</li>
    crc = hSelectNode( OptionsListe , iIndex )
    
    ' if the name of the node at given position is ok ...
    if ( crc = cName ) then
        printlog( CFN & "Exit: The node was found: " & crc )
        hSelectOptionsItem() = 0
        exit function 

    ' if the name is not ok, try to find the node by name
    else 
        printlog( CFN & "Node not found at expected position, retrying" )
        irc = hSelectNodeByName( OptionsListe , cName )

        ' if the node is found it is just at the wrong position - bad but not deadly
        if ( irc > 0 ) then
            qaerrorlog( CFN & "Exit: Node <" & cName & "> found at pos: " & irc )
            hSelectOptionsItem() = 1
            exit function

        ' if the node was not found at all this is really bad.
        else
            warnlog( CFN & "Exit: Node does not exist: " & cName )
            hSelectOptionsItem() = 3
            exit function            
        endif
    endif
    
    '///</ul>

end function


'*******************************************************************************

function hIsExtensionAlreadyInstalled( cName as string ) as boolean

    '///<h3>Check if an extension is already installed</h3>
    '///<i>The function opents the Extension Manager, finds (or not) the extension
    '///+ and returns an appropriate return value. The Extension Manager is
    '///+ closed again at the end.</i><br><br>
    
    '///<u>Input:</u>
    '///<ol>
    '///+<li>Name of the extension (String)</li>
    '///</ol>
    
    '///<u>Return Value:</u><br>
    '///<ol>
    '///+<li>is the extension allredy installed (boolean)</li>
    '///<ul>
    '///+<li>false = Extension is not installed</li>
    '///+<li>true = Extension is installed</li>
    '///</ul>
    '///</ol>
    
    const CFN = "hIsExtensionAlreadyInstalled::"
    printlog( CFN & "Enter with option (Name).: " & cName  )
    
    '///<u>Description:</u>
    '///<ul>
    
    '///+<li>Open ToolsExtensionManager</li>
    printlog( CFN & "Open Extension Manager" )
    ToolsPackageManager
    
    '///+<li>Check if hSelectExtensionName returns 0, if so return false otherwise return true</li>
    if ( hSelectExtensionName( cName ) = 0 ) then
        hIsExtensionAlreadyInstalled() = false    
    else
        hIsExtensionAlreadyInstalled() = true
    endif
    
    '///+<li>Close the Extension Manager with OK</li>
    printlog( CFN & "Close Extension Manager with OK" )
    kontext "PackageManager"
    PackageManager.OK()
    '///</ul>
    
end function


'*******************************************************************************

function hDisableUserExtensionGUI( extension_name as string ) as integer


    '///<h3>Disable a userspace extension by name via GUI</h3>
    '///<i>This function tries to disable an Extension specified by name. There
    '///+ are several reasons why this might fail so it is required to evaluate the
    '///+ returnvalue as the function will not print any warnings. If the function 
    '///+ fails this is most likely caused by a) the extension already being
    '///+ disabled or b) the extension belonging to the shared layer.<br>
    '///+ The Extension Manager needs to be open when function is called.</i><br><br>

    '///<u>Parameter(s):</u><br>
    '///<ol>

    '///+<li>Name of the extension to be disabled (string)</li>
    '///<ul>
    '///+<li>Only extensions from the user layer can be disabled</li>
    '///</ul>

    '///</ol>


    '///<u>Returns:</u><br>
    '///<ol>
    '///+<li>Errorcondition (integer)</li>
    '///<ul>
    '///+<li>0 = Extension was found and disabled</li>
    '///+<li>1 = Extension does not exist</li>
    '///+<li>2 = Extension could not be disabled</li>
    '///</ul>
    '///</ol>

    const CFN = "hDisableUserExtensionGUI::"
    printlog( CFN & "Enter" )
    dim irc as integer
    dim iTry as integer

    '///<u>Description:</u>
    '///<ul>
    '///+<li>Open Tools/Extension Manger</li>
    ToolsPackageManager
    
    '///+<li>Find the extension, if not found return 1</li>
    kontext "PackageManager"
    irc = hSelectExtensionName( extension_name )
    if ( irc = 0 ) then
        printlog( CFN & "Extension was not found" )
        irc = 1
    else
    
        '///+<li>Try to disable the extension, if not possible return 2</li>
        if ( disable.isEnabled( 1 ) ) then
            disable.click()
            irc = 0
        else
            printlog( CFN & "Could not disable extension. Possible reasons:" )
            printlog( CFN & "1) It is already disabled." )
            printlog( CFN & "2) It is a shared extension." )
            irc = 2
        endif
    endif
    
    '///+<li>Close the Extension Manager</li>
    '///<ul>


    '///+<li>Trying to execute .close() which might be blocked by the progress bar</li>
    for iTry = 1 to 20
        try
            kontext "PackageManager"
            PackageManager.ok()
            exit for
        catch
            printlog( CFN & "Waiting for progressbar ... " )
            sleep( 1 )
        endcatch
    next iTry
    '///</ul>
    
    if ( irc = 0 ) then
        printlog( CFN & "Disabling extension took " & iTry & " seconds." )
        printlog( CFN & "Restarting the office to make the change take effect" )
        call exitRestartTheOffice()
    endif

    '///</ul>    
    printlog( CFN & "Exit with result: " & irc )
    hDisableUserExtensionGUI() = irc

end function


'*******************************************************************************

function hExpandExtensionTreeGUI() as integer


    '///<h3>Expand the top nodes in the extension tree</h3>
    '///<i>This function expands the two default nodes in the Extension manager
    '///+ &quot;OpenOffice.org Extensions&quot; and &quot;My Extensions&quot;
    '///+ so all installed extensions but not their components are visible.<br>
    '///+ The Extension Manager has to be open when calling this function.</i><br><br>

    '///<u>Parameter(s):</u><br>
    '///<ol>
    '///+<li>No input parameters</li>
    '///</ol>


    '///<u>Returns:</u><br>
    '///<ol>
    '///+<li>Number of extensions listed after the top nodes have been expanded</li>
    '///<ul>
    '///+<li>0 : any error (Reserved, currently unused)</li>
    '///+<li>&gt; 0 : Number of visible nodes</li>
    '///</ul>
    '///</ol>

    const CFN = "hExpandExtensionTreeGUI::"
    printlog( CFN & "Enter" )
    dim irc as integer
        irc = 0

    '///<u>Description:</u>
    '///<ul>
    
    kontext "PackageManager"
    
    '///+<li>Expand the shared extensions node</li>
    kontext "PackageManager"
    BrowsePackages.typeKeys( "<DOWN>" )
    sleep( 1 )
    BrowsePackages.typeKeys( "<SPACE>" )
    sleep( 1 )
    BrowsePackages.typeKeys( "<RIGHT>" )
    sleep( 1 )
    
    '///+<li>Expand &quot;My Extensions&quot;</li>
    kontext "PackageManager"
    BrowsePackages.typeKeys( "<UP>" )
    sleep( 1 )
    BrowsePackages.typeKeys( "<SPACE>" )
    sleep( 1 )    
    BrowsePackages.typeKeys( "<RIGHT>" )
    sleep( 1 )
    
    
    '///+<li>Retrieve the number of currently visible nodes</li>
    irc = BrowsePackages.getItemCount()
    '///</ul>

    printlog( CFN & "Exit with result: " & irc )
    hExpandExtensionTreeGUI() = irc

end function
