/*************************************************************************
 *
 *  $RCSfile: PropertyInfo.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:16:52 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _PROPERTYINFO_HXX_
#include "PropertyInfo.hxx"
#endif

#include <windows.h>
#include <stdio.h>


/**************************************************************************
 *
 * helper functions
 *
 **************************************************************************
 */

// get an instance of com.sun.star.config.SpecialConfigManager

inline Reference< XConfigManager > getSpecialConfigManager(const Reference< XMultiServiceFactory >& xMgr)
	throw(RuntimeException, ServiceNotRegisteredException)
{
	try
	{
		Reference< XInterface > xIfc = xMgr->
			createInstance(OUString::createFromAscii("com.sun.star.config.SpecialConfigManager"));

		if(xIfc.is())
		{
			return Reference< XConfigManager >::query(xIfc);
		}

		return Reference< XConfigManager >();
	}

    catch(RuntimeException)
	{
        throw;
	}

    catch(Exception)
	{
		ServiceNotRegisteredException xcept;

		xcept.Message = OUString::createFromAscii("com.sun.star.config.SpecialConfigManager");

        throw xcept;
	}
}

//------------------------------------------------------------------------
//  get an instance of com.sun.star.registry.SimpleRegistry

inline Reference< XSimpleRegistry > getSimpleRegistry(const Reference< XMultiServiceFactory >& xMgr)
    throw(RuntimeException, ServiceNotRegisteredException)
{
    try
	{
		Reference< XInterface > xIfc = xMgr->
			createInstance(OUString::createFromAscii("com.sun.star.registry.SimpleRegistry"));

		if(xIfc.is())
		{
			return Reference< XSimpleRegistry >::query(xIfc);
		}

		return Reference< XSimpleRegistry >();
	}

    catch(RuntimeException)
	{
        throw;
	}

    catch(Exception)
	{
		ServiceNotRegisteredException xcept;

		xcept.Message = OUString::createFromAscii("com.sun.star.registry.SimpleRegistry");

        throw xcept;
	}
}

//------------------------------------------------------------------------
// Get the URL from the USER/CONFIG-directory

inline OUString getUserConfigURL(const Reference< XConfigManager >& xSpecialConfigMgr)
{
	Reference< XSimpleRegistry > xSimReg=
		Reference< XSimpleRegistry >::query(xSpecialConfigMgr);

    if(xSimReg.is())
	{
		Reference< XRegistryKey > xRootKey = xSimReg->getRootKey();

		if(xRootKey.is())
		{
			OUString aURL;

            Reference< XRegistryKey > xRegKey = xRootKey->
				openKey(OUString::createFromAscii("Directories/UserConfig-Path"));

            if(xRegKey.is())
			{
				if(xRegKey->getValueType() == RegistryValueType_STRING)
				{
					aURL = xRegKey->getStringValue();
				}

                xRegKey->closeKey();
			}

            return xSpecialConfigMgr->substituteVariables(aURL);
		}
	}

    return OUString();
}

//------------------------------------------------------------------------
// Open the registry propertyinfo.rdb

inline void openRegistry(const Reference< XSimpleRegistry >& xRegistry,
						 const Reference< XConfigManager >& xSpecialConfigMgr)

    throw(InvalidRegistryException)
{
    OSL_ASSERT(xRegistry.is());
    OSL_ASSERT(xSpecialConfigMgr.is());

	// retrieve file location of user config dir
	OUString aURL = getUserConfigURL(xSpecialConfigMgr);

	if(aURL.getLength())
	{
		// expand the url with "/desktop.rdb"
		aURL += OUString::createFromAscii(PropertyInfoRdb);

		// open the registry database
		xRegistry->open(aURL, sal_False, sal_True);
	}
}

//------------------------------------------------------------------------
// Concat two sequences

template <class Type>
static Sequence< Type > concat( const Sequence< Type > &aPrefix, const Sequence< Type > &aSuffix )
{
	int	i, j;
	int pLen = aPrefix.getLength();
	int sLen = aSuffix.getLength();
	Sequence< Type >	aResult( pLen + sLen );

	for ( i = 0, j = 0; j < pLen; aResult[i++] = aPrefix[j++] );
	for ( j = 0; j < sLen; aResult[i++] = aSuffix[j++] );

	return aResult;
}

/////////////////////////////////////////////////////////////////////////////
//
//	OPropertyInfoProviderService
//
/////////////////////////////////////////////////////////////////////////////

// Search in the list with the shemes for a special scheme
list< ContentTypeList >::iterator OPropertyInfoProviderService::searchScheme(const OUString &aSchemeName)
{
	list< ContentTypeList >::iterator	maItScheme;

	for (maItScheme=m_aList.begin(); maItScheme!=m_aList.end(); maItScheme++)
	{
		if (maItScheme->aSchemeName==aSchemeName)
			return maItScheme;
	}

	return maItScheme;
}

//------------------------------------------------------------------------
// Search in the list with the ContentTypes for a special ContentType

list< ProviderList >::iterator OPropertyInfoProviderService::searchContentType(const list< ContentTypeList >::iterator &aItScheme, const OUString &aContentTypeName)
{
	list< ProviderList >::iterator	maItContentType;

	for (maItContentType=aItScheme->aContentTypeList.begin(); maItContentType!=aItScheme->aContentTypeList.end(); maItContentType++)
	{
		if (maItContentType->aContentTypeName==aContentTypeName)
			return maItContentType;
	}

	return maItContentType;
}

//------------------------------------------------------------------------
// Search in the list with the Provider for a special Provider

list< Reference< XLogicalContentPropertyInfoProvider > >::iterator OPropertyInfoProviderService::searchProvider(const list< ProviderList >::iterator &aItContentType, const Reference< XLogicalContentPropertyInfoProvider >& aProvider)
{
	list< Reference< XLogicalContentPropertyInfoProvider > >::iterator	maItProvider;

	for (maItProvider=aItContentType->aProviderList.begin(); maItProvider!=aItContentType->aProviderList.end(); maItProvider++)
	{
		Reference< XServiceInfo > xInfoProviderList=Reference< XServiceInfo >::query(*maItProvider);
		Reference< XServiceInfo > xInfoProvider=Reference< XServiceInfo >::query(aProvider);
		if (xInfoProviderList.is() && xInfoProvider.is())
		{
			Sequence< OUString >	aProviderListSupportServices = xInfoProviderList->getSupportedServiceNames();
			Sequence< OUString >	aProviderSupportServices = xInfoProvider->getSupportedServiceNames();
			if (aProviderListSupportServices.getLength()>0 && aProviderSupportServices.getLength()>0)
			{
				if (aProviderListSupportServices.getConstArray()[0] == aProviderSupportServices.getConstArray()[0])
					return maItProvider;
			}
		}
	}

	return maItProvider;
}

//------------------------------------------------------------------------
// Insert the new scheme at the end of the list with the different schemes

list< ContentTypeList >::iterator OPropertyInfoProviderService::registerScheme(const OUString &aSchemeName)
{
	ContentTypeList		maContentTypeList;
	
	maContentTypeList.aSchemeName=aSchemeName;

	m_aList.push_back(maContentTypeList);

	return --m_aList.end();
}

//------------------------------------------------------------------------
// Insert the new ContentType at the end of the list with the different ContentTypes

list< ProviderList >::iterator OPropertyInfoProviderService::registerContentType(const list< ContentTypeList >::iterator &aItScheme, const OUString &aContentTypeName)
{
	ProviderList		maProviderList;

	maProviderList.aContentTypeName=aContentTypeName;

	aItScheme->aContentTypeList.push_back(maProviderList);

	return --aItScheme->aContentTypeList.end();
}

//------------------------------------------------------------------------
// Insert the new Provider at the end of the list with the different Providers

void OPropertyInfoProviderService::registerProvider(const list< ProviderList >::iterator &aItContentType, const Reference < XLogicalContentPropertyInfoProvider >& Provider)
{
	aItContentType->aProviderList.push_back(Provider);
}

//------------------------------------------------------------------------
// Insert the  Provider at the end of the list with the different Providers

void OPropertyInfoProviderService::registerSortedProviderList(const list< ProviderList >::iterator &aItContentTypeList,  list< ProviderPriority > aProviderPriorityList )
{
	list< ProviderPriority >::iterator	aItPriorityList=aProviderPriorityList.begin();
	list< ProviderPriority >::iterator	aItMinPriorityList;
	Reference< XLogicalContentPropertyInfoProvider >	aProvider;

	while (aProviderPriorityList.size()!=0)
	{
		long aMinPriority=2^31;

		for(aItPriorityList = aProviderPriorityList.begin() ; aItPriorityList!=aProviderPriorityList.end(); aItPriorityList++)
		{
			if (aItPriorityList->lPriority<=aMinPriority)	
			{
				aItMinPriorityList=aItPriorityList;
				aMinPriority=aItPriorityList->lPriority;
				aProvider=aItPriorityList->aProvider;
			}
		}
		aProviderPriorityList.erase(aItMinPriorityList);
		registerProvider(aItContentTypeList, aProvider);
	}
}
	

//------------------------------------------------------------------------
// QuickSort - Algorithm for a sequence of Properties

static PropertyValues quicksort(const PropertyValues& seqPropertyValue, const sal_Int32 left, const sal_Int32 right )
{
	PropertyValues	sPropertyValue(seqPropertyValue);
	PropertyValue	aHelpPropertyValue;
	sal_Int32	nPropertyValue = sPropertyValue.getLength();
	OUString	aString(sPropertyValue.getArray()[(left+right)/2].Name);

	sal_Int32	i = left;
	sal_Int32	j = right;
	
	if (i>=j)
	{
		return sPropertyValue;
	}
	
	do
	{
		while ((sPropertyValue.getArray()[i].Name).compareTo(aString)<0)
		{
			i++;
		}
		
		while ((sPropertyValue.getArray()[j].Name).compareTo(aString)>0)
		{
			j--;
		}

		if (i <= j )
		{
			aHelpPropertyValue=sPropertyValue.getArray()[i];
			sPropertyValue.getArray()[i]=sPropertyValue.getArray()[j];
			sPropertyValue.getArray()[j]=aHelpPropertyValue;			
			i++;
			j--;
		}
	} while (i < j);

	if (left < j) 
	{
		sPropertyValue = quicksort(sPropertyValue, left, j);
	}
	
	if (i < right) 
	{
		sPropertyValue = quicksort(sPropertyValue, i, right);
	}

	return sPropertyValue;
}

//------------------------------------------------------------------------
// Search a Property in a given Sequence of Properties

static sal_Bool findPropertyValue (const PropertyValues& sPropertyValue, sal_Int32 left, sal_Int32 right, const PropertyValue &aPropertyValue)
{
	sal_Bool	fFound=sal_False;
	sal_Int32	i = (left+right)/2;
	sal_Int32	aCompare;

	aCompare=aPropertyValue.Name.compareTo(sPropertyValue.getConstArray()[i].Name);

	if (aCompare==0)
	{
		return sal_True;
	}

	if (i==left)
	{
		if (aPropertyValue.Name.compareTo(sPropertyValue.getConstArray()[right].Name)==0)
		{
			return sal_True;
		}
		else
		{
			return sal_False;
		}
	}
		
	if (aCompare<0)
	{
		fFound=findPropertyValue(sPropertyValue, left, i, aPropertyValue);
	}
	else if (aCompare>0)
	{
		fFound=findPropertyValue(sPropertyValue, i, right, aPropertyValue);
	}

	return fFound;
}

//------------------------------------------------------------------------
// Merge two sequences of properties , if there a property is double prefer the property of the first sequence (first Sequence must be sorted, second not)

PropertyValues OPropertyInfoProviderService::mergeProperties(const PropertyValues &sPrimaryPropertyValue, const PropertyValues &sSecondaryPropertyValue )
{
	PropertyValues	sPropertyValue(sPrimaryPropertyValue);
	PropertyValues	sHelpPropertyValue(sSecondaryPropertyValue.getLength());
	sal_Int32	nAddedProperties=0;

	if (sPrimaryPropertyValue.getLength() != 0)
	{
		// Merging the sequences
		if (sSecondaryPropertyValue.getLength() != 0)
		{
				for (sal_Int32 i=0; i<sSecondaryPropertyValue.getLength(); i++)
				{
					if (! findPropertyValue(sPropertyValue, 0, sPropertyValue.getLength()-1, sSecondaryPropertyValue.getConstArray()[i]))
					{
						sHelpPropertyValue.getArray()[nAddedProperties]=sSecondaryPropertyValue.getConstArray()[i];
						nAddedProperties++;						
					}
				}				

			if (nAddedProperties>0)
			{
				sHelpPropertyValue.realloc(nAddedProperties);
				sPropertyValue=concat(sPropertyValue, sHelpPropertyValue);

				return quicksort(sPropertyValue,0,sPropertyValue.getLength()-1);
			}
			else
			{
				return sPropertyValue;
			}

		}
	}
	else if (sSecondaryPropertyValue.getLength() != 0)
	{
		sPropertyValue=sSecondaryPropertyValue;	

		return quicksort(sPropertyValue,0,sPropertyValue.getLength()-1);
	}

	return sPropertyValue;
}

//------------------------------------------------------------------------
// Load the Scheme-ContentType-List from the Registry
						
void OPropertyInfoProviderService::loadPropertyInfoProviderList() throw(InvalidRegistryException, IllegalArgumentException, RuntimeException)
{
	list< ContentTypeList >::iterator									aItSchemeList;								// iterator for the Schemes
	list< ProviderList >::iterator										aItContentTypeList;							// iterator for the ContentTypes
	list< Reference< XLogicalContentPropertyInfoProvider > >::iterator	aItProviderList;							// iterator for the Providers
	list< ProviderPriority >											aPriorityList;

    OSL_ASSERT(m_xRegistry.is());

    if(!m_xRegistry->isValid())
	{
        // throws InvalidRegistryException if it fails
        openRegistry(m_xRegistry, m_xConfigMgr);
	}

    // query the registry's root key
	Reference< XRegistryKey > xRootKey = m_xRegistry->getRootKey();

	if(xRootKey.is())
	{
		Sequence< OUString > aSchemeKeyList=xRootKey->getKeyNames();
		for (sal_Int32 i=0, imax=aSchemeKeyList.getLength(); i < imax ; i++)
		{
			// open the Scheme-Key if exists
			Reference< XRegistryKey > xSchemeKey = xRootKey->openKey(aSchemeKeyList.getConstArray()[i]);

			OSL_ASSERT(xSchemeKey.is());

			if (xSchemeKey.is())
			{
				OUString aSchemeName=xSchemeKey->getKeyName().copy(1);
				aItSchemeList=registerScheme(aSchemeName);

				Sequence< OUString > aContentTypeKeyList=xSchemeKey->getKeyNames();
				for (sal_Int32 j=0, jmax=aContentTypeKeyList.getLength(); j < jmax ; j++)
				{
					// open the Content-Type-Key if exists
					Reference< XRegistryKey > xContentTypeKey = xRootKey->openKey(aContentTypeKeyList.getConstArray()[j]);
	
					OSL_ASSERT(xContentTypeKey.is());

					if (xContentTypeKey.is())
					{
						OUString aContentTypeName=xContentTypeKey->getKeyName().copy(aSchemeName.getLength()+2);
						aItContentTypeList=registerContentType(aItSchemeList, aContentTypeName);
						
						Sequence< OUString >			sKeyName=xContentTypeKey->getKeyNames();
						
						for (sal_Int32 k=0; k<sKeyName.getLength(); k++)
						{
							OUString aProviderName=sKeyName.getArray()[k].copy(aSchemeName.getLength()+aContentTypeName.getLength()+3);
							Reference< XRegistryKey > xProviderKey = xRootKey->openKey(sKeyName.getConstArray()[k]);

							OSL_ASSERT(xProviderKey.is());

							if (xProviderKey.is())
							{
								Reference< XInterface > xIfc(m_xMgr->createInstance(aProviderName ));
								OSL_ASSERT(xIfc.is());
						
								if (xIfc.is())		
								{
									Reference< XLogicalContentPropertyInfoProvider > xLProv( xIfc, UNO_QUERY );
									OSL_ASSERT(xLProv.is());
								
									if (xLProv.is())		
									{
										ProviderPriority	aPriority;
										aPriority.aProvider=xLProv;
										aPriority.lPriority=xProviderKey->getLongValue();

										aPriorityList.push_back(aPriority);
									}									
								}
								xProviderKey->closeKey();
							}							
						}
						if (k!=0)
						{
							registerSortedProviderList(aItContentTypeList, aPriorityList);
							aPriorityList.clear();
						}
						xContentTypeKey->closeKey();
					}					
				}				
				xSchemeKey->closeKey();
			}
		}
	}	
}

//------------------------------------------------------------------------
// Register the Provider into the registry (with Scheme and ContentType)

void OPropertyInfoProviderService::registerProviderInRegistry( const Reference < XLogicalContentPropertyInfoProvider >& Provider, const SchemeContentTypePair& SchemeContentType, list< ProviderList >::iterator aItContentTypeList) throw(InvalidRegistryException, IllegalArgumentException, RuntimeException)
{
	long	lMaxPriority;
	
	// query the registry's root key
	Reference< XRegistryKey > xRootKey = m_xRegistry->getRootKey();

	if(xRootKey.is())
	{
		// open the Scheme-Key if exists
		Reference< XRegistryKey > xSchemeKey = xRootKey->openKey(SchemeContentType.Scheme);

		if (!xSchemeKey.is())
		{
			// create a new SchemeKey
			xSchemeKey=xRootKey->createKey(SchemeContentType.Scheme);
		}
	
		// open the Content-Type-Key if exists
		Reference< XRegistryKey > xContentTypeKey = xSchemeKey->openKey(SchemeContentType.ContentType);

		if (!xContentTypeKey.is())
		{
			// create a new ContentTypeKey
			xContentTypeKey=xSchemeKey->createKey(SchemeContentType.ContentType);
			xContentTypeKey->setLongValue(0);
		}
		
		Reference< XServiceInfo > xInfo=Reference< XServiceInfo >::query(Provider);
		if (xInfo.is())
		{
			// open the Provider-Key if exists
			OUString aProviderName;
			if ((xInfo->getSupportedServiceNames()).getLength()>0) 
			{
				aProviderName=(xInfo->getSupportedServiceNames()).getConstArray()[0];
				Reference< XRegistryKey > xProviderKey = xContentTypeKey->openKey(aProviderName);
				if (!xProviderKey.is())
				{
					// create a new Provider
					xProviderKey=xContentTypeKey->createKey(aProviderName);
					lMaxPriority=xContentTypeKey->getLongValue();
					xContentTypeKey->setLongValue(lMaxPriority+1);
					xProviderKey->setLongValue(lMaxPriority+1);
					xProviderKey->closeKey();
				}
				else
				{
					lMaxPriority=xContentTypeKey->getLongValue();
					if (xProviderKey->getLongValue()!=lMaxPriority)
					{
						xContentTypeKey->setLongValue(lMaxPriority+1); 
						xProviderKey->setLongValue(lMaxPriority+1);
						xProviderKey->closeKey();
					}
				}
			}
		}
		xContentTypeKey->closeKey();
		xSchemeKey->closeKey();		
	}
}

//------------------------------------------------------------------------
// Deregister the Provider into the registry (with Scheme and ContentType)

void OPropertyInfoProviderService::deregisterProviderInRegistry( const Reference < XLogicalContentPropertyInfoProvider >& Provider, const SchemeContentTypePair& SchemeContentType, list< ProviderList >::iterator	aItContentTypeList )
{
    // query the registry's root key
	Reference< XRegistryKey > xRootKey = m_xRegistry->getRootKey();

	if(xRootKey.is())
	{
		// open the Scheme-Key if exists
		Reference< XRegistryKey > xSchemeKey = xRootKey->openKey(SchemeContentType.Scheme);

		OSL_ASSERT(xSchemeKey.is());
		if (xSchemeKey.is())
		{
			// open the ContentType-Key if exists
			Reference< XRegistryKey > xContentTypeKey = xSchemeKey->openKey(SchemeContentType.ContentType);

			OSL_ASSERT(xContentTypeKey.is());
			if (xContentTypeKey.is())
			{
				Reference< XServiceInfo > xInfo=Reference< XServiceInfo >::query(Provider);
				if (xInfo.is())
				{
					// open the Provider-Key if exists
					if ((xInfo->getSupportedServiceNames()).getLength()>0) 
					{
						OUString aProviderName=(xInfo->getSupportedServiceNames()).getConstArray()[0];
						Reference< XRegistryKey > xProviderKey = xContentTypeKey->openKey(aProviderName);
						if (xProviderKey.is())
						{
							xProviderKey->closeKey();
							xContentTypeKey->deleteKey(aProviderName);
				
							if ((xContentTypeKey->getKeyNames()).getLength()==0)
							{
								// destroy the ContentType-Key
								xContentTypeKey->closeKey();
								xSchemeKey->deleteKey(SchemeContentType.ContentType);
	
								Sequence< OUString > sContentTypeNames=xSchemeKey->getKeyNames();
								if (sContentTypeNames.getLength()==0)
								{
									// destroy the Scheme-Key
									xSchemeKey->closeKey();
									xRootKey->deleteKey(SchemeContentType.Scheme);
								}
							}
						}
					}
				}
				if (xContentTypeKey->isValid())
				{
					xContentTypeKey->closeKey();
				}
			}
			if (xSchemeKey->isValid())
			{
				xSchemeKey->closeKey();
			}
		}
	}
}

//------------------------------------------------------------------------

OPropertyInfoProviderService::OPropertyInfoProviderService(const Reference< XMultiServiceFactory >& xMgr): m_xMgr( xMgr )
{
	if ( !m_aList.empty())
		m_aList.clear();

	// create the services needed
	m_xRegistry  = getSimpleRegistry(m_xMgr);
    m_xConfigMgr = getSpecialConfigManager(m_xMgr);
	
	loadPropertyInfoProviderList();
}



OPropertyInfoProviderService::~OPropertyInfoProviderService()
{
	OSL_ASSERT(m_xRegistry.is());

    // close propertyinfo.rdb
    if(m_xRegistry->isValid())
        m_xRegistry->close();

}

//-----------------------------------------------------------------------------
//	XInterface implementation
//-----------------------------------------------------------------------------

Any SAL_CALL OPropertyInfoProviderService::queryInterface( const Type& type ) throw( RuntimeException )
{
	Any	aRet = ::cppu::queryInterface(
		type,
		SAL_STATIC_CAST( XLogicalContentPropertyInfoManagerProvider*, this ),
		SAL_STATIC_CAST( XLogicalContentPropertyInfoManager*, this ),
		SAL_STATIC_CAST( XServiceInfo*, this )
		);

	return ( aRet.hasValue() ? aRet : OWeakObject::queryInterface( type ) );
}

//-----------------------------------------------------------------------------
//	XLogicalContentPropertyInfoProvider implementation
//-----------------------------------------------------------------------------

PropertyValues SAL_CALL OPropertyInfoProviderService::getPropertyValuesWithFlags(const Reference <XContent>& Content, sal_Int32 nContentPropertyInfoFlags) throw( RuntimeException )
{
	PropertyValues														sPropertyValue;									// main sequence of Properties
	PropertyValues														sSchemeDefaultContentTypePropertyValue;			// sequence of Properties with specified Scheme and Default ContentType
	PropertyValues														sContentTypeDefaultSchemePropertyValue;			// sequence of Properties with specified ContentType and default scheme
	PropertyValues														sDefaultSchemeDefaultContentTypePropertyValue;	// sequence of Properties with default Scheme and default ContentType
	PropertyValues														sHelpPropertyValue;								// help sequence of Properties
	list< ContentTypeList >::iterator									aItSchemeList;								// iterator of the Schemes
	list< ProviderList >::iterator										aItContentTypeList;							// iterator of the ContentTypes
	list< Reference< XLogicalContentPropertyInfoProvider > >::iterator	aItProviderList;							// iterator of the Providers
	
	OUString						aDefault(L"*");
	OUString						aDefaultBlank(L"");

	Reference< XContentIdentifier >	xIdentifier = Content->getIdentifier();

	SchemeContentTypePair			aSchemeContentTypePair;

	aSchemeContentTypePair.Scheme=xIdentifier->getContentProviderScheme();
	if (aSchemeContentTypePair.Scheme==aDefaultBlank)
	{
		aSchemeContentTypePair.Scheme=aDefault;
	}

	aSchemeContentTypePair.ContentType=Content->getContentType();
	if (aSchemeContentTypePair.ContentType==aDefaultBlank)
	{
		aSchemeContentTypePair.ContentType=aDefault;
	}

	// Search for a XLogicalContentPropertyValueInfoProvider with the specified Schmeme
	aItSchemeList=searchScheme(aSchemeContentTypePair.Scheme);
	
	// Scheme / ContentType = Scheme / ContentType
	if (aItSchemeList!=m_aList.end())		// Scheme is registered
	{
		// Search for the specified ContentType
		aItContentTypeList=searchContentType(aItSchemeList, aSchemeContentTypePair.ContentType);	
	
		if (aItContentTypeList!=aItSchemeList->aContentTypeList.end())		// ContentType is registered
		{
			sal_Int32	nProvider=aItContentTypeList->aProviderList.size();
			aItProviderList=aItContentTypeList->aProviderList.end();
			
			for (sal_Int32 i = nProvider; i != 0; i--)
			{
				aItProviderList--;
				sHelpPropertyValue=(*aItProviderList)->getPropertyValues(Content,aSchemeContentTypePair );
				sPropertyValue=mergeProperties(sPropertyValue, sHelpPropertyValue);
			}			
		}
		
		// Scheme / ContentType = Scheme / "*"
		if ( (aSchemeContentTypePair.ContentType != aDefault) && !(nContentPropertyInfoFlags & ContentPropertyInfoFlags::NODEFAULTCONTENTTYPEMERGING ) )
		{
			// Search for the default ContentType "*"
			aItContentTypeList=searchContentType(aItSchemeList, aDefault);
			if (aItContentTypeList!=aItSchemeList->aContentTypeList.end())		// "*" is registered
			{
				sal_Int32	nProvider=aItContentTypeList->aProviderList.size();
				aItProviderList=aItContentTypeList->aProviderList.end();
			
				for (sal_Int32 i = nProvider; i != 0; i--)
				{
					aItProviderList--;
					sHelpPropertyValue=(*aItProviderList)->getPropertyValues(Content, aSchemeContentTypePair);
					sSchemeDefaultContentTypePropertyValue=mergeProperties(sSchemeDefaultContentTypePropertyValue, sHelpPropertyValue);
				}
			}
		}
	}

	if ( (aSchemeContentTypePair.Scheme != aDefault) && !(nContentPropertyInfoFlags & ContentPropertyInfoFlags::NODEFAULTSCHEMEMERGING) )
	{
		// Search for a XLogicalContentPropertyValueInfoProvider with the specified Schmeme
		aItSchemeList=searchScheme(aDefault);
		
		// Scheme / ContentType = "*" / ContentType
		if (aItSchemeList!=m_aList.end())		// Scheme is registered
		{
			// Search for the specified ContentType
			aItContentTypeList=searchContentType(aItSchemeList, aSchemeContentTypePair.ContentType);	
	
			if (aItContentTypeList!=aItSchemeList->aContentTypeList.end())		// ContentType is registered
			{
				sal_Int32	nProvider=aItContentTypeList->aProviderList.size();
				aItProviderList=aItContentTypeList->aProviderList.end();
			
				for (sal_Int32 i = nProvider; i != 0; i--)
				{
					aItProviderList--;
					sHelpPropertyValue=(*aItProviderList)->getPropertyValues(Content, aSchemeContentTypePair);
					sContentTypeDefaultSchemePropertyValue=mergeProperties(sContentTypeDefaultSchemePropertyValue, sHelpPropertyValue);
				}			
			}
		
			// Scheme / ContentType = "*" / "*" ( only if the ContentPropertyValueInfoFlag isn't set to NODEFAULTSMERGING )
			if ( (aSchemeContentTypePair.ContentType != aDefault) && ((nContentPropertyInfoFlags & MASK_CONTENTPROPERTYINFOFLAGS_MERGING )== 0 ) )
			{
				// Search for the default ContentType "*"
				aItContentTypeList=searchContentType(aItSchemeList, aDefault);
				if (aItContentTypeList!=aItSchemeList->aContentTypeList.end())		// "*" is registered
				{
					sal_Int32	nProvider=aItContentTypeList->aProviderList.size();
					aItProviderList=aItContentTypeList->aProviderList.end();
			
					for (sal_Int32 i = nProvider; i != 0; i--)
					{
						aItProviderList--;
						sHelpPropertyValue=(*aItProviderList)->getPropertyValues(Content, aSchemeContentTypePair);
						sDefaultSchemeDefaultContentTypePropertyValue=mergeProperties(sDefaultSchemeDefaultContentTypePropertyValue, sHelpPropertyValue);
					}
				}
			}
		}	
	}

	// Merge the "*"/ContentType-Sequence to the Scheme/ContentType-Sequence first if the ContentPropertyValueInfoFlag is set to PREFERCONTENTTYPE
	if ( (sContentTypeDefaultSchemePropertyValue.getLength() != 0) && (nContentPropertyInfoFlags & ContentPropertyInfoFlags::PREFERCONTENTTYPE) )
	{
		sPropertyValue=mergeProperties(sPropertyValue, sContentTypeDefaultSchemePropertyValue );
	}
	
	// Merge the "Scheme/*-Sequence to the Scheme/ContentType-Sequence 

	if ( sSchemeDefaultContentTypePropertyValue.getLength() != 0 )
	{
		sPropertyValue=mergeProperties(sPropertyValue, sSchemeDefaultContentTypePropertyValue );
	}
	
	// Merge the "*"/ContentType-Sequence to the Scheme/ContentType-Sequence first if the ContentPropertyValueInfoFlag is set to PREFERCONTENTTYPE
	if ( (sContentTypeDefaultSchemePropertyValue.getLength() != 0) && !(nContentPropertyInfoFlags & ContentPropertyInfoFlags::PREFERCONTENTTYPE) )
	{
		sPropertyValue=mergeProperties(sPropertyValue, sContentTypeDefaultSchemePropertyValue );
	}

	// Merge the "*"/"*"-Sequence to the Scheme/ContentType-Sequence
	if (sDefaultSchemeDefaultContentTypePropertyValue.getLength() != 0)
	{
		sPropertyValue=mergeProperties(sPropertyValue, sDefaultSchemeDefaultContentTypePropertyValue );
	}

	return sPropertyValue;
}

PropertyValues SAL_CALL OPropertyInfoProviderService::getPropertyValues(const Reference <XContent>& Content, const SchemeContentTypePair& SchemeContentType) throw( RuntimeException )
{
	return getPropertyValuesWithFlags(Content, sal_False);
}

//-----------------------------------------------------------------------------
//  XLogicalContentPropertyInfoManager implementation
//-----------------------------------------------------------------------------

void SAL_CALL OPropertyInfoProviderService::registerLogicalPropertyInfoProvider( const Reference< XLogicalContentPropertyInfoProvider >& Provider, const Sequence< SchemeContentTypePair >& SchemeContentTypePairs, sal_Bool OverwriteExisting ) throw(RuntimeException)
{
	list< ContentTypeList >::iterator									aItSchemeList;										// Iterator of the Schemes
	list< ProviderList >::iterator										aItContentTypeList;									// Iterator of the ContentTypes
	list< Reference< XLogicalContentPropertyInfoProvider > >::iterator	aItProviderList;									// Iterator of the Providers
	OUString															aDefault(L"*");										// Default scheme / ContentType
	OUString															aDefaultBlank(L"");									// Default scheme / ContentType


	for (sal_Int32 nSchemeContentTypePairs=0; nSchemeContentTypePairs < SchemeContentTypePairs.getLength(); nSchemeContentTypePairs++)
	{
		SchemeContentTypePair	aSchemeContentTypePair(SchemeContentTypePairs.getConstArray()[nSchemeContentTypePairs]);

		if (aSchemeContentTypePair.Scheme==aDefaultBlank)
		{
			aSchemeContentTypePair.Scheme=aDefault;
		}
		
		if (aSchemeContentTypePair.ContentType==aDefaultBlank)
		{
			aSchemeContentTypePair.ContentType=aDefault;
		}

		aItSchemeList=searchScheme(aSchemeContentTypePair.Scheme);
		
		// Scheme not registered
		if (aItSchemeList==m_aList.end())
		{
			aItSchemeList=registerScheme(aSchemeContentTypePair.Scheme);
		}
		
		aItContentTypeList=searchContentType(aItSchemeList, aSchemeContentTypePair.ContentType);	

		// ContentType not registered
		if (aItContentTypeList==aItSchemeList->aContentTypeList.end())
		{
			aItContentTypeList=registerContentType(aItSchemeList, aSchemeContentTypePair.ContentType);
		}

		if (OverwriteExisting)
		{
			// Clear the ProviderList
			aItContentTypeList->aProviderList.clear();
		}
		// Add Provider to ProviderList
		registerProvider(aItContentTypeList, Provider);
		registerProviderInRegistry( Provider, aSchemeContentTypePair, aItContentTypeList);
	}
}

//-----------------------------------------------------------------------------

void SAL_CALL OPropertyInfoProviderService::deregisterLogicalPropertyInfoProvider( const Reference< XLogicalContentPropertyInfoProvider >& Provider, const Sequence< SchemeContentTypePair >& SchemeContentTypePairs ) throw(IllegalArgumentException, RuntimeException)
{
	list< ContentTypeList >::iterator									aItSchemeList;							// Iterator of the Schemes
	list< ProviderList >::iterator										aItContentTypeList;						// Iterator of the ContentTypes
	list< Reference< XLogicalContentPropertyInfoProvider > >::iterator	aItProviderList;						// Iterator of the Providers

	OUString															aDefault(L"*");							// Default scheme / ContentType
	OUString															aDefaultBlank(L"");						// Default scheme / ContentType
	

	for (sal_Int32 nSchemeContentTypePairs=0; nSchemeContentTypePairs < SchemeContentTypePairs.getLength(); nSchemeContentTypePairs++)
	{
		SchemeContentTypePair	aSchemeContentTypePair(SchemeContentTypePairs.getConstArray()[nSchemeContentTypePairs]);

		if (aSchemeContentTypePair.Scheme==aDefaultBlank)
		{
			aSchemeContentTypePair.Scheme=aDefault;
		}
		
		if (aSchemeContentTypePair.ContentType==aDefaultBlank)
		{
			aSchemeContentTypePair.ContentType=aDefault;
		}

		aItSchemeList=searchScheme(aSchemeContentTypePair.Scheme);

		// Scheme registered
		if (aItSchemeList!=m_aList.end())
		{
			aItContentTypeList=searchContentType(aItSchemeList, aSchemeContentTypePair.ContentType);	

			// ContentType registered
			if (aItContentTypeList!=aItSchemeList->aContentTypeList.end())
			{
				aItProviderList=searchProvider(aItContentTypeList, Provider);

				// Provider registered 
				if (aItProviderList!=aItContentTypeList->aProviderList.end())
				{
					// Erase Provider from the ProviderList
					aItContentTypeList->aProviderList.erase(aItProviderList);
					// Erase Provider from the Registry
					deregisterProviderInRegistry( Provider, aSchemeContentTypePair, aItContentTypeList);

					if (aItContentTypeList->aProviderList.size()==0)
					{
						// Erase ContentType from the ContentTypeList, if there's no Provider for this ContentType
						aItSchemeList->aContentTypeList.erase(aItContentTypeList);
						
						if ( aItSchemeList->aContentTypeList.size()==0 )
						{
							// Erase Scheme from the SchemeList, if there's no ContentType for this Scheme
							m_aList.erase(aItSchemeList);
						}
					}
				}
			}
		}
	}
}

//-----------------------------------------------------------------------------
//  XServiceInfo implementation
//-----------------------------------------------------------------------------

OUString SAL_CALL OPropertyInfoProviderService::getImplementationName() throw(RuntimeException)
{
	 return L"com.sun.star.extensions.StandardContentPropertyInfo";
}

//-----------------------------------------------------------------------------

sal_Bool SAL_CALL OPropertyInfoProviderService::supportsService(const OUString& aServiceName) throw(RuntimeException)
{
	Sequence< OUString > aServiceNameList = getSupportedServiceNames();
	const OUString    *pNameArray       = aServiceNameList.getArray();

	for( sal_Int32 i = 0; i < aServiceNameList.getLength(); i++ )
		if( pNameArray[i] == aServiceName)
			return sal_True;

	return sal_False;
}

//-----------------------------------------------------------------------------

Sequence< OUString > SAL_CALL OPropertyInfoProviderService::getSupportedServiceNames() throw(RuntimeException)
{
	Sequence<OUString> aServiceNameList( 1 );

	aServiceNameList.getArray()[0] = L"com.sun.star.hierarchy.LogicalContentPropertyInfo";

	return aServiceNameList;
}

