/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: font.cxx,v $
 *
 *  $Revision: 1.13 $
 *
 *  last change: $Author: hr $ $Date: 2006/06/19 19:22:34 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif
#ifndef _VCOMPAT_HXX
#include <tools/vcompat.hxx>
#endif
#ifndef _DEBUG_HXX
#include <tools/debug.hxx>
#endif
#ifndef _SV_FONT_HXX
#include <font.hxx>
#endif
#ifndef _SV_IMPFONT_HXX
#include <impfont.hxx>
#endif
#ifndef _VCL_FONTCFG_HXX
#include <fontcfg.hxx>
#endif
#ifndef _VCL_OUTDEV_H
#include <outdev.h> // just for ImplGetEnglishSearchFontName! TODO: move it
#endif

// =======================================================================

DBG_NAME( Font )

// -----------------------------------------------------------------------

Impl_Font::Impl_Font() :
	maColor( COL_TRANSPARENT ),
	maFillColor( COL_TRANSPARENT )
{
	mnRefCount			= 1;
	meCharSet			= RTL_TEXTENCODING_DONTKNOW;
	meLanguage			= LANGUAGE_DONTKNOW;
	meCJKLanguage		= LANGUAGE_DONTKNOW;
	meFamily			= FAMILY_DONTKNOW;
	mePitch 			= PITCH_DONTKNOW;
	meAlign 			= ALIGN_TOP;
	meWeight			= WEIGHT_DONTKNOW;
	meWidthType 		= WIDTH_DONTKNOW;
	meItalic			= ITALIC_NONE;
	meUnderline 		= UNDERLINE_NONE;
	meStrikeout 		= STRIKEOUT_NONE;
	meRelief			= RELIEF_NONE;
	meEmphasisMark		= EMPHASISMARK_NONE;
	mnOrientation		= 0;
	mnKerning			= 0;
	mbWordLine			= false;
	mbOutline			= false;
	mbShadow			= false;
	mbVertical			= false;
	mbTransparent		= true;
	mbConfigLookup		= false;
}

// -----------------------------------------------------------------------

Impl_Font::Impl_Font( const Impl_Font& rImplFont )
:   maFamilyName( rImplFont.maFamilyName ),
	maStyleName( rImplFont.maStyleName ),
	maSize( rImplFont.maSize ),
	maColor( rImplFont.maColor ),
	maFillColor( rImplFont.maFillColor )
{
	mnRefCount			= 1;
	meCharSet			= rImplFont.meCharSet;
	meLanguage			= rImplFont.meLanguage;
	meCJKLanguage		= rImplFont.meCJKLanguage;
	meFamily			= rImplFont.meFamily;
	mePitch 			= rImplFont.mePitch;
	meAlign 			= rImplFont.meAlign;
	meWeight			= rImplFont.meWeight;
	meWidthType 		= rImplFont.meWidthType;
	meItalic			= rImplFont.meItalic;
	meUnderline 		= rImplFont.meUnderline;
	meStrikeout 		= rImplFont.meStrikeout;
	meRelief			= rImplFont.meRelief;
	meEmphasisMark		= rImplFont.meEmphasisMark;
	mnOrientation		= rImplFont.mnOrientation;
	mnKerning			= rImplFont.mnKerning;
	mbWordLine			= rImplFont.mbWordLine;
	mbOutline			= rImplFont.mbOutline;
	mbShadow			= rImplFont.mbShadow;
	mbVertical			= rImplFont.mbVertical;
	mbTransparent		= rImplFont.mbTransparent;
	mbConfigLookup		= rImplFont.mbConfigLookup;
}

// -----------------------------------------------------------------------

bool Impl_Font::operator==( const Impl_Font& rOther ) const
{
    // equality tests split up for easier debugging
    if( (meWeight   != rOther.meWeight)
    ||  (meItalic   != rOther.meItalic)
    ||  (meFamily   != rOther.meFamily)
    ||  (mePitch    != rOther.mePitch) )
        return false;

    if( (meCharSet     != rOther.meCharSet)
    ||  (meLanguage    != rOther.meLanguage)
    ||  (meCJKLanguage != rOther.meCJKLanguage)
    ||  (meAlign       != rOther.meAlign) )
        return false;

    if( (maSize         != rOther.maSize)
    ||  (mnOrientation  != rOther.mnOrientation)
    ||  (mbVertical     != rOther.mbVertical) )
        return false;

    if( (maFamilyName   != rOther.maFamilyName)
    ||  (maStyleName    != rOther.maStyleName) )
        return false;

    if( (maColor        != rOther.maColor)
    ||  (maFillColor    != rOther.maFillColor) )
        return false;

    if( (meUnderline    != rOther.meUnderline)
    ||  (meStrikeout    != rOther.meStrikeout)
    ||  (meRelief       != rOther.meRelief)
    ||  (meEmphasisMark != rOther.meEmphasisMark)
    ||  (mbWordLine     != rOther.mbWordLine)
    ||  (mbOutline      != rOther.mbOutline)
    ||  (mbShadow       != rOther.mbShadow)
    ||  (mnKerning      != rOther.mnKerning)
    ||  (mbTransparent  != rOther.mbTransparent) )
        return false;

    return true;
}

// -----------------------------------------------------------------------

void Impl_Font::AskConfig()
{
    if( mbConfigLookup )
        return;

    mbConfigLookup = true;

    // prepare the FontSubst configuration lookup
    const vcl::FontSubstConfiguration* pFontSubst = vcl::FontSubstConfiguration::get();

    String      aShortName;
    String      aFamilyName;
    ULONG       nType = 0;
    FontWeight  eWeight = WEIGHT_DONTKNOW;
    FontWidth   eWidthType = WIDTH_DONTKNOW;
    String      aMapName = maFamilyName;
    ImplGetEnglishSearchFontName( aMapName );
    vcl::FontSubstConfiguration::getMapName( aMapName,
        aShortName, aFamilyName, eWeight, eWidthType, nType );

    // lookup the font name in the configuration
    const vcl::FontNameAttr* pFontAttr = pFontSubst->getSubstInfo( aMapName );

    // if the direct lookup failed try again with an alias name
    if ( !pFontAttr && (aShortName != aMapName) )
        pFontAttr = pFontSubst->getSubstInfo( aShortName );

    if( pFontAttr )
    {
        // the font was found in the configuration
        if( meFamily == FAMILY_DONTKNOW )
        {
            if ( pFontAttr->Type & IMPL_FONT_ATTR_SERIF )
                meFamily = FAMILY_ROMAN;
            else if ( pFontAttr->Type & IMPL_FONT_ATTR_SANSSERIF )
                meFamily = FAMILY_SWISS;
            else if ( pFontAttr->Type & IMPL_FONT_ATTR_TYPEWRITER )
                meFamily = FAMILY_MODERN;
            else if ( pFontAttr->Type & IMPL_FONT_ATTR_ITALIC )
                meFamily = FAMILY_SCRIPT;
            else if ( pFontAttr->Type & IMPL_FONT_ATTR_DECORATIVE )
                meFamily = FAMILY_DECORATIVE;
        }

        if( mePitch == PITCH_DONTKNOW )
        {
            if ( pFontAttr->Type & IMPL_FONT_ATTR_FIXED )
                mePitch = PITCH_FIXED;
        }
    }

    // if some attributes are still unknown then use the FontSubst magic
    if( meFamily == FAMILY_DONTKNOW )
    {
        if( nType & IMPL_FONT_ATTR_SERIF )
            meFamily = FAMILY_ROMAN;
        else if( nType & IMPL_FONT_ATTR_SANSSERIF )
            meFamily = FAMILY_SWISS;
        else if( nType & IMPL_FONT_ATTR_TYPEWRITER )
            meFamily = FAMILY_MODERN;
        else if( nType & IMPL_FONT_ATTR_ITALIC )
            meFamily = FAMILY_SCRIPT;
        else if( nType & IMPL_FONT_ATTR_DECORATIVE )
            meFamily = FAMILY_DECORATIVE;
    }

    if( meWeight == WEIGHT_DONTKNOW )
        meWeight = eWeight;
    if( meWidthType == WIDTH_DONTKNOW )
        meWidthType = eWidthType;
}

// =======================================================================

void Font::MakeUnique()
{
	// create a copy if others still reference it
	if ( mpImplFont->mnRefCount != 1 )
	{
		if ( mpImplFont->mnRefCount )
			mpImplFont->mnRefCount--;
		mpImplFont = new Impl_Font( *mpImplFont );
	}
}

// -----------------------------------------------------------------------

Font::Font()
{
	DBG_CTOR( Font, NULL );

	static Impl_Font aStaticImplFont;
	// RefCount is zero for static objects
	aStaticImplFont.mnRefCount = 0;
	mpImplFont = &aStaticImplFont;
}

// -----------------------------------------------------------------------

Font::Font( const Font& rFont )
{
	DBG_CTOR( Font, NULL );
	DBG_CHKOBJ( &rFont, Font, NULL );
	DBG_ASSERT( rFont.mpImplFont->mnRefCount < 0xFFFE, "Font: RefCount overflow" );

	mpImplFont = rFont.mpImplFont;
	// do not count static objects (where RefCount is zero)
	if ( mpImplFont->mnRefCount )
		mpImplFont->mnRefCount++;
}

// -----------------------------------------------------------------------

Font::Font( const String& rFamilyName, const Size& rSize )
{
	DBG_CTOR( Font, NULL );

	mpImplFont				= new Impl_Font;
	mpImplFont->maFamilyName= rFamilyName;
	mpImplFont->maSize		= rSize;
}

// -----------------------------------------------------------------------

Font::Font( const String& rFamilyName, const String& rStyleName, const Size& rSize )
{
	DBG_CTOR( Font, NULL );

	mpImplFont				= new Impl_Font;
	mpImplFont->maFamilyName= rFamilyName;
	mpImplFont->maStyleName = rStyleName;
	mpImplFont->maSize		= rSize;
}

// -----------------------------------------------------------------------

Font::Font( FontFamily eFamily, const Size& rSize )
{
	DBG_CTOR( Font, NULL );

	mpImplFont				= new Impl_Font;
	mpImplFont->meFamily	= eFamily;
	mpImplFont->maSize		= rSize;
}

// -----------------------------------------------------------------------

Font::~Font()
{
	DBG_DTOR( Font, NULL );

    // decrement reference counter and delete if last reference
    // if the object is not static (Refcounter==0)
	if ( mpImplFont->mnRefCount )
	{
		if ( mpImplFont->mnRefCount == 1 )
			delete mpImplFont;
		else
			mpImplFont->mnRefCount--;
	}
}

// -----------------------------------------------------------------------

void Font::SetColor( const Color& rColor )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->maColor != rColor )
    {
        MakeUnique();
        mpImplFont->maColor = rColor;
    }
}

// -----------------------------------------------------------------------

void Font::SetFillColor( const Color& rColor )
{
	DBG_CHKTHIS( Font, NULL );

	MakeUnique();
	mpImplFont->maFillColor = rColor;
	if ( rColor.GetTransparency() )
		mpImplFont->mbTransparent = true;
}

// -----------------------------------------------------------------------

void Font::SetTransparent( BOOL bTransparent )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->mbTransparent != bTransparent )
    {
        MakeUnique();
        mpImplFont->mbTransparent = bTransparent;
    }
}

// -----------------------------------------------------------------------

void Font::SetAlign( FontAlign eAlign )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meAlign != eAlign )
    {
        MakeUnique();
        mpImplFont->meAlign = eAlign;
    }
}

// -----------------------------------------------------------------------

void Font::SetName( const String& rFamilyName )
{
	DBG_CHKTHIS( Font, NULL );

	MakeUnique();
	mpImplFont->maFamilyName = rFamilyName;
}

// -----------------------------------------------------------------------

void Font::SetStyleName( const String& rStyleName )
{
	DBG_CHKTHIS( Font, NULL );

	MakeUnique();
	mpImplFont->maStyleName = rStyleName;
}

// -----------------------------------------------------------------------

void Font::SetSize( const Size& rSize )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->maSize != rSize )
    {
        MakeUnique();
        mpImplFont->maSize = rSize;
    }
}

// -----------------------------------------------------------------------

void Font::SetFamily( FontFamily eFamily )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meFamily != eFamily )
    {
        MakeUnique();
        mpImplFont->meFamily = eFamily;
    }
}

// -----------------------------------------------------------------------

void Font::SetCharSet( CharSet eCharSet )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meCharSet != eCharSet )
    {
        MakeUnique();
        mpImplFont->meCharSet = eCharSet;
    }
}

// -----------------------------------------------------------------------

void Font::SetLanguage( LanguageType eLanguage )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meLanguage != eLanguage )
    {
        MakeUnique();
        mpImplFont->meLanguage = eLanguage;
    }
}

// -----------------------------------------------------------------------

void Font::SetCJKContextLanguage( LanguageType eLanguage )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meCJKLanguage != eLanguage )
    {
        MakeUnique();
        mpImplFont->meCJKLanguage = eLanguage;
    }
}

// -----------------------------------------------------------------------

void Font::SetPitch( FontPitch ePitch )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->mePitch != ePitch )
    {
        MakeUnique();
        mpImplFont->mePitch = ePitch;
    }
}

// -----------------------------------------------------------------------

void Font::SetOrientation( short nOrientation )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->mnOrientation != nOrientation )
    {
        MakeUnique();
        mpImplFont->mnOrientation = nOrientation;
    }
}

// -----------------------------------------------------------------------

void Font::SetVertical( BOOL bVertical )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->mbVertical != bVertical )
    {
        MakeUnique();
        mpImplFont->mbVertical = bVertical;
    }
}

// -----------------------------------------------------------------------

void Font::SetKerning( FontKerning nKerning )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->mnKerning != nKerning )
    {
        MakeUnique();
        mpImplFont->mnKerning = nKerning;
    }
}

// -----------------------------------------------------------------------

BOOL Font::IsKerning() const
{
    return (mpImplFont->mnKerning & KERNING_FONTSPECIFIC) != 0;
}

// -----------------------------------------------------------------------

void Font::SetWeight( FontWeight eWeight )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meWeight != eWeight )
    {
        MakeUnique();
        mpImplFont->meWeight = eWeight;
    }
}

// -----------------------------------------------------------------------

void Font::SetWidthType( FontWidth eWidth )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meWidthType != eWidth )
    {
        MakeUnique();
        mpImplFont->meWidthType = eWidth;
    }
}

// -----------------------------------------------------------------------

void Font::SetItalic( FontItalic eItalic )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meItalic != eItalic )
    {
        MakeUnique();
        mpImplFont->meItalic = eItalic;
    }
}

// -----------------------------------------------------------------------

void Font::SetOutline( BOOL bOutline )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->mbOutline != bOutline )
    {
        MakeUnique();
        mpImplFont->mbOutline = bOutline;
    }
}

// -----------------------------------------------------------------------

void Font::SetShadow( BOOL bShadow )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->mbShadow != bShadow )
    {
        MakeUnique();
        mpImplFont->mbShadow = bShadow;
    }
}

// -----------------------------------------------------------------------

void Font::SetUnderline( FontUnderline eUnderline )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meUnderline != eUnderline )
    {
        MakeUnique();
        mpImplFont->meUnderline = eUnderline;
    }
}

// -----------------------------------------------------------------------

void Font::SetStrikeout( FontStrikeout eStrikeout )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meStrikeout != eStrikeout )
    {
        MakeUnique();
        mpImplFont->meStrikeout = eStrikeout;
    }
}

// -----------------------------------------------------------------------

void Font::SetRelief( FontRelief eRelief )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meRelief != eRelief )
    {
        MakeUnique();
        mpImplFont->meRelief = eRelief;
    }
}

// -----------------------------------------------------------------------

void Font::SetEmphasisMark( FontEmphasisMark eEmphasisMark )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->meEmphasisMark != eEmphasisMark )
    {
        MakeUnique();
        mpImplFont->meEmphasisMark = eEmphasisMark;
    }
}

// -----------------------------------------------------------------------

void Font::SetWordLineMode( BOOL bWordLine )
{
	DBG_CHKTHIS( Font, NULL );

    if( mpImplFont->mbWordLine != bWordLine )
    {
        MakeUnique();
        mpImplFont->mbWordLine = bWordLine;
    }
}

// -----------------------------------------------------------------------

Font& Font::operator=( const Font& rFont )
{
	DBG_CHKTHIS( Font, NULL );
	DBG_CHKOBJ( &rFont, Font, NULL );
	DBG_ASSERT( rFont.mpImplFont->mnRefCount < 0xFFFE, "Font: RefCount overflow" );

	// Zuerst Referenzcounter erhoehen, damit man sich selbst zuweisen kann
	// RefCount == 0 fuer statische Objekte
	if ( rFont.mpImplFont->mnRefCount )
		rFont.mpImplFont->mnRefCount++;

	// Wenn es keine statischen ImplDaten sind, dann loeschen, wenn es
	// die letzte Referenz ist, sonst Referenzcounter decrementieren
	if ( mpImplFont->mnRefCount )
	{
		if ( mpImplFont->mnRefCount == 1 )
			delete mpImplFont;
		else
			mpImplFont->mnRefCount--;
	}

	mpImplFont = rFont.mpImplFont;

	return *this;
}

// -----------------------------------------------------------------------

BOOL Font::operator==( const Font& rFont ) const
{
    DBG_CHKTHIS( Font, NULL );
    DBG_CHKOBJ( &rFont, Font, NULL );

    if( mpImplFont == rFont.mpImplFont )
        return TRUE;
    if( *mpImplFont == *rFont.mpImplFont )
        return TRUE;

    return FALSE;
}

// -----------------------------------------------------------------------

void Font::Merge( const Font& rFont )
{
	if ( rFont.GetName().Len() )
	{
		SetName( rFont.GetName() );
		SetStyleName( rFont.GetStyleName() );
		SetCharSet( GetCharSet() );
		SetLanguage( rFont.GetLanguage() );
		SetCJKContextLanguage( rFont.GetCJKContextLanguage() );
		// don't use access methods here, might lead to AskConfig(), if DONTKNOW
		SetFamily( rFont.mpImplFont->meFamily );
		SetPitch( rFont.mpImplFont->mePitch );
	}

	// don't use access methods here, might lead to AskConfig(), if DONTKNOW
	if ( rFont.mpImplFont->meWeight != WEIGHT_DONTKNOW )
		SetWeight( rFont.GetWeight() );
	if ( rFont.mpImplFont->meItalic != ITALIC_DONTKNOW )
		SetItalic( rFont.GetItalic() );
	if ( rFont.mpImplFont->meWidthType != WIDTH_DONTKNOW )
		SetWidthType( rFont.GetWidthType() );


	if ( rFont.GetSize().Height() )
		SetSize( rFont.GetSize() );
	if ( rFont.GetUnderline() != UNDERLINE_DONTKNOW )
	{
		SetUnderline( rFont.GetUnderline() );
		SetWordLineMode( rFont.IsWordLineMode() );
	}
	if ( rFont.GetStrikeout() != STRIKEOUT_DONTKNOW )
	{
		SetStrikeout( rFont.GetStrikeout() );
		SetWordLineMode( rFont.IsWordLineMode() );
	}

	// Defaults?
	SetOrientation( rFont.GetOrientation() );
	SetVertical( rFont.IsVertical() );
	SetEmphasisMark( rFont.GetEmphasisMark() );
	SetKerning( rFont.IsKerning() );
	SetOutline( rFont.IsOutline() );
	SetShadow( rFont.IsShadow() );
	SetRelief( rFont.GetRelief() );
}

void Font::GetFontAttributes( ImplFontAttributes& rAttrs ) const
{
    // #i56788# Use members directly, don't risc config access.    
    rAttrs.maName = mpImplFont->maFamilyName;
    rAttrs.maStyleName = mpImplFont->maStyleName;
    rAttrs.meFamily = mpImplFont->meFamily;
    rAttrs.mePitch = mpImplFont->mePitch;
    rAttrs.meItalic = mpImplFont->meItalic;
    rAttrs.meWeight = mpImplFont->meWeight;
    rAttrs.meWidthType = WIDTH_DONTKNOW;
    rAttrs.mbSymbolFlag= (mpImplFont->meCharSet == RTL_TEXTENCODING_SYMBOL);
}


// -----------------------------------------------------------------------

SvStream& operator>>( SvStream& rIStm, Impl_Font& rImpl_Font )
{
	VersionCompat	aCompat( rIStm, STREAM_READ );
	UINT16			nTmp16;
	BOOL			bTmp;
	BYTE			nTmp8;

	rIStm.ReadByteString( rImpl_Font.maFamilyName, rIStm.GetStreamCharSet() );
	rIStm.ReadByteString( rImpl_Font.maStyleName, rIStm.GetStreamCharSet() );
	rIStm >> rImpl_Font.maSize;

	rIStm >> nTmp16; rImpl_Font.meCharSet = (rtl_TextEncoding) nTmp16;
	rIStm >> nTmp16; rImpl_Font.meFamily = (FontFamily) nTmp16;
	rIStm >> nTmp16; rImpl_Font.mePitch = (FontPitch) nTmp16;
	rIStm >> nTmp16; rImpl_Font.meWeight = (FontWeight) nTmp16;
	rIStm >> nTmp16; rImpl_Font.meUnderline = (FontUnderline) nTmp16;
	rIStm >> nTmp16; rImpl_Font.meStrikeout = (FontStrikeout) nTmp16;
	rIStm >> nTmp16; rImpl_Font.meItalic = (FontItalic) nTmp16;
	rIStm >> nTmp16; rImpl_Font.meLanguage = (LanguageType) nTmp16;
	rIStm >> nTmp16; rImpl_Font.meWidthType = (FontWidth) nTmp16;

	rIStm >> rImpl_Font.mnOrientation;

	rIStm >> bTmp; rImpl_Font.mbWordLine = bTmp;
	rIStm >> bTmp; rImpl_Font.mbOutline = bTmp;
	rIStm >> bTmp; rImpl_Font.mbShadow = bTmp;
	rIStm >> nTmp8; rImpl_Font.mnKerning = nTmp8;

    if( aCompat.GetVersion() >= 2 )
    {
        rIStm >> nTmp8;		rImpl_Font.meRelief = (FontRelief)nTmp8;
        rIStm >> nTmp16;	rImpl_Font.meCJKLanguage = (LanguageType)nTmp16;
        rIStm >> bTmp;		rImpl_Font.mbVertical = bTmp;
        rIStm >> nTmp16;	rImpl_Font.meEmphasisMark = (FontEmphasisMark)nTmp16;
    }
	// Relief
	// CJKContextLanguage

	return rIStm;
}

// -----------------------------------------------------------------------

SvStream& operator<<( SvStream& rOStm, const Impl_Font& rImpl_Font )
{
	VersionCompat aCompat( rOStm, STREAM_WRITE, 2 );
	rOStm.WriteByteString( rImpl_Font.maFamilyName, rOStm.GetStreamCharSet() );
	rOStm.WriteByteString( rImpl_Font.maStyleName, rOStm.GetStreamCharSet() );
	rOStm << rImpl_Font.maSize;

	rOStm << (UINT16) GetStoreCharSet( rImpl_Font.meCharSet );
	rOStm << (UINT16) rImpl_Font.meFamily;
	rOStm << (UINT16) rImpl_Font.mePitch;
	rOStm << (UINT16) rImpl_Font.meWeight;
	rOStm << (UINT16) rImpl_Font.meUnderline;
	rOStm << (UINT16) rImpl_Font.meStrikeout;
	rOStm << (UINT16) rImpl_Font.meItalic;
	rOStm << (UINT16) rImpl_Font.meLanguage;
	rOStm << (UINT16) rImpl_Font.meWidthType;

	rOStm << rImpl_Font.mnOrientation;

	rOStm << (BOOL) rImpl_Font.mbWordLine;
	rOStm << (BOOL) rImpl_Font.mbOutline;
	rOStm << (BOOL) rImpl_Font.mbShadow;
	rOStm << (BYTE) rImpl_Font.mnKerning;

    // new in version 2
    rOStm << (BYTE)		rImpl_Font.meRelief;
    rOStm << (UINT16)	rImpl_Font.meCJKLanguage;
    rOStm << (BOOL)		rImpl_Font.mbVertical;
    rOStm << (UINT16)	rImpl_Font.meEmphasisMark;

	return rOStm;
}

// -----------------------------------------------------------------------

SvStream& operator>>( SvStream& rIStm, Font& rFont )
{
	rFont.MakeUnique();
	return( rIStm >> *rFont.mpImplFont );
}

// -----------------------------------------------------------------------

SvStream& operator<<( SvStream& rOStm, const Font& rFont )
{
	return( rOStm << *rFont.mpImplFont );
}

// -----------------------------------------------------------------------

// the inlines from the font.hxx header are now instantiated for pImpl-ification
// TODO: reformat
const Color& Font::GetColor() const { return mpImplFont->maColor; }
const Color& Font::GetFillColor() const { return mpImplFont->maFillColor; }
BOOL Font::IsTransparent() const { return mpImplFont->mbTransparent; }
FontAlign Font::GetAlign() const { return mpImplFont->meAlign; }
const String& Font::GetName() const { return mpImplFont->maFamilyName; }
const String& Font::GetStyleName() const { return mpImplFont->maStyleName; }
const Size& Font::GetSize() const { return mpImplFont->maSize; }
void Font::SetHeight( long nHeight ) { SetSize( Size( mpImplFont->maSize.Width(), nHeight ) ); }
long Font::GetHeight() const { return mpImplFont->maSize.Height(); }
void Font::SetWidth( long nWidth ) { SetSize( Size( nWidth, mpImplFont->maSize.Height() ) ); }
long Font::GetWidth() const { return mpImplFont->maSize.Width(); }
rtl_TextEncoding Font::GetCharSet() const { return mpImplFont->meCharSet; }
LanguageType Font::GetLanguage() const { return mpImplFont->meLanguage; }
LanguageType Font::GetCJKContextLanguage() const { return mpImplFont->meCJKLanguage; }
short Font::GetOrientation() const { return mpImplFont->mnOrientation; }
BOOL Font::IsVertical() const { return mpImplFont->mbVertical; }
FontKerning Font::GetKerning() const { return mpImplFont->mnKerning; }
FontPitch Font::GetPitch() const { return mpImplFont->GetPitch(); }
FontWeight Font::GetWeight() const { return mpImplFont->GetWeight(); }
FontWidth Font::GetWidthType() const { return mpImplFont->GetWidthType(); }
FontItalic Font::GetItalic() const { return mpImplFont->GetItalic(); }
FontFamily Font::GetFamily() const { return mpImplFont->GetFamily(); }
BOOL Font::IsOutline() const { return mpImplFont->mbOutline; }
BOOL Font::IsShadow() const { return mpImplFont->mbShadow; }
FontRelief Font::GetRelief() const { return mpImplFont->meRelief; }
FontUnderline Font::GetUnderline() const { return mpImplFont->meUnderline; }
FontStrikeout Font::GetStrikeout() const { return mpImplFont->meStrikeout; }
FontEmphasisMark Font::GetEmphasisMark() const { return mpImplFont->meEmphasisMark; }
BOOL Font::IsWordLineMode() const { return mpImplFont->mbWordLine; }
BOOL Font::IsSameInstance( const Font& rFont ) const { return (mpImplFont == rFont.mpImplFont); }

