/*
** Copyright (C) 2000 Alan McIvor <alan@mcivor.gen.nz>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <cstdio>
#include <stdarg.h>

using namespace std;

#include "resource.h"
#include "projectExceptions.h"

RESOURCE::RESOURCE( char * id, char * name ):
  NAMED_ID(id, name)
{
    mEfficiency = 1.0;
    mRate = -1.0;

    is_group = 0;
}


void RESOURCE::Error(char *fmt, ...) 
{
    char buf[1024];
    va_list args;

    va_start(args, fmt);
    vsprintf(buf, fmt, args);
    va_end(args);

    string sid = id();
    throw RESOURCEException( sid, buf );
}


void RESOURCE::setEfficiency(double e)
{
    if ( e <= 0.0 )
	Error("Invalid value (%f) for efficiency", e );

    mEfficiency = e;
}



void RESOURCE::setRate(double r)
{ 
    if ( r < 0.0 )
	Error("Invalid value (%f) for rate", r );

    if ( mRate != -1.0 )
	Error("Setting rate to %f after already set to %f", r, mRate);

    mRate = r; 
}



bool RESOURCE::IsAvailable( int start, int finish )
{
    for ( ResourceTimeBlockIterator tb = mBooked.begin(); 
	  tb != mBooked.end(); tb++ )
    {
	if ( tb->overlap( start, finish ) )
	{
	    return false;
	}
    }

    return true;
}

bool RESOURCE::IsUsed( int dayNo )
{
    return ! IsAvailable( dayNo, dayNo );
}


bool RESOURCE::reverseIsAvailable( int start, int finish )
{
    for ( ResourceTimeBlockIterator tb = mRbooked.begin(); 
	  tb != mRbooked.end(); tb++ )
    {
	if ( tb->overlap( start, finish ) )
	{
	    return false;
	}
    }

    return true;
}


void RESOURCE::AddNote(const string & note)
{
    mNotes.push_back(note);
}

void RESOURCE::addTimeBlock(TASK *t, int start, int finish, TimeBlock::Type type)
{
    mBooked.push_back(ResourceTimeBlock(t, start, finish, type));
}


void RESOURCE::addReverseTimeBlock(TASK *t, int start, int finish, TimeBlock::Type type)
{
    mRbooked.push_back(ResourceTimeBlock(t, start, finish, type));
}


double RESOURCE::sDefaultRate = -1.0;

void RESOURCE::setDefaultRate(double rate)
{
    sDefaultRate = rate;
}

double RESOURCE::rate()
{
    if ( mRate != -1.0 )
	return mRate;
    if ( sDefaultRate != -1 )
	return sDefaultRate;
    Error("Neither resource rate nor default rate set");
    /*NOTREACHED*/
    return 0.0;
}
