//                                               -*- C++ -*-
/**
 * @file  ComputedNumericalMathEvaluationImplementation.cxx
 * @brief Abstract top-level class for all distributions
 *
 * (C) Copyright 2005-2010 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 */

#include "ComputedNumericalMathEvaluationImplementation.hxx"
#include "ComputedNumericalMathEvaluationImplementationFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "Cache.hxx"
#include "Log.hxx"
#include "WrapperData.hxx"
#include "WrapperObject.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      using Common::Log;

      CLASSNAMEINIT(ComputedNumericalMathEvaluationImplementation);

      static Common::Factory<ComputedNumericalMathEvaluationImplementation> RegisteredFactory("ComputedNumericalMathEvaluationImplementation");

      /* Default constructor */
      ComputedNumericalMathEvaluationImplementation::ComputedNumericalMathEvaluationImplementation(const String & name,
                                                                                                   const WrapperFile & file)
        /* throw(WrapperInternalException) */
        : NumericalMathEvaluationImplementation(),
          p_function_(0),
          p_state_(0)
      {
        setName(name);
        const WrapperData data = file.getWrapperData();
        if (! data.isValid()) throw WrapperInternalException(HERE) << "The wrapper data are not valid";

        if (data.getFunctionDescription().provided_) {
          p_function_.reset(new WrapperObject( data.getLibraryPath(),
                                               data.getFunctionDescription().name_,
                                               data,
                                               WrapperObject::FUNCTION ));
        }

        if (p_function_.isNull()) throw WrapperInternalException(HERE) << "Unable to allocate wrapper";

        // Initialize the state into the wrapper
        p_state_ = p_function_->createNewState();
        NumericalMathEvaluationImplementation::setDescription(getDescription());

        // Activate the cache only if the external code id expensive: only the user knows it.
        NumericalMathEvaluationImplementation::disableCache();
      }


      /* Copy constructor */
      ComputedNumericalMathEvaluationImplementation::ComputedNumericalMathEvaluationImplementation(const ComputedNumericalMathEvaluationImplementation & other)
        /* throw(WrapperInternalException) */
        : NumericalMathEvaluationImplementation(other),
          p_function_(other.p_function_),
          p_state_(0)
      {
        if (p_function_.isNull()) throw WrapperInternalException(HERE) << "Unable to allocate wrapper";

        // Initialize the state into the wrapper
        p_state_ = p_function_->createNewState();
      }


      /* Virtual constructor */
      ComputedNumericalMathEvaluationImplementation * ComputedNumericalMathEvaluationImplementation::clone() const
      {
        return new ComputedNumericalMathEvaluationImplementation(*this);
      }

      /* Destructor */
      ComputedNumericalMathEvaluationImplementation::~ComputedNumericalMathEvaluationImplementation()
      {
        // Call the finalization function before destruction
        p_function_->finalize( p_state_ );

        // Delete the state into the wrapper
        p_function_->deleteState( p_state_ );
      }


      /* Comparison operator */
      Bool ComputedNumericalMathEvaluationImplementation::operator ==(const ComputedNumericalMathEvaluationImplementation & other) const
      {
        return true;
      }


      /* String converter */
      String ComputedNumericalMathEvaluationImplementation::__repr__() const {
        OSS oss;
        oss << "class=" << ComputedNumericalMathEvaluationImplementation::GetClassName()
            << " name=" << getName();
        return oss;
      }





      /* State accessor */
      void * ComputedNumericalMathEvaluationImplementation::getState() const
      {
        return p_state_;
      }






      /* Here is the interface that all derived class must implement */

      /* Operator () */
      ComputedNumericalMathEvaluationImplementation::NumericalPoint
      ComputedNumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
      /* throw(InvalidArgumentException,InternalException) */
      {
        try {
          // First, initialize the external code on first invocation
          if (callsNumber_ == 0) p_function_->initialize( p_state_ );
          // Specific code if cache is enabled
          if (p_cache_->isEnabled())
            {
              CacheKeyType inKey = in.getCollection();
              // Execute the wrapper
              if ( p_cache_->hasKey( inKey ) )
                return NumericalPoint::ImplementationType( p_cache_->find( inKey ) );
              else {
                ++callsNumber_;
                NumericalPoint out = p_function_->execute( p_state_, in );
                CacheValueType outValue = out.getCollection();
                p_cache_->add( inKey, outValue );
                return out;
              }
            } // If cache is enabled
          else
            {
              ++callsNumber_;
              NumericalPoint out = p_function_->execute( p_state_, in );
              return out;
            } // Cache disabled
        } // try
        catch (InvalidArgumentException & ex) {
          throw;
        }
        catch (WrapperInternalException & ex) {
          throw InternalException(HERE) << ex;
        }
      }

      /* Operator () */
      ComputedNumericalMathEvaluationImplementation::NumericalSample
      ComputedNumericalMathEvaluationImplementation::operator() (const NumericalSample & in) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        try {
          const UnsignedLong size(in.getSize());
          // The sample out will store all the results as if there was no specific action for multiple input points or already computed values
          NumericalSample out( size, getOutputDimension() );
          // First, initialize the external code on first invocation
          if (callsNumber_ == 0) p_function_->initialize( p_state_ );
          // If the cache is enabled, it means that each evaluation is costly, so avoid any evaluation as much as possible
          if (p_cache_->isEnabled())
            {
              // First, remove all the points that have already been computed
              // We build a new sample without the already computed points
              NumericalSample newIn( 0, in.getDimension() );
              for(UnsignedLong i = 0; i < size; ++i) if ( ! p_cache_->hasKey( in[i].getCollection() ) ) newIn.add( in[i] );
              const UnsignedLong newSize(newIn.getSize());
              // Execute the wrapper if needed
              NumericalSample newOut( 0, getOutputDimension() );

              // If there is still something to do
              if (newSize > 0)
                {
                  // We shrink newIn and remove double entries
                  std::sort( newIn.begin(), newIn.end() );
                  NumericalSample::iterator last = std::unique( newIn.begin(), newIn.end() );
                  newIn.erase( last, newIn.end() );
                  callsNumber_ += newSize;
                  newOut = p_function_->execute( p_state_, newIn );
                } // If there is something to do

              // We use a secondary cache to hold the computed values
              CacheType tempCache( newSize );
              tempCache.enable();
              for(UnsignedLong i = 0; i < newSize; ++i) tempCache.add( newIn[i].getCollection(), newOut[i].getCollection() );

              // We gather the computed values into a new output sample
              for(UnsignedLong i = 0; i < size; ++i)
                {
                  // The output values were either in the cache before the call to the evaluation operator
                  if ( p_cache_->hasKey( in[i].getCollection() ) )
                    {
                      out[i] = NumericalPoint::ImplementationType( p_cache_->find( in[i].getCollection() ) );
                    }
                  // or they are in the new temporary cache that store the unique points that have been evaluated
                  else if ( tempCache.hasKey( in[i].getCollection() ) )
                    {
                      out[i] = NumericalPoint::ImplementationType( tempCache.find( in[i].getCollection() ) );
                    }
                  // or something strange occured
                  else throw InternalException(HERE) << "Error in computing sample. Some values can not be retrieved from cache.";
                } // Loop over the output sample
              // We add the computed values into the cache AFTER having read the cache because
              // older values may be flushed
              //for(UnsignedLong i=0; i<newIn.getSize(); ++i) p_cache_->add( newIn[i], newOut[i] );
              p_cache_->merge( tempCache );
              // return the gathered sample
              return out;
            } // If the cache is enabled
          else
            {
              callsNumber_ += size;
              return p_function_->execute( p_state_, in );
            }
        } // try
        catch (InvalidArgumentException & ex) {
          throw;
        }
        catch (WrapperInternalException & ex) {
          throw InternalException(HERE) << ex;
        }
      }



      /* Accessor for input point dimension */
      UnsignedLong ComputedNumericalMathEvaluationImplementation::getInputDimension() const
      /* throw(InternalException) */
      {
        UnsignedLong inputDimension = 0;

        try {
          inputDimension = p_function_->getInNumericalPointDimension( p_state_ );
        }
        catch (WrapperInternalException & ex) {
          throw InternalException(HERE) << ex;
        }

        return inputDimension;
      }



      /* Accessor for output point dimension */
      UnsignedLong ComputedNumericalMathEvaluationImplementation::getOutputDimension() const
      /* throw(InternalException) */
      {
        UnsignedLong outputDimension = 0;

        try {
          outputDimension = p_function_->getOutNumericalPointDimension( p_state_ );
        }
        catch (WrapperInternalException & ex) {
          throw InternalException(HERE) << ex;
        }

        return outputDimension;
      }

      /* Accessor for output point dimension */
      ComputedNumericalMathEvaluationImplementation::Description ComputedNumericalMathEvaluationImplementation::getDescription() const
      /* throw(InternalException) */
      {
        Description description;

        try {
          // Here, we get only the description of the input variable
          description = p_function_->getDescription();
          if (description.getSize() == getInputDimension())
            {
              // Put generic names for the output description if something they are missing in the wrapper
              for (UnsignedLong i = 0; i < getOutputDimension(); ++i) description.add(OSS() << "y" << i);
            }
          // If the description does not match the dimensions, error
          if (description.getSize() != getInputDimension() + getOutputDimension()) throw InternalException(HERE) << "Error: the description " << description << " does not match the dimensions of the function. Here, input dimension=" << getInputDimension() << " and output dimension=" << getOutputDimension() << ". Check the wrapper description.";
        }
        catch (WrapperInternalException & ex) {
          throw InternalException(HERE) << ex;
        }

        return description;
      }

      /* Method save() stores the object through the StorageManager */
      void ComputedNumericalMathEvaluationImplementation::save(StorageManager::Advocate & adv) const
      {
        NumericalMathEvaluationImplementation::save(adv);
      }

      /* Method load() reloads the object from the StorageManager */
      void ComputedNumericalMathEvaluationImplementation::load(StorageManager::Advocate & adv)
      {
        NumericalMathEvaluationImplementation::load(adv);
        *this = *static_cast<ComputedNumericalMathEvaluationImplementation*>(ComputedNumericalMathEvaluationImplementationFactory::getInstance()->buildObject(getName()));
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
