//                                               -*- C++ -*-
/**
 *  @file  GraphImplementation.hxx
 *  @brief GraphImplementation implements graphic devices for plotting through R
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: GraphImplementation.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_GRAPHIMPLEMENTATION_HXX
#define OPENTURNS_GRAPHIMPLEMENTATION_HXX

#include "PersistentObject.hxx"
#include "PersistentCollection.hxx"
#include "Drawable.hxx"
#include "Exception.hxx"
#include "Interval.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Graph
    {
      /**
       * @class GraphImplementation
       *
       * GraphImplementation implements graphic devices for plotting through R,
       * and manages drawables to be plotted on the same window
       */

      class GraphImplementation :
	public Common::PersistentObject
      {

	CLASSNAME;
	
      public:

	typedef Drawable::Description                Description;
	typedef Common::InvalidArgumentException     InvalidArgumentException;
	typedef Common::InternalException            InternalException;
	typedef Common::StorageManager               StorageManager;
	typedef Type::NumericalPoint                 BoundingBox;
	typedef Type::Interval                       Interval;
	typedef Type::Collection<Drawable>           DrawableCollection;
	typedef Type::PersistentCollection<Drawable> DrawablePersistentCollection;

	// Default string value
	static const String NoSpecifiedLabel;

	// Default dimensions for the bitmap
	static const UnsignedLong DefaultWidth;
	static const UnsignedLong DefaultHeight;
	// Default font size for the legend
	static const NumericalScalar DefaultLegendFontSize;

	static const UnsignedLong BoundingBoxSize;

	static Description ValidLegendPositions;
	static Bool IsFirstInitialization;

	enum Format { ALL = 0, PNG, EPS, FIG, PDF };

	/** Default constructor */
	explicit GraphImplementation();

	/** Constructor with parameters */
	GraphImplementation(const String & title,
			    const String & xTitle,
			    const String & yTitle,
			    const Bool showAxes,
			    const String & legendPosition = NoSpecifiedLabel,
			    const NumericalScalar legendFontSize = DefaultLegendFontSize) /* throw(InvalidArgumentException) */;
	

	/** Virtual constructor */
	virtual GraphImplementation * clone() const;

	/** String converter */
	String __repr__() const;

	/** Adds a drawable instance to the collection of drawables contained in GraphImplementation */
	void addDrawable(const Drawable & aDrawable);

	/** Drawables accessor */
	DrawableCollection getDrawables() const;
	void setDrawables(const DrawableCollection & drawableCollection);

	/** Individual drawable accessor */
	Drawable getDrawable(const UnsignedLong index) const;
	void setDrawable(const Drawable & drawable, const UnsignedLong index);


	/** Hide or show x and y axes */
	void setAxes(const Bool showAxes);

	/** Accessor for showAxes_ */
	Bool getAxes() const;
	
	/** Hide or show grid */
	void setGrid(const Bool showGrid);

	/** Accessor for showGrid_ */
	Bool getGrid() const;
	
	/** Accesor for xTitle */
	String getXTitle() const;

	/** Accessor for xTitle */
	void setXTitle(const String & title);
	
	/** Accessor for yTitle */
	String getYTitle() const;
	
	/** Accessor for yTitle */
	void setYTitle(const String & title);

	/** Accesor for title */
	String getTitle() const;
	
	/** Accesor for title */
	void setTitle(const String & title);

	/** Accessor for path */
	String getPath() const;

	/** Accessor for file name */
	String getFileName() const;

	/** The method that generates the graphic files */
	void draw(const String & path,
                  const String & file,
                  const NumericalScalar width,
                  const NumericalScalar height,
		  const Format format = ALL) /* throw(InternalException) */;

	/** The method that generates the graphic files */
	void draw(const String & file,
                  const NumericalScalar width = DefaultWidth,
                  const NumericalScalar height = DefaultHeight,
		  const Format format = ALL) /* throw(InternalException) */;

	/** Get the R command corresponding to the graph */
	String getRCommand() const;

	/** Clean temporary files */
	void clean();

	/** The method returning absolute path of the bitmap graphic file */
	String getBitmap() const;

	/** The method returning absolute path of the postscript graphic file */
	String getPostscript() const;
	
	/** The method returning absolute path of the PDF graphic file */
	String getPDF() const;
	
	/** The method returning absolute path of the vectorial graphic file */
	String getVectorial() const;
	
	/** Get the bounding box of the whole plot */
	BoundingBox getBoundingBox() const;
	
	/** Set the bounding box of the whole plot */
	void setBoundingBox(const BoundingBox & boundingBox);

	/** Set the bounding box of the whole plot */
	void setBoundingBox(const Interval & boundingBox);

	/** Automatic bounding box accessor */
	Bool getAutomaticBoundingBox() const;
	void setAutomaticBoundingBox(const Bool automaticBoundingBox);

	/** Get the legend position */
	String getLegendPosition() const;

	/** Set the legend position */
	void setLegendPosition(const String & position) /* throw(InvalidArgumentException) */;
	
	/** Get the legend font size */
	NumericalScalar getLegendFontSize() const;

	/** Set the legend font size */
	void setLegendFontSize(const NumericalScalar legendFontSize) /* throw(InvalidArgumentException) */;
	
	/** Check for legend position validity */
	Bool isValidLegendPosition(const String & position) const;

	/** Initialize valid legend positions **/
	static void initializeValidLegendPositions();

 	/** Method save() stores the object through the StorageManager */
	void save(StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(StorageManager::Advocate & adv);

	/** Gives all the valid legend positions */
	static Description GetValidLegendPositions();

      private :
	/** Compute the best bounding box to enclose all the drawables */
	void computeBoundingBox() const;

	/** Make R legend command */
	String makeRLegendCommand() const;

	/** Make R header command */
	String makeRHeaderCommand() const;

	/** Make R core command */
	String makeRCoreCommand() const;

	/** GraphImplementationic main title */
	String title_;

	/** Legend position */
	String legendPosition_;

	/** Legend font size */
	NumericalScalar legendFontSize_;

	/** x axis title */
	String xTitle_;

	/** y axis title */
	String yTitle_;

	/** if TRUE, displays the axes on the graphic plot. if FALSE, hides the axes */
	Bool showAxes_;

	/** if TRUE, displays a grid on the graphic plot. if FALSE, hides the grid */
	Bool showGrid_;

	/** Is the bounding box automatically computed. */
	Bool automaticBoundingBox_;

	/** Current bounding box */
	/* The format is:
	 * bb[0] = xmin
	 * bb[1] = xmax
	 * bb[2] = ymin
	 * bb[3] = ymax
	 */
	mutable BoundingBox boundingBox_;

	/** The drawables to be plotted */
	DrawablePersistentCollection drawablesCollection_;

	/** Path to latest files generated */
	String path_;

	/** Name of the graphic file*/
	String fileName_;

      }; /* class GraphImplementation */

    }/* namespace GraphImplementation */

  }/* namespace Base */

}/* namespace OpenTURNS */

#endif /* OPENTURNS_GRAPHIMPLEMENTATION_HXX */
