//                                               -*- C++ -*-
/**
 *  @file  LogNormalFactory.cxx
 *  @brief Factory for LogNormal distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: LogNormalFactory.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "LogNormalFactory.hxx"
#include "LogNormal.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Model::DistributionImplementationFactory::NumericalPoint NumericalPoint;

      /* Default constructor */
      LogNormalFactory::LogNormalFactory():
	DistributionImplementationFactory()
      {
	// Nothing to do
      }

      /* Virtual constructor */
      LogNormalFactory * LogNormalFactory::clone() const
      {
	return new LogNormalFactory(*this);
      }


      /* Here is the interface that all derived class must implement */

      LogNormalFactory::DistributionImplementation * LogNormalFactory::buildImplementation(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a LogNormal distribution from an empty sample";
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a LogNormal distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
	NumericalScalar gamma(sample.getMin()[0]);
	NumericalScalar meanLog(0.0);
	NumericalScalar varLog(0.0);
	UnsignedLong size(sample.getSize());
	if (gamma > 0.0)
	  {
            gamma *= NumericalScalar(size) / NumericalScalar(size + 1.0);
	  }
	else
	  {
            gamma *= NumericalScalar(size + 1.0) / NumericalScalar(size);
	  }
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    if (sample[i][0] > gamma)
	      {
		NumericalScalar logX(log(sample[i][0] - gamma));
		NumericalScalar meanLogOld(meanLog);
		meanLog = (logX + i * meanLog) / (i + 1.0);
		NumericalScalar factor((meanLogOld - logX) / (i + 1.0));
		varLog = i * varLog / (i + 1.0) + i * factor * factor;
	      }
	  }
	if (varLog <= 0.0) throw InternalException(HERE) << "Error: can build a LogNormal distribution only from a sample with varLog > 0, here varLog=" << varLog;
	return LogNormal(meanLog, sqrt(varLog), gamma).clone();
      }

      LogNormalFactory::DistributionImplementation * LogNormalFactory::buildImplementation(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
	try {
	  LogNormal distribution;
	  distribution.setParametersCollection(parameters);
	  return distribution.clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot build a LogNormal distribution from the given parameters";
	  }
      }

      LogNormalFactory::DistributionImplementation * LogNormalFactory::buildImplementation() const /* throw(InvalidArgumentException, InternalException) */
      {
	return LogNormal().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
