//                                               -*- C++ -*-
/**
 *  @file  LibraryLoader.hxx
 *  @brief The class LibraryLoader loads and manages all dynamic libraries
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2012-03-20 06:42:21 +0100 (Tue, 20 Mar 2012) $
 *  Id:      $Id: LibraryLoader.hxx 2428 2012-03-20 05:42:21Z lebrun $
 */
#ifndef OPENTURNS_LIBRARYLOADER_HXX
#define OPENTURNS_LIBRARYLOADER_HXX

#include <iostream>              // for std::ostream
#include <map>                   // for std::map
#include "OStream.hxx"
#include "Library.hxx"
#include "Lockable.hxx"


BEGIN_NAMESPACE_OPENTURNS



/**
 * @class LibraryLoader
 * @brief Loads and manages all dynamic libraries
 * @ingroup Wrapper
 *
 * The LibraryLoader class acts as a general factory to load and unload
 * dynamic libraries (Unix shared objects or Windows DLL) on demand.
 * The class is based on the 'singleton' design pattern.
 * Libraries are loaded only once and handles are stored in a map.
 * @see Library
 * @todo Library reference counting
 * @toto Release method definition and implementation
 */

class LibraryLoader
  : public Lockable
{
  CLASSNAME;
public:


  /** The accessor to the Singleton element */
  static LibraryLoader & GetInstance();

  /* Destructor */
  ~LibraryLoader();

  /* String converter */
  String __repr__() const;
  String __str__(const String & offset = "") const;

  /** @brief Loads the library at path
   * This method takes the path, either relative or absolute, to a dynamic loadable object
   * and tries to lod it. If the object was already loaded, the previous reference is returned.
   * It not the object is searched for on the filesystem. If it can't be found a
   * FileNotFoundException is raised. If it is found but can't be loaded a DynamicLibraryException
   * is raised.
   * @throw FileNotFoundException
   * @throw DynamicLibraryException
   * @param path A path to the object to be loaded
   * @result An handle to the Library usable afterwars by other Open TURNS objects
   */
  Library load(const FileName & path);

protected:

private:

  /**
   * The map that stores all the handles on loaded dynamic libraries
   * Key is the path of the library, value is the handle onto the library.
   * This map is protected against concurrent accesses
   */
  typedef std::map<FileName,LibraryHandle> CatalogType;
  CatalogType libraryCatalog_;

  /** The unique instance of the Singleton model */
  static LibraryLoader * p_instance_;

  /** Default constructor */
  LibraryLoader();

  /** Copy constructor */
  LibraryLoader(const LibraryLoader & other);

  /** Assignment */
  LibraryLoader & operator =(const LibraryLoader & other);

}

  ; /* class LibraryLoader */

/**
 * @fn std::ostream & operator <<(std::ostream & os, const LibraryLoader & obj)
 * @brief Output stream converter
 * @param os A STL output stream object
 * @param obj The object read by \em os
 * @return A reference to \em os
 */
std::ostream & operator <<(std::ostream & os, const LibraryLoader & obj);
OStream & operator <<(OStream & OS, const LibraryLoader & obj);

END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_LIBRARYLOADER_HXX */
