//                                               -*- C++ -*-
/**
 *  @file  PostAnalyticalImportanceSampling.cxx
 *  @brief PostAnalyticalImportanceSampling is an implementation of the  importance sampling Monte Carlo simulation method in standard space
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-04-15 16:24:44 +0200 (mar, 15 avr 2008) $
 *  Id:      $Id: PostAnalyticalImportanceSampling.cxx 789 2008-04-15 14:24:44Z lebrun $
 */
#include "PostAnalyticalImportanceSampling.hxx"
#include "NumericalPoint.hxx"
#include "ComparisonOperatorImplementation.hxx"

BEGIN_NAMESPACE_OPENTURNS



/*
 * @class PostAnalyticalImportanceSampling
 */

CLASSNAMEINIT(PostAnalyticalImportanceSampling);


/* Constructor with parameters */
PostAnalyticalImportanceSampling::PostAnalyticalImportanceSampling(const AnalyticalResult & analyticalResult):
  PostAnalyticalSimulation(analyticalResult)
{
  // Nothing to do
}

/* Virtual constructor */
PostAnalyticalImportanceSampling * PostAnalyticalImportanceSampling::clone() const
{
  return new PostAnalyticalImportanceSampling(*this);
}

/* Compute the block sample */
NumericalSample PostAnalyticalImportanceSampling::computeBlockSample()
{
  const UnsignedLong blockSize(getBlockSize());
  const NumericalPoint standardSpaceDesignPoint(analyticalResult_.getStandardSpaceDesignPoint());
  // Get the threshold and the reliability index
  const NumericalScalar threshold(event_.getThreshold());
  // First, compute a sample of the importance distribution. It is simply
  // the standard distribution translated to the design point
  NumericalSample inputSample(standardDistribution_.getNumericalSample(blockSize));
  inputSample.translate(standardSpaceDesignPoint);
  // Then, evaluate the function on this sample
  NumericalSample blockSample(getEvent().getImplementation()->getFunction()(inputSample));
  // realizedEventSample = NumericalSample(blockSize_, inputSample.getDimension());
  // Then, modify in place this sample to take into account the change in the input distribution
  for (UnsignedLong i = 0; i < blockSize; ++i)
    {
      const Bool isRealized(getEvent().getOperator()(blockSample[i][0], threshold));
      // If the event has occured
      if (isRealized)
        {
          // If the event occured, the value is p_initial(x[i]) / p_importance(x[i])
          const NumericalScalar weight(standardDistribution_.computePDF(inputSample[i]) / standardDistribution_.computePDF(inputSample[i] - standardSpaceDesignPoint));
          blockSample[i][0] = weight;
        }
      else blockSample[i][0] = 0.0;
    }
  return blockSample;
}

/* String converter */
String PostAnalyticalImportanceSampling::__repr__() const
{
  OSS oss;
  oss << "class=" << PostAnalyticalImportanceSampling::GetClassName()
      << " derived from " << PostAnalyticalSimulation::__repr__();
  return oss;
}

END_NAMESPACE_OPENTURNS
