//                                               -*- C++ -*-
/**
 *  @file  LogNormal.cxx
 *  @brief The LogNormal distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: LogNormal.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <cmath>
#include "LogNormal.hxx"
#include "DistFunc.hxx"
#include "SpecFunc.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"
#include "Normal.hxx"

BEGIN_NAMESPACE_OPENTURNS




CLASSNAMEINIT(LogNormal);

static Factory<LogNormal> RegisteredFactory("LogNormal");

/* Default constructor */
LogNormal::LogNormal()
  : NonEllipticalDistribution("LogNormal"),
    muLog_(0.0),
    sigmaLog_(1.0),
    gamma_(0.0),
    // 1 / SQRT(2Pi)
    normalizationFactor_(0.39894228040143267794),
    H_(pow(M_PI / (2.0 * sigmaLog_), 2.0) / 2.0 - 0.5 * log(2.0 * M_PI))
{
  setDimension(1);
  setIntegrationNodesNumber(128);
  computeRange();
}

/* Default constructor */
LogNormal::LogNormal(const NumericalScalar arg1,
                     const NumericalScalar arg2,
                     const NumericalScalar gamma,
                     const ParameterSet set)
  : NonEllipticalDistribution("LogNormal"),
    muLog_(0.0),
    sigmaLog_(0.0),
    gamma_(gamma),
    normalizationFactor_(0.0),
    H_(0.0)
{
  // Adapt the integration nodes number to the needs of the characteristic function integration
  setIntegrationNodesNumber(128);
  switch (set) {
  case MUSIGMA_LOG:
    // This call set also the range.
    setMuLogSigmaLog(arg1, arg2);
    break;

  case MUSIGMA:
    // This call set also the range.
    setMuSigma(arg1, arg2);
    break;

  case MU_SIGMAOVERMU:
    if (arg1 == 0.0) throw InvalidArgumentException(HERE) << "Error: mu cannot be null in the parameter set (mu, sigmaOverMu)";
    // This call set also the range.
    setMuSigma(arg1, arg1 * arg2);
    break;

  default:
    throw InvalidArgumentException(HERE) << "Invalid parameter set argument";

  } /* end switch */
  normalizationFactor_ = 1.0 / (sigmaLog_ * sqrt(2.0 * M_PI));
  setDimension(1);
}

/* Comparison operator */
Bool LogNormal::operator ==(const LogNormal & other) const
{
  if (this == &other) return true;
  return (muLog_ == other.muLog_) && (sigmaLog_ == other.sigmaLog_) && (gamma_ == other.gamma_);
}

/* String converter */
String LogNormal::__repr__() const
{
  OSS oss;
  oss << "class=" << LogNormal::GetClassName()
      << " name=" << getName()
      << " dimension=" << getDimension()
      << " muLog=" << muLog_
      << " sigmaLog=" << sigmaLog_
      << " gamma=" << gamma_;
  return oss;
}

String LogNormal::__str__(const String & offset) const
{
  OSS oss;
  oss << offset << getClassName() << "(muLog = " << muLog_ << ", sigmaLog = " << sigmaLog_ << ", gamma = " << gamma_ << ")";
  return oss;
}

/* Virtual constructor */
LogNormal * LogNormal::clone() const
{
  return new LogNormal(*this);
}

/* Compute the numerical range of the distribution given the parameters values */
void LogNormal::computeRange()
{
  NumericalPoint lowerBound(1, gamma_);
  const NumericalPoint upperBound(computeUpperBound());
  const Interval::BoolCollection finiteLowerBound(1, true);
  const Interval::BoolCollection finiteUpperBound(1, false);
  setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
}

/* Get one realization of the distribution */
NumericalPoint LogNormal::getRealization() const
{
  return NumericalPoint(1, gamma_ + exp(muLog_ + sigmaLog_ * DistFunc::rNormal()));
}



/* Get the DDF of the distribution */
NumericalPoint LogNormal::computeDDF(const NumericalPoint & point) const
{
  NumericalScalar x(point[0] - gamma_);
  // Here we keep the bound within the special case as the distribution is continuous
  if (x <= 0.0) return NumericalPoint(1, 0.0);
  NumericalScalar v(sigmaLog_ * sigmaLog_);
  return NumericalPoint(1, (muLog_ - log(x) - v) / (v *x) * computePDF(point));
}


/* Get the PDF of the distribution */
NumericalScalar LogNormal::computePDF(const NumericalPoint & point) const
{
  NumericalScalar x(point[0] - gamma_);
  // Here we keep the bound within the special case as the distribution is continuous
  if (x <= 0.0) return 0.0;
  NumericalScalar logX((log(x) - muLog_) / sigmaLog_);
  return normalizationFactor_ * exp(-0.5 * logX * logX) / x;
}


/* Get the CDF of the distribution */
NumericalScalar LogNormal::computeCDF(const NumericalPoint & point,
                                      const Bool tail) const
{
  NumericalScalar x(point[0] - gamma_);
  // Here we keep the bound within the special case as the distribution is continuous
  if (x <= 0.0) return (tail ? 1.0 : 0.0);
  NumericalScalar logX((log(x) - muLog_) / sigmaLog_);
  return DistFunc::pNormal(logX, tail);
}

/* Compute the integrand that is involved in the computation of the characteristic function */
NumericalComplex LogNormal::characteristicIntegrand(const NumericalScalar t,
                                                    const NumericalScalar nu) const
{
  return exp(NumericalComplex(-nu * exp(sigmaLog_ * t) - 0.5 * t * t, -M_PI_2 * t /sigmaLog_));
}

/* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X))
 * Algorithm adapted from:
 * John A. Gubner, "A New Formula for Lognormal Characteristic Functions",
 * IEEE transactions on vehicular technology, vol. 55, no. 5, September 2006.
 */
NumericalComplex LogNormal::computeCharacteristicFunction(const NumericalScalar x,
                                                          const Bool logScale) const
{
  // Quick return for null argument
  if (x == 0.0)
    {
      if (logScale) return 0.0;
      return 1.0;
    }
  static const NumericalSample legendreNodesAndWeights(getGaussNodesAndWeights());
  NumericalComplex value(0.0);
  // Compute the characteristic function for the positive arguments
  const NumericalScalar nu(fabs(x) * exp(muLog_));
  const NumericalScalar tmax(-SpecFunc::LambertW(sigmaLog_ * sigmaLog_ * nu) / (sqrt(2.0) * sigmaLog_));
  const NumericalScalar vMax(abs(characteristicIntegrand(tmax, nu)));
  // Find the lower bound of the integration interval
  NumericalScalar a(tmax);
  NumericalScalar vA(vMax);
  const NumericalScalar pdfEpsilon(ResourceMap::GetAsNumericalScalar("DistributionImplementation-DefaultPDFEpsilon"));
  do
    {
      a -= sigmaLog_;
      vA = abs(characteristicIntegrand(a, nu));
    }
  while (vA / vMax > pdfEpsilon);
  // Find the upper bound of the integration interval
  NumericalScalar b(tmax);
  NumericalScalar vB(vMax);
  do
    {
      b += sigmaLog_;
      vB = abs(characteristicIntegrand(b, nu));
    }
  while (vB / vMax > pdfEpsilon);
  const NumericalScalar halfLength(0.5 * (b - a));
  for (UnsignedLong i = 0; i < legendreNodesAndWeights.getDimension(); ++i)
    {
      value += legendreNodesAndWeights[1][i] * characteristicIntegrand(a + (1.0 + legendreNodesAndWeights[0][i]) * halfLength, nu);
    }
  value *= halfLength;
  value = log(value);
  value += H_ + NumericalComplex(0.0, x * gamma_);
  // Use symmetry for negative arguments
  if (x < 0.0) value = conj(value);
  if (logScale) return value;
  return exp(value);
}

/* Get the PDFGradient of the distribution */
NumericalPoint LogNormal::computePDFGradient(const NumericalPoint & point) const
{
  NumericalScalar x(point[0] - gamma_);
  NumericalPoint pdfGradient(3, 0.0);
  // Here we keep the bound within the special case as the distribution is continuous
  if (x <= 0.0) return pdfGradient;
  NumericalScalar logX((log(x) - muLog_) / sigmaLog_);
  NumericalScalar pdf(normalizationFactor_ * exp(-0.5 * logX * logX) / x);
  pdfGradient[0] = pdf * logX / sigmaLog_;
  pdfGradient[1] = pdf * (logX - 1.0) * (logX + 1.0) / sigmaLog_;
  pdfGradient[2] = pdf * (1.0 + logX / sigmaLog_) / x;
  return pdfGradient;
}

/* Get the CDFGradient of the distribution */
NumericalPoint LogNormal::computeCDFGradient(const NumericalPoint & point) const
{
  NumericalScalar x(point[0] - gamma_);
  NumericalPoint cdfGradient(3, 0.0);
  // Here we keep the bound within the special case as the distribution is continuous
  if (x <= 0.0) return cdfGradient;
  NumericalScalar logX((log(x) - muLog_) / sigmaLog_);
  NumericalScalar pdf(normalizationFactor_ * exp(-0.5 * logX * logX) / x);
  cdfGradient[0] = -x * pdf;
  cdfGradient[1] = -logX * x * pdf;
  cdfGradient[2] = -pdf;
  return cdfGradient;
}

/* Get the quantile of the distribution */
NumericalScalar LogNormal::computeScalarQuantile(const NumericalScalar prob,
                                                 const Bool tail,
                                                 const NumericalScalar precision) const
{
  return gamma_ + exp(muLog_ + sigmaLog_ * DistFunc::qNormal(prob, tail));
}

/* Compute the mean of the distribution */
void LogNormal::computeMean() const
{
  mean_ = NumericalPoint(1, getMu());
  isAlreadyComputedMean_ = true;
}

/* Get the standard deviation of the distribution */
NumericalPoint LogNormal::getStandardDeviation() const
{
  return NumericalPoint(1, getSigma());
}

/* Get the skewness of the distribution */
NumericalPoint LogNormal::getSkewness() const
{
  NumericalScalar expSigmaLog2(exp(sigmaLog_ * sigmaLog_));
  return NumericalPoint(1, (expSigmaLog2 + 2.0) * sqrt(expSigmaLog2 - 1.0));
}

/* Get the kurtosis of the distribution */
NumericalPoint LogNormal::getKurtosis() const
{
  NumericalScalar expSigmaLog2(exp(sigmaLog_ * sigmaLog_));
  return NumericalPoint(1, -3.0 + expSigmaLog2 * expSigmaLog2 * (3.0 + expSigmaLog2 * (2.0 + expSigmaLog2)));
}

/* Get the moments of the standardized distribution */
NumericalPoint LogNormal::getStandardMoment(const UnsignedLong n) const
{
  return NumericalPoint(1, exp(n * muLog_ + 0.5 * pow(n * sigmaLog_, 2)));
}

/* Compute the covariance of the distribution */
void LogNormal::computeCovariance() const
{
  covariance_ = CovarianceMatrix(1);
  const NumericalScalar expSigmaLog2(exp(sigmaLog_ * sigmaLog_));
  covariance_(0, 0) = expSigmaLog2 * exp(2.0 * muLog_) * (expSigmaLog2 - 1.0);
  isAlreadyComputedCovariance_ = true;
}

/* Parameters value and description accessor */
LogNormal::NumericalPointWithDescriptionCollection LogNormal::getParametersCollection() const
{
  NumericalPointWithDescriptionCollection parameters(1);
  NumericalPointWithDescription point(3);
  Description description(point.getDimension());
  point[0] = muLog_;
  point[1] = sigmaLog_;
  point[2] = gamma_;
  description[0] = "muLog";
  description[1] = "sigmaLog";
  description[2] = "gamma";
  point.setDescription(description);
  point.setName(getDescription()[0]);
  parameters[0] = point;
  return parameters;
}

void LogNormal::setParametersCollection(const NumericalPointCollection & parametersCollection)
{
  *this = LogNormal(parametersCollection[0][0], parametersCollection[0][1], parametersCollection[0][2]);
}


/* Interface specific to LogNormal */

/* MuLogSigmaLog accessor */
void LogNormal::setMuLogSigmaLog(const NumericalScalar muLog,
                                 const NumericalScalar sigmaLog)
{
  if (sigmaLog <= 0.) throw InvalidArgumentException(HERE) << "SigmaLog MUST be positive, here sigmaLog=" << sigmaLog;
  if ((muLog != muLog_) || (sigmaLog != sigmaLog_))
    {
      muLog_ = muLog;
      sigmaLog_ = sigmaLog;
      // Check if the parameters values are not crazy
      static const Interval range(Normal().getRange());
      const NumericalScalar rMin(muLog_ + range.getUpperBound()[0] * sigmaLog_);
      const NumericalScalar rMax(muLog_ + range.getLowerBound()[0] * sigmaLog_);
      if ((rMin >= SpecFunc::LogMaxNumericalScalar) ||
          (rMax <= SpecFunc::LogMinNumericalScalar)) throw InvalidArgumentException(HERE) << "MuLog and SigmaLog lead to a LogNormal distribution with a too much wide range";
      H_ = pow(M_PI / (2.0 * sigmaLog_), 2.0) / 2.0 - 0.5 * log(2.0 * M_PI);
      isAlreadyComputedMean_ = false;
      isAlreadyComputedCovariance_ = false;
      computeRange();
    }
}

/* MuLog accessor */
void LogNormal::setMuLog(const NumericalScalar muLog)
{
  if (muLog != muLog_)
    {
      muLog_ = muLog;
      // Check if the parameters values are not crazy
      static const Interval range(Normal().getRange());
      const NumericalScalar rMin(muLog_ + range.getUpperBound()[0] * sigmaLog_);
      const NumericalScalar rMax(muLog_ + range.getLowerBound()[0] * sigmaLog_);
      if ((rMin >= SpecFunc::LogMaxNumericalScalar) ||
          (rMax <= SpecFunc::LogMinNumericalScalar)) throw InvalidArgumentException(HERE) << "MuLog and SigmaLog lead to a LogNormal distribution with a too much wide range";
      isAlreadyComputedMean_ = false;
      isAlreadyComputedCovariance_ = false;
      computeRange();
    }
}

NumericalScalar LogNormal::getMuLog() const
{
  return muLog_;
}


/* SigmaLog accessor */
void LogNormal::setSigmaLog(const NumericalScalar sigmaLog)
{
  if (sigmaLog <= 0.) throw InvalidArgumentException(HERE) << "SigmaLog MUST be positive, here sigmaLog=" << sigmaLog;
  if (sigmaLog != sigmaLog_)
    {
      sigmaLog_ = sigmaLog;
      // Check if the parameters values are not crazy
      static const Interval range(Normal().getRange());
      const NumericalScalar rMin(muLog_ + range.getUpperBound()[0] * sigmaLog_);
      const NumericalScalar rMax(muLog_ + range.getLowerBound()[0] * sigmaLog_);
      if ((rMin >= SpecFunc::LogMaxNumericalScalar) ||
          (rMax <= SpecFunc::LogMinNumericalScalar)) throw InvalidArgumentException(HERE) << "MuLog and SigmaLog lead to a LogNormal distribution with a too much wide range";
      H_ = pow(M_PI / (2.0 * sigmaLog_), 2.0) / 2.0 - 0.5 * log(2.0 * M_PI);
      isAlreadyComputedMean_ = false;
      isAlreadyComputedCovariance_ = false;
      computeRange();
    }
}

NumericalScalar LogNormal::getSigmaLog() const
{
  return sigmaLog_;
}

NumericalScalar LogNormal::getMu() const
{
  return gamma_ + exp(muLog_ + 0.5 * sigmaLog_ * sigmaLog_);
}


/* Sigma accessor */
void LogNormal::setMuSigma(const NumericalScalar mu,
                           const NumericalScalar sigma)
{
  if (sigma <= 0.0) throw InvalidArgumentException(HERE) << "Error: sigma must be > 0, here sigma=" << sigma;
  if (mu <= gamma_) throw InvalidArgumentException(HERE) << "Error: mu must be greater than gamma, here mu=" << mu << " and gamma=" << gamma_;
  NumericalScalar shift(mu - gamma_);
  NumericalScalar shiftSquared(shift * shift);
  NumericalScalar deltaSquareRoot(sqrt(shiftSquared + sigma * sigma));
  // This call takes care of the range and the mean and covariance flags
  setMuLogSigmaLog(log(shiftSquared / deltaSquareRoot), sqrt(2 * log(deltaSquareRoot / shift)));
}

NumericalScalar LogNormal::getSigma() const
{
  NumericalScalar expSigmaLog2(exp(sigmaLog_ * sigmaLog_));
  return exp(muLog_) * sqrt(expSigmaLog2 * (expSigmaLog2 - 1.0));
}

/* Gamma accessor */
void LogNormal::setGamma(const NumericalScalar gamma)
{
  if (gamma != gamma_)
    {
      gamma_ = gamma;
      isAlreadyComputedMean_ = false;
      // The covariance does not depends on gamma
      computeRange();
    }
}

NumericalScalar LogNormal::getGamma() const
{
  return gamma_;
}

/* SigmaOverMu accessor */
NumericalScalar LogNormal::getSigmaOverMu() const
{
  NumericalScalar mu(getMu());
  if (mu == 0.0) throw NotDefinedException(HERE) << "Error: trying to get sigmaOverMu with mu equals to zero";
  return getSigma() / mu;
}

/* Method save() stores the object through the StorageManager */
void LogNormal::save(Advocate & adv) const
{
  NonEllipticalDistribution::save(adv);
  adv.saveAttribute( "muLog_", muLog_ );
  adv.saveAttribute( "sigmaLog_", sigmaLog_ );
  adv.saveAttribute( "gamma_", gamma_ );
  adv.saveAttribute( "normalizationFactor_", normalizationFactor_ );
  adv.saveAttribute( "H_", H_ );
}

/* Method load() reloads the object from the StorageManager */
void LogNormal::load(Advocate & adv)
{
  NonEllipticalDistribution::load(adv);
  adv.loadAttribute( "muLog_", muLog_ );
  adv.loadAttribute( "sigmaLog_", sigmaLog_ );
  adv.loadAttribute( "gamma_", gamma_ );
  adv.loadAttribute( "normalizationFactor_", normalizationFactor_ );
  adv.loadAttribute( "H_", H_ );
  computeRange();
}

END_NAMESPACE_OPENTURNS
