
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk.h,v 1.49 2006/01/19 07:42:01 mschwerin Exp $
 *
 */

#ifndef HAVE_ODK_H
#define HAVE_ODK_H


#include <xine.h>

#include "event.h"
#include "meta_info.h"

/*
 * alignments for text
 */

#define ODK_ALIGN_LEFT              (0x001)
#define ODK_ALIGN_CENTER            (0x002)
#define ODK_ALIGN_RIGHT             (0x004)
#define ODK_ALIGN_TOP               (0x008)
#define ODK_ALIGN_VCENTER           (0x010)
#define ODK_ALIGN_BOTTOM            (0x020)

#define PIXMAP_SIMPLE_ARROW_UP      (1)
#define PIXMAP_SIMPLE_ARROW_DOWN    (2)
#define PIXMAP_POWEROFF             (3)
#define PIXMAP_HOME                 (4)
#define PIXMAP_FOLDER               (5)
#define PIXMAP_PLAYLIST             (6)

#define AUDIO_CHANNEL_OFF           (-2)
#define AUDIO_CHANNEL_AUTO          (-1)

#define SPU_CHANNEL_OFF             (-2)
#define SPU_CHANNEL_AUTO            (-1)


/*
 * operation modes
 */
typedef enum {
    ODK_MODE_NULL,
    ODK_MODE_NORMAL,
    ODK_MODE_LOGO
} odk_mode_t;

/*
 * stream parameters
 */
typedef enum {
    ODK_PARAM_POSITION,         /* 0..100 */
    ODK_PARAM_SPEED = XINE_PARAM_SPEED, /* see below */
    ODK_PARAM_AUDIO_OFFSET = XINE_PARAM_AV_OFFSET,      /* unit: 1/90000 sec */
    ODK_PARAM_AUDIO_CHANNEL = XINE_PARAM_AUDIO_CHANNEL_LOGICAL, /* -1 => auto, -2 => off */
    ODK_PARAM_SPU_CHANNEL = XINE_PARAM_SPU_CHANNEL,     /* -1 => auto, -2 => off */
    ODK_PARAM_SPU_OFFSET = XINE_PARAM_SPU_OFFSET,       /* unit: 1/9 ms */
    ODK_PARAM_AUDIO_VOLUME = XINE_PARAM_AUDIO_VOLUME,   /* 0..100 */
    ODK_PARAM_AUDIO_MUTE = XINE_PARAM_AUDIO_MUTE,       /* 1=>mute, 0=>unmute */
    ODK_PARAM_VO_DEINTERLACE = XINE_PARAM_VO_DEINTERLACE,       /* bool */
    ODK_PARAM_VO_ASPECT_RATIO = XINE_PARAM_VO_ASPECT_RATIO,     /* see below */
    ODK_PARAM_VO_HUE = XINE_PARAM_VO_HUE,       /* 0..65535 */
    ODK_PARAM_VO_SATURATION = XINE_PARAM_VO_SATURATION, /* 0..65535 */
    ODK_PARAM_VO_CONTRAST = XINE_PARAM_VO_CONTRAST,     /* 0..65535 */
    ODK_PARAM_VO_BRIGHTNESS = XINE_PARAM_VO_BRIGHTNESS, /* 0..65535 */
    ODK_PARAM_VO_ZOOM_X = XINE_PARAM_VO_ZOOM_X, /* 0..400 */
    ODK_PARAM_VO_ZOOM_Y = XINE_PARAM_VO_ZOOM_Y  /* 0..400 */
} odk_stream_param_t;

typedef enum {
    ODK_STREAM_INFO_HAS_VIDEO = XINE_STREAM_INFO_HAS_VIDEO,
    ODK_STREAM_INFO_HAS_AUDIO = XINE_STREAM_INFO_HAS_AUDIO,
    ODK_STREAM_INFO_MAX_AUDIO_CHANNEL = XINE_STREAM_INFO_MAX_AUDIO_CHANNEL,
    ODK_STREAM_INFO_MAX_SPU_CHANNEL = XINE_STREAM_INFO_MAX_SPU_CHANNEL
} odk_stream_info_t;

/*
 * stream speeds
 */
typedef enum {
    ODK_SPEED_PAUSE = XINE_SPEED_PAUSE,
    ODK_SPEED_SLOW_4 = XINE_SPEED_SLOW_4,
    ODK_SPEED_SLOW_2 = XINE_SPEED_SLOW_2,
    ODK_SPEED_NORMAL = XINE_SPEED_NORMAL,
    ODK_SPEED_FAST_2 = XINE_SPEED_FAST_2,
    ODK_SPEED_FAST_4 = XINE_SPEED_FAST_4
} odk_speed_t;

/*
 * aspect ratios
 */
typedef enum {
    ODK_VO_ASPECT_AUTO = XINE_VO_ASPECT_AUTO,
    ODK_VO_ASPECT_SQUARE = XINE_VO_ASPECT_SQUARE,
    ODK_VO_ASPECT_4_3 = XINE_VO_ASPECT_4_3,
    ODK_VO_ASPECT_ANAMORPHIC = XINE_VO_ASPECT_ANAMORPHIC,
    ODK_VO_ASPECT_DVB = XINE_VO_ASPECT_DVB
} odk_aspect_ratio_t;

/*
 * public data struct for input devices
 */
typedef struct {
    pthread_t thread;
} odk_input_t;

/*
 * opaque odk data type
 */
typedef struct odk_s odk_t;

typedef void (*odk_event_handler_t) (void *data, oxine_event_t * ev);

/*
 * ***************************************************************************
 * The following methods are for creating and freeing the odk object.
 * ***************************************************************************
 */
odk_t *odk_init (xine_t * xine);
void odk_free (odk_t * odk);

/*
 * ***************************************************************************
 * The following methods are related to event handling and the eventloop.
 * ***************************************************************************
 */
void odk_run (odk_t * odk);
void odk_exit (odk_t * odk);

void odk_listen_to (odk_t * odk, const char *id);

void odk_set_event_handler (odk_t * odk, odk_event_handler_t cb, void *data);

void odk_oxine_event_send (odk_t * odk, oxine_event_t * event);
void odk_xine_event_send (odk_t * odk, int type);

void odk_forward_events_to_xine (odk_t * odk, int forward);

/*
 * ***************************************************************************
 * The following methods are used by the oxine toolkit (otk) for drawing,
 * showing and hiding the OSD.
 * ***************************************************************************
 */
void odk_osd_show (odk_t * odk);
void odk_osd_hide (odk_t * odk);
void odk_osd_clear (odk_t * odk);

uint8_t *odk_get_bitmap (int type);

void odk_draw_point (odk_t * odk, int x, int y, int color);
void odk_draw_line (odk_t * odk, int x1, int y1, int x2, int y2, int color);
void odk_draw_rect (odk_t * odk, int x1, int y1, int x2, int y2, int color,
                    int filled);
void odk_draw_text (odk_t * odk, int x, int y, const char *text,
                    int alignment, int color);
void odk_draw_bitmap (odk_t * odk, uint8_t * bitmap, int x1, int y1,
                      int width, int height, uint8_t * palette_map);


void odk_get_text_size (odk_t * odk, const char *text,
                        int *width, int *height);
void odk_osd_set_font (odk_t * odk, const char *font, int font_size);
int odk_osd_alloc_text_palette (odk_t * odk, uint32_t fg_color,
                                uint8_t fg_trans, uint32_t bg_color,
                                uint8_t bg_trans, uint32_t bo_color,
                                uint8_t bo_trans);
int odk_osd_get_color (odk_t * odk, uint32_t color, uint8_t trans);
int odk_is_osd_visible (odk_t * odk);
void odk_osd_get_user_color (odk_t * odk, char *name, uint32_t * color,
                             uint8_t * trans);
void odk_osd_draw_image (odk_t * odk, const char *mrl,
                         int x, int y, int width, int height);
void odk_osd_hide_image (odk_t * odk);

/*
 * ***************************************************************************
 * The following methods are for window control.
 * ***************************************************************************
 */
void odk_show_window (odk_t * odk, int fullscreen);
void odk_hide_window (odk_t * odk);
void odk_set_fullscreen (odk_t * odk, int fullscreen);
int odk_is_fullscreen (odk_t * odk);

/*
 * ***************************************************************************
 * The following methods are stream related.
 * ***************************************************************************
 */
int odk_play (odk_t * odk, const char *title, const char *mrl, int mode);
void odk_stop (odk_t * odk);
int odk_eject (odk_t * odk);

int odk_get_stream_param (odk_t * odk, odk_stream_param_t param);
char *odk_get_stream_param_as_string (odk_t * odk, odk_stream_param_t param);
char *odk_get_stream_param_name (odk_t * odk, odk_stream_param_t param);
int odk_set_stream_param (odk_t * odk, odk_stream_param_t param, int value);
int odk_change_stream_param (odk_t * odk, odk_stream_param_t param, int how,
                             int min, int max);

char *odk_get_current_mrl (odk_t * odk);

int odk_current_is_logo_mode (odk_t * odk);
int odk_current_is_normal_mode (odk_t * odk);

int odk_current_is_cdda (odk_t * odk);
int odk_current_is_dvd (odk_t * odk);
int odk_current_is_vcd (odk_t * odk);
int odk_current_is_dvb (odk_t * odk);
int odk_current_has_video (odk_t * odk);
int odk_current_has_chapters (odk_t * odk);

char *odk_get_meta_info (odk_t * odk, meta_info_t type);
uint32_t odk_get_stream_info (odk_t * odk, odk_stream_info_t info);

int odk_get_audio_lang (odk_t * odk, int channel, char *lang);
int odk_get_spu_lang (odk_t * odk, int channel, char *lang);

int odk_get_pos_length (odk_t * odk, int *pos, int *time, int *length);
int odk_supports_video_driver (const char *video_driver);

#endif
