
#include "config.h"

#include <assert.h>

#include "heap.h"
#include "joystick.h"
#include "lirc.h"
#include "logger.h"
#include "odk_private.h"
#include "odk.h"
#include "scheduler.h"

/* 
 * ***************************************************************************
 * Name:            odk_set_event_handler
 * Access:          public
 *
 * Description:     Sets the event handling function.
 * ***************************************************************************
 */
void
odk_set_event_handler (odk_t * odk,
                       odk_event_handler_t event_handler,
                       void *event_handler_data)
{
    odk->event_handler = event_handler;
    odk->event_handler_data = event_handler_data;
}


/* 
 * ***************************************************************************
 * Name:            odk_forward_events_to_xine
 * Access:          public
 *
 * Description:     Are events supposed to be forwarded to the xine engine.
 * ***************************************************************************
 */
void
odk_forward_events_to_xine (odk_t * odk, int forward)
{
    odk->forward_events_to_xine = forward;
}


/* 
 * ***************************************************************************
 * Name:            odk_xine_event_send
 * Access:          public
 *
 * Description:     Send an event to the xine engine.
 * ***************************************************************************
 */
void
odk_xine_event_send (odk_t * odk, int type)
{
    if (!odk->forward_events_to_xine)
        return;

    xine_event_t ev;
    ev.type = type;
    ev.data = NULL;
    ev.data_length = 0;
    xine_event_send (odk->win->stream, &ev);
}


/* 
 * ***************************************************************************
 * Name:            odk_oxine_event_send
 * Access:          public
 *
 * Description:     Sends an event the registered event handler.
 * ***************************************************************************
 */
void
odk_oxine_event_send (odk_t * odk, oxine_event_t * event)
{
    if (odk->event_handler)
        odk->event_handler (odk->event_handler_data, event);
}


/*
 * ***************************************************************************
 * Name:            image_finish_sender
 * Access:          private
 *
 * Description:     Sends an playback finished event to oxine after a short
 *                  delay. This is used when displaying images.
 * ***************************************************************************
 */
static void
image_finish_sender (void *odk_p)
{
    odk_t *odk = (odk_t *) odk_p;

    oxine_event_t ev;
    ev.type = OXINE_EVENT_PLAYBACK_FINISHED;
    odk_oxine_event_send (odk, &ev);
}


/* 
 * ***************************************************************************
 * Name:            scale_mouse_position
 * Access:          private
 *
 * Description:     Scales the mouse position to the current video size.
 * ***************************************************************************
 */
static void
scale_mouse_position (odk_t * odk, int *x, int *y)
{
    /* Scale position. */
    x11_rectangle_t rect;
    rect.x = *x;
    rect.y = *y;
    rect.w = 0;
    rect.h = 0;

    if (xine_port_send_gui_data (odk->win->video_port,
                                 XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO,
                                 (void *) &rect) == -1) {
        return;
    }

    *x = rect.x;
    *y = rect.y;
}


/* 
 * ***************************************************************************
 * Name:            odk_event_handler
 * Access:          private
 *
 * Description:     Handles events coming from the current video window and
 *                  the input plugins (LIRC, joystick).
 * ***************************************************************************
 */
static void
odk_event_handler (void *odk_p, oxine_event_t * event)
{
    odk_t *odk = (odk_t *) odk_p;

    switch (event->type) {
        case OXINE_EVENT_FRAME_FORMAT_CHANGED:
            odk_osd_adapt_size (odk);
            break;
        case OXINE_EVENT_BUTTON:
        case OXINE_EVENT_MOTION:
            /* Send an event to the xine engine. */
            if (odk->forward_events_to_xine) {
                int vx = event->data.where.x;
                int vy = event->data.where.y;
                scale_mouse_position (odk, &vx, &vy);

                xine_input_data_t inp;
                inp.x = vx;
                inp.y = vy;
                inp.button = event->source.button;

                xine_event_t xine_event;
                xine_event.type = event->type;
                xine_event.data = &inp;
                xine_event.data_length = sizeof (inp);

                xine_event_send (odk->win->stream, &xine_event);
            }

            /* Scale x and y to video size. */
            if (!odk->use_unscaled_osd || !odk->is_unscaled_osd_available) {
                int vx = event->data.where.x;
                int vy = event->data.where.y;
                scale_mouse_position (odk, &vx, &vy);

                event->data.where.x = vx;
                event->data.where.y = vy;
            }

            /* Scale x any y to OSD size. */
            {
                event->data.where.x =
                    (int) ((double) event->data.where.x / odk->hscale);
                event->data.where.y =
                    (int) ((double) event->data.where.y / odk->vscale);
            }

            break;
        case OXINE_EVENT_KEY:
            /* Send an event to the xine engine. */
            if (odk->forward_events_to_xine) {
                xine_event_t xine_event;

                xine_event.stream = odk->win->stream;
                xine_event.type = event->source.key;
                xine_event.data = NULL;
                xine_event.data_length = 0;

                xine_event_send (odk->win->stream, &xine_event);
            }
            break;
        default:
            break;
    }

    /* Send the event to the frontend. */
    odk_oxine_event_send (odk, event);
}


/* 
 * ***************************************************************************
 * Name:            xine_event_cb
 * Access:          private
 *
 * Description:     Handles events coming from the xine engine.
 * ***************************************************************************
 */
static void
xine_event_cb (void *odk_p, const xine_event_t * xine_event)
{
    odk_t *odk = (odk_t *) odk_p;

    switch (xine_event->type) {
        case OXINE_EVENT_PLAYBACK_FINISHED:
            if (odk->current_mode == ODK_MODE_NULL)
                return;
            if (odk->current_mode == ODK_MODE_LOGO)
                return;
            if (xine_event->stream == odk->background_stream)
                return;

            /* Restart the animation stream */
            if (xine_event->stream == odk->animation_stream) {
                xine_play (odk->animation_stream, 0, 0);
                return;
            }

            /* If its an image we start a finish job. */
            if (is_image (odk->current_mrl)) {
                odk->image_finish_job =
                    schedule_job (odk->image_change_delay * 1000,
                                  image_finish_sender, odk);
                return;
            }

            /* The main stream has ended. */
            if (xine_event->stream == odk->win->stream) {
                /* If we have alternative streams waiting we play one of them. */
                if (playlist_length (odk->current_alternatives) > 0) {
                    /* FIXME: We currently clear the alternatives when we stop
                     *        the current stream in odk_play. This means we
                     *        only try the first alternative. */
                    playlist_t *alts = odk->current_alternatives;
                    playitem_t *alt = playlist_first (alts);
                    while (alt) {
                        oxine_event_t ev;
                        ev.type = OXINE_EVENT_WAITING_FOR_ALTERNATIVE;
                        odk_oxine_event_send (odk, &ev);
                        if (odk_play (odk, alt->mrl, NULL, odk->current_mode))
                            break;
                        alt = playlist_next (alts, alt);
                    }

                    /* If we were not able to play one of the alternative
                     * streams, we send a stop event. */
                    if (!alt) {
                        oxine_event_t ev;
                        ev.type = OXINE_EVENT_PLAYBACK_ERROR;
                        odk_oxine_event_send (odk, &ev);
                    }
                }

                /* Else we inform the frontend about the ending of the stream. */
                else {
                    oxine_event_t ev;
                    ev.type = xine_event->type;
                    odk_oxine_event_send (odk, &ev);
                }
            }
            break;
        case OXINE_EVENT_CHANNELS_CHANGED:
            /* We use this event to find out if the meta info has changed
             * without a new track starting (e.g. web-radio). */
            if (xine_event->stream == odk->win->stream) {
                char *current_title =
                    odk_get_meta_info (odk, META_INFO_TITLE);
                if (current_title && !odk->current_title) {
                    odk->current_title = current_title;
                    oxine_event_t ev;
                    ev.type = OXINE_EVENT_PLAYBACK_STARTED;
                    odk_oxine_event_send (odk, &ev);
                }

                else if (!current_title && odk->current_title) {
                    ho_free (odk->current_title);
                    odk->current_title = NULL;
                    oxine_event_t ev;
                    ev.type = OXINE_EVENT_PLAYBACK_STARTED;
                    odk_oxine_event_send (odk, &ev);
                }

                else if (current_title && odk->current_title
                         && (strcmp (current_title,
                                     odk->current_title) != 0)) {
                    ho_free (odk->current_title);
                    odk->current_title = current_title;
                    oxine_event_t ev;
                    ev.type = OXINE_EVENT_PLAYBACK_STARTED;
                    odk_oxine_event_send (odk, &ev);
                }

                else if (current_title) {
                    ho_free (current_title);
                }
            }
            break;
        case XINE_EVENT_PROGRESS:
            if (xine_event->stream == odk->win->stream) {
                xine_progress_data_t *pevent =
                    (xine_progress_data_t *) xine_event->data;
                debug ("%s [%d%%]", pevent->description, pevent->percent);
            }
            break;
        case XINE_EVENT_MRL_REFERENCE:
            /* If we receive an alternative stream, we add it to a list of
             * alternatives. When receiving the first alternative stream, we
             * send an event telling the frontend, that we are waiting for the
             * stream. */
            if (xine_event->stream == odk->win->stream) {
                xine_mrl_reference_data_t *ref =
                    (xine_mrl_reference_data_t *) xine_event->data;

                debug ("Got alternative MRL [%d]: '%s'",
                       ref->alternative, ref->mrl);

                playlist_insert (odk->current_alternatives,
                                 ref->alternative, ref->mrl, ref->mrl);
            }
            break;
        default:
            break;
    }
}


/* 
 * ***************************************************************************
 * Name:            odk_listen_to
 * Access:          public
 *
 * Description:     Tells odk to listen to an input plugin.
 * ***************************************************************************
 */
void
odk_listen_to (odk_t * odk, const char *id)
{
    if (!strcmp (id, "lirc")) {
        if (!odk->lirc)
            odk->lirc = start_lirc (odk, odk_event_handler);
    }
    if (!strcmp (id, "js")) {
        if (!odk->js)
            odk->js = start_js (odk, odk_event_handler);
    }
}


/* 
 * ***************************************************************************
 * Name:            odk_event_init
 * Access:          private
 *
 * Description:     Initialize ODK event handling.
 * ***************************************************************************
 */
void
odk_event_init (odk_t * odk)
{
#ifdef DEBUG
    assert (odk);
    assert (odk->win);
    assert (odk->win->stream);
    assert (odk->background_stream);
    assert (odk->animation_stream);
#endif

    odk->image_finish_job = 0;
    odk->forward_events_to_xine = 0;
    odk->win->set_event_handler (odk->win, odk_event_handler, odk);

    xine_event_queue_t *q0 = xine_event_new_queue (odk->win->stream);
    xine_event_queue_t *q1 = xine_event_new_queue (odk->background_stream);
    xine_event_queue_t *q2 = xine_event_new_queue (odk->animation_stream);

    xine_event_create_listener_thread (q0, xine_event_cb, odk);
    xine_event_create_listener_thread (q1, xine_event_cb, odk);
    xine_event_create_listener_thread (q2, xine_event_cb, odk);
}


/* 
 * ***************************************************************************
 * Name:            odk_event_free
 * Access:          private
 *
 * Description:     Free ODK event handling.
 * ***************************************************************************
 */
void
odk_event_free (odk_t * odk)
{
    if (odk->lirc)
        stop_lirc (odk->lirc);
    if (odk->js)
        stop_js (odk->js);
}
